#include "onig4j_OnigRegex.h"
#include "onig4j_OnigRegion.h"
#include "onig4j_OnigCaptureTreeNode.h"
#include "onig4j_OnigSyntaxType.h"
#include "oniguruma.h"
#include <malloc.h>

#ifdef __cplusplus
extern "C" {
#endif

/* oniguruma UTF-16 encoding */
static OnigEncoding ONIG_ENCODING_UTF16 = 0;

#define IS_UNICODE_LITTLE_UNMARKED() (ONIG_ENCODING_UTF16 == ONIG_ENCODING_UTF16_LE)

/* number of bytes to number of UTF-16 characters */
#define JAVA_LENGTH(n)   ((jint)(n > 0 ? ((n) / sizeof(jchar)) : n))

/* number of UTF-16 characters to number of bytes */
#define BYTE_LENGTH(n)   (((int)(n)) * sizeof(jchar))

/* number of UTF-16 characters to number of bytes */
#define onigenc_str_bytelen(env, str)   BYTE_LENGTH((*env)->GetStringLength(env, str))

/* index range check */
#define IS_VALIDITY_RANGE(n, end)    (((int)(n)) >= 0 && ((int)(n)) < (end))

/* convert pointer to jlong */
static jlong _jlong(void *p)
{
    jvalue val;
    val.l = p;
    return val.j;
}

/* convert jlong to pointer */
static void * _voidp(jlong j)
{
    jvalue val;
    val.j = j;
    return val.l;
}

/* throw new exception */
static void ThrowException(JNIEnv *env, const char *className, const char* message)
{
    jthrowable ex = (*env)->FindClass(env, className);
    if (ex) {
        (*env)->ThrowNew(env, ex, message);
        (*env)->DeleteLocalRef(env, ex);
    }
}

#define ThrowOutOfMemoryError(env)                  ThrowException(env, "java/lang/OutOfMemoryError", "Not enough memory")
#define ThrowIllegalStateException(env)             ThrowException(env, "java/lang/IllegalStateException", "handle is invalid")
#define ThrowArrayIndexOutOfBoundsException(env)    ThrowException(env, "java/lang/ArrayIndexOutOfBoundsException", "the index is out of range")

/* global JavaVM object */
static JavaVM *jvm = 0;

/* field of onig4j.OnigHandle.handle */
static jfieldID fid_OnigHandle_handle = 0;

/* class of onig4j.OnigRegex */
static jclass clsOnigRegex = 0;

/* class of onig4j.OnigSyntaxType */
static jclass clsOnigSyntaxType = 0;

/* method of onig4j.OnigSyntaxType constructor */
static jmethodID mid_OnigSyntaxType_init = 0;

/* class of onig4j.OnigCaptureTreeNode */
static jclass clsOnigCaptureTreeNode = 0;

/* new onig4j.OnigCaptureTreeNode object */
static jobject NewOnigCaptureTreeNode(JNIEnv *env, OnigCaptureTreeNode *node)
{
    if (node)
    {
        static jmethodID mid_init = 0;
        if (!mid_init)
        {
            clsOnigCaptureTreeNode = (*env)->FindClass(env, "onig4j/OnigCaptureTreeNode");
            if (clsOnigCaptureTreeNode)
            {
                // get MethodID of constractor
                mid_init = (*env)->GetMethodID(env, clsOnigCaptureTreeNode, "<init>", "(JIIIII)V");
                if (mid_init)
                {
                    clsOnigCaptureTreeNode = (*env)->NewGlobalRef(env, clsOnigCaptureTreeNode);
                }
                else
                {
                    (*env)->DeleteLocalRef(env, clsOnigCaptureTreeNode);
                    clsOnigCaptureTreeNode = 0;
                    return NULL;
                }
            }
            else
            {
                return NULL;
            }
        }

        // new onig4j.OnigCaptureTreeNode object
        return (*env)->NewObject(env, clsOnigCaptureTreeNode, mid_init, _jlong(node), node->group,
                                    JAVA_LENGTH(node->beg), JAVA_LENGTH(node->end),
                                    node->allocated, node->num_childs);
    }
    else
    {
        ThrowIllegalStateException(env);
    }
    return NULL;
}

static void SetOnigCaptureTreeNode(JNIEnv *env, jobject jRegion, regex_t *reg, OnigRegion *region)
{
    static jfieldID fid_node = 0;
    jobject node = NULL;
    
    if (!fid_node)
    {
        jclass clsOnigRegion = (*env)->FindClass(env, "onig4j/OnigRegion");
        if (clsOnigRegion)
        {
            fid_node = (*env)->GetFieldID(env, clsOnigRegion, "node", "Lonig4j/OnigCaptureTreeNode;");
            (*env)->DeleteLocalRef(env, clsOnigRegion);
            if (!clsOnigRegion)
            {
                return;
            }
        }
    }

    if (onig_get_syntax_op2(onig_get_syntax(reg)) & ONIG_SYN_OP2_ATMARK_CAPTURE_HISTORY)
    {
        node = NewOnigCaptureTreeNode(env, onig_get_capture_tree(region));
    }
    if (!(*env)->ExceptionCheck(env))
    {
        (*env)->SetObjectField(env, jRegion, fid_node, node);
    }
}

/* decides encoding from the byte order */
JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *vm, void *reserved)
{
    JNIEnv *env;
    // require JNI ver 1.2 or later
    if ((*vm)->GetEnv(vm, (void **)&env, JNI_VERSION_1_2) == JNI_OK)
    {
        /*
         * Decides encoding from the byte order.
         * @see http://www.kk.iij4u.or.jp/~kondo/wave/swab.html
         */
        int const i = 1;
        if (*((char *) &i))
        {
            // little endian
            ONIG_ENCODING_UTF16 = ONIG_ENCODING_UTF16_LE;
            //onigenc_set_default_encoding(ONIG_ENCODING_UTF16_LE);
            return JNI_VERSION_1_2;
        }
        else if(*((char *) &i + (sizeof(int) - 1)))
        {
            // big endian
            ONIG_ENCODING_UTF16 = ONIG_ENCODING_UTF16_BE;
            //onigenc_set_default_encoding(ONIG_ENCODING_UTF16_BE);
            return JNI_VERSION_1_2;
        }
    }
    return JNI_ERR;
}

/* delete references and invoke onig_end() function */
JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *vm, void *reserved)
{
    JNIEnv *env;
    if ((*vm)->AttachCurrentThread(vm, (void **)&env, 0) == JNI_OK) {
        jclass classes[]
                = {
                        clsOnigRegex,
                        clsOnigSyntaxType,
                        clsOnigCaptureTreeNode,
                  };
        int const end = sizeof(classes) / sizeof(classes[0]);
        int i;
        for (i = 0; i < end; ++i) {
            if (classes[i]) {
                (*env)->DeleteGlobalRef(env, classes[i]);
            }
        }
    }

    onig_end();
}

static void * GetHandle(JNIEnv *env, jobject obj)
{
    if (!fid_OnigHandle_handle)
    {
        jclass clsOnigHandle = (*env)->FindClass(env, "onig4j/OnigHandle");
        if (clsOnigHandle)
        {
            fid_OnigHandle_handle = (*env)->GetFieldID(env, clsOnigHandle, "handle", "J");
            (*env)->DeleteLocalRef(env, clsOnigHandle);
            if (!fid_OnigHandle_handle)
            {
                return NULL;
            }
        }
    }

    return _voidp((*env)->GetLongField(env, obj, fid_OnigHandle_handle));
}

static void * GetActiveHandle(JNIEnv *env, jobject obj)
{
    void *p = GetHandle(env, obj);
    if (!p && !(*env)->ExceptionCheck(env))
    {
        ThrowIllegalStateException(env);
    }
    return p;
}

#define GetOnigSyntaxType(env, obj) (GetActiveHandle(env, obj))
#define GetOnigRegex(env, obj)      (GetActiveHandle(env, obj))
#define GetOnigRegion(env, obj)     (GetActiveHandle(env, obj))

static void SetHandle(JNIEnv *env, jobject obj, void *p)
{
    (*env)->SetLongField(env, obj, fid_OnigHandle_handle, _jlong(p));
}

static void CloseHandle(JNIEnv *env, jobject obj)
{
    (*env)->SetLongField(env, obj, fid_OnigHandle_handle, 0);
}

#if defined(_WIN32_WCE) || defined(WIN32)
static void onig_null_warn(const char* s) { };
#endif

static void InvokeFireWarning(const char *msg, int verbose)
{
    if (jvm)
    {
        JNIEnv *env;
        if ((*jvm)->AttachCurrentThread(jvm, (void **)&env, 0) == JNI_OK)
        {
            static jmethodID mid_fireWarning = 0;
            static jmethodID mid_fireWarningVerbose = 0;
            jstring jmsg = NULL;
            jmethodID *method = (verbose ? &mid_fireWarningVerbose : &mid_fireWarning);
            if (!*method)
            {
                clsOnigRegex = (*env)->FindClass(env, "onig4j/OnigRegex");
                if (clsOnigRegex)
                {
                    *method = (*env)->GetStaticMethodID(
                                    env, clsOnigRegex,
                                    (verbose ? "fireWarningVerbose" : "fireWarning"),
                                    "(Ljava/lang/String;)V"
                                );

                    if (*method)
                    {
                        clsOnigRegex = (*env)->NewGlobalRef(env, clsOnigRegex);
                    }
                    else
                    {
                        (*env)->DeleteLocalRef(env, clsOnigRegex);
                        clsOnigRegex = 0;
                        (*jvm)->DetachCurrentThread(jvm);
                        return;
                    }
                }
                else
                {
                    (*jvm)->DetachCurrentThread(jvm);
                    return;
                }
            }

            if (msg)
            {
                jmsg = (*env)->NewStringUTF(env, msg);
            }
            if (!(*env)->ExceptionCheck(env))
            {
                (*env)->CallStaticVoidMethod(env, clsOnigRegex, *method, jmsg);
            }
            (*jvm)->DetachCurrentThread(jvm);
        }
    }
}

static void xWarnFunc(const char *msg)
{
    InvokeFireWarning(msg, 0);
}

static void xVerbWarnFunc(const char* msg)
{
    InvokeFireWarning(msg, 1);
}

/* structure for onig4j.OnigRegex.Callbakc object */
typedef struct _REGEX_CALLBACK_OBJECT
{
    JNIEnv *env;
    jobject callback;
    jobject regex;
} REGEX_CALLBACK_OBJECT, *PREGEX_CALLBACK_OBJECT;

/* Call from onig_foreach_name() function */
static int xForeachName(const UChar* name, const UChar* name_end, int cnt, int *list, regex_t* reg, void *p)
{
    static jmethodID mid_call = 0;
    jintArray groups;
    JNIEnv *env = ((PREGEX_CALLBACK_OBJECT)p)->env;

    if (!mid_call)
    {
        jclass clazz = (*env)->FindClass(env, "onig4j/OnigRegex$Callback");
        if (clazz)
        {
            // get MethodID of call() method
            mid_call = (*env)->GetMethodID(env, clazz, "call", "(Ljava/lang/String;[ILonig4j/OnigRegex;)I");
            (*env)->DeleteLocalRef(env, clazz);
            if (!mid_call)
            {
                return -1;
            }
        }
    }

    groups = (*env)->NewIntArray(env, cnt);
    if (groups)
    {
        if (cnt > 0)
        {
            (*env)->SetIntArrayRegion(env, groups, 0, cnt, (jint *)list);
        }
        return (int)(*env)->CallIntMethod(
                                    env,
                                    ((PREGEX_CALLBACK_OBJECT)p)->callback,
                                    mid_call,
                                    (*env)->NewString(env, (const jchar *)name,
                                    onigenc_strlen(ONIG_ENCODING_UTF16, name, name_end)),
                                    groups,
                                    ((PREGEX_CALLBACK_OBJECT)p)->regex
                                );
    }
    else
    {
        return -1;
    }
}

static int LoadOnigSyntaxTypeClass(JNIEnv *env)
{
    clsOnigSyntaxType = (*env)->FindClass(env, "onig4j/OnigSyntaxType");
    if (clsOnigSyntaxType)
    {
        mid_OnigSyntaxType_init = (*env)->GetMethodID(env, clsOnigSyntaxType, "<init>", "(J)V");
        if (mid_OnigSyntaxType_init)
        {
            clsOnigSyntaxType = (*env)->NewGlobalRef(env, clsOnigSyntaxType);
            return 1;
        }
        else
        {
            (*env)->DeleteLocalRef(env, clsOnigSyntaxType);
            clsOnigSyntaxType = 0;
        }
    }
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
/// onig4j.OnigRegex //////////////////////////////////////////////////////////

/*
 * Class:     onig4j_OnigRegex
 * Method:    useUnicodeLittleUnmarked
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_onig4j_OnigRegex_useUnicodeLittleUnmarked
  (JNIEnv *env, jclass clazz)
{
    return (IS_UNICODE_LITTLE_UNMARKED() ? JNI_TRUE : JNI_FALSE);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_error_code_to_str
 * Signature: (ILjava/lang/String;)Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_onig4j_OnigRegex_onig_1error_1code_1to_1str
  (JNIEnv *env, jclass clazz, jint ecode, jstring jParam)
{
    jstring msg = NULL;
    OnigErrorInfo einfo = { ONIG_ENCODING_UTF16 };
    UChar err_buff[ONIG_MAX_ERROR_MESSAGE_LEN] = {0};

    if (jParam)
    {
        einfo.par = (OnigUChar *)(*env)->GetStringChars(env, jParam, 0);
        if (!einfo.par)
        {
            return 0;
        }
        einfo.par_end = einfo.par + onigenc_str_bytelen(env, jParam);
    }

    if (onig_error_code_to_str(err_buff, (int)ecode, &einfo) >= 0)
    {
        msg = (*env)->NewStringUTF(env, (const char *)err_buff);
    }

    if (einfo.par)
    {
        (*env)->ReleaseStringChars(env, jParam, (const jchar *)einfo.par);
    }

    return msg;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_new
 * Signature: (Ljava/lang/String;ILonig4j/OnigSyntaxType;[Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1new
  (JNIEnv *env, jobject this, jstring jPattern, jint options, jobject syntax, jobjectArray errmsg)
{
    const OnigUChar *pattern;
    OnigErrorInfo einfo;
    int ret;
    OnigSyntaxType *syn;
    regex_t *reg = GetHandle(env, this);
    
    if ((*env)->ExceptionCheck(env))
    {
        return 0;
    }

    syn = GetOnigSyntaxType(env, syntax);
    if (!syn)
    {
        return 0;
    }

    pattern = (const OnigUChar *)(*env)->GetStringChars(env, jPattern, 0);
    if (!pattern) {
        return 0;
    }

    ret = onig_new(
                &reg, pattern, pattern + onigenc_str_bytelen(env, jPattern),
                (OnigOptionType)options, ONIG_ENCODING_UTF16, syn,
                (errmsg ? &einfo : NULL)
            );

    if (pattern) {
        (*env)->ReleaseStringChars(env, jPattern, (const jchar *)pattern);
    }

    if (ret == ONIG_NORMAL)
    {
        SetHandle(env, this, reg);
    }
    else if (errmsg && (*env)->GetArrayLength(env, errmsg) != 0)
    {
        jstring msg;
        UChar err_buff[ONIG_MAX_ERROR_MESSAGE_LEN] = {0};
        onig_error_code_to_str(err_buff, ret, &einfo);
        msg = (*env)->NewStringUTF(env, (const char *)err_buff);
        if (msg) {
            (*env)->SetObjectArrayElement(env, errmsg, 0, msg);
        }
    }

    return (jint)ret;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_new_deluxe
 * Signature: (Ljava/lang/String;ILonig4j/OnigSyntaxType;I[Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1new_1deluxe
 (JNIEnv *env, jobject this, jstring jPattern, jint options, jobject syntax, jint flag, jobjectArray errmsg)
{
    const OnigUChar *pattern;
    OnigErrorInfo einfo;
    int ret;
    OnigCompileInfo ci
            = {
                5,
                ONIG_ENCODING_UTF16,
                ONIG_ENCODING_UTF16,
                GetOnigSyntaxType(env, syntax),
                (OnigOptionType)options,
                (OnigCaseFoldType)flag,
            };
    regex_t *reg = GetHandle(env, this);
    
    if ((*env)->ExceptionCheck(env))
    {
        return 0;
    }

    if (!ci.syntax)
    {
        return 0;
    }

    pattern = (const OnigUChar *)(*env)->GetStringChars(env, jPattern, 0);
    if (!pattern) {
        return 0;
    }

    ret = onig_new_deluxe(&reg, pattern, pattern + onigenc_str_bytelen(env, jPattern),
            &ci, (errmsg ? &einfo : NULL));

    if (pattern) {
        (*env)->ReleaseStringChars(env, jPattern, (const jchar *)pattern);
    }

    if (ret == ONIG_NORMAL)
    {
        SetHandle(env, this, reg);
    }
    else if (errmsg && (*env)->GetArrayLength(env, errmsg) != 0)
    {
        jstring msg;
        UChar err_buff[ONIG_MAX_ERROR_MESSAGE_LEN] = {0};
        onig_error_code_to_str(err_buff, ret, &einfo);
        msg = (*env)->NewStringUTF(env, (const char *)err_buff);
        if (msg) {
            (*env)->SetObjectArrayElement(env, errmsg, 0, msg);
        }
    }

    return (jint)ret;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_free
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigRegex_onig_1free
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    if (p)
    {
        onig_free(p);
        CloseHandle(env, this);
    }
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_noname_group_capture_is_active
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_onig4j_OnigRegex_onig_1noname_1group_1capture_1is_1active
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    return (p && onig_noname_group_capture_is_active(p) ? JNI_TRUE : JNI_FALSE);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_search
 * Signature: (Ljava/lang/String;IIILonig4j/OnigRegion;I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1search
  (JNIEnv *env, jobject this, jstring jStr, jint jEnd, jint jStart, jint jRange, jobject jRegion, jint option)
{
    const OnigUChar *str;
    const OnigUChar *start, *range, *end;
    OnigRegion *region = NULL;
    int ret;
    regex_t *reg = GetOnigRegex(env, this);
    if (!reg)
    {
        return 0;
    }

    if (jRegion)
    {
        region = GetOnigRegion(env, jRegion);
        if (!region)
        {
            return 0;
        }
    }

    str = (const OnigUChar *)(*env)->GetStringChars(env, jStr, 0);
    if (!str) {
        return 0;
    }
    end = str + BYTE_LENGTH(jEnd);
    start = str + BYTE_LENGTH(jStart);
    range = str + BYTE_LENGTH(jRange);

    ret = onig_search(reg, str, end, start, range, region, (OnigOptionType)option);

    if (str) {
        (*env)->ReleaseStringChars(env, jStr, (const jchar *)str);
    }

    if (jRegion)
    {
        SetOnigCaptureTreeNode(env, jRegion, reg, region);
    }

    return JAVA_LENGTH(ret);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_match
 * Signature: (Ljava/lang/String;IILonig4j/OnigRegion;I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1match
  (JNIEnv *env, jobject this, jstring jStr, jint jEnd, jint jAt, jobject jRegion, jint option)
{
    const OnigUChar *str;
    const OnigUChar *end, *at;
    OnigRegion *region = NULL;
    int ret;
    regex_t *reg = GetOnigRegex(env, this);

    if (!reg)
    {
        return 0;
    }

    if (jRegion)
    {
        region = GetOnigRegion(env, jRegion);
        if (!region)
        {
            return 0;
        }
    }

    str = (const OnigUChar *)(*env)->GetStringChars(env, jStr, 0);
    if (!str) {
        return 0;
    }
    end = str + BYTE_LENGTH(jEnd);
    at = str + BYTE_LENGTH(jAt);

    ret = onig_match(reg, str, end, at, region, (OnigOptionType)option);

    if (str) {
        (*env)->ReleaseStringChars(env, jStr, (const jchar *)str);
    }

    if (jRegion)
    {
        SetOnigCaptureTreeNode(env, jRegion, reg, region);
    }

    return JAVA_LENGTH(ret);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_split
 * Signature: (Ljava/lang/String;IILjava/util/Collection;)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1split
  (JNIEnv *env, jobject this, jstring jStr, jint limit, jint option, jobject list)
{
    regex_t *reg = GetOnigRegex(env, this);
    if (reg && jStr && list)
    {
        static jmethodID mid_add = 0;
        const OnigUChar *str;

        if (!mid_add)
        {
            jclass clazz = (*env)->FindClass(env, "java/util/Collection");
            if (clazz)
            {
                mid_add = (*env)->GetMethodID(env, clazz, "add", "(Ljava/lang/Object;)Z");
                (*env)->DeleteLocalRef(env, clazz);
                if (!mid_add)
                {
                    return 0;
                }
            }
            else
            {
                return 0;
            }
        }

        str = (const OnigUChar *)(*env)->GetStringChars(env, jStr, 0);
        if (str) {
            OnigRegion *region = onig_region_new();
            if (region)
            {
                const OnigUChar *start = str;
                const OnigUChar *end = str + onigenc_str_bytelen(env, jStr);
                const OnigUChar *range = end;
                int offset = 0;
                int ret;
                while ((ret = onig_search(reg, str, end, start + offset, range, region, (OnigOptionType)option)) >= 0)
                {
                    jstring s = (*env)->NewString(env, (const jchar *)start, onigenc_strlen(ONIG_ENCODING_UTF16, start, str + ret));
                    if (s)
                    {
                        int const regionEnd = region->end[0];
                        (*env)->CallBooleanMethod(env, list, mid_add, s);
                        start = str + regionEnd;
                        --limit;
                        if (limit == 0 || (*env)->ExceptionCheck(env))
                        {
                            break;
                        }
                        if (ret == regionEnd) {  // region->beg[0] == region->end[0]
                            // sLȂǂ̋EK\╝[̐ǂ݁Eǂ݂Ƀ}b`ꍇ
                            // [vɊׂ̂h߂ɌJnʒu1炷
                            // TODO TQ[gyAɑΉĂ邩vmFI
                            // MEMO onigenc_get_right_adjust_char_head()֐gΗǂH
                            offset = sizeof(jchar);
                            if (start + offset >= end) {
                                break;
                            }
                        }
                    }
                    else
                    {
                        break;
                    }
                }
                if (!(*env)->ExceptionCheck(env) && /*(limit == 0 || ret < 0) &&*/ (start <= end))
                {
                    jstring s = (*env)->NewString(env, (const jchar *)start, onigenc_strlen(ONIG_ENCODING_UTF16, start, end));
                    if (s)
                    {
                        (*env)->CallBooleanMethod(env, list, mid_add, s);
                    }
                }
                onig_region_free(region, 1);
                return ret;
            }
            else
            {
                ThrowOutOfMemoryError(env);
            }
        }
    }
    return 0;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_name_to_backref_number
 * Signature: (Ljava/lang/String;Lonig4j/OnigRegion;)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1name_1to_1backref_1number
  (JNIEnv *env, jobject this, jstring jName, jobject jRegion)
{
    int ret = -1;
    regex_t *reg = GetOnigRegex(env, this);
    if (reg && jName)
    {
        OnigRegion *region = NULL;
        const OnigUChar *name;
        if (jRegion)
        {
            region = GetOnigRegion(env, jRegion);
            if (!region)
            {
                return 0;
            }
        }

        name = (const OnigUChar *)(*env)->GetStringChars(env, jName, 0);
        if (name) {
            ret = onig_name_to_backref_number(reg, name, name + onigenc_str_bytelen(env, jName), region);

            (*env)->ReleaseStringChars(env, jName, (const jchar *)name);
        }
    }

    return ret;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_name_to_group_numbers
 * Signature: (Ljava/lang/String;)[I
 */
JNIEXPORT jintArray JNICALL Java_onig4j_OnigRegex_onig_1name_1to_1group_1numbers
  (JNIEnv *env, jobject this, jstring jName)
{
    jintArray ret = NULL;
    regex_t *reg = GetOnigRegex(env, this);
    if (reg && jName)
    {
        const OnigUChar *name = (const OnigUChar *)(*env)->GetStringChars(env, jName, 0);
        if (name) {
            int *list;
            int const cnt = onig_name_to_group_numbers(reg, name, name + onigenc_str_bytelen(env, jName), &list);

            (*env)->ReleaseStringChars(env, jName, (const jchar *)name);

            ret = (*env)->NewIntArray(env, (cnt > 0 ? cnt : 0));
            if (ret && cnt > 0)
            {
                (*env)->SetIntArrayRegion(env, ret, 0, cnt, (jint *)list);
            }
        }
    }

    return ret;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_number_of_names
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1number_1of_1names
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    return (p ? (jint)onig_number_of_names(p) : -1);
}
/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_get_encoding_name
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_onig4j_OnigRegex_onig_1get_1encoding_1name
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    if (p)
    {
        OnigEncoding enc = onig_get_encoding(p);
        if (enc && enc->name)
        {
            return (*env)->NewStringUTF(env, enc->name);
        }
    }
    return NULL;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_get_options
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1get_1options
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    return (p ? (jint)onig_get_options(p) : -1);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_get_case_fold_flag
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1get_1case_1fold_1flag
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    return (p ? (jint)onig_get_case_fold_flag(p) : -1);
}

#define BUILTIN_SYNTAX_NAME(syntax) \
            (syntax == ONIG_SYNTAX_ASIS ? "ONIG_SYNTAX_ASIS" \
            : (syntax == ONIG_SYNTAX_POSIX_BASIC ? "ONIG_SYNTAX_POSIX_BASIC" \
            : (syntax == ONIG_SYNTAX_POSIX_EXTENDED ? "ONIG_SYNTAX_POSIX_EXTENDED" \
            : (syntax == ONIG_SYNTAX_EMACS ? "ONIG_SYNTAX_EMACS" \
            : (syntax == ONIG_SYNTAX_GREP ? "ONIG_SYNTAX_GREP" \
            : (syntax == ONIG_SYNTAX_GNU_REGEX ? "ONIG_SYNTAX_GNU_REGEX" \
            : (syntax == ONIG_SYNTAX_JAVA ? "ONIG_SYNTAX_JAVA" \
            : (syntax == ONIG_SYNTAX_PERL ? "ONIG_SYNTAX_PERL" \
            : (syntax == ONIG_SYNTAX_PERL_NG ? "ONIG_SYNTAX_PERL_NG" \
            : (syntax == ONIG_SYNTAX_RUBY ? "ONIG_SYNTAX_RUBY" : NULL))))))))))

///*
// * Class:     onig4j_OnigRegex
// * Method:    onig_get_syntax
// * Signature: ()Lonig4j/OnigSyntaxType;
// */
//JNIEXPORT jobject JNICALL Java_onig4j_OnigRegex_onig_1get_1syntax
//  (JNIEnv *env, jobject this)
//{
//    regex_t *reg = GetOnigRegex(env, this);
//    if (reg)
//    {
//        if (clsOnigSyntaxType || LoadOnigSyntaxTypeClass(env))
//        {
//            jobject syntax = NULL;
//            OnigSyntaxType *syn = onig_get_syntax(reg);
//            const char *name = BUILTIN_SYNTAX_NAME(syn);
//            if (name)
//            {
//                jfieldID field = (*env)->GetStaticFieldID(env, clsOnigSyntaxType, name, "Lonig4j/OnigSyntaxType;");
//                if (field)
//                {
//                    syntax = (*env)->GetStaticObjectField(env, clsOnigSyntaxType, field);
//                }
//            }
//            else
//            {
//                syntax = (*env)->NewObject(env, clsOnigSyntaxType, mid_OnigSyntaxType_init, _jlong(syn));
//            }
//
//            return syntax;
//        }
//    }
//    return NULL;
//}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_number_of_captures
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1number_1of_1captures
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    return (p ? (jint)onig_number_of_captures(p) : -1);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_number_of_capture_histories
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1number_1of_1capture_1histories
  (JNIEnv *env, jobject this)
{
    regex_t *p = GetOnigRegex(env, this);
    return (p ? (jint)onig_number_of_capture_histories(p) : -1);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_region_new
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_onig_1region_1new
  (JNIEnv *env, jclass clazz)
{
    OnigRegion *p = onig_region_new();
    if (p)
    {
        return _jlong(p);
    }
    else
    {
        ThrowOutOfMemoryError(env);
        return 0;
    }
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_get_default_case_fold_flag
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1get_1default_1case_1fold_1flag
  (JNIEnv *env, jclass clazz)
{
    return (jint)onig_get_default_case_fold_flag();
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_set_default_case_fold_flag
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1set_1default_1case_1fold_1flag
  (JNIEnv *env, jclass clazz, jint flag)
{
    return (jint)onig_set_default_case_fold_flag(flag);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_get_match_stack_limit_size
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1get_1match_1stack_1limit_1size
  (JNIEnv *env, jclass clazz)
{
    return (jint)onig_get_match_stack_limit_size();
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_set_match_stack_limit_size
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1set_1match_1stack_1limit_1size
  (JNIEnv *env, jclass clazz, jint size)
{
    return (jint)onig_set_match_stack_limit_size((unsigned int)size);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_version
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_onig4j_OnigRegex_onig_1version
  (JNIEnv *env, jclass clazz)
{
    const char *version = onig_version();
    if (version)
    {
        return (*env)->NewStringUTF(env, version);
    }

    return NULL;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_copyright
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_onig4j_OnigRegex_onig_1copyright
  (JNIEnv *env, jclass clazz)
{
    const char *copyright = onig_copyright();
    if (copyright)
    {
        return (*env)->NewStringUTF(env, copyright);
    }

    return NULL;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_set_warn_func
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_onig4j_OnigRegex_onig_1set_1warn_1func
  (JNIEnv *env, jclass clazz)
{
    if (jvm || ((*env)->GetJavaVM(env, &jvm) == JNI_OK))
    {
        onig_set_warn_func(&xWarnFunc);
        return JNI_TRUE;
    }
    return JNI_FALSE;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_clear_warn_func
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigRegex_onig_1clear_1warn_1func
  (JNIEnv *env, jclass clazz)
{
    onig_set_warn_func(ONIG_NULL_WARN);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_set_verb_warn_func
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_onig4j_OnigRegex_onig_1set_1verb_1warn_1func
  (JNIEnv *env, jclass clazz)
{
    if (jvm || ((*env)->GetJavaVM(env, &jvm) == JNI_OK))
    {
        onig_set_verb_warn_func(&xVerbWarnFunc);
        return JNI_TRUE;
    }
    return JNI_FALSE;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_clear_verb_warn_func
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigRegex_onig_1clear_1verb_1warn_1func
  (JNIEnv *env, jclass clazz)
{
    onig_set_verb_warn_func(ONIG_NULL_WARN);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    onig_foreach_name
 * Signature: (Lonig4j/OnigRegex$Callback;)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegex_onig_1foreach_1name
  (JNIEnv *env, jobject this, jobject callback)
{
    regex_t *p = GetOnigRegex(env, this);
    if (p)
    {
        REGEX_CALLBACK_OBJECT arg = { env, callback, this };
        return (jint)onig_foreach_name(p, &xForeachName, &arg);
    }
    return -1;
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_ASIS
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1ASIS
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_ASIS);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_POSIX_BASIC
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1POSIX_1BASIC
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_POSIX_BASIC);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_POSIX_EXTENDED
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1POSIX_1EXTENDED
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_POSIX_EXTENDED);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_EMACS
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1EMACS
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_EMACS);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_GREP
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1GREP
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_GREP);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_GNU_REGEX
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1GNU_1REGEX
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_GNU_REGEX);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_JAVA
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1JAVA
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_JAVA);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_PERL
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1PERL
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_PERL);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_PERL_NG
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1PERL_1NG
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_PERL_NG);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_RUBY
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1RUBY
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_RUBY);
}

/*
 * Class:     onig4j_OnigRegex
 * Method:    ONIG_SYNTAX_DEFAULT
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigRegex_ONIG_1SYNTAX_1DEFAULT
  (JNIEnv *env, jclass clazz)
{
    return _jlong(ONIG_SYNTAX_DEFAULT);
}

///////////////////////////////////////////////////////////////////////////////
/// onig4j.OnigRegion /////////////////////////////////////////////////////////

/* structure for onig4j.OnigRegion.Callbakc object */
typedef struct _REGION_CALLBACK_OBJECT
{
    JNIEnv *env;
    jobject callback;
} REGION_CALLBACK_OBJECT, *PREGION_CALLBACK_OBJECT;

/* Call from onig_capture_tree_traverse() function */
static int xTraverseCaptureTree(int group, int begin, int end, int level, int at, void *p)
{
    static jmethodID mid_call = 0;
    JNIEnv *env = ((PREGION_CALLBACK_OBJECT)p)->env;

    if (!mid_call)
    {
        jclass clazz = (*env)->FindClass(env, "onig4j/OnigRegion$Callback");
        if (clazz)
        {
            // get MethodID of call() method
            mid_call = (*env)->GetMethodID(env, clazz, "call", "(IIIII)I");
            (*env)->DeleteLocalRef(env, clazz);
            if (!mid_call)
            {
                return -1;
            }
        }
    }

    return (int)(*env)->CallIntMethod(
                                env,
                                ((PREGION_CALLBACK_OBJECT)p)->callback,
                                mid_call,
                                (jint)group,
                                JAVA_LENGTH(begin),
                                JAVA_LENGTH(end),
                                (jint)level,
                                (jint)at
                            );
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    allocated
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_allocated
  (JNIEnv *env, jobject this)
{
    OnigRegion *p = GetOnigRegion(env, this);
    return (p ? (jint)(p->allocated) : -1);
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    count
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_count
  (JNIEnv *env, jobject this)
{
    OnigRegion *p = GetOnigRegion(env, this);
    return (p ? (jint)p->num_regs : -1);
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    begin
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_begin
  (JNIEnv *env, jobject this, jint i)
{
    OnigRegion *p = GetOnigRegion(env, this);
    if (p)
    {
        if  (IS_VALIDITY_RANGE(i, p->num_regs))
        {
            return JAVA_LENGTH(p->beg[(int)i]);
        }
        else
        {
            ThrowArrayIndexOutOfBoundsException(env);
        }
    }
    return 0;
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    end
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_end
  (JNIEnv *env, jobject this, jint i)
{
    OnigRegion *p = GetOnigRegion(env, this);
    if (p)
    {
        if (IS_VALIDITY_RANGE(i, p->num_regs))
        {
            return JAVA_LENGTH(p->end[(int)i]);
        }
        else
        {
            ThrowArrayIndexOutOfBoundsException(env);
        }
    }
    return 0;
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_region_free
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigRegion_onig_1region_1free
  (JNIEnv *env, jobject this, jint free_self)
{
    OnigRegion *p = GetOnigRegion(env, this);
    if (p)
    {
        onig_region_free(p, (int)free_self);
        CloseHandle(env, this);
    }
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_region_clear
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigRegion_onig_1region_1clear
  (JNIEnv *env, jobject this)
{
    OnigRegion *p = GetOnigRegion(env, this);
    if (p)
    {
        onig_region_clear(p);
    }
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_region_copy
 * Signature: (Lonig4j/OnigRegion;)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigRegion_onig_1region_1copy
  (JNIEnv *env, jobject this, jobject target)
{
    OnigRegion *from = GetOnigRegion(env, this);
    if (from && target)
    {
        OnigRegion *to = GetOnigRegion(env, target);
        if (to)
        {
            onig_region_copy(to, from);
            SetHandle(env, target, to);
        }
    }
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_region_resize
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_onig_1region_1resize
  (JNIEnv *env, jobject this, jint n)
{
    OnigRegion *p = GetOnigRegion(env, this);
    return (p ? (jint)onig_region_resize(p, (int)n) : 0);
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_region_resize_clear
 * Signature: (I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_onig_1region_1resize_1clear
  (JNIEnv *env, jobject this, jint n)
{
    OnigRegion *p = GetOnigRegion(env, this);
    if (p)
    {
        int const ret = onig_region_resize(p, (int)n);
        if (ret == ONIG_NORMAL)
        {
            onig_region_clear(p);
        }
        return (jint)ret;
    }
    return 0;
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_get_capture_tree
 * Signature: ()Lonig4j/OnigCaptureTreeNode;
 */
JNIEXPORT jobject JNICALL Java_onig4j_OnigRegion_onig_1get_1capture_1tree
  (JNIEnv *env, jobject this)
{
    OnigRegion *p = GetOnigRegion(env, this);
    return (p ? NewOnigCaptureTreeNode(env, onig_get_capture_tree(p)) : NULL);
}

/*
 * Class:     onig4j_OnigRegion
 * Method:    onig_capture_tree_traverse
 * Signature: (Lonig4j/OnigRegion$Callback;I)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigRegion_onig_1capture_1tree_1traverse
  (JNIEnv *env, jobject this, jobject callback, jint at)
{
    OnigRegion *p = GetOnigRegion(env, this);
    if (p)
    {
        REGION_CALLBACK_OBJECT arg = { env, callback };
        return (jint)onig_capture_tree_traverse(p, (int)at, &xTraverseCaptureTree, &arg);
    }
    return -1;
}

///////////////////////////////////////////////////////////////////////////////
/// onig4j.OnigCaptureTreeNode ////////////////////////////////////////////////

/*
 * Class:     onig4j_OnigCaptureTreeNode
 * Method:    getChildNode
 * Signature: (JI)Lonig4j/OnigCaptureTreeNode;
 */
JNIEXPORT jobject JNICALL Java_onig4j_OnigCaptureTreeNode_getChildNode
  (JNIEnv *env, jclass clazz, jlong handle, jint i)
{
    OnigCaptureTreeNode *p = (handle ? (OnigCaptureTreeNode *)_voidp(handle) : NULL);
    if (p)
    {
        if (IS_VALIDITY_RANGE(i, p->num_childs))
        {
            return NewOnigCaptureTreeNode(env, p->childs[i]);
        }
        else
        {
            ThrowArrayIndexOutOfBoundsException(env);
        }
    }
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
/// onig4j.OnigSyntaxType ////////////////////////////////////////////////////

#define IS_NOT_BUILTIN_SYNTAX(syntax) \
    (syntax != ONIG_SYNTAX_ASIS \
     && syntax != ONIG_SYNTAX_POSIX_BASIC \
     && syntax != ONIG_SYNTAX_POSIX_EXTENDED \
     && syntax != ONIG_SYNTAX_EMACS \
     && syntax != ONIG_SYNTAX_GREP \
     && syntax != ONIG_SYNTAX_GNU_REGEX \
     && syntax != ONIG_SYNTAX_JAVA \
     && syntax != ONIG_SYNTAX_PERL \
     && syntax != ONIG_SYNTAX_PERL_NG \
     && syntax != ONIG_SYNTAX_RUBY)

#define onig_syntax_new()   ((OnigSyntaxType *)malloc(sizeof(OnigSyntaxType)))

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    free
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_free
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *syntax = GetOnigSyntaxType(env, this);
    if (syntax && IS_NOT_BUILTIN_SYNTAX(syntax))
    {
        free(syntax);
        CloseHandle(env, this);
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_syntax_new
 * Signature: ()J
 */
JNIEXPORT jlong JNICALL Java_onig4j_OnigSyntaxType_onig_1syntax_1new
  (JNIEnv *env, jclass clazz)
{
    OnigSyntaxType *p = onig_syntax_new();
    if (p)
    {
        // initialize all fields by 0
        onig_set_syntax_op(p, 0);
        onig_set_syntax_op2(p, 0);
        onig_set_syntax_behavior(p, 0);
        onig_set_syntax_options(p, 0);
        onig_set_meta_char(p, ONIG_META_CHAR_ESCAPE, ONIG_INEFFECTIVE_META_CHAR);
        onig_set_meta_char(p, ONIG_META_CHAR_ANYCHAR, ONIG_INEFFECTIVE_META_CHAR);
        onig_set_meta_char(p, ONIG_META_CHAR_ANYTIME, ONIG_INEFFECTIVE_META_CHAR);
        onig_set_meta_char(p, ONIG_META_CHAR_ZERO_OR_ONE_TIME, ONIG_INEFFECTIVE_META_CHAR);
        onig_set_meta_char(p, ONIG_META_CHAR_ONE_OR_MORE_TIME, ONIG_INEFFECTIVE_META_CHAR);
        onig_set_meta_char(p, ONIG_META_CHAR_ANYCHAR_ANYTIME, ONIG_INEFFECTIVE_META_CHAR);
        return _jlong(p);
    }
    else
    {
        ThrowOutOfMemoryError(env);
        return 0;
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_copy_syntax
 * Signature: ()Lonig4j/OnigSyntaxType;
 */
JNIEXPORT jobject JNICALL Java_onig4j_OnigSyntaxType_onig_1copy_1syntax
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *from = GetOnigSyntaxType(env, this);
    if (from)
    {
        if (clsOnigSyntaxType || LoadOnigSyntaxTypeClass(env))
        {
            OnigSyntaxType *to = onig_syntax_new();
            if (to)
            {
                // new onig4j.OnigSyntaxType object
                onig_copy_syntax(to, from);
                return (*env)->NewObject(env, clsOnigSyntaxType, mid_OnigSyntaxType_init, _jlong(to));
            }
            else
            {
                ThrowOutOfMemoryError(env);
            }
        }
    }
    return NULL;
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_set_default_syntax
 * Signature: (Lonig4j/OnigSyntaxType;)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigSyntaxType_onig_1set_1default_1syntax
  (JNIEnv *env, jclass clazz, jobject syntax)
{
    return (jint)onig_set_default_syntax((syntax ? GetOnigSyntaxType(env, syntax) : NULL));
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_get_syntax_op
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigSyntaxType_onig_1get_1syntax_1op
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    return (p ? (jint)onig_get_syntax_op(p) : 0);
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_get_syntax_op2
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigSyntaxType_onig_1get_1syntax_1op2
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    return (p ? (jint)onig_get_syntax_op2(p) : 0);
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_get_syntax_behavior
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigSyntaxType_onig_1get_1syntax_1behavior
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    return (p ? (jint)onig_get_syntax_behavior(p) : 0);
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_get_syntax_options
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigSyntaxType_onig_1get_1syntax_1options
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    return (p ? (jint)onig_get_syntax_options(p) : 0);
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_set_syntax_op
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1set_1syntax_1op
  (JNIEnv *env, jobject this, jint op)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_op(p, (unsigned int)op);
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_set_syntax_op2
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1set_1syntax_1op2
  (JNIEnv *env, jobject this, jint op2)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_op2(p, (unsigned int)op2);
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_set_syntax_behavior
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1set_1syntax_1behavior
  (JNIEnv *env, jobject this, jint behavior)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_behavior(p, (unsigned int)behavior);
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_set_syntax_options
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1set_1syntax_1options
  (JNIEnv *env, jobject this, jint options)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_options(p, (OnigOptionType)options);
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_set_meta_char
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_onig4j_OnigSyntaxType_onig_1set_1meta_1char
  (JNIEnv *env, jobject this, jint what, jint codePoint)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        return (jint)onig_set_meta_char(p, (unsigned int)what, (OnigCodePoint)codePoint);
    }
    return (jint)ONIG_NORMAL;
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_add_syntax_op
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1add_1syntax_1op
  (JNIEnv *env, jobject this, jint op)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_op(p, (onig_get_syntax_op(p) | (OnigOptionType)op));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_add_syntax_op2
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1add_1syntax_1op2
  (JNIEnv *env, jobject this, jint op2)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_op2(p, (onig_get_syntax_op2(p) | (OnigOptionType)op2));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_add_syntax_behavior
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1add_1syntax_1behavior
  (JNIEnv *env, jobject this, jint behavior)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_behavior(p, (onig_get_syntax_behavior(p) | (OnigOptionType)behavior));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_add_syntax_options
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1add_1syntax_1options
  (JNIEnv *env, jobject this, jint options)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_options(p, (onig_get_syntax_options(p) | (OnigOptionType)options));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_remove_syntax_op
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1remove_1syntax_1op
  (JNIEnv *env, jobject this, jint op)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_op(p, (onig_get_syntax_op(p) & ~((OnigOptionType)op)));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_remove_syntax_op2
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1remove_1syntax_1op2
  (JNIEnv *env, jobject this, jint op2)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_op2(p, (onig_get_syntax_op2(p) & ~((OnigOptionType)op2)));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_remove_syntax_behavior
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1remove_1syntax_1behavior
  (JNIEnv *env, jobject this, jint behavior)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_behavior(p, (onig_get_syntax_behavior(p) & ~((OnigOptionType)behavior)));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    onig_remove_syntax_options
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_onig4j_OnigSyntaxType_onig_1remove_1syntax_1options
  (JNIEnv *env, jobject this, jint options)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    if (p && IS_NOT_BUILTIN_SYNTAX(p))
    {
        onig_set_syntax_options(p, (onig_get_syntax_options(p) & ~((OnigOptionType)options)));
    }
}

/*
 * Class:     onig4j_OnigSyntaxType
 * Method:    isBuiltIn
 * Signature: ()Z
 */
JNIEXPORT jboolean JNICALL Java_onig4j_OnigSyntaxType_isBuiltIn
  (JNIEnv *env, jobject this)
{
    OnigSyntaxType *p = GetOnigSyntaxType(env, this);
    return (p && !IS_NOT_BUILTIN_SYNTAX(p) ? JNI_TRUE : JNI_FALSE);
}

#ifdef __cplusplus
}
#endif
