package onig4j;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;

/**
 * OnigOptionType enum class.
 * @author calico
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
 */
public enum OnigOptionType {
    /** no option */
    ONIG_OPTION_NONE(0),
    /** ambiguity match on */
    ONIG_OPTION_IGNORECASE(1),
    /** extended pattern form */
    ONIG_OPTION_EXTEND(ONIG_OPTION_IGNORECASE.value << 1),
    /** '.' match with newline<br/>(Equals to the java.util.regex.Pattern.DOTALL option of Java) */
    ONIG_OPTION_MULTILINE(ONIG_OPTION_EXTEND.value << 1),
    /** '^' -> '\A', '$' -> '\Z' */
    ONIG_OPTION_SINGLELINE(ONIG_OPTION_MULTILINE.value << 1),
    /** find longest match */
    ONIG_OPTION_FIND_LONGEST(ONIG_OPTION_SINGLELINE.value << 1),
    /** ignore empty match */
    ONIG_OPTION_FIND_NOT_EMPTY(ONIG_OPTION_FIND_LONGEST.value << 1),
    /**
     * clear ONIG_OPTION_SINGLELINE which is enabled on ONIG_SYNTAX_POSIX_BASIC, ONIG_SYNTAX_POSIX_EXTENDED, ONIG_SYNTAX_PERL, ONIG_SYNTAX_PERL_NG, ONIG_SYNTAX_JAVA.<br/>
     * (Equals to the java.util.regex.Pattern.MULTILINE option of Java)
     * @see java.util.regex.Pattern#MULTILINE
     */
    ONIG_OPTION_NEGATE_SINGLELINE(ONIG_OPTION_FIND_NOT_EMPTY.value << 1),
    /** only named group captured */
    ONIG_OPTION_DONT_CAPTURE_GROUP(ONIG_OPTION_NEGATE_SINGLELINE.value << 1),
    /** named and no-named group captured */
    ONIG_OPTION_CAPTURE_GROUP(ONIG_OPTION_DONT_CAPTURE_GROUP.value << 1),
    /** string head(str) isn't considered as begin of line */
    ONIG_OPTION_NOTBOL(ONIG_OPTION_CAPTURE_GROUP.value << 1),
    /** string end (end) isn't considered as end of line */
    ONIG_OPTION_NOTEOL(ONIG_OPTION_NOTBOL.value << 1),
    /** region argument is regmatch_t[] of POSIX API */
    ONIG_OPTION_POSIX_REGION(ONIG_OPTION_NOTEOL.value << 1),
    /**
     * hit to Java line terminators (UNICODE only).<br/>
     * NOTE: This is an original option. If it wants to use this option, the Oniruguma extended version 5.9.1.1 or later is required. (Not exist in the Oniguruma version 5.9.1)
     * @see <a href="http://java.sun.com/javase/6/docs/api/java/util/regex/Pattern.html#lt">Line terminators</a>
     * @see <a href="http://java.sun.com/javase/ja/6/docs/ja/api/java/util/regex/Pattern.html#lt">行末記号</a>
     */
    ONIG_OPTION_JAVA_LINES(ONIG_OPTION_POSIX_REGION.value << 1),
    /**
     * clear ONIG_OPTION_JAVA_LINES which is enabled on ONIG_SYNTAX_JAVA.<br/>
     * (Equals to the java.util.regex.Pattern.UNIX_LINES option of Java)<br/>
      * NOTE: This is an original option. If it wants to use this option, the Oniruguma extended version 5.9.1.1 or later is required. (Not exist in the Oniguruma version 5.9.1)
     * @see #ONIG_OPTION_JAVA_LINES
     * @see java.util.regex.Pattern#UNIX_LINES
     */
    ONIG_OPTION_NEGATE_JAVA_LINES(ONIG_OPTION_JAVA_LINES.value << 1),
    /**
     * case-insensitive matching assumes that only characters in the US-ASCII charset are being matched.<br/>
     * NOTE: This is an original option. If it wants to use this option, the Oniruguma extended version 5.9.1.1 or later is required. (Not exist in the Oniguruma version 5.9.1)
     */
    ONIG_OPTION_CASE_FOLD_ASCII_ONLY(ONIG_OPTION_NEGATE_JAVA_LINES.value << 1),
    /**
     * clear ONIG_OPTION_CASE_FOLD_ASCII_ONLY which is enabled on ONIG_SYNTAX_JAVA.<br/>
     * (Equals to the java.util.regex.Pattern.UNICODE_CASE option of Java)<br/>
     * NOTE: This is an original option. If it wants to use this option, the Oniruguma extended version 5.9.1.1 or later is required. (Not exist in the Oniguruma version 5.9.1)
     * @see #ONIG_OPTION_CASE_FOLD_ASCII_ONLY
     * @see java.util.regex.Pattern#UNICODE_CASE
     */
    ONIG_OPTION_NEGATE_CASE_FOLD_ASCII_ONLY(ONIG_OPTION_CASE_FOLD_ASCII_ONLY.value << 1);
    
    final int value;
    
    private OnigOptionType(int value) {
        this.value = value;
    }
    
//    static OnigOptionType valueOf(final int value) {
//        for (final OnigOptionType option : OnigOptionType.values()) {
//            if (value == option.value) {
//                return option;
//            }
//        }
//        throw new IllegalArgumentException("No enum const class " + OnigOptionType.class.getName() + ".value(" + value + ")");
//    }

    static EnumSet<OnigOptionType> valuesOf(final int values) {
        final List<OnigOptionType> options = new ArrayList<OnigOptionType>();
        for (final OnigOptionType option : EnumSet.range(ONIG_OPTION_IGNORECASE, ONIG_OPTION_POSIX_REGION)) {
            if ((values & option.value) == option.value) {
                options.add(option);
            }
        }
        return (options.size() != 0 ? EnumSet.copyOf(options) : EnumSet.of(ONIG_OPTION_NONE));
    }

    static int values(Collection<OnigOptionType> options) {
        if (options != null) {
            int values = ONIG_OPTION_NONE.value;
            for (final OnigOptionType option : options) {
                if (option != null) {
                    values |= option.value;
                }
            }
            return values;
        }
        return ONIG_OPTION_NONE.value;
    }

    static int values(OnigOptionType... options) {
        if (options != null) {
            int values = ONIG_OPTION_NONE.value;
            for (final OnigOptionType option : options) {
                if (option != null) {
                    values |= option.value;
                }
            }
            return values;
        }
        return ONIG_OPTION_NONE.value;
    }
    
    public static final int ONIG_OPTION_MAXBIT = ONIG_OPTION_NEGATE_CASE_FOLD_ASCII_ONLY.value;
}
