package onig4j;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.EventListener;
import java.util.List;
import java.util.regex.PatternSyntaxException;

/**
 * Oniguruma regex class
 * @author calico
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.txt">Oniguruma API</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/RE.txt">Oniguruma Regular Expressions</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.ja.txt">鬼車インターフェース</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/RE.ja.txt">鬼車 正規表現</a>
 * @see <a href="http://homepage3.nifty.com/k-takata/diary/perl595_onig.txt">Perl 5.9.5 で拡張された正規表現の概要 および、その Oniguruma との比較</a>
 */
public class OnigRegex extends OnigHandle {
    
    /** warning listener */
    private static WarningListener warnListener;
    /** verbose warning listener */
    private static WarningListener verbListener;
    
    static {
        // load JNI library
        System.loadLibrary("onig4j");
        
//        final java.util.logging.Logger log = java.util.logging.Logger.getLogger(OnigRegex.class.getName());
//        if (log.isLoggable(java.util.logging.Level.INFO)) {
//            log.info("encoding is UTF-16" + (useUnicodeLittleUnmarked() ? "LE" : "BE"));
//        }
//        if (log.isLoggable(java.util.logging.Level.WARNING)) {
//            final WarningListener warn
//                    = new WarningListener() {
//                            public void warning(String msg) {
//                                log.warning(msg);
//                            }
//                        };
//            setWarningListener(warn);
//            setVerboseWarningListener(warn);
//        }
    }
    
    /** regex pattern string */
    private final String pattern;
    
    /** syntax object */
    private OnigSyntaxType syntax;
    
    /**
     * invoke onig_new() function with ONIG_OPTION_NONE and ONIG_SYNTAX_DEFAULT.
     * @param pattern The regex pattern string
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, onig4j.OnigOptionType...)
     */
    public OnigRegex(String pattern) {
        this(pattern, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE, OnigOptionType.ONIG_OPTION_NONE.value);
    }
    
    /**
     * invoke onig_new() function with ONIG_SYNTAX_DEFAULT.
     * @param pattern The regex pattern string
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, onig4j.OnigOptionType...)
     */
    public OnigRegex(String pattern, OnigOptionType... options) {
        this(pattern, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new() function with ONIG_SYNTAX_DEFAULT.
     * @param pattern The regex pattern string
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, java.util.Collection)
     */
    public OnigRegex(String pattern, Collection<OnigOptionType> options) {
        this(pattern, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function with ONIG_SYNTAX_DEFAULT.
     * @param pattern The regex pattern string
     * @param flag The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, onig4j.OnigCaseFoldType, onig4j.OnigOptionType...)
     */
    public OnigRegex(String pattern, OnigCaseFoldType flag, OnigOptionType... options) {
        this(pattern, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigCaseFoldType.value(flag), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function with ONIG_SYNTAX_DEFAULT.
     * @param pattern The regex pattern string
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, onig4j.OnigCaseFoldType, onig4j.OnigOptionType...)
     */
    public OnigRegex(String pattern, OnigCaseFoldType[] flags, OnigOptionType... options) {
        this(pattern, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigCaseFoldType.values(flags), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function with ONIG_SYNTAX_DEFAULT.
     * @param pattern The regex pattern string
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, onig4j.OnigCaseFoldType, java.util.Collection)
     */
    public OnigRegex(String pattern, Collection<OnigCaseFoldType> flags, Collection<OnigOptionType> options) {
        this(pattern, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigCaseFoldType.values(flags), OnigOptionType.values(options));
    }

    /**
     * invoke onig_new() function with ONIG_OPTION_NONE.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, onig4j.OnigOptionType...)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax) {
        this(pattern, syntax, OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE, OnigOptionType.ONIG_OPTION_NONE.value);
    }
    
    /**
     * invoke onig_new() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, OnigOptionType... options) {
        this(pattern, syntax, OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, Collection<OnigOptionType> options) {
        this(pattern, syntax, OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flag The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, OnigCaseFoldType flag, OnigOptionType... options) {
        this(pattern, syntax, OnigCaseFoldType.value(flag), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, OnigCaseFoldType[] flags, OnigOptionType... options) {
        this(pattern, syntax, OnigCaseFoldType.values(flags), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, Collection<OnigCaseFoldType> flags, OnigOptionType... options) {
        this(pattern, syntax, OnigCaseFoldType.values(flags), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flag The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, OnigCaseFoldType flag, Collection<OnigOptionType> options) {
        this(pattern, syntax, OnigCaseFoldType.value(flag), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, OnigCaseFoldType[] flags, Collection<OnigOptionType> options) {
        this(pattern, syntax, OnigCaseFoldType.values(flags), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode. (null is allowed)
     * @param options The compile time options
     * @see #OnigRegex(java.lang.String, onig4j.OnigSyntaxType, int, int)
     */
    public OnigRegex(String pattern, OnigSyntaxType syntax, Collection<OnigCaseFoldType> flags, Collection<OnigOptionType> options) {
        this(pattern, syntax, OnigCaseFoldType.values(flags), OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_new() or onig_new_deluxe() function.
     * @param pattern The regex pattern string
     * @param syntax The pattern syntax definition (null is allowed)
     * @param flags The character matching case fold bit flag for ONIG_OPTION_IGNORECASE mode.
     * @param options The compile time options
     */
    protected OnigRegex(String pattern, OnigSyntaxType syntax, int flags, int options) {
        if (pattern == null) {
            throw new NullPointerException("pattern is null");
        }
        if (syntax == null) {
            throw new NullPointerException("syntax is null");
        }
        
        final String [] errmsg = new String[1];
        final int ret
                = (flags == OnigCaseFoldType.ONIGENC_CASE_FOLD_NONE
                    ? onig_new(pattern, options, syntax, errmsg)
                    : onig_new_deluxe(pattern, options, syntax, flags, errmsg));
        if (ret < ONIG_MISMATCH) {
            throwOnigError(ret, errmsg[0], pattern);
        }
        this.pattern = pattern;
        this.syntax = syntax;
    }
    
    /**
     * invoke onig_free() function.
     */
    @Override
    protected final void free() {
        onig_free();
        syntax = null;
    }

    /**
     * invoke onig_name_to_backref_number() function.
     * @param name The group name
     * @param region The region object (null is allowed)
     * @return The group number
     */
    public int getBackRefNumber(String name, OnigRegion region) {
        if (name == null) {
            throw new NullPointerException("name is null");
        }
        if (region != null && region.count() == 0) {
            throw new IllegalStateException("region is empty");
        }
        
        final int ret = onig_name_to_backref_number(name, region);
        if (ret < ONIG_MISMATCH) {
            throwOnigError(ret, onig_error_code_to_str(ret, name), name);
        }
        return ret;
    }
    
    /**
     * invoke onig_name_to_group_numbers() function.
     * @param name The group name
     * @return The number of groups for the name
     */
    public int[] getGroupNumbers(String name) {
        if (name == null) {
            throw new NullPointerException("name is null");
        }
        return onig_name_to_group_numbers(name);
    }
    
    /**
     * Returns the charset object of this instance.<br/>
     * NOTE: The onig_get_encoding() function is called.
     * @return The charset object
     */
    public Charset getEncoding() {
        return Charset.forName(onig_get_encoding_name());
    }
    
    /**
     * invoke onig_get_options() function.
     * @return The value of OnigOptionType
     */
    public EnumSet<OnigOptionType> getOptionType() {
        return OnigOptionType.valuesOf(onig_get_options());
    }
    
    /**
     * invoke onig_get_case_fold_flag() function.
     * @return The value of OnigCaseFoldType
     */
    public EnumSet<OnigCaseFoldType> getCaseFoldType() {
        return OnigCaseFoldType.valuesOf(onig_get_case_fold_flag());
    }
    
    /**
     * Returns the syntax object of this instance.<br/>
     * NOTE: The onig_get_syntax() function is not called.
     * @return The OnigSyntaxType object
     */
    public OnigSyntaxType getSyntaxType() {
        return syntax;
//        return onig_get_syntax();
    }
    
    /**
     * Returns the regular expression from which this pattern was compiled.
     * @return The source of this pattern
     */
    public String pattern() {
        return pattern;
    }
    
    /**
     * invoke onig_noname_group_capture_is_active() function.
     * @return true if the noname group capture is active
     */
    public boolean isActiveNonameGroupCaputure() {
        return onig_noname_group_capture_is_active();
    }
    
    /**
     * invoke search(input, input.length(), 0, input.length(), null, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input) {
        final int end = input.length();
        return search(input, end, 0, end, null, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, input.length(), 0, input.length(), null, options) method.
     * @param input The character sequence to be searched
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, OnigOptionType... options) {
        final int end = input.length();
        return search(input, end, 0, end, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), 0, input.length(), null, options) method.
     * @param input The character sequence to be searched
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, Collection<OnigOptionType> options) {
        final int end = input.length();
        return search(input, end, 0, end, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), 0, input.length(), region, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param region The region object for return group match range info (null is allowed)
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, OnigRegion region) {
        final int end = input.length();
        return search(input, end, 0, end, region, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, input.length(), 0, input.length(), region, options) method.
     * @param input The character sequence to be searched
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, OnigRegion region, OnigOptionType... options) {
        final int end = input.length();
        return search(input, end, 0, end, region, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), 0, input.length(), region, options) method.
     * @param input The character sequence to be searched
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, OnigRegion region, Collection<OnigOptionType> options) {
        final int end = input.length();
        return search(input, end, 0, end, region, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, input.length(), null, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start) {
        final int end = input.length();
        return search(input, end, start, end, null, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, input.length(), start, input.length(), region, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param region The region object for return group match range info (null is allowed)
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, OnigRegion region) {
        final int end = input.length();
        return search(input, end, start, end, region, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, input.length(), start, input.length(), null, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, OnigOptionType... options) {
        final int end = input.length();
        return search(input, end, start, end, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), input.length(), end, null, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, Collection<OnigOptionType> options) {
        final int end = input.length();
        return search(input, end, start, end, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, input.length(), region, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, OnigRegion region, OnigOptionType... options) {
        final int end = input.length();
        return search(input, end, start, end, region, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, input.length(), region, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, OnigRegion region, Collection<OnigOptionType> options) {
        final int end = input.length();
        return search(input, end, start, end, region, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, range, null, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, int range) {
        return search(input, input.length(), start, range, null, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, input.length(), start, range, region, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, int range, OnigRegion region) {
        return search(input, input.length(), start, range, region, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, input.length(), start, range, null, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, int range, OnigOptionType... options) {
        return search(input, input.length(), start, range, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, range, null, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, int range, Collection<OnigOptionType> options) {
        return search(input, input.length(), start, range, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, range, region, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, int range, OnigRegion region, OnigOptionType... options) {
        return search(input, input.length(), start, range, region, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, input.length(), start, range, region, options) method.
     * @param input The character sequence to be searched
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int start, int range, OnigRegion region, Collection<OnigOptionType> options) {
        return search(input, input.length(), start, range, region, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, end, start, range, null, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int end, int start, int range) {
        return search(input, end, start, range, null, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, end, start, range, region, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int end, int start, int range, OnigRegion region) {
        return search(input, end, start, range, region, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke search(input, end, start, range, null, options) method.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int end, int start, int range, OnigOptionType... options) {
        return search(input, end, start, range, null, OnigOptionType.values(options));
    }

    /**
     * invoke search(input, end, start, range, null, options) method.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int end, int start, int range, Collection<OnigOptionType> options) {
        return search(input, end, start, range, null, OnigOptionType.values(options));
    }

    /**
     * invoke onig_search() function.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int end, int start, int range, OnigRegion region, OnigOptionType... options) {
        return search(input, end, start, range, region, OnigOptionType.values(options));
    }

    /**
     * invoke onig_search() function.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     * @see #search(java.lang.CharSequence, int, int, int, onig4j.OnigRegion, int)
     */
    public int search(CharSequence input, int end, int start, int range, OnigRegion region, Collection<OnigOptionType> options) {
        return search(input, end, start, range, region, OnigOptionType.values(options));
    }

    /**
     * invoke onig_search() function.
     * @param input The character sequence to be searched
     * @param end The terminate index of character sequence, exclusive
     * @param start The search start index of character sequence, inclusive
     * @param range The search terminate index of character sequence<br/>in forward search  (start <= searched string < range)</br>in backward search (range <= searched string <= start)
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The index of the first character matched
     */
    protected int search(CharSequence input, int end, int start, int range, OnigRegion region, int options) {
        if (input == null) {
            throw new NullPointerException("input is null");
        }
        final int ret = onig_search(input.toString(), end, start, range, region, options);
        if (ret < ONIG_MISMATCH) {
            throwOnigError(ret, onig_error_code_to_str(ret, input.toString()), input.toString());
        }
        return ret;
    }
    
    /**
     * invoke match(input, input.length(), at, null, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be matched
     * @param at The match index of character sequence
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int at) {
        return match(input, input.length(), at, null, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke match(input, input.length(), at, region, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be matched
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int at, OnigRegion region) {
        return match(input, input.length(), at, region, OnigOptionType.ONIG_OPTION_NONE.value);
    }

    /**
     * invoke match(input, input.length(), at, null, options) method.
     * @param input The character sequence to be matched
     * @param at The match index of character sequence
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int at, OnigOptionType... options) {
        return match(input, input.length(), at, null, OnigOptionType.values(options));
    }

    /**
     * invoke match(input, input.length(), at, null, options) method.
     * @param input The character sequence to be matched
     * @param at The match index of character sequence
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int at, Collection<OnigOptionType> options) {
        return match(input, input.length(), at, null, OnigOptionType.values(options));
    }

    /**
     * invoke match(input, input.length(), at, region, options) method.
     * @param input The character sequence to be matched
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int at, OnigRegion region, OnigOptionType... options) {
        return match(input, input.length(), at, region, OnigOptionType.values(options));
    }

    /**
     * invoke match(input, input.length(), at, region, options) method.
     * @param input The character sequence to be matched
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int at, OnigRegion region, Collection<OnigOptionType> options) {
        return match(input, input.length(), at, region, OnigOptionType.values(options));
    }
    
    /**
     * invoke match(input, end, at, null, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int end, int at) {
        return match(input, end, at, null, OnigOptionType.ONIG_OPTION_NONE.value);
    }
    
    /**
     * invoke match(input, end, at, region, ONIG_OPTION_NONE) method.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int end, int at, OnigRegion region) {
        return match(input, end, at, region, OnigOptionType.ONIG_OPTION_NONE.value);
    }
    
    /**
     * invoke match(input, end, at, null, options) method.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int end, int at, OnigOptionType... options) {
        return match(input, end, at, null, OnigOptionType.values(options));
    }
    
    /**
     * invoke match(input, end, at, null, options) method.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int end, int at, Collection<OnigOptionType> options) {
        return match(input, end, at, null, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_match() function.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int end, int at, OnigRegion region, OnigOptionType... options) {
        return match(input, end, at, region, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_match() function.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The matched character count
     * @see #match(java.lang.CharSequence, int, int, onig4j.OnigRegion, int)
     */
    public int match(CharSequence input, int end, int at, OnigRegion region, Collection<OnigOptionType> options) {
        return match(input, end, at, region, OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_match() function.
     * @param input The character sequence to be matched
     * @param end The terminate index of character sequence, exclusive
     * @param at The match index of character sequence
     * @param region The region object for return group match range info (null is allowed)
     * @param options The search time option
     * @return The matched character count
     */
    protected int match(CharSequence input, int end, int at, OnigRegion region, int options) {
        if (input == null) {
            throw new NullPointerException("input is null");
        }
        if (at < 0 || at > end) {
            throw new IndexOutOfBoundsException("the at parameter is out of range");
        }
        final int ret = onig_match(input.toString(), end, at, region, options);
        if (ret < ONIG_MISMATCH) {
            throwOnigError(ret, onig_error_code_to_str(ret, input.toString()), input.toString());
        }
        return ret;
    }
    
    /**
     * invoke split(input, 0, OnigOptionType.ONIG_OPTION_NONE) method.
     * @param input The character sequence to be split
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int, onig4j.OnigOptionType...)
     */
    public List<String> split(CharSequence input) {
        return split(input, 0, OnigOptionType.ONIG_OPTION_NONE.value);
    }
    
    /**
     * invoke split(input, limit, OnigOptionType.ONIG_OPTION_NONE) method.
     * @param input The character sequence to be split
     * @param limit The result threshold, as described above
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int, onig4j.OnigOptionType...)
     */
    public List<String> split(CharSequence input, int limit) {
        return split(input, limit, OnigOptionType.ONIG_OPTION_NONE.value);
    }
    
    /**
     * invoke split(input, 0, option) method.
     * @param input The character sequence to be split
     * @param options The search time option
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int, java.util.Collection)
     */
    public List<String> split(CharSequence input, Collection<OnigOptionType> options) {
        return split(input, 0, OnigOptionType.values(options));
    }
    
    /**
     * invoke split(input, 0, option) method.
     * @param input The character sequence to be split
     * @param options The search time option
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int, onig4j.OnigOptionType...)
     */
    public List<String> split(CharSequence input, OnigOptionType... options) {
        return split(input, 0, OnigOptionType.values(options));
    }
    
    /**
     * Splits the given string around matches of this pattern.
     * @param input The character sequence to be split
     * @param limit The result threshold, as described above
     * @param options The search time option
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int, int)
     */
    public List<String> split(CharSequence input, int limit, Collection<OnigOptionType> options) {
        return split(input, limit, OnigOptionType.values(options));
    }
    
    /**
     * Splits the given string around matches of this pattern.
     * @param input The character sequence to be split
     * @param limit The result threshold, as described above
     * @param options The search time option
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see #split(java.lang.CharSequence, int, int)
     */
    public List<String> split(CharSequence input, int limit, OnigOptionType... options) {
        return split(input, limit, OnigOptionType.values(options));
    }
    
    /**
     * Splits the given string around matches of this pattern.
     * @param input The character sequence to be split
     * @param limit The result threshold, as described above
     * @param options The search time option
     * @return The array of strings computed by splitting the input around matches of this pattern
     * @see java.util.regex.Pattern#split(java.lang.CharSequence, int)
     */
    protected List<String> split(CharSequence input, int limit, int options) {
        if (input == null) {
            throw new NullPointerException("input is null");
        }
        final List<String> list = new ArrayList<String>();
        final int ret = onig_split(input.toString(), limit, options, list);
        if (ret < ONIG_MISMATCH) {
            throwOnigError(ret, onig_error_code_to_str(ret, input.toString()), input.toString());
        }        
        if (limit == 0) {
            // discards a trailing empty strings
            for (int i = list.size(); i > 0 && list.get(--i).length() == 0; list.remove(i)) {
                // nothing
            }
//            for (final ListIterator<String> iter = list.listIterator(list.size());
//                    iter.hasPrevious() && iter.previous().length() == 0;
//                    iter.remove()) {
//                // nothing
//            }
        }
        return list;
    }
    
    /**
     * invoke onig_number_of_names() function.
     * @return The number of names defined in the pattern (Multiple definitions of one name is counted as one)
     */
    public int nameCount() {
        return onig_number_of_names();
    }
    
    /**
     * invoke onig_number_of_captures() function.
     * @return The number of capture group in the pattern
     */
    public int captureCount() {
        return onig_number_of_captures();
    }
    
    /**
     * invoke onig_number_of_capture_histories() function.
     * @return The number of capture history defined in the pattern
     */
    public int captureHistoryCount() {
        return onig_number_of_capture_histories();
    }

    /**
     * Callback interface for onig_foreach_name() function.
     * @see onig4j.OnigRegex#foreachName(onig4j.OnigRegex.Callback)
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.txt">Oniguruma API</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.ja.txt">鬼車インターフェース</a>
     */
    public static interface Callback {
        /**
         * Called from onig_foreach_name() function.
         * @param name The group name
         * @param groups The group number's list
         * @param regex The regex object
         * @return if func does not return 0, then iteration is stopped
         */
        int call(String name, int[] groups, OnigRegex regex);
    }
    
    /**
     * invoke onig_foreach_name() function.
     * @param callback The callback object
     * @return The callback object return value
     */
    public int foreachName(Callback callback) {
        if (callback == null) {
            throw new NullPointerException("callback is null");
        }
        return onig_foreach_name(callback);
    }
    
    /**
     * invoke onig_get_match_stack_limit_size() function.
     * @return the maximum number of stack size (default: 0 == unlimited)
     */
    public static int getMatchStackLimitSize() {
        return onig_get_match_stack_limit_size();
    }

    /**
     * invoke onig_set_match_stack_limit_size() function.
     * @param size the maximum number of stack size (size = 0: unlimited)
     */
    public static void setMatchStackLimitSize(int size) {
        final int ret = onig_set_match_stack_limit_size(size);
        if (ret != ONIG_NORMAL) {
            throwOnigError(ret, String.valueOf(size));
        }
    }
    
    /**
     * invoke onig_version() function.
     * @return The version string.  (ex. "5.9.1")
     */
    public static String getVersion() {
        return onig_version();
    }
    
    /**
     * invoke onig_copyright() function.
     * @return The copyright string
     */
    public static String getCopyright() {
        return onig_copyright();        
    }
    
    /**
     * invoke onig_error_code_to_str() function.
     * @param ecode The error code returned by other API functions.
     * @param param The parameter value (null is allowed)
     * @return The error message string
     */
    public static String getErrorMessage(int ecode, String param) {
        return onig_error_code_to_str(ecode, param);
    }
    
    /**
     * Warning message listener interface for onig_set_warn_func() and onig_set_verb_warn_func() functions.
     * @author calico
     * @see #setWarningListener(onig4j.OnigRegex.WarningListener)
     * @see #setVerboseWarningListener(onig4j.OnigRegex.WarningListener)
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.txt">Oniguruma API</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.ja.txt">鬼車インターフェース</a>
     */
    public static interface WarningListener extends EventListener {
        /**
         * Called from onig_new() or onig_new_deluxe() function.
         * @param msg The warning message string
         */
        void warning(String msg);
    }
    
    /**
     * invoke onig_set_warn_func() function.
     * @param listener The warning message listener object
     * @return true if it's successful
     */
    public static boolean setWarningListener(WarningListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener is null");
        }
        warnListener = listener;
        return onig_set_warn_func();
    }

    /**
     * clear onig_set_warn_func() listener object.
     */
    public static void clearWarningListener() {
        onig_clear_warn_func();
        warnListener = null;
    }
    
    /**
     * invoke onig_set_verb_warn_func() function.
     * @param listener The warning message listener object
     * @return true if it's successful
     */
    public static boolean setVerboseWarningListener(WarningListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener is null");
        }
        verbListener = listener;
        return onig_set_verb_warn_func();
    }
    
    /**
     * clear onig_set_verb_warn_func() listener object.
     */
    public static void clearVerboseWarningListener() {
        onig_clear_verb_warn_func();
        verbListener = null;
    }
    
    private static String decodeString(String str) {
        final OnigRegex regex = new OnigRegex("(?<=\\/)(?:\\\\x\\h{2})+(?=\\/)");
        final OnigRegion region = new OnigRegion();
        int len = str.length();
        final int pos = regex.search(str, len, 0, len, region);
        if (pos < 0) {
            region.close();
            regex.close();
            return str;
        }

        final String enc = regex.onig_get_encoding_name();
        regex.close();        
        final int end = region.end(0);
        region.close();
        
        final String pattern = str.substring(pos, end);
        final StringBuilder decoded = new StringBuilder(len);
        decoded.append(str.substring(0, pos));
        final ByteArrayOutputStream bout = new ByteArrayOutputStream();
        len = pattern.length();
        for (int i = 2; i < len; i += 4) {
            final String hex = pattern.substring(i, i + 2);
            bout.write(Integer.parseInt(hex, 16));
        }
        try {
            decoded.append(new String(bout.toByteArray(), enc));
            decoded.append(str.substring(end));
            return decoded.toString();
            
        } catch (UnsupportedEncodingException ex) {
            return str;
        }
    }
    
    protected static void fireWarning(String msg) {
        if (warnListener != null) {
            warnListener.warning(decodeString(msg));
        }
    }
    
    protected static void fireWarningVerbose(String msg) {
        if (verbListener != null) {
            verbListener.warning(decodeString(msg));
        }
    }
    
    static void throwOnigError(int ecode) {
        throwOnigError(ecode, onig_error_code_to_str(ecode, null), null);
    }
    
    private static void throwOnigError(int ecode, String msg, String pattern) {
        if (isSyntaxError(ecode)) {
            throw new PatternSyntaxException(msg, pattern, -1);
        }
        throwOnigError(ecode, msg);
    }
    
    private static void throwOnigError(int ecode, String msg) {
        switch (ecode) {
            case ONIGERR_MEMORY:
                throw new OutOfMemoryError(msg);

            case ONIGERR_MATCH_STACK_LIMIT_OVER:
                throw new StackOverflowError(msg);

            case ONIG_NO_SUPPORT_CONFIG:
            case ONIGERR_INVALID_ARGUMENT:
                throw new IllegalArgumentException(msg);
        }
        throw new InternalError(msg);
    }
    
    /**
     * This method is in stead of ONIG_IS_PATTERN_ERROR() macro.
     * @param ecode The return code
     * @return true if the ecode is syntax error
     */
    public static boolean isSyntaxError(int ecode) {
        return ((ecode) <= -100 && (ecode) > -1000);
    }
    
//    public final static int ONIGENC_CODE_TO_MBC_MAXLEN = 7;
//    public final static int ONIGENC_MBC_CASE_FOLD_MAXLEN = 18;
//    public final static int ONIGENC_CTYPE_NEWLINE = 0;
//    public final static int ONIGENC_CTYPE_ALPHA = 1;
//    public final static int ONIGENC_CTYPE_BLANK = 2;
//    public final static int ONIGENC_CTYPE_CNTRL = 3;
//    public final static int ONIGENC_CTYPE_DIGIT = 4;
//    public final static int ONIGENC_CTYPE_GRAPH = 5;
//    public final static int ONIGENC_CTYPE_LOWER = 6;
//    public final static int ONIGENC_CTYPE_PRINT = 7;
//    public final static int ONIGENC_CTYPE_PUNCT = 8;
//    public final static int ONIGENC_CTYPE_SPACE = 9;
//    public final static int ONIGENC_CTYPE_UPPER = 10;
//    public final static int ONIGENC_CTYPE_XDIGIT = 11;
//    public final static int ONIGENC_CTYPE_WORD = 12;
//    public final static int ONIGENC_CTYPE_ALNUM = 13;
//    public final static int ONIGENC_CTYPE_ASCII = 14;
//    public final static int ONIGENC_MAX_STD_CTYPE = 14;
//    public final static int ONIG_NREGION = 10;
//    public final static int ONIG_MAX_BACKREF_NUM = 1000;
//    public final static int ONIG_MAX_REPEAT_NUM = 100000;
//    public final static int ONIG_MAX_MULTI_BYTE_RANGES_NUM = 10000;
//    public final static int ONIG_MAX_ERROR_MESSAGE_LEN = 90;
    
    /** normal return */
    public static final int ONIG_NORMAL = 0;
    /** mismatch */
    public static final int ONIG_MISMATCH = -1;
    /** no support in this configuration */
    public static final int ONIG_NO_SUPPORT_CONFIG = -2;
    
    /* internal error */
    /** fail to memory allocation */
    public static final int ONIGERR_MEMORY = -5;
    /** undefined type (bug) */
    public static final int ONIGERR_TYPE_BUG = -6;
    /** internal parser error (bug) */
    public static final int ONIGERR_PARSER_BUG = -11;
    /** stack error (bug) */
    public static final int ONIGERR_STACK_BUG = -12;
    /** undefined bytecode (bug) */
    public static final int ONIGERR_UNDEFINED_BYTECODE = -13;
    /** unexpected bytecode (bug) */
    public static final int ONIGERR_UNEXPECTED_BYTECODE = -14;
    /** match-stack limit over */
    public static final int ONIGERR_MATCH_STACK_LIMIT_OVER = -15;
    /** default multibyte-encoding is not setted */
    public static final int ONIGERR_DEFAULT_ENCODING_IS_NOT_SETTED = -21;
    /** can't convert to wide-char on specified multibyte-encoding */
    public static final int ONIGERR_SPECIFIED_ENCODING_CANT_CONVERT_TO_WIDE_CHAR = -22;
    
    /* general error */
    /** invalid argument */
    public static final int ONIGERR_INVALID_ARGUMENT = -30;
    
    /* syntax error */
    /** end pattern at left brace */
    public static final int ONIGERR_END_PATTERN_AT_LEFT_BRACE = -100;
    /** end pattern at left bracket */
    public static final int ONIGERR_END_PATTERN_AT_LEFT_BRACKET = -101;
    /** empty char-class */
    public static final int ONIGERR_EMPTY_CHAR_CLASS = -102;
    /** premature end of char-class */
    public static final int ONIGERR_PREMATURE_END_OF_CHAR_CLASS = -103;
    /** end pattern at escape */
    public static final int ONIGERR_END_PATTERN_AT_ESCAPE = -104;
    /** end pattern at meta */
    public static final int ONIGERR_END_PATTERN_AT_META = -105;
    /** end pattern at control */
    public static final int ONIGERR_END_PATTERN_AT_CONTROL = -106;
    /** invalid meta-code syntax */
    public static final int ONIGERR_META_CODE_SYNTAX = -108;
    /** invalid control-code syntax */
    public static final int ONIGERR_CONTROL_CODE_SYNTAX = -109;
    /** char-class value at end of range */
    public static final int ONIGERR_CHAR_CLASS_VALUE_AT_END_OF_RANGE = -110;
    /** char-class value at start of range */
    public static final int ONIGERR_CHAR_CLASS_VALUE_AT_START_OF_RANGE = -111;
    /** unmatched range specifier in char-class */
    public static final int ONIGERR_UNMATCHED_RANGE_SPECIFIER_IN_CHAR_CLASS = -112;
    /** target of repeat operator is not specified */
    public static final int ONIGERR_TARGET_OF_REPEAT_OPERATOR_NOT_SPECIFIED = -113;
    /** target of repeat operator is invalid */
    public static final int ONIGERR_TARGET_OF_REPEAT_OPERATOR_INVALID = -114;
    /** nested repeat operator */
    public static final int ONIGERR_NESTED_REPEAT_OPERATOR = -115;
    /** unmatched close parenthesis */
    public static final int ONIGERR_UNMATCHED_CLOSE_PARENTHESIS = -116;
    /** end pattern with unmatched parenthesis */
    public static final int ONIGERR_END_PATTERN_WITH_UNMATCHED_PARENTHESIS = -117;
    /** end pattern in group */
    public static final int ONIGERR_END_PATTERN_IN_GROUP = -118;
    /** undefined group option */
    public static final int ONIGERR_UNDEFINED_GROUP_OPTION = -119;
    /** invalid POSIX bracket type */
    public static final int ONIGERR_INVALID_POSIX_BRACKET_TYPE = -121;
    /** invalid pattern in look-behind */
    public static final int ONIGERR_INVALID_LOOK_BEHIND_PATTERN = -122;
    /** invalid repeat range {lower,upper} */
    public static final int ONIGERR_INVALID_REPEAT_RANGE_PATTERN = -123;

    /* values error (syntax error) */
    /** too big number */
    public static final int ONIGERR_TOO_BIG_NUMBER = -200;
    /** too big number for repeat range */
    public static final int ONIGERR_TOO_BIG_NUMBER_FOR_REPEAT_RANGE = -201;
    /** upper is smaller than lower in repeat range */
    public static final int ONIGERR_UPPER_SMALLER_THAN_LOWER_IN_REPEAT_RANGE = -202;
    /** empty range in char class */
    public static final int ONIGERR_EMPTY_RANGE_IN_CHAR_CLASS = -203;
    /** mismatch multibyte code length in char-class range */
    public static final int ONIGERR_MISMATCH_CODE_LENGTH_IN_CLASS_RANGE = -204;
    /** too many multibyte code ranges are specified */
    public static final int ONIGERR_TOO_MANY_MULTI_BYTE_RANGES = -205;
    /** too short multibyte code string */
    public static final int ONIGERR_TOO_SHORT_MULTI_BYTE_STRING = -206;
    /** too big backref number */
    public static final int ONIGERR_TOO_BIG_BACKREF_NUMBER = -207;
    /** invalid backref number/name */
    public static final int ONIGERR_INVALID_BACKREF = -208;
    /** numbered backref/call is not allowed. (use name) */
    public static final int ONIGERR_NUMBERED_BACKREF_OR_CALL_NOT_ALLOWED = -209;
    /** too long wide-char value */
    public static final int ONIGERR_TOO_LONG_WIDE_CHAR_VALUE = -212;
    /** group name is empty */
    public static final int ONIGERR_EMPTY_GROUP_NAME = -214;
    /** invalid group name &lt;%n&gt; */
    public static final int ONIGERR_INVALID_GROUP_NAME = -215;
    /** invalid char in group name &lt;%n&gt;</br>(invalid char in group number &lt;%n&gt;) */
    public static final int ONIGERR_INVALID_CHAR_IN_GROUP_NAME = -216;
    /** undefined name &lt;%n&gt; reference */
    public static final int ONIGERR_UNDEFINED_NAME_REFERENCE = -217;
    /** undefined group &lt;%n&gt; reference */
    public static final int ONIGERR_UNDEFINED_GROUP_REFERENCE = -218;
    /** multiplex defined name &lt;%n&gt; */
    public static final int ONIGERR_MULTIPLEX_DEFINED_NAME = -219;
    /** multiplex definition name &lt;%n&gt; call */
    public static final int ONIGERR_MULTIPLEX_DEFINITION_NAME_CALL = -220;
    /** never ending recursion */
    public static final int ONIGERR_NEVER_ENDING_RECURSION = -221;
    /** group number is too big for capture history */
    public static final int ONIGERR_GROUP_NUMBER_OVER_FOR_CAPTURE_HISTORY = -222;
    /** invalid character property name {%n} */
    public static final int ONIGERR_INVALID_CHAR_PROPERTY_NAME = -223;
    /** invalid code point value */
    public static final int ONIGERR_INVALID_CODE_POINT_VALUE = -400;
    /** UNKOWN */
    public static final int ONIGERR_INVALID_WIDE_CHAR_VALUE = -400;
    /** too big wide-char value */
    public static final int ONIGERR_TOO_BIG_WIDE_CHAR_VALUE = -401;
    /** not supported encoding combination */
    public static final int ONIGERR_NOT_SUPPORTED_ENCODING_COMBINATION = -402;
    /** invalid combination of options */
    public static final int ONIGERR_INVALID_COMBINATION_OF_OPTIONS = -403;
    
    /* errors related to thread */
    /** over thread pass limit count */
    public static final int ONIGERR_OVER_THREAD_PASS_LIMIT_COUNT = -1001;
    
    private static native boolean useUnicodeLittleUnmarked();
    private static native String onig_error_code_to_str(int ecode, String param);
    private native int onig_new(String pattern, int option, OnigSyntaxType syntax, String[] errmsg);
    private native int onig_new_deluxe(String pattern, int option, OnigSyntaxType syntax, int case_flag, String[] errmsg);
    private native void onig_free();
    private native boolean onig_noname_group_capture_is_active();
    private native int onig_search(String str, int end, int start, int range, OnigRegion region, int option);
    private native int onig_match(String str, int end, int at, OnigRegion region, int option);
    private native int onig_split(String str, int limit, int options, Collection<String> list);
    private native int onig_name_to_backref_number(String name, OnigRegion region);
    private native int[] onig_name_to_group_numbers(String name);
    private native int onig_number_of_names();
    private native String onig_get_encoding_name();
    private native int onig_get_options();
    private native int onig_get_case_fold_flag();
//    private native OnigSyntaxType onig_get_syntax();
    private native int onig_number_of_captures();
    private native int onig_number_of_capture_histories();
    // MEMO: OnigRegionのコンストラクタが呼ばれた際にstaticイニシャライザでJNIをロードさせるためにOnigRegexのメソッドとして定義している
    static native long onig_region_new();
    static native int onig_get_default_case_fold_flag();
    static native int onig_set_default_case_fold_flag(int flag);
    private static native int onig_get_match_stack_limit_size();
    private static native int onig_set_match_stack_limit_size(int size);
    private static native String onig_version();
    private static native String onig_copyright();
    private static native boolean onig_set_warn_func();
    private static native void onig_clear_warn_func();
    private static native boolean onig_set_verb_warn_func();
    private static native void onig_clear_verb_warn_func();
    private native int onig_foreach_name(Callback callback);

    // MEMO: OnigSyntaxTypeがロードされる際にstaticイニシャライザでJNIをロードさせるためにOnigRegexのメソッドとして定義している
    static native long ONIG_SYNTAX_ASIS();
    static native long ONIG_SYNTAX_POSIX_BASIC();
    static native long ONIG_SYNTAX_POSIX_EXTENDED();
    static native long ONIG_SYNTAX_EMACS();
    static native long ONIG_SYNTAX_GREP();
    static native long ONIG_SYNTAX_GNU_REGEX();
    static native long ONIG_SYNTAX_JAVA();
    static native long ONIG_SYNTAX_PERL();
    static native long ONIG_SYNTAX_PERL_NG();
    static native long ONIG_SYNTAX_RUBY();
    static native long ONIG_SYNTAX_DEFAULT();
}
