package onig4j;

import static onig4j.OnigRegex.onig_region_new;

/**
 * Oniguruma region class
 * @author calico
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.txt">Oniguruma API</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.ja.txt">鬼車インターフェース</a>
 */
public class OnigRegion extends OnigHandle implements Cloneable {

    private volatile OnigCaptureTreeNode node;
    
    /**
     * invoke onig_region_new() function.
     */
    public OnigRegion() {
        this.handle = onig_region_new();
    }
    
    /**
     * invoke onig_region_free(1) function.
     */
    @Override
    protected final void free() {
        onig_region_free(1);
        if (node != null) {
            node.close();
            node = null;
        }
    }
    
    /**
     * invoke onig_region_clear() function.
     * @see #resizeClear(int)
     */
    public void clear() {
        onig_region_clear();
        if (node != null) {
            node.close();
            node = null;
        }
    }
    
    /**
     * invoke onig_region_copy() function.
     * @return a clone of this instance
     */
    @Override
    public OnigRegion clone() {
        final OnigRegion clone = new OnigRegion();
        onig_region_copy(clone);
        if (node != null) {
            clone.node = clone.onig_get_capture_tree();
        }
        return clone;
    }
    
    /**
     * invoke onig_get_capture_tree() function.
     * @return The root node of capture history data tree
     * @see onig4j.OnigSyntaxType#isActiveCaputureHistory()
     */
    public OnigCaptureTreeNode getCaptureTreeNode() {
        return node;
    }
    
    /**
     * invoke onig_region_resize() funcation.
     * @param n The new size
     * @see #resizeClear(int)
     */
    public void resize(int n) {
        final int ret = onig_region_resize(n);
        if (ret != OnigRegex.ONIG_NORMAL) {
            OnigRegex.throwOnigError(ret);
        }
    }
    
    /**
     * invoke onig_region_resize() and onig_region_clear() funcation.
     * @param n The new size
     */
    public void resizeClear(int n) {
        final int ret = onig_region_resize_clear(n);
        if (ret != OnigRegex.ONIG_NORMAL) {
            OnigRegex.throwOnigError(ret);
        }
    }
    
    /**
     * Returns the value of OnigRegion.allocated.
     * @return The value of OnigRegion.allocated
     */
    public native int allocated();
    
    /**
     * Returns the value of OnigRegion.num_regs.
     * @return The value of OnigRegion.num_regs
     */
    public native int count();
    
    /**
     * Returns the value of OnigRegion.beg[index].
     * @param index The index value
     * @return The value of OnigRegion.beg[index]
     */
    public native int begin(int index);
    
    /**
     * Returns the value of OnigRegion.end[index].
     * @param index The index value
     * @return The value of OnigRegion.end[index]
     */
    public native int end(int index);
    
    /**
     * Callback interface for onig_capture_tree_traverse() function.
     * @see onig4j.OnigRegion#traverseCaptureTree(onig4j.OnigRegion.Callback, int)
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.txt">Oniguruma API</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/API.ja.txt">鬼車インターフェース</a>
     */
    public static interface Callback {
        /**
         * Called from onig_capture_tree_traverse() function.
         * @param group The group number
         * @param begin The capture start position
         * @param end The capture end position
         * @param level The nest level (from 0)
         * @param at The callback position (ONIG_TRAVERSE_CALLBACK_AT_FIRST or ONIG_TRAVERSE_CALLBACK_AT_LAST)
         * @return if func does not return 0, then traverse is stopped
         * @see #ONIG_TRAVERSE_CALLBACK_AT_FIRST
         * @see #ONIG_TRAVERSE_CALLBACK_AT_LAST
         */
        int call(int group, int begin, int end, int level, int at);
    }
    
    /**
     * invoke onig_capture_tree_traverse() function.
     * @param callback The callback object
     * @param at The callback position (ONIG_TRAVERSE_CALLBACK_AT_FIRST or ONIG_TRAVERSE_CALLBACK_AT_LAST or ONIG_TRAVERSE_CALLBACK_AT_BOTH)
     * @return The callback object return value
     */
    public int traverseCaptureTree(Callback callback, int at) {
        return onig_capture_tree_traverse(callback, at);
    }
    
    /** callback first, then traverse children */
    public final static int ONIG_TRAVERSE_CALLBACK_AT_FIRST = 1;
    /** traverse children first, then callback */
    public final static int ONIG_TRAVERSE_CALLBACK_AT_LAST = 2;
    /** callback first, then traverse children, and at last callback again */
    public final static int ONIG_TRAVERSE_CALLBACK_AT_BOTH
            = (ONIG_TRAVERSE_CALLBACK_AT_FIRST | ONIG_TRAVERSE_CALLBACK_AT_LAST);
    /** mismatched position value of region */
    public final static int ONIG_REGION_NOTPOS = -1;

    private native void onig_region_free(int free_self);
    private native void onig_region_clear();
    private native void onig_region_copy(OnigRegion to);
    private native int onig_region_resize(int n);
    private native int onig_region_resize_clear(int n);
    private native OnigCaptureTreeNode onig_get_capture_tree();
    private native int onig_capture_tree_traverse(Callback callback, int at);
}
