package onig4j;

import java.util.Collection;
import static onig4j.OnigRegex.ONIG_NORMAL;
import static onig4j.OnigRegex.ONIG_SYNTAX_ASIS;
import static onig4j.OnigRegex.ONIG_SYNTAX_POSIX_BASIC;
import static onig4j.OnigRegex.ONIG_SYNTAX_POSIX_EXTENDED;
import static onig4j.OnigRegex.ONIG_SYNTAX_EMACS;
import static onig4j.OnigRegex.ONIG_SYNTAX_GREP;
import static onig4j.OnigRegex.ONIG_SYNTAX_GNU_REGEX;
import static onig4j.OnigRegex.ONIG_SYNTAX_JAVA;
import static onig4j.OnigRegex.ONIG_SYNTAX_PERL;
import static onig4j.OnigRegex.ONIG_SYNTAX_PERL_NG;
import static onig4j.OnigRegex.ONIG_SYNTAX_RUBY;
import static onig4j.OnigRegex.ONIG_SYNTAX_DEFAULT;

/**
 * Pattern syntax class.
 * @author calico
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/">Oniguruma</a>
 * @see <a href="http://www.geocities.jp/kosako3/oniguruma/index_ja.html">鬼車</a>
 */
public final class OnigSyntaxType extends OnigHandle implements Cloneable {
    
    /** plain text */
    public static final OnigSyntaxType ONIG_SYNTAX_ASIS = new OnigSyntaxType(ONIG_SYNTAX_ASIS());
    /** POSIX Basic RE */
    public static final OnigSyntaxType ONIG_SYNTAX_POSIX_BASIC = new OnigSyntaxType(ONIG_SYNTAX_POSIX_BASIC());
    /** POSIX Extended RE */
    public static final OnigSyntaxType ONIG_SYNTAX_POSIX_EXTENDED = new OnigSyntaxType(ONIG_SYNTAX_POSIX_EXTENDED());
    /** Emacs */
    public static final OnigSyntaxType ONIG_SYNTAX_EMACS = new OnigSyntaxType(ONIG_SYNTAX_EMACS());
    /** grep */
    public static final OnigSyntaxType ONIG_SYNTAX_GREP = new OnigSyntaxType(ONIG_SYNTAX_GREP());
    /** GNU regex */
    public static final OnigSyntaxType ONIG_SYNTAX_GNU_REGEX = new OnigSyntaxType(ONIG_SYNTAX_GNU_REGEX());
    /**
     * Java (Sun java.util.regex)<br/>
     * NOTE: If the Oniruguma extended version 5.9.1.1 or later is used, the ONIG_SYN_OP2_OPTION_JAVA, the ONIG_OPTION_JAVA_LINES and the ONIG_OPTION_CASE_FOLD_ASCII_ONLY are added.
     * @see #ONIG_SYN_OP2_OPTION_JAVA
     * @see OnigOptionType#ONIG_OPTION_JAVA_LINES
     * @see OnigOptionType#ONIG_OPTION_CASE_FOLD_ASCII_ONLY
     */
    public static final OnigSyntaxType ONIG_SYNTAX_JAVA = new OnigSyntaxType(ONIG_SYNTAX_JAVA());
    /** Perl */
    public static final OnigSyntaxType ONIG_SYNTAX_PERL = new OnigSyntaxType(ONIG_SYNTAX_PERL());
    /** Perl + named group */
    public static final OnigSyntaxType ONIG_SYNTAX_PERL_NG = new OnigSyntaxType(ONIG_SYNTAX_PERL_NG());
    /** Ruby */
    public static final OnigSyntaxType ONIG_SYNTAX_RUBY = new OnigSyntaxType(ONIG_SYNTAX_RUBY());
    /**  default (== Ruby) */
    public static final OnigSyntaxType ONIG_SYNTAX_DEFAULT = ONIG_SYNTAX_RUBY;
    
    /**
     * Allocates a OnigSyntaxType object.<br/>
     * NOTE: All fields have been initialized by 0.
     * @see #clone()
     */
    public OnigSyntaxType() {
        this(onig_syntax_new());
    }
    
    /**
     * Allocates a OnigSyntaxType object with the handle argument.
     * @param handle The pointer of OnigSyntaxType
     * @see #clone()
     */
    private OnigSyntaxType(long handle) {
        this.handle = handle;
    }
    
    /**
     * invoke onig_copy_syntax() function.
     * @return a clone of this instance
     */
    @Override
    public OnigSyntaxType clone() {
        return onig_copy_syntax();
    }

    /**
     * invoke free() function.<br/>
     * NOTE: Ignored if the built-in type.
     */
    protected native void free();

    /**
     * Returns the capture histories activity.
     * @return true if the capture histories is active in the pattern syntax
     * @see #ONIG_SYN_OP2_ATMARK_CAPTURE_HISTORY
     */
    public boolean isActiveCaputureHistory() {
        return ((onig_get_syntax_op2() & ONIG_SYN_OP2_ATMARK_CAPTURE_HISTORY) != 0);
    }
    
    /**
     * invoke onig_get_syntax_op() function.
     * @return The return value of onig_get_syntax_op() function
     */
    public int getOperator() {
        return onig_get_syntax_op();
    }
    
    /**
     * invoke onig_get_syntax_op2() function.
     * @return The return value of onig_get_syntax_op2() function
     */
    public int getOperator2() {
        return onig_get_syntax_op2();
    }
    
    /**
     * invoke onig_get_syntax_behavior() function.
     * @return The return value of onig_get_syntax_behavior() function
     */
    public int getBehavior() {
        return onig_get_syntax_behavior();
    }
    
    /**
     * invoke onig_get_syntax_options() function.
     * @return The return value of onig_get_syntax_options() function
     */
    public int getOptions() {
        return onig_get_syntax_options();
    }
    
    /**
     * invoke onig_set_syntax_op() function.
     * NOTE: Ignored if the built-in type.
     * @param op The parameter value
     */
    public void setOperator(int op) {
        onig_set_syntax_op(op);
    }
    
    /**
     * invoke onig_set_syntax_op2() function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param op2 The parameter value
     */
    public void setOperator2(int op2) {
        onig_set_syntax_op2(op2);
    }
    
    /**
     * invoke onig_set_syntax_behavior() function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param behavior The parameter value
     */
    public void setBehavior(int behavior) {
        onig_set_syntax_behavior(behavior);
    }
    
    /**
     * invoke onig_set_syntax_options() function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void setOptions(int options) {
        onig_set_syntax_options(options);
    }
    
    /**
     * invoke onig_set_syntax_options() function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void setOptions(OnigOptionType... options) {
        onig_set_syntax_options(OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_set_syntax_options() function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void setOptions(Collection<OnigOptionType> options) {
        onig_set_syntax_options(OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_set_syntax_op(onig_get_syntax_op() | op) function.
     * NOTE: Ignored if the built-in type.
     * @param op The parameter value
     */
    public void addOperator(int op) {
        onig_add_syntax_op(op);
    }
    
    /**
     * invoke onig_set_syntax_op2(onig_get_syntax_op2() | op2) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param op2 The parameter value
     */
    public void addOperator2(int op2) {
        onig_add_syntax_op2(op2);
    }
    
    /**
     * invoke onig_set_syntax_behavior(onig_get_syntax_behavior() | behavior) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param behavior The parameter value
     */
    public void addBehavior(int behavior) {
        onig_add_syntax_behavior(behavior);
    }
    
    /**
     * invoke onig_set_syntax_options(onig_get_syntax_options() | options) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void addOptions(int options) {
        onig_add_syntax_options(options);
    }
    
    /**
     * invoke onig_set_syntax_options(onig_get_syntax_options() | options) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void addOptions(OnigOptionType... options) {
        onig_add_syntax_options(OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_set_syntax_options(onig_get_syntax_options() | options) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void addOptions(Collection<OnigOptionType> options) {
        onig_add_syntax_options(OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_set_syntax_op(onig_get_syntax_op() & ~op) function.
     * NOTE: Ignored if the built-in type.
     * @param op The parameter value
     */
    public void removeOperator(int op) {
        onig_remove_syntax_op(op);
    }
    
    /**
     * invoke onig_set_syntax_op2(onig_get_syntax_op2() & ~op2) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param op2 The parameter value
     */
    public void removeOperator2(int op2) {
        onig_remove_syntax_op2(op2);
    }
    
    /**
     * invoke onig_set_syntax_behavior(onig_get_syntax_behavior() & ~behavior) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param behavior The parameter value
     */
    public void removeBehavior(int behavior) {
        onig_remove_syntax_behavior(behavior);
    }
    
    /**
     * invoke onig_set_syntax_options(onig_get_syntax_options() & ~options) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void removeOptions(int options) {
        onig_remove_syntax_options(options);
    }
    
    /**
     * invoke onig_set_syntax_options(onig_get_syntax_options() & ~options) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void removeOptions(OnigOptionType... options) {
        onig_remove_syntax_options(OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_set_syntax_options(onig_get_syntax_options() & ~options) function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param options The parameter value
     */
    public void removeOptions(Collection<OnigOptionType> options) {
        onig_remove_syntax_options(OnigOptionType.values(options));
    }
    
    /**
     * invoke onig_set_meta_char() function.<br/>
     * NOTE: Ignored if the built-in type.
     * @param what The specifies which meta character it is. (ONIG_META_CHAR_ESCAPE, ONIG_META_CHAR_ANYCHAR, ONIG_META_CHAR_ANYTIME, ONIG_META_CHAR_ZERO_OR_ONE_TIME, ONIG_META_CHAR_ONE_OR_MORE_TIME, ONIG_META_CHAR_ANYCHAR_ANYTIME)
     * @param codePoint The code point of meta character or ONIG_INEFFECTIVE_META_CHAR
     * @see #ONIG_META_CHAR_ESCAPE
     * @see #ONIG_META_CHAR_ANYCHAR
     * @see #ONIG_META_CHAR_ANYTIME
     * @see #ONIG_META_CHAR_ZERO_OR_ONE_TIME
     * @see #ONIG_META_CHAR_ONE_OR_MORE_TIME
     * @see #ONIG_META_CHAR_ANYCHAR_ANYTIME
     * @see #ONIG_INEFFECTIVE_META_CHAR
     * @see java.lang.Character
     */
    public void setMetaChar(int what, int codePoint) {
        final int ret = onig_set_meta_char(what, codePoint);
        if (ret != ONIG_NORMAL) {
            throw new IllegalArgumentException(OnigRegex.getErrorMessage(ret, null));
        }
    }
    
    /**
     * Returns the ONIG_SYNTAX_DEFAULT object.
     * @return The default syntax object
     */
    public static OnigSyntaxType getDefaultSyntax() {
        return new OnigSyntaxType(ONIG_SYNTAX_DEFAULT());
    }
    
    /**
     * invoke onig_set_default_syntax() function.<br/>
     * When the parameter is null, the default value is reset.
     * @param syntax The OnigSyntaxType object (null is allowed)
     */
    public static void setDefaultSyntax(OnigSyntaxType syntax) {
        final int ret = onig_set_default_syntax(syntax);
        if (ret != ONIG_NORMAL) {
            throw new IllegalArgumentException(OnigRegex.getErrorMessage(ret, null));
        }
    }

    static OnigSyntaxType[] values() {
        return new OnigSyntaxType[] {
                    ONIG_SYNTAX_ASIS,
                    ONIG_SYNTAX_POSIX_BASIC,
                    ONIG_SYNTAX_POSIX_EXTENDED,
                    ONIG_SYNTAX_EMACS,
                    ONIG_SYNTAX_GREP,
                    ONIG_SYNTAX_GNU_REGEX,
                    ONIG_SYNTAX_JAVA,
                    ONIG_SYNTAX_PERL,
                    ONIG_SYNTAX_PERL_NG,
                    ONIG_SYNTAX_RUBY,
                };
    }
    
    private static native long onig_syntax_new();
    private native OnigSyntaxType onig_copy_syntax();
    private static native int onig_set_default_syntax(OnigSyntaxType syntax);

    private native int onig_get_syntax_op();
    private native int onig_get_syntax_op2();
    private native int onig_get_syntax_behavior();
    private native int onig_get_syntax_options();

    private native void onig_set_syntax_op(int op);
    private native void onig_set_syntax_op2(int op2);
    private native void onig_set_syntax_behavior(int behavior);
    private native void onig_set_syntax_options(int options);
    private native int onig_set_meta_char(int what, int codePoint);

    private native void onig_add_syntax_op(int op);
    private native void onig_add_syntax_op2(int op2);
    private native void onig_add_syntax_behavior(int behavior);
    private native void onig_add_syntax_options(int options);

    private native void onig_remove_syntax_op(int op);
    private native void onig_remove_syntax_op2(int op2);
    private native void onig_remove_syntax_behavior(int behavior);
    private native void onig_remove_syntax_options(int options);
    
    native boolean isBuiltIn();
    
    /* syntax (operators) */
    public final static int ONIG_SYN_OP_VARIABLE_META_CHARACTERS = (1 << 0);
    /** . */
    public final static int ONIG_SYN_OP_DOT_ANYCHAR = (1 << 1);
    /** * */
    public final static int ONIG_SYN_OP_ASTERISK_ZERO_INF = (1 << 2);
    public final static int ONIG_SYN_OP_ESC_ASTERISK_ZERO_INF = (1 << 3);
    /** + */
    public final static int ONIG_SYN_OP_PLUS_ONE_INF = (1 << 4);
    public final static int ONIG_SYN_OP_ESC_PLUS_ONE_INF = (1 << 5);
    /** ? */
    public final static int ONIG_SYN_OP_QMARK_ZERO_ONE = (1 << 6);
    public final static int ONIG_SYN_OP_ESC_QMARK_ZERO_ONE = (1 << 7);
    /** {lower,upper} */
    public final static int ONIG_SYN_OP_BRACE_INTERVAL = (1 << 8);
    /** \{lower,upper\} */
    public final static int ONIG_SYN_OP_ESC_BRACE_INTERVAL = (1 << 9);
    /** | */
    public final static int ONIG_SYN_OP_VBAR_ALT = (1 << 10);
    /** \| */
    public final static int ONIG_SYN_OP_ESC_VBAR_ALT = (1 << 11);
    /** (...) */
    public final static int ONIG_SYN_OP_LPAREN_SUBEXP = (1 << 12);
    /** \(...\) */
    public final static int ONIG_SYN_OP_ESC_LPAREN_SUBEXP = (1 << 13);
    /** \A, \Z, \z */
    public final static int ONIG_SYN_OP_ESC_AZ_BUF_ANCHOR = (1 << 14);
    /** \G */
    public final static int ONIG_SYN_OP_ESC_CAPITAL_G_BEGIN_ANCHOR = (1 << 15);
    /** \num */
    public final static int ONIG_SYN_OP_DECIMAL_BACKREF = (1 << 16);
    /** [...] */
    public final static int ONIG_SYN_OP_BRACKET_CC = (1 << 17);
    /** \w, \W */
    public final static int ONIG_SYN_OP_ESC_W_WORD = (1 << 18);
    /** {@literal \<. \>} */
    public final static int ONIG_SYN_OP_ESC_LTGT_WORD_BEGIN_END = (1 << 19);
    /** \b, \B */
    public final static int ONIG_SYN_OP_ESC_B_WORD_BOUND = (1 << 20);
    /** \s, \S */
    public final static int ONIG_SYN_OP_ESC_S_WHITE_SPACE = (1 << 21);
    /** \d, \D */
    public final static int ONIG_SYN_OP_ESC_D_DIGIT = (1 << 22);
    /** ^, $ */
    public final static int ONIG_SYN_OP_LINE_ANCHOR = (1 << 23);
    /** [:xxxx:] */
    public final static int ONIG_SYN_OP_POSIX_BRACKET = (1 << 24);
    /** ??, *?, +?, {n,m}? */
    public final static int ONIG_SYN_OP_QMARK_NON_GREEDY = (1 << 25);
    /** \n, \r, \t, \a ... */
    public final static int ONIG_SYN_OP_ESC_CONTROL_CHARS = (1 << 26);
    /** \cx */
    public final static int ONIG_SYN_OP_ESC_C_CONTROL = (1 << 27);
    /** \OOO */
    public final static int ONIG_SYN_OP_ESC_OCTAL3 = (1 << 28);
    /** \xHH */
    public final static int ONIG_SYN_OP_ESC_X_HEX2 = (1 << 29);
    /* \x{7HHHHHHH} */
    public final static int ONIG_SYN_OP_ESC_X_BRACE_HEX8 = (1 << 30);
    
    /** \Q...\E */
    public final static int ONIG_SYN_OP2_ESC_CAPITAL_Q_QUOTE = (1 << 0);
    /** (?...) */
    public final static int ONIG_SYN_OP2_QMARK_GROUP_EFFECT = (1 << 1);
    /** (?imsx), (?-imsx) */
    public final static int ONIG_SYN_OP2_OPTION_PERL = (1 << 2);
    /** (?imx), (?-imx)  */
    public final static int ONIG_SYN_OP2_OPTION_RUBY = (1 << 3);
    /** ?+, *+, ++ */
    public final static int ONIG_SYN_OP2_PLUS_POSSESSIVE_REPEAT = (1 << 4);
    /** {n,m}+ */
    public final static int ONIG_SYN_OP2_PLUS_POSSESSIVE_INTERVAL = (1 << 5);
    /** [...&&..[..]..] */
    public final static int ONIG_SYN_OP2_CCLASS_SET_OP = (1 << 6);
    /** {@literal (?<name>...)} */
    public final static int ONIG_SYN_OP2_QMARK_LT_NAMED_GROUP = (1 << 7);
    /** {@literal \k<name>} */
    public final static int ONIG_SYN_OP2_ESC_K_NAMED_BACKREF = (1 << 8);
    /** {@literal \g<name>, \g<n>} */
    public final static int ONIG_SYN_OP2_ESC_G_SUBEXP_CALL = (1 << 9);
    /** {@literal (?@..),(?@<x>..)} */
    public final static int ONIG_SYN_OP2_ATMARK_CAPTURE_HISTORY = (1 << 10);
    /** \C-x */
    public final static int ONIG_SYN_OP2_ESC_CAPITAL_C_BAR_CONTROL = (1 << 11);
    /** \M-x */
    public final static int ONIG_SYN_OP2_ESC_CAPITAL_M_BAR_META = (1 << 12);
    /** \v as VTAB */
    public final static int ONIG_SYN_OP2_ESC_V_VTAB = (1 << 13);
    /** \\uHHHH */
    public final static int ONIG_SYN_OP2_ESC_U_HEX4 = (1 << 14);
    /** \`, \' */
    public final static int ONIG_SYN_OP2_ESC_GNU_BUF_ANCHOR = (1 << 15);
    /** \p{...}, \P{...} */
    public final static int ONIG_SYN_OP2_ESC_P_BRACE_CHAR_PROPERTY = (1 << 16);
    /** \p{^..}, \P{^..} */
    public final static int ONIG_SYN_OP2_ESC_P_BRACE_CIRCUMFLEX_NOT = (1 << 17);
    /** \h, \H */
    public final static int ONIG_SYN_OP2_ESC_H_XDIGIT = (1 << 19);
    /** \ */
    public final static int ONIG_SYN_OP2_INEFFECTIVE_ESCAPE = (1 << 20);
    /**
     * (?idmsux), (?-idmsux)<br/>
     * NOTE: This is an original option. If it wants to use this option, the Oniruguma extended version 5.9.1.1 or later is required. (Not exist in the Oniguruma version 5.9.1)
     */
    public final static int ONIG_SYN_OP2_OPTION_JAVA = (1 << 31);
    
    /* syntax (behavior) */
    /** not implemented */
    public final static int ONIG_SYN_CONTEXT_INDEP_ANCHORS = (1 << 31);
    /** ?, *, +, {n,m} */
    public final static int ONIG_SYN_CONTEXT_INDEP_REPEAT_OPS = (1 << 0);
    /** error or ignore */
    public final static int ONIG_SYN_CONTEXT_INVALID_REPEAT_OPS = (1 << 1);
    /** ...)... */
    public final static int ONIG_SYN_ALLOW_UNMATCHED_CLOSE_SUBEXP = (1 << 2);
    /** {??? */
    public final static int ONIG_SYN_ALLOW_INVALID_INTERVAL = (1 << 3);
    /** {,n} => {0,n} */
    public final static int ONIG_SYN_ALLOW_INTERVAL_LOW_ABBREV = (1 << 4);
    /** /(\1)/,/\1()/ .. */
    public final static int ONIG_SYN_STRICT_CHECK_BACKREF = (1 << 5);
    /** {@literal (?<=a|bc)} */
    public final static int ONIG_SYN_DIFFERENT_LEN_ALT_LOOK_BEHIND = (1 << 6);
    /**
     * see doc/RE
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/RE.txt">Oniguruma Regular Expressions</a>
     * @see <a href="http://www.geocities.jp/kosako3/oniguruma/doc/RE.ja.txt">鬼車 正規表現</a>
     */
    public final static int ONIG_SYN_CAPTURE_ONLY_NAMED_GROUP = (1 << 7);
    /* {@literal (?<x>)(?<x>)} */
    public final static int ONIG_SYN_ALLOW_MULTIPLEX_DEFINITION_NAME = (1 << 8);
    /** a{n}?=(?:a{n})? */
    public final static int ONIG_SYN_FIXED_INTERVAL_IS_GREEDY_ONLY = (1 << 9);
    
    /* syntax (behavior) in char class [...] */
    /** [^...] */
    public final static int ONIG_SYN_NOT_NEWLINE_IN_NEGATIVE_CC = (1 << 20);
    /** [..\w..] etc.. */
    public final static int ONIG_SYN_BACKSLASH_ESCAPE_IN_CC = (1 << 21);
    public final static int ONIG_SYN_ALLOW_EMPTY_RANGE_IN_CC = (1 << 22);
    /** [0-9-a]=[0-9\-a] */
    public final static int ONIG_SYN_ALLOW_DOUBLE_RANGE_OP_IN_CC = (1 << 23);
    
    /* syntax (behavior) warning */
    /** [,-,] */
    public final static int ONIG_SYN_WARN_CC_OP_NOT_ESCAPED = (1 << 24);
    /** (?:a*)+ */
    public final static int ONIG_SYN_WARN_REDUNDANT_NESTED_REPEAT = (1 << 25);
    
    /* meta character specifiers (onig_set_meta_char()) */
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_META_CHAR_ESCAPE = 0;
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_META_CHAR_ANYCHAR = 1;
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_META_CHAR_ANYTIME = 2;
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_META_CHAR_ZERO_OR_ONE_TIME = 3;
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_META_CHAR_ONE_OR_MORE_TIME = 4;
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_META_CHAR_ANYCHAR_ANYTIME = 5;
    /**
     * @see #setMetaChar(int, int)
     */
    public final static int ONIG_INEFFECTIVE_META_CHAR = 0;
}
