package onig4j;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;
import onig4j.OnigRegex.Callback;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class OnigRegexTest {

    @Test
    public void newOnigRegexDefault() {
        OnigRegex regex = new OnigRegex("a(.*)b|[e-f]+");
        regex.close();
    }

    @Test
    public void newOnigRegexJava() {
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        regex.close();
    }

    @Test(expected = java.util.regex.PatternSyntaxException.class)
    public void newOnigRegexThrowPatternSyntaxException() {
        try {
            OnigRegex regex = new OnigRegex("\\p{InGreek}");
            regex.close();
            
        } catch (RuntimeException ex) {
            System.out.println(ex.getMessage());
            assertNotNull(ex.getMessage());
            throw ex;
        }
    }
    
    @Test
    public void getBackRefNumber() {
        OnigRegex regex = new OnigRegex("(?<foo>a*)(?<bar>b*)(?<foo>c*)");
        
        int num = regex.getBackRefNumber("foo", null);
        assertEquals(3, num);

        num = regex.getBackRefNumber("bar", null);
        assertEquals(2, num);

        regex.close();
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void getBackRefNumberThrowIllegalArgumentException() {
        OnigRegex regex = new OnigRegex("(?<foo>a*)(?<bar>b*)(?<foo>c*)");
        
        int num = regex.getBackRefNumber("buz", null);

        regex.close();
    }

    @Test
    public void getGroupNumbers() {
        OnigRegex regex = new OnigRegex("(?<foo>a*)(?<bar>b*)(?<foo>c*)");
        
        int[] list = regex.getGroupNumbers("foo");
        assertEquals(2, list.length);
        assertEquals(1, list[0]);
        assertEquals(3, list[1]);

        list = regex.getGroupNumbers("bar");
        assertEquals(1, list.length);
        assertEquals(2, list[0]);

        list = regex.getGroupNumbers("buz");
        assertEquals(0, list.length);
        
        regex.close();
    }
    
    @Test
    public void getEncoding() {
        final String pattern = "a(.*)b|[e-f]+";
        OnigRegex regex = new OnigRegex(pattern);
        assertEquals(Charset.forName("UTF-16LE"), regex.getEncoding());
        regex.close();
    }
    
    @Test
    public void getOptionType() {
        final String pattern = "a(.*)b|[e-f]+";
        OnigRegex regex = null;
        OnigOptionType option = OnigOptionType.ONIG_OPTION_NONE;
        EnumSet<OnigOptionType> options = null;
        
        regex = new OnigRegex(pattern);
        options = regex.getOptionType();
        assertEquals(1, options.size());
        assertTrue(options.contains(option));
        regex.close();

        option = OnigOptionType.ONIG_OPTION_EXTEND;
        regex = new OnigRegex(pattern, option);
        options = regex.getOptionType();
        assertEquals(1, options.size());
        assertTrue(options.contains(option));
        regex.close();

        option = OnigOptionType.ONIG_OPTION_MULTILINE;
        regex = new OnigRegex(pattern, option);
        options = regex.getOptionType();
        assertEquals(1, options.size());
        assertTrue(options.contains(option));
        regex.close();
    }
    
    @Test
    public void getCaseFoldType() {
        final String pattern = "a(.*)b|[e-f]+";
        OnigRegex regex = null;
        
        regex = new OnigRegex(pattern);
        assertTrue(regex.getCaseFoldType().contains(OnigCaseFoldType.ONIGENC_CASE_FOLD_MIN));
        regex.close();
    }
    
    @Test
    public void getSyntaxType() {
        final String pattern = "a(.*)b|[e-f]+";
        OnigSyntaxType syntax = null;
        OnigRegex regex = null;
        
        regex = new OnigRegex(pattern);
        syntax = regex.getSyntaxType();
        assertEquals(OnigSyntaxType.ONIG_SYNTAX_DEFAULT, syntax);
        
        regex = new OnigRegex(pattern, OnigSyntaxType.ONIG_SYNTAX_JAVA);
        syntax = regex.getSyntaxType();
        
        assertEquals(OnigSyntaxType.ONIG_SYNTAX_JAVA, syntax);
        
        regex.close();
    }
    
    @Test
    public void search() {
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        OnigRegion region = new OnigRegion();
        int ret = OnigRegex.ONIG_MISMATCH;
        
        System.out.println("node is " + region.getCaptureTreeNode());
        
//        ret = regex.search("abcdEfg", region, (1 << 10));
        ret = regex.search("abcdEfg", region);
        assertEquals(4, ret);

        System.out.printf("allocated is %d\n", region.allocated());
        System.out.printf("count is %d\n", region.count());
        
        System.out.printf("onig_number_of_capture_histories is %d\n", regex.captureHistoryCount());
        if (regex.captureHistoryCount() > 0) {
            OnigCaptureTreeNode root = region.getCaptureTreeNode();
        }
        
        System.out.printf("match at %d\n", ret);
        for (int i = 0; i < region.count(); i++) {
            System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        }
        
        ret = regex.search("abcdefg", region);
        assertEquals(OnigRegex.ONIG_MISMATCH, ret);
        
        regex.close();
        region.close();
    }

    @Test
    public void searchLineTerminator() {
        final String pattern = "$";
        final String input = "い\rろ\nは\r\nに\u0085ほ\u2028へ\u2029と";
        
        final Collection<OnigOptionType> options
                = EnumSet.of(
                        OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE
                    );
        final OnigRegex regex
                = new OnigRegex(pattern, OnigSyntaxType.ONIG_SYNTAX_JAVA, options);

        assertEquals(1, regex.search(input, 0));
        assertEquals(1, regex.search(input, 1));
        assertEquals(3, regex.search(input, 2));
        assertEquals(3, regex.search(input, 3));
        assertEquals(5, regex.search(input, 4));
        assertEquals(5, regex.search(input, 5));
        assertEquals(8, regex.search(input, 6));
        // MEMO 鬼車改造版でもJavaの正規表現と同じくCR＋LFのLFにはマッチしない
        assertEquals(8, regex.search(input, 7));
        assertEquals(8, regex.search(input, 8));
        assertEquals(10, regex.search(input, 9));
        assertEquals(10, regex.search(input, 10));
        assertEquals(12, regex.search(input, 11));
        assertEquals(12, regex.search(input, 12));
        assertEquals(14, regex.search(input, 13));
        assertEquals(14, regex.search(input, 14));
        
        regex.close();
    }

    @Test
    public void match() {
        OnigRegex regex = new OnigRegex("a(.*)b|[e-f]+");
        OnigRegion region = new OnigRegion();
        int ret = OnigRegex.ONIG_MISMATCH;

        ret = regex.search("zzzzaffffffffb", region);
        assertEquals(4, ret);
        
        ret = regex.match("zzzzaffffffffb", 4, region);
        assertEquals(10, ret);

        System.out.printf("allocated at %d\n", region.allocated());
        System.out.printf("match bytes is %d\n", ret);
        for (int i = 0; i < region.count(); i++) {
            System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        }
        
        regex.close();
        region.close();
    }

    @Test
    public void matchWithFindLongest() {
        final String regex = "foo(?:bar)?(?:barbaz)?";
        final String input = "foobarbaz";

        OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_DEFAULT);
        OnigRegion region = new OnigRegion();
        int ret = reg.search(input, region);
        assertEquals(0, ret);
        System.out.println("without ONIG_OPTION_FIND_LONGEST: " + input.substring(region.begin(0), region.end(0)));
        assertEquals(0, region.begin(0));
        assertEquals(6, region.end(0));
        
        reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_DEFAULT, OnigOptionType.ONIG_OPTION_FIND_LONGEST);
        ret = reg.search(input, region);
        assertEquals(0, ret);
        System.out.println("with ONIG_OPTION_FIND_LONGEST: " + input.substring(region.begin(0), region.end(0)));
        assertEquals(0, region.begin(0));
        assertEquals(9, region.end(0));
    }

    @Test
    public void matchLineTerminator() {
        final String pattern = "$";
        final String input = "い\rろ\nは\r\nに\u0085ほ\u2028へ\u2029と";
        
        final java.util.regex.Matcher matcher
                = java.util.regex.Pattern.compile(
                        pattern,
                        java.util.regex.Pattern.MULTILINE
                    ).matcher(input);

        assertFalse(matcher.lookingAt());
        matcher.region(1, input.length());
        assertTrue(matcher.lookingAt());
        matcher.region(2, input.length());
        assertFalse(matcher.lookingAt());
        matcher.region(3, input.length());
        assertTrue(matcher.lookingAt());
        matcher.region(4, input.length());
        assertFalse(matcher.lookingAt());
        matcher.region(5, input.length());
        assertTrue(matcher.lookingAt());
        matcher.region(6, input.length());
        // MEMO Javaの正規表現ライブラリでは行末記号'$'で検索してもCR＋LFのLFにはマッチしない
        assertFalse(matcher.lookingAt());
        matcher.region(7, input.length());
        assertFalse(matcher.lookingAt());
        matcher.region(8, input.length());
        assertTrue(matcher.lookingAt());
        matcher.region(9, input.length());
        assertFalse(matcher.lookingAt());
        matcher.region(10, input.length());
        assertTrue(matcher.lookingAt());
        matcher.region(11, input.length());
        assertFalse(matcher.lookingAt());
        matcher.region(12, input.length());
        assertTrue(matcher.lookingAt());
        matcher.region(13, input.length());
        assertFalse(matcher.lookingAt());
        matcher.region(14, input.length());
        assertTrue(matcher.lookingAt());
        
        final Collection<OnigOptionType> options
                = EnumSet.of(
                        OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE
                    );
        final OnigRegex regex
                = new OnigRegex(pattern, OnigSyntaxType.ONIG_SYNTAX_JAVA, options);

        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 0));
        assertEquals(0, regex.match(input, 1));
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 2));
        assertEquals(0, regex.match(input, 3));
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 4));
        assertEquals(0, regex.match(input, 5));
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 6));
        // MEMO 鬼車改造版でもJavaの正規表現と同じくCR＋LFのLFにはマッチしない
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 7));
        assertEquals(0, regex.match(input, 8));
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 9));
        assertEquals(0, regex.match(input, 10));
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 11));
        assertEquals(0, regex.match(input, 12));
        assertEquals(OnigRegex.ONIG_MISMATCH, regex.match(input, 13));
        assertEquals(0, regex.match(input, 14));
        
        regex.close();
    }
    
    @Test
    public void getVersion() throws IOException {
        final Properties prop = new Properties();
        prop.load(new FileInputStream("nbproject/project.properties"));
        assertEquals(prop.getProperty("product.version"), OnigRegex.getVersion());
    }
    
    @Test
    public void setWarningListener() {
        OnigRegex.clearVerboseWarningListener();
        final OnigRegex.WarningListener listener
                = new OnigRegex.WarningListener() {
                        public void warning(String msg) {
                            Logger.getLogger(OnigRegex.WarningListener.class.getName()).warning(msg);
                            assertNotNull(msg);
                        }
                    };
        assertTrue(OnigRegex.setWarningListener(listener));
        OnigRegex regex = new OnigRegex("あいうえお]");
        regex.close();
    }
    
    @Test
    public void setVerboseWarningListener() {
        OnigRegex.clearWarningListener();
        final OnigRegex.WarningListener listener
                = new OnigRegex.WarningListener() {
                        public void warning(String msg) {
                            Logger.getLogger(OnigRegex.WarningListener.class.getName()).severe(msg);
                            assertNotNull(msg);
                        }
                    };
        assertTrue(OnigRegex.setVerboseWarningListener(listener));
        OnigRegex regex = new OnigRegex("(?:.?)?");
        regex.close();
    }
    
    @Test
    public void onig_new_deluxe() {
        OnigRegex regex = new OnigRegex("[あ-お]", OnigCaseFoldType.ONIGENC_CASE_FOLD_MIN, OnigOptionType.ONIG_OPTION_IGNORECASE);
        
        assertEquals(2, regex.search("abうdefg"));
        
        regex.close();
    }
    
    @Test
    public void captureHistories() {
        OnigRegex regex = new OnigRegex("a(.*)b|[e-f]+");
        OnigRegion region = new OnigRegion();
        int ret = OnigRegex.ONIG_MISMATCH;
        OnigCaptureTreeNode capture = null;
        
        ret = regex.search("abcdefg", region);
        capture = region.getCaptureTreeNode();
        System.out.println("node is " + capture);
        assertNull(capture);
        
        OnigSyntaxType syntax = OnigSyntaxType.ONIG_SYNTAX_DEFAULT.clone();
        syntax.addOperator2(OnigSyntaxType.ONIG_SYN_OP2_ATMARK_CAPTURE_HISTORY);
        
        regex = new OnigRegex("(?@a)*", syntax);
//        ret = regex.search("abcdefg", region, (1 << 10));
        ret = regex.search("abcdefg", region);

        capture = region.getCaptureTreeNode();
        System.out.println("node is " + capture);
        assertNotNull(capture);
        OnigCaptureTreeNode[] children = capture.getChildNodes();
        assertNotNull(children);
        
        System.out.printf("allocated is %d\n", region.allocated());
        System.out.printf("count is %d\n", region.count());
        
        System.out.printf("onig_number_of_capture_histories is %d\n", regex.captureHistoryCount());
        if (regex.captureHistoryCount() > 0) {
            OnigCaptureTreeNode root = region.getCaptureTreeNode();
        }
        
        System.out.printf("match at %d\n", ret);
        for (int i = 0; i < region.count(); i++) {
            System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        }
        
        ret = regex.search("abcdefg", region);
        assertEquals(0, ret);
        
        regex.close();
        region.close();
        syntax.close();
    }
    
    @Test
    public void nameCount() {
        OnigRegex regex = new OnigRegex("a(.*)b|[e-f]+");

        assertEquals(0, regex.nameCount());        
        regex.close();
        
        regex = new OnigRegex("(?<hoge>.*)+");

        assertEquals(1, regex.nameCount());
        
        regex.close();
    }
    
    @Test
    public void foreachName() {
        final OnigRegex regex = new OnigRegex("(?<hogｅ>.*)+(?<ふご>\\h)?(?<hogｅ>.*)?");

        assertEquals(2, regex.nameCount());
        
        final Callback callbakc
                = new OnigRegex.Callback() {
                        public int call(String name, int[] groups, OnigRegex reg) {
                            assertEquals(regex, reg);
                            
                            if ("ふご".equals(name)) {
                                assertEquals(1, groups.length);
                                assertEquals(2, groups[0]);
                                
                            } else {
                                assertEquals("hogｅ", name);
                                assertEquals(2, groups.length);
                                assertEquals(1, groups[0]);
                                assertEquals(3, groups[1]);
                            }
                            return 0;
                        }
                    };
                
        int ret = regex.foreachName(callbakc);
        assertEquals(0, ret);
        
        regex.close();
    }
    
    @Test
    public void comments() {
        final String[] patterns
                = new String[] {
                        "a(?#....\\\\JJJJ)b",
                        "a(?#...\\\\JJJJ)b",
                        "a(?#..\\\\JJJJ)b",
                        "a(?#.\\\\JJJJ)b",
                        "a(?#\\\\JJJJ)b",
                    };
        for (final String pattern : patterns) {
            final OnigRegex regex = new OnigRegex(pattern);
            final OnigRegion region = new OnigRegion();

            int ret = regex.search("ab", region);
            assertEquals(0, ret);
            assertEquals(2, region.end(0));

            regex.close();
        }
    }

    @Test
    public void splitComma() {
        final String regex = ",";
        final String input = "abc,ｄ,eｆg,h,ｉ,,j,,,,ｋ,,,";
        final String[] answer
                = new String[] { "abc", "ｄ", "eｆg", "h", "ｉ", "", "j", "", "", "", "ｋ", };
        
        OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA);
        final List<String> values = reg.split(input, 0);
        for (final String v : values) {
            System.out.println("splitComma1:[" + v + "]");
        }
        assertEquals(answer.length, values.size());
        for (int i = 0; i < answer.length; ++i) {
            assertEquals(answer[i], values.get(i));
        }

        final List<String> values2 = reg.split(input, -1);
        for (final String v : values2) {
            System.out.println("splitComma2:[" + v + "]");
        }
        assertEquals(answer.length, values2.size() - 3);
        for (int i = 0; i < answer.length; ++i) {
            assertEquals(answer[i], values.get(i));
        }
    }

    @Test
    public void splitLineTerminator() {
        final String regex = "$";
        final String input = "い\rろ\nは\r\nに\u0085ほ\u2028へ\u2029と";
        final String[][] answers
                = new String[][] {
                        { "い", "\rろ", "\nは", "\r\nに", "\u0085ほ", "\u2028へ", "\u2029と", "" },
                        { "い\rろ", "\nは\r", "\nに\u0085ほ\u2028へ\u2029と", "" },
                    };

        final Collection<OnigOptionType> options
                = EnumSet.of(OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
        for (final String[] answer : answers) {
            final OnigRegex reg
                    = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA, options);
            final List<String> values = reg.split(input, -1);
            for (final String v : values) {
                System.out.println("splitEOL:[" + v + "]");
            }
            assertEquals(answer.length, values.size());
            for (int i = 0; i < answer.length; ++i) {
                assertEquals(answer[i], values.get(i));
            }
            options.add(OnigOptionType.ONIG_OPTION_NEGATE_JAVA_LINES);
        }
    }

    @Test
    public void splitJavaEOL() {
        final String regex = "(?=[\r\u0085\u2028\u2029]|(?<!\r)\n|\\z)";
        final String input = "い\rろ\nは\r\nに\u0085ほ\u2028へ\u2029と";
        final String[] answer
                = new String[] { "い", "\rろ", "\nは", "\r\nに", "\u0085ほ", "\u2028へ", "\u2029と", "" };

        final OnigRegex reg
                = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
        final List<String> values = reg.split(input, -1);
        for (final String v : values) {
            System.out.println("splitJavaEOL:[" + v + "]");
        }
        assertEquals(answer.length, values.size());
        for (int i = 0; i < answer.length; ++i) {
            assertEquals(answer[i], values.get(i));
        }        
    }
    
    @Test
    public void checkPriority() {
        final String[] regexs
                = new String[] {
                        "world",
                        "(?-i:World)",
                        "(?-i)WORLD",
                    };
        final String input = "Hello, world!";
        final boolean[] answers
                = new boolean[] { true, false, false, };
        // MEMO 鬼車ではAPIに指定したオプションより埋め込みフラグ表現が優先されるようだ。

        int i = 0;
        for (final String regex : regexs) {
            final OnigRegex reg
                    = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_IGNORECASE);
            final boolean answer = answers[i++];
            assertEquals(answer, reg.search(input) >= 0);
        }
    }
    
    @Test
    public void getCopyright() {
        System.out.println(OnigRegex.getCopyright());
        assertNotNull(OnigRegex.getCopyright());
    }

    @Test
    public void unixLines() {
        final String[] regex
                = new String[] { "(?m-d)$", "(?md:$)", "(?md)$", "((?md)$)", };
        final String input = "い\rろ\nは\r\nに\u0085ほ\u2028へ\u2029と";
        final String[][] answers
                = new String[][] {
                        java.util.regex.Pattern.compile(regex[0]).split(input, -1),
                        java.util.regex.Pattern.compile(regex[1]).split(input, -1),
                        java.util.regex.Pattern.compile(regex[2]).split(input, -1),
                        java.util.regex.Pattern.compile(regex[3]).split(input, -1),
//                        { "い", "\rろ", "\nは", "\r\nに", "\u0085ほ", "\u2028へ", "\u2029と", "" },
//                        { "い\rろ", "\nは\r", "\nに\u0085ほ\u2028へ\u2029と", "" },
//                        { "い", "\rろ", "\nは", "\r\nに", "\u0085ほ", "\u2028へ", "\u2029と", "" },
                    };

        int r = 0;
        for (final String[] answer : answers) {
            final OnigRegex reg
                    = new OnigRegex(regex[r++], OnigSyntaxType.ONIG_SYNTAX_JAVA);
            final List<String> values = reg.split(input, -1);
            for (final String v : values) {
                System.out.println(v);
            }
            assertEquals(answer.length, values.size());
            for (int i = 0; i < answer.length; ++i) {
                assertEquals(answer[i], values.get(i));
            }
        }
    }
}