package onig4j;

import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class OnigRegionTest {

    @Test
    public void searchMatch() {
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        int ret = OnigRegex.ONIG_MISMATCH;
                
        ret = regex.search("abcdEfg", region);
        System.out.printf("match at %d\n", ret);
        assertEquals(4, ret);
        
        System.out.printf("count is %d\n", region.count());
        assertEquals(1, region.count());

        int i = 0;
        System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        assertEquals(4, region.begin(i));
        assertEquals(5, region.end(i));
        
        System.out.printf("allocated at %d\n", region.allocated());
        
        ret = regex.search("abcdefg", region);
        assertEquals(OnigRegex.ONIG_MISMATCH, ret);
        
        regex.close();
        region.close();
    }

    @Test
    public void searchMismatch() {
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        int ret = OnigRegex.ONIG_MISMATCH;
                
        ret = regex.search("abcdefg", region);
        assertEquals(OnigRegex.ONIG_MISMATCH, ret);
        
        regex.close();
        region.close();
    }

    @Test
    public void matchMatch() {
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("あ(.*)い|[お-か]+");
        int ret = OnigRegex.ONIG_MISMATCH;

        ret = regex.match("んんんんあかかかかかかかかい", 4, region);
        System.out.printf("match bytes is %d\n", ret);
        assertEquals(10, ret);

        System.out.printf("count is %d\n", region.count());
        assertEquals(2, region.count());
        
        int i = 0;
        System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        assertEquals(4, region.begin(i));
        assertEquals(14, region.end(i));

        ++i;
        System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        assertEquals(5, region.begin(i));
        assertEquals(13, region.end(i));
        
        regex.close();
        region.close();
    }

    @Test
    public void traverseCaptureTree() {
        OnigSyntaxType syntax = OnigSyntaxType.ONIG_SYNTAX_DEFAULT.clone();
        syntax.setOperator2(syntax.getOperator2() | OnigSyntaxType.ONIG_SYN_OP2_ATMARK_CAPTURE_HISTORY);
        
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("(?@x(?@\\d+))+", syntax);
        int ret = OnigRegex.ONIG_MISMATCH;

        ret = regex.search("x00x00x00", region);
        System.out.printf("ret is %d\n", ret);
        assertEquals(0, ret);

        System.out.printf("count is %d\n", region.count());
        assertEquals(3, region.count());
        
        OnigRegion.Callback callback
                = new OnigRegion.Callback() {
                        public int call(int group, int begin, int end, int level, int at) {
                            System.out.printf("group = %d, begin = %d, end = %d, level = &d, at = %d\n", group, begin, end, level, at);
                            return 0;
                        }
                    };
        region.traverseCaptureTree(callback, OnigRegion.ONIG_TRAVERSE_CALLBACK_AT_BOTH);
                    
        int i = 0;
        System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        assertEquals(0, region.begin(i));
        assertEquals(9, region.end(i));

        ++i;
        System.out.printf("%d: (%d-%d)\n", i, region.begin(i), region.end(i));
        assertEquals(6, region.begin(i));
        assertEquals(9, region.end(i));
        
        regex.close();
        region.close();
        syntax.close();
    }
    
    @Test
    public void clear() {
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        int ret = OnigRegex.ONIG_MISMATCH;
                
        ret = regex.search("abcdEfg", region);
        assertEquals(4, ret);
        
        region.clear();
        for (int i = 0; i < region.count(); ++i) {
            assertEquals(OnigRegion.ONIG_REGION_NOTPOS, region.begin(i));
            assertEquals(OnigRegion.ONIG_REGION_NOTPOS, region.end(i));
        }
    }
    
    @Test
    public void resize() {
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        int ret = OnigRegex.ONIG_MISMATCH;
                
        ret = regex.search("abcdEfg", region);
        assertEquals(1, region.count());
        
        region.resize(0);
        assertEquals(0, region.count());
        
        region.resize(20);
        assertEquals(20, region.count());
        assertEquals(ret, region.begin(0));
    }
    
    @Test
    public void resizeClear() {
        OnigRegion region = new OnigRegion();
        OnigRegex regex = new OnigRegex("\\p{Lu}", OnigSyntaxType.ONIG_SYNTAX_JAVA);
        int ret = OnigRegex.ONIG_MISMATCH;
                
        ret = regex.search("abcdEfg", region);
        assertEquals(1, region.count());
        
        region.resize(0);
        assertEquals(0, region.count());
        
        region.resizeClear(20);
        assertEquals(20, region.count());
        for (int i = 0; i < region.count(); ++i) {
            assertEquals(OnigRegion.ONIG_REGION_NOTPOS, region.begin(i));
            assertEquals(OnigRegion.ONIG_REGION_NOTPOS, region.end(i));
        }
    }
}