package onig4j.java.util.regex;

import java.util.List;
import onig4j.OnigOptionType;
import onig4j.OnigRegex;
import onig4j.OnigSyntaxType;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class PatternTest {

    @Test
    public void split() {
        final String regex = ",";
        final String input = "abc,ｄ,eｆg,h,ｉ,,j,,,,ｋ,,,";
        
        final String[] values = java.util.regex.Pattern.compile(regex).split(input);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }
        assertEquals(11, values.length);
        
        Pattern pattern = Pattern.compile(regex);
        final String[] values2 = pattern.split(input);
        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }
        
        OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA);
        final List<String> values3 = reg.split(input, 0, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        assertEquals(values.length, values3.size());
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values3.get(i));
        }
    }

    @Test
    public void split2() {
//        final String regex = ",\\p{Alpha}";
        final String regex = ",[a-zA-Z]";
        final String input = "abc,ｄ,eｆg,h,ｉ,,j,,,,ｋ,,,";
        
//        final String[] values
//                = java.util.regex.Pattern.compile(regex).split(input);
        final String[] values
                = java.util.regex.Pattern.compile(regex, java.util.regex.Pattern.UNICODE_CASE).split(input);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }
        assertEquals(4, values.length);
        
        OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA);
        final List<String> values3 = reg.split(input, 0, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        assertEquals(values.length, values3.size());
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values3.get(i));
        }
         
        Pattern pattern = Pattern.compile(regex);
        final String[] values2 = pattern.split(input);
        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }
   }

    @Test
    public void split3() {
        int options = java.util.regex.Pattern.MULTILINE;
//        options |= java.util.regex.Pattern.UNIX_LINES;
        final String regex = "$";
//        final String regex = "(\n|\r\n|\r|\u0085|\u2028|\u2029)";
//        final String regex = "\\b";
        final String input = "あいうえお\nかきくけこ\r\nさしすせそ\rたちつてと\u0085なにぬねの\u2028はひふへほ\u2029まみむめも";
        
        final String[] values
                = java.util.regex.Pattern.compile(regex, options).split(input, -1);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }
        
//        org.apache.regexp.RE re = new org.apache.regexp.RE(regex, org.apache.regexp.RE.MATCH_MULTILINE);
//        final String[] values4 = re.split(input);
//        for (final String v : values4) {
//            System.out.println("regexp:[" + v + "]");
//        }
        
        OnigRegex reg = new OnigRegex("(?:(?=[\r\u0085\u2028\u2029]|(?<!\r)\n)|\\Z)", OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
        final List<String> values3 = reg.split(input, -1, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        assertEquals(values.length, values3.size());
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values3.get(i));
        }
        
        Pattern pattern = Pattern.compile(regex, options);
        final String[] values2 = pattern.split(input, -1);
        for (final String v : values2) {
            System.out.println("values2:[" + v + "]");
        }
        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }
   }

    @Test
    public void split4() {
        final int options = 0;
        final String regex = "\\bdef";
        final String input = "abcdefghijklmn define";
        
        final String[] values
                = java.util.regex.Pattern.compile(regex, options).split(input);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }
        
        Pattern pattern = Pattern.compile(regex, options);
        final String[] values2 = pattern.split(input, -1);
        for (final String v : values2) {
            System.out.println("values2:[" + v + "]");
        }
//        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }
        
        OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA);
        final List<String> values3 = reg.split(input, -1, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        assertEquals(values.length, values3.size());
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values3.get(i));
        }
   }

    @Test
    public void split5() {
        final int options = java.util.regex.Pattern.MULTILINE;
//        final int options = java.util.regex.Pattern.MULTILINE | java.util.regex.Pattern.UNIX_LINES;
        
        // TODO 行末記号の否定を [^$] とした場合はJavaの正規表現クラスと異なる結果を返してしまう
//        final String regex = "[^$]";
        final String regex = "(?!$)";
        
        final String input = "\r\nあいうえお\nかきくけこ\r\nさしすせそ\rたちつてと\u0085なにぬねの\u2028はひふへほ\u2029まみむめも";
        
        final String[] values
                = java.util.regex.Pattern.compile(regex, options).split(input, -1);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }
        
        Pattern pattern = Pattern.compile(regex, options);
        final String[] values2 = pattern.split(input, -1);
        for (final String v : values2) {
            System.out.println("values2:[" + v + "]");
        }
        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }
        
        OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE, OnigOptionType.ONIG_OPTION_FIND_LONGEST);
        final List<String> values3 = reg.split(input, -1, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        if ((options & java.util.regex.Pattern.UNIX_LINES) != 0) {
            assertEquals(values2.length, values3.size());
            for (int i = 0; i < values2.length; ++i) {
                assertEquals(values2[i], values3.get(i));
            }
        }
    }

    @Test
    public void split6() {
        final int options = java.util.regex.Pattern.MULTILINE;
        final String regex = "お$";
        final String input = "\r\nあいうえお\nかきくけこ\r\nさしすせそ\rたちつてお\u0085なにぬねの\u2028はひふへほ\u2029まみむめも";
        
        final String[] values
                = java.util.regex.Pattern.compile(regex, options).split(input, -1);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }
        
        Pattern pattern = Pattern.compile(regex, options);
        final String[] values2 = pattern.split(input, -1);
        for (final String v : values2) {
            System.out.println("values2:[" + v + "]");
        }
        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }

        final String regex2 = "お(?:(?=[\r\u0085\u2028\u2029]|(?<!\r)\n)|\\Z)";
        OnigRegex reg = new OnigRegex(regex2, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
        final List<String> values3 = reg.split(input, -1, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        assertEquals(values.length, values3.size());
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values3.get(i));
        }
    }

    @Test
    public void split7() {
        final int options = java.util.regex.Pattern.MULTILINE;
        final String regex = "$";
//        final String input = "いろはabcにbcほへとcちりぬるを";
        final String input = "い\rろ\nは\r\nに\u0085ほ\u2028へ\u2029と";
        
        final String[] values
                = java.util.regex.Pattern.compile(regex, options).split(input, -1);
        for (final String v : values) {
            System.out.println("values :[" + v + "]");
        }

        final String regex2 = "(?:(?=[\r\u0085\u2028\u2029]|(?<!\r)\n)|\\Z)";
        OnigRegex reg = new OnigRegex(regex2, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_NEGATE_SINGLELINE);
        final List<String> values3 = reg.split(input, -1, OnigOptionType.ONIG_OPTION_NONE);
        for (final String v : values3) {
            System.out.println("values3:[" + v + "]");
        }
        assertEquals(values.length, values3.size());
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values3.get(i));
        }
        
        Pattern pattern = Pattern.compile(regex, options);
        final String[] values2 = pattern.split(input, -1);
        for (final String v : values2) {
            System.out.println("values2:[" + v + "]");
        }
        assertEquals(values.length, values2.length);
        for (int i = 0; i < values.length; ++i) {
            assertEquals(values[i], values2[i]);
        }
    }

    @Test
    public void split8() {
        final String regex = "\\b";
        final String[] inputs
                = new String[] {
                        "Hello, Oniguruma for Java!",
                        "こんにちは、　鬼車 for Java！",
                    };
        for (final String input : inputs) {
            final String[] values
                    = java.util.regex.Pattern.compile(regex).split(input, -1);
            for (final String v : values) {
                System.out.println("values :[" + v + "]");
            }

            OnigRegex reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA);
            final List<String> values3 = reg.split(input, -1, OnigOptionType.ONIG_OPTION_NONE);
            for (final String v : values3) {
                System.out.println("values3:[" + v + "]");
            }
            assertEquals(values.length, values3.size());
            for (int i = 0; i < values.length; ++i) {
                assertEquals(values[i], values3.get(i));
            }

            Pattern pattern = Pattern.compile(regex);
            final String[] values2 = pattern.split(input, -1);
            for (final String v : values2) {
                System.out.println("values2:[" + v + "]");
            }
            assertEquals(values.length, values2.length);
            for (int i = 0; i < values.length; ++i) {
                assertEquals(values[i], values2[i]);
            }
        }
    }
    
    @Test
    public void unicodeEscape() {
        final String regex = "\uuuuuuu3042い\\u3046え\\u304a";
        final String input = "かきくけこあ\u3044うえ\u304Aさしすせそあいう\u3048おたちつてとあいうえおなにぬねの";
     
        final java.util.regex.Matcher matcher = java.util.regex.Pattern.compile(regex).matcher(input);
        assertTrue(matcher.find());
        assertEquals(5, matcher.start());
        assertEquals("あいうえお", matcher.group());
        assertTrue(matcher.find());
        assertEquals(15, matcher.start());
        assertTrue(matcher.find());
        assertEquals(25, matcher.start());
        assertFalse(matcher.find());
        
        final Matcher matcher2 = Pattern.compile(regex).matcher(input);
        assertTrue(matcher2.find());
        assertEquals(5, matcher2.start());
        assertEquals("あいうえお", matcher2.group());
        assertTrue(matcher2.find());
        assertEquals(15, matcher2.start());
        assertTrue(matcher2.find());
        assertEquals(25, matcher2.start());
        assertFalse(matcher2.find());
    }
    
    @Test
    public void comments() {
        final String[] patterns
                = new String[] {
//                        " a bc d\\ #ghijklmn\r\r ef ", // NG
                        " abc  de f ",
                        " (?x) abc def #ghijklmn ",
                        " a bc d ef #ghijklmn ",
                        " a bc d #ghijklmn\n ef ",
                        " a bc d #ghijklmn\r ef ",
                        " a bc d #ghijklmn\r\n ef ",
//                        " a bc d #ghijklmn\u0085 ef ", // NG
//                        " a bc d #ghijklmn\u2028 ef ", // NG
//                        " a bc d #ghijklmn\u2029 ef ", // NG
                        " a bc d #ghijklmn\n\r ef ", // OK
                        " a bc d #ghijklmn\n\n ef ", // OK
                        " a bc d #ghijklmn\r\r ef ", // OK
//                        " a bc d \\#ghijklmn\r\r ef ", // NG
                    };
        
        final String input = "abcdef";
//        final int options = java.util.regex.Pattern.COMMENTS;
        final int options = java.util.regex.Pattern.COMMENTS | java.util.regex.Pattern.UNIX_LINES;
     
        for (final String regex : patterns) {
            java.util.regex.Matcher matcher = java.util.regex.Pattern.compile(regex).matcher(input);
            System.out.println("matcher is " + matcher);
            if (matcher.find()) {
                assertEquals(0, matcher.start());
            }

            final Matcher matcher2 = Pattern.compile(regex, options).matcher(input);
            assertTrue(matcher2.find());
            assertEquals(0, matcher2.start());

            matcher = java.util.regex.Pattern.compile(regex, options).matcher(input);
            System.out.println("matcher is " + matcher);
            assertTrue("regex is " + regex, matcher.find());
            assertEquals(0, matcher.start());
        }
    }
    
    @Test
    public void canonicalEqual() {
        final String[] patterns
                = new String[] {
                        "a\u030A",
                        "\u00C5",
                        "A\u030A",
                        "\u212B",
                        
                    };
        final String input = "\u212B\u00E5A\u030A\u00C5";
        final int options = java.util.regex.Pattern.CANON_EQ;
        final int[] answers
                = new int[] {
                    1,
                    2,
                    2,
                    2,
                };
     
        int i = 0;
        for (final String regex : patterns) {
            final int answer = answers[i];
            
            java.util.regex.Matcher matcher = java.util.regex.Pattern.compile(regex).matcher(input);
            System.out.println("matcher is " + matcher);
            if (matcher.find()) {
                assertEquals("i = " + i, (i == 1 ? 4 : (i == 2 ? 2 : 0)), matcher.start());
            }

            matcher = java.util.regex.Pattern.compile(regex, options).matcher(input);
            System.out.println("matcher is " + matcher);
            assertTrue("regex is " + regex, matcher.find());
            assertEquals(answer, matcher.start());
            final int end1 = matcher.end();
            final String s1 = matcher.group();

            final Matcher matcher2 = Pattern.compile(regex, options).matcher(input);
            assertTrue(matcher2.find());
            assertEquals(answer, matcher2.start());
            final int end2 = matcher2.end();
            final String s2 = matcher2.group();
            
            assertEquals(end1, end2);
            assertEquals(s1, s2);
            
            ++i;
        }
    }
    
    @Test
    public void canonicalEqual2() {
        final String[] patterns
                = new String[] {
                        "が",
                        "ぱ",
                        "\u3099",
                    };
        final String input = "は\u309aか\u3099";
        final int options = java.util.regex.Pattern.CANON_EQ;
        final int[] answers
                = new int[] {
                    2,
                    0,
                    3,
                };
     
        int i = 0;
        for (final String regex : patterns) {
            final int answer = answers[i];
            
            final java.util.regex.Matcher matcher
                    = java.util.regex.Pattern.compile(regex, options).matcher(input);
            System.out.println("matcher is " + matcher);
            assertTrue("regex is " + regex, matcher.find());
            System.out.println("matcher is " + matcher);
            assertEquals(answer, matcher.start());
            final int end1 = matcher.end();
            final String s1 = matcher.group();

            final Matcher matcher2 = Pattern.compile(regex, options).matcher(input);
            assertTrue(matcher2.find());
            assertEquals(answer, matcher2.start());
            final int end2 = matcher2.end();
            final String s2 = matcher2.group();
            
            assertEquals(end1, end2);
            assertEquals(s1, s2);
            
            ++i;
        }
    }
    
    @Test
    public void pattern() {
        final String[] regex
                = new String[] {
                        "\uuuuuuu3042い\\u3046え\\u304a",
                        "が、",
                    };
        final int[] option
                = new int[] {
                    Pattern.COMMENTS,
                    Pattern.CANON_EQ,
                };
                
        for (int i = 0; i < regex.length; ++i) {
            final java.util.regex.Pattern pattern = java.util.regex.Pattern.compile(regex[i], option[i]);
            final Pattern pattern2 = Pattern.compile(regex[i], option[i]);

            assertEquals(pattern.pattern(), pattern2.pattern());
            assertEquals(pattern.pattern(), pattern.toString());
            assertEquals(pattern2.pattern(), pattern2.toString());
        }
    }
    
    @Test
    public void caseInsensitive() {
        final String[] regexs
                = new String[] {
                        "Hello",
                        "Hello",
                        "HELLO",
                        "AπΦω",
                        "(?iu)aπφω",
                        "AπΦω",
                        "(?i)πφω",
                        "(?iu:πφω)",
                        "(?i-u)πφω",
                        "A(?iu:πΦω)",
                        "A(?iu:πΦω)",
                    };
        final int[] options
                = new int[] {
                    0,
                    Pattern.CASE_INSENSITIVE,
                    Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE,
                    Pattern.CASE_INSENSITIVE,
                    0,
                    Pattern.CASE_INSENSITIVE,
                    0,
                    0,
                    0,
                    0,
                    0,
                };
        final String[] input
                = new String[] {
                    "HELLO, world!",
                    "hello, world!",
                    "Hello, world!",
                    "aπΦω",
                    "AΠΦΩ",
                    "aΠΦΩ",
                    "ΠΦΩ",
                    "ΠΦΩ",
                    "ΠΦΩ",
                    "aΠΦΩ",
                    "AΠΦΩ",
                };
        final boolean[] answer
                = new boolean[] {
                    false,
                    true,
                    true,
                    true,
                    true,
                    false,
                    false,
                    true,
                    false,
                    false,
                    true,
                };
                
        for (int i = 0; i < regexs.length; ++i) {
            final String regex = regexs[i];
            final int option = options[i];
            final java.util.regex.Matcher matcher = java.util.regex.Pattern.compile(regex, option).matcher(input[i]);
            assertEquals(answer[i], matcher.find());
            
            final Matcher matcher2 = Pattern.compile(regex, option).matcher(input[i]);
            // TODO 現在の実装ではPattern.UNICODE_CASEが指定されていない場合でも指定された場合と同じ振る舞いしか出来ない（鬼車のAPIで対応できないため）。
            assertEquals("regex = /" + regex + "/ option = " + option, answer[i], matcher2.find());
            // MEMO java.nio.charset.Charset#defaultCharset()で取得したエンコーディングをOnigEncodingとして使用するようにするべきか？
            
            if (i != 4) {
                OnigRegex reg = null;
                if ((option & Pattern.CASE_INSENSITIVE) == 0) {
                    reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA);
                } else {
                    if ((option & Pattern.UNICODE_CASE) != 0) {
                        reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_IGNORECASE, OnigOptionType.ONIG_OPTION_NEGATE_CASE_FOLD_ASCII_ONLY);
                    } else {
                        reg = new OnigRegex(regex, OnigSyntaxType.ONIG_SYNTAX_JAVA, OnigOptionType.ONIG_OPTION_IGNORECASE);
                    }
                }
                assertEquals(answer[i], (reg.search(input[i]) >= 0));
            }
        }
    }
    
    @Test
    public void unixLines() {
        int options = java.util.regex.Pattern.MULTILINE;
//        options |= java.util.regex.Pattern.UNIX_LINES;
        final String[] regexList = new String[] { "(?-d)$", "(?d)$", "(?d:$)" };
//        final String[] regexList = new String[] { "(?-d)$", "(?d)$", };
        final String input = "あいうえお\nかきくけこ\r\nさしすせそ\rたちつてと\u0085なにぬねの\u2028はひふへほ\u2029まみむめも";
        
        for (final String regex : regexList) {
            final String[] values
                    = java.util.regex.Pattern.compile(regex, options).split(input, -1);
            for (final String v : values) {
                System.out.println("values :[" + v + "]");
            }

            Pattern pattern = Pattern.compile(regex, options);
            final String[] values2 = pattern.split(input, -1);
            for (final String v : values2) {
                System.out.println("values2:[" + v + "]");
            }
            assertEquals(values.length, values2.length);
            for (int i = 0; i < values.length; ++i) {
                assertEquals(values[i], values2[i]);
            }
        }
    }
    
    @Test
    public void unicodeCase() {
//        final String regex = "(?iu-d)πφω(?:(?ud:あいう)えお)*";
        final String regex = "(?iu-d)πφω(?:(?-i:あいう)えお)*";
        final String input = "ΠΦΩ";

        final java.util.regex.Matcher matcher = java.util.regex.Pattern.compile(regex).matcher(input);
        assertTrue(matcher.find());

        final Matcher matcher2 = Pattern.compile(regex).matcher(input);
        assertTrue(matcher2.find());
    }
    
    @Test
    public void quote() {
        final String[] regexs
                = new String[] {
                        "Abc",
                        "^I am .* \\.$",
                        "(?iu-d)πφω(?:(?ud:あいう)えお)*",
                        "(?-d)$",
                        "(?d:$)",
                        "\uuuuuuu3042い\\u3046え\\u304a",
                    };
        for (final String regex : regexs) {
            final String quoted1 = java.util.regex.Pattern.quote(regex);
//            System.out.println(regex + " = " + quoted1);
            final String quoted2 = Pattern.quote(regex);
            assertEquals(quoted1, quoted2);
        }
    }
    
    @Test
    public void literal() {
        final String[] regexs
                = new String[] {
                        "Abc",
                        "I am .*\\.",
                        "^I am .*\\.$",
                        "\uuuuuuu3042い\\u3046え\\u304a",
                        "\uuuuuuu3042い\\u3046え\\u304a",
                        "a\u030A",                        
                    };
        final String[] inputs
                = new String[] {
                    "Abc",
                    "I am ROBOT.",
                    "^i Am .*\\.$",
                    "あい\\u3046え\\u304a",
                    "あいうえお",
                    "\u00E5",
                };
        final boolean[] answers
                = new boolean[] {
                    true,
                    false,
                    true,
                    true,
                    false,
                    false,
                };
        int options = java.util.regex.Pattern.LITERAL;
        options |= java.util.regex.Pattern.CASE_INSENSITIVE;
        options |= java.util.regex.Pattern.CANON_EQ;

        int i = 0;
        for (final String regex : regexs) {
            final String input = inputs[i];
            final boolean answer = answers[i];
            System.out.println("regex  = " + regex);
            System.out.println("input  = " + input);
            System.out.println("answer = " + answer);
            
            final java.util.regex.Matcher matcher1
                    = java.util.regex.Pattern.compile(regex, options).matcher(input);
            assertEquals(answer, matcher1.find());
            
            final Matcher matcher2 = Pattern.compile(regex, options).matcher(input);
            assertEquals(answer, matcher2.find());
            
            ++i;
        }
    }
    
    @Test
    public void checkPriority() {
        final String[] regexs
                = new String[] {
                        "hello, world!",
                        "(?-i:Hello, World!)",
                        "(?-i)Hello, World!",
                    };
        final String input = "Hello, world!";
        final boolean[] answers
                = new boolean[] { true, false, false, };
        final int options = java.util.regex.Pattern.CASE_INSENSITIVE;

        int i = 0;
        for (final String regex : regexs) {
            final java.util.regex.Matcher matcher
                    = java.util.regex.Pattern.compile(regex, options).matcher(input);
            final boolean answer = answers[i++];
            assertEquals(answer, matcher.find());
        }        
    }
}