/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.security;

import org.opengion.fukurou.util.LogWriter;

import javax.crypto.spec.SecretKeySpec;
import javax.crypto.Cipher;

/**
 * HybsCryptography は、セキュリティ強化の為の Hybs独自の暗号化クラスです。
 *
 * このクラスは、暗号化キーを受け取り、それに基づいて暗号化/復号化を行います。
 * ここでの暗号化は、秘密キー方式でバイト文字列に変換されたものを、１６進アスキー文字に
 * 直して、扱っています。よって、暗号化/復号化共に、文字列として扱うことが可能です。
 *
 * @og.rev 4.0.0 (2005/08/31) 新規追加
 * @og.group ライセンス管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class HybsCryptography {
	private final SecretKeySpec sksSpec ;
	private static final String CIPHER_TYPE = "Blowfish" ;

	// 注意：秘密キーは、８の倍数でないといけない。
	private static final String HYBS_CRYPT_KEY = "2a5a88891d37ae59" ;

	/**
	 * 内部設定の秘密鍵を使用して，暗号化を行うオブジェクトを構築します。
	 * ここでの暗号化は、Java標準のセキュリティパッケージを使用しています。
	 */
	public HybsCryptography() {
		sksSpec = new SecretKeySpec( HYBS_CRYPT_KEY.getBytes(), CIPHER_TYPE );
	}

	/**
	 * 秘密鍵の文字列を受け取って，暗号化を行うオブジェクトを構築します。
	 * ここでの暗号化は、Java標準のセキュリティパッケージを使用しています。
	 * 秘密鍵のサイズを、8 の倍数 (32 以上 448 以下) にする必要があります。
	 *
	 * @param  cryptKey String   暗号化を行う秘密鍵
	 */
	public HybsCryptography( final String cryptKey ) {
		sksSpec = new SecretKeySpec( cryptKey.getBytes(), CIPHER_TYPE );
	}

	/**
	 * セキュリティカラムのDBTyepに対してHybs独自の暗号化を行います。
	 * 暗号化されたデータは、通常 byte 文字ですが、１６進数アスキー文字列に変換
	 * したものを返します。
	 * この暗号化では、引数が null の場合は、ゼロ文字列を返します。
	 *
	 * @param  org String 暗号化を行う元の文字列
	 * @return  String  暗号化された文字列(HEXADECIMAL化)
	 */
	public String encrypt( final String org ) {
		if( org == null || org.length() == 0 ) { return ""; }

		try {
			Cipher cipher = Cipher.getInstance( CIPHER_TYPE );
			cipher.init( Cipher.ENCRYPT_MODE, sksSpec );
			byte[] encrypted = cipher.doFinal( org.getBytes() );

			return byte2hexa( encrypted );
		}
		catch( javax.crypto.IllegalBlockSizeException	ex ) { throw new RuntimeException( ex ); }
		catch( java.security.InvalidKeyException		ex ) { throw new RuntimeException( ex ); }
		catch( java.security.NoSuchAlgorithmException	ex ) { throw new RuntimeException( ex ); }
		catch( javax.crypto.BadPaddingException			ex ) { throw new RuntimeException( ex ); }
		catch( javax.crypto.NoSuchPaddingException		ex ) { throw new RuntimeException( ex ); }
	}

	/**
	 * セキュリティカラムのDBTyepに対してHybs独自の復号化を行います。
	 * ここでの復号化は、encrypt で暗号化された文字を戻す場合に使用します。
	 * この復号化では、null は復号化できないため、ゼロ文字列を返します。
	 *
	 * @param  hex String 復号化を行う暗号化された１６進数アスキー文字列
	 * @return  String  復号化された元の文字列
	 */
	public String decrypt( final String hex ) {
		if( hex == null || hex.length() == 0 ) { return ""; }

		try {
			Cipher cipher = Cipher.getInstance( CIPHER_TYPE );
			cipher.init( Cipher.DECRYPT_MODE, sksSpec );
			byte[] encrypted = hexa2byte( hex );
			byte[] decrypted = cipher.doFinal( encrypted );
			return new String( decrypted );
		}
		catch( javax.crypto.IllegalBlockSizeException	ex ) { throw new RuntimeException( ex ); }
		catch( java.security.InvalidKeyException		ex ) { throw new RuntimeException( ex ); }
		catch( java.security.NoSuchAlgorithmException	ex ) { throw new RuntimeException( ex ); }
		catch( javax.crypto.BadPaddingException			ex ) { throw new RuntimeException( ex ); }
		catch( javax.crypto.NoSuchPaddingException		ex ) { throw new RuntimeException( ex ); }
	}
	/**
	 * 数字から１６進文字に変換するテーブルです。
	 */
	private static final char[] hexadecimal =
		{ '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
		  'a', 'b', 'c', 'd', 'e', 'f' };

	/**
	 * バイト配列を１６進数アスキー文字列に変換します。
	 *
	 * バイト配列を、２文字の0～9,a～ｆのアスキーに変換されます。
	 * これにより、すべての文字を、アスキー化できます。
	 * アスキー化で、上位が0F以下の場合でも、0 を出すことで、固定長に変換します。
	 *
	 * よって、入力バイトの２倍のlength()を持ったStringを作成します。
	 *
	 * @param	input バイト配列
	 * @return	１６進数アスキー文字列
	 */
	public static String byte2hexa( final byte[] input ) {
		String rtn = null;
		if( input != null ) {
			int len = input.length ;
			char[] ch = new char[len*2];
			for( int i=0; i<len; i++ ) {
				int high = ((input[i] & 0xf0) >> 4);
				int low  = (input[i] & 0x0f);
				ch[i*2]   = hexadecimal[high];
				ch[i*2+1] = hexadecimal[low];
			}
			rtn =  new String(ch);
		}
		return rtn;
	}

	/**
	 * １６進数アスキー文字列をバイト配列に変換します。
	 *
	 * ２文字の0～9,a～ｆのアスキー文字列を、バイト配列に変換されます。
	 *
	 * よって、入力Stringの１/２倍のlengthを持ったバイト配列を作成します。
	 *
	 * @param	input １６進数アスキー文字列
	 * @return	バイト配列
	 */
	public static byte[] hexa2byte( final String input ) {
		byte[] rtn = null;
		if( input != null ) {
			int len = input.length() ;
			rtn = new byte[len/2];
			for( int i=0; i<len/2; i++ ) {
				char ch = input.charAt( i*2 );
				int high = ( ch < 'a' ) ? ch-'0' : ch-'a'+10 ;
				ch = input.charAt( i*2+1 );
				int low  = ( ch < 'a' ) ? ch-'0' : ch-'a'+10 ;
				rtn[i] = (byte)(high << 4 | low);
			}
		}
		return rtn;
	}

	/**
	 * 暗号化のテストを行う為のメインメソッド
	 *
	 * java HybsCryptography KEY TEXT で起動します。
	 *   KEY  : 秘密鍵(8 の倍数 (32 以上 448 以下)文字)
	 *   TEXT : 変換する文字列
	 *
	 * @param	args String[]	引数配列
	 */
	public static void main( final String[] args ) throws Exception {
		if( args.length != 2 ) {
			LogWriter.log( "java HybsCryptography KEY TEXT" );
			LogWriter.log( "  KEY  : 秘密鍵(8 の倍数 (32 以上 448 以下)文字)" );
			LogWriter.log( "  TEXT : 変換する文字列" );
			return;
		}

		HybsCryptography cript = new HybsCryptography( args[0] );

		System.out.println( "IN   TEXT : " + args[1] );

		String hexa = cript.encrypt( args[1] );
		System.out.println( "HEXA TEXT : " + hexa );

		String data = cript.decrypt( hexa );
		System.out.println( "OUT  DATA : " + data );
	}
}
