/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.File;
import java.util.Map;
import java.util.HashMap;
import java.util.Locale ;

/**
 * FileMap は、ファイルを読み取って、キー情報から、ファイルへのリンクを作成するための
 * 情報を返します。
 * ファイルそのものは、指定のディレクトリをすべて読み取り、拡張子以外の部分を、キーとして
 * 登録します。(キーは大文字に統一されます。)
 * 実際のファイルの拡張子は、リンク作成時の処理で付与されます。
 * 例えば、HELPファイルを、XXXX.html や、XXXX.htm 、XXXX.pdf など、色々な形態で作成した
 * 場合でも、キーとしては、XXXX で存在チェックをかけることができるようになります。
 *
 * ファイルは、一旦すべて読み取ってメモリ上で管理されます。
 * ディレクトリの再読取が必要な場合は、オブジェクトを再作成する必要があります。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class FileMap {
	private final String directory;
	private final Map<String,String> map = new HashMap<String,String>();

	/**
	 * 読み取るディレクトリを指定して、オブジェクトを構築します。
	 *
	 * @param  dir String
	 */
	public FileMap( final String dir ) {
		directory = dir;

		makeFileMap();
	}

	/**
	 * 読み取るディレクトリを指定して、オブジェクトを構築します。
	 *
	 */
	private void makeFileMap() {
		File dir = new File( directory );
		if( ! dir.exists() ) {
			String errMsg = "指定のディレクトリは、存在しません。[" + directory + "]";
			throw new IllegalArgumentException( errMsg );
		}

		if( ! dir.isDirectory() ) {
			String errMsg = "指定のキーは、ディレクトリではありません。[" + directory + "]";
			throw new IllegalArgumentException( errMsg );
		}

		String[] files = dir.list();
		for( int i=0; i<files.length; i++ ) {
			String key  = files[i].toUpperCase( Locale.JAPAN ) ;

			int idx = key.lastIndexOf( '.' );
			if( idx >= 0 ) {
				map.put( key.substring( 0,idx ), files[i] );
			}
			else {
				map.put( key, files[i] );
			}
		}
	}

	/**
	 * 指定のキーのファイルが存在しているかどうかを返します。
	 * 存在している場合は、true , 存在していない場合は、false になります。
	 *
	 * @param   key String
	 * @return  boolean 存在しているかどうか(true:存在する/false:存在しない)
	 * @throws  IllegalArgumentException
	 */
	public boolean exists( final String key ) {
		if( key == null ) {
			String errMsg = "キーが指定されていません。" ;
			throw new IllegalArgumentException( errMsg );
		}

		return map.containsKey( key.toUpperCase( Locale.JAPAN ) );
	}

	/**
	 * キーに対応したファイル名を返します。
	 * 指定のキーに対するファイル名が存在しない場合は、null を返します。
	 *
	 * @param   key String
	 * @return  String ファイル名(ディレクトリパスは含まず)
	 */
	public String getFilename( final String key ) {
		if( key == null ) {
			return null ;
	//		String errMsg = "キーが指定されていません。" ;
	//		throw new IllegalArgumentException( errMsg );
		}

		return map.get( key.toUpperCase( Locale.JAPAN ) );
	}
}
