/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;

/**
 * FixLengthData.java は、固定長データを作成するための簡易クラスです。
 *
 * データの項目(String[])を、それぞれの中で最大桁数にあわせて、スペース埋めします。
 * 各項目間に、追加するスペース数は、setAddLength( int[] ) メソッドで、
 * 各項目のタイプ(半角文字、全角混在、数字)の指定は、setType( int[] ) メソッド行います。
 *
 * このクラスは同期処理は保障されていません。
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */

public final class FixLengthData {

	/** 項目タイプの定義変数	{@value}	*/
	public static final int X = 0 ;		// X:半角文字
	public static final int S = 1 ;		// S:数字(前ゼロ)
	public static final int K = 2 ;		// K:半角全角混在

	private static final String CR = System.getProperty("line.separator");
	public static final String ENCODE = "Windows-31J" ;

	private final int[]	addLen	;		// 各データ間に追加するスペースを設定する。
	private final int[]	type	;		// 各データの固定長形式。 0:半角文字 1:数字(前ゼロ) 2:半角全角混在
	private final int	size	;		// データの個数
	private String	encode	= ENCODE;

	private int[]		maxLen	= null;		// 内部変数。各データの最大長を記憶する。
	private String[]	fill_X	= null;		// スペースの文字列
	private String[]	fill_S	= null;		// ゼロの文字列
	private final List<String[]> list = new ArrayList<String[]>();

	/**
	 * データの項目数を指定して、オブジェクトを構築します。
	 *
	 * @param   len int データの項目数
	 */
	public FixLengthData( final int len ) {
		size	= len ;
		addLen	= new int[size];
		type	= new int[size];
		maxLen	= new int[size];
	}

	/**
	 * データの全角混在時に文字列長を算出するのに使用する エンコード方式を指定します。
	 * 固定長では、基本的には、全角２Byte 半角１Byte で換算すべきです。
	 * 設定値が、null または、ゼロ文字列の場合は、NullPointerException が throw されます。
	 * 初期値は、"Windows-31J" です。
	 *
	 * @param   encode String 全角混在時の固定長文字数算出エンコード
	 * @throws  IllegalArgumentException 引数が null または、ゼロ文字列の場合
	 */
	public void setEncode( final String encode ) {
		if( encode == null || encode.length() == 0 ) {
			String errMsg = "エンコードに、null または、ゼロ文字列は、指定できません。";
			throw new IllegalArgumentException( errMsg );
		}

		this.encode = encode;
	}

	/**
	 * データの項目に対応した、固定時の間に挿入する空白文字数を指定します。
	 * 初期値は、0 です。
	 *
	 * @param   inAddLen int[] データのカラム数
	 * @throws  IllegalArgumentException 引数のデータ件数が、コンストラクタで指定した数と異なる場合
	 */
	public void setAddLength( final int[] inAddLen ) {
		int len = (inAddLen == null) ? 0 : inAddLen.length ;
		if( len != size ) {
			String errMsg = "引数のデータ件数が、コンストラクタで指定した数と異なります。"
								+ "SIZE=[" + size + "] , 引数長=[" + len + "]" ;
			throw new IllegalArgumentException( errMsg );
		}

		System.arraycopy( inAddLen,0,addLen,0,size );
	}

	/**
	 * データの各項目のタイプ(半角文字、数字)を指定します。
	 * X:半角文字の場合は、データを前方に、余った分を後方にスペースを埋めます。
	 * S:数字(前ゼロ)の場合は、データを後方に、余った分をゼロを前方に埋めます。
	 * K:半角全角混在の場合は、encode で文字数を求めるとともに、X:半角文字と同様の処理を行います。
	 * 初期値は、X:半角文字 です。
	 *
	 * @param   inType int[] データの各項目のタイプ
	 * @see #X
	 * @see #S
	 * @throws  IllegalArgumentException 引数のデータ件数が、コンストラクタで指定した数と異なる場合
	 */
	public void setType( final int[] inType ) {
		int len = (inType == null) ? 0 : inType.length ;
		if( len != size ) {
			String errMsg = "引数のデータ件数が、コンストラクタで指定した数と異なります。"
								+ "SIZE=[" + size + "] , 引数長=[" + len + "]" ;
			throw new IllegalArgumentException( errMsg );
		}

		System.arraycopy( inType,0,type,0,size );
	}

	/**
	 * データの各項目に対応した配列データを設定します。
	 *
	 * @param   inData String[] データの各項目の配列データ
	 * @throws  IllegalArgumentException 引数のデータ件数が、コンストラクタで指定した数と異なる場合
	 */
	public void addListData( final String[] inData ) {
		int len = (inData == null) ? 0 : inData.length ;
		if( len != size ) {
			String errMsg = "引数のデータ件数が、コンストラクタで指定した数と異なります。"
								+ "SIZE=[" + size + "] , 引数長=[" + len + "]" ;
			throw new IllegalArgumentException( errMsg );
		}

		// 最大データ長の取得のみ行っておきます。
		try {
			for( int i=0; i<size; i++ ) {
				if( inData[i] != null ) {
					if( type[i] == K ) {
						len = inData[i].getBytes( encode ).length ;
					}
					else {
						len = inData[i].length();
					}
					if( maxLen[i] < len ) { maxLen[i] = len; }
				}
			}
		}
		catch( UnsupportedEncodingException ex ) {
			String errMsg = "データの変換に失敗しました。[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		list.add( inData );
	}

	/**
	 * 指定の行に対する固定文字数に設定された文字列を返します。
	 * 引数の行番号は、addListData(String[])メソッドで登録された順番です。
	 *
	 * @param   line int 行番号(addListData で登録した順)
	 * @return  String 固定文字数に設定された文字列
	 */
	public String getFixData( final int line ) {
		if( fill_X == null ) { makeSpace(); }	// 初期処理

		String[] data = list.get( line );
		StringBuilder rtn = new StringBuilder();
		for( int i=0; i<size; i++ ) {
			String dt = ( data[i] == null ) ? "" : data[i] ;
			switch( type[i] ) {
				case S: // ゼロで埋めてから、文字を出力する。
						rtn.append( fill_S[i].substring( dt.length() ) );
						rtn.append( dt );
						break;
				case X: // 文字を出力してから、スペースで埋める。
						rtn.append( dt );
						rtn.append( fill_X[i].substring( dt.length() ) );
						break;
				case K: // 全角を含む文字を出力してから、スペースで埋める。
						try {
							int len = dt.getBytes( encode ).length ;
							rtn.append( dt );
							rtn.append( fill_X[i].substring( len ) );
						}
						catch( UnsupportedEncodingException ex ) {
							String errMsg = "データの変換に失敗しました。[" + encode + "]" ;
							throw new RuntimeException( errMsg,ex );
						}
						break;
				default: // 基本的にありえない
						String errMsg = "不正な種別が指定されました[" + type[i] + "]" ;
						throw new RuntimeException( errMsg );
				//		break;
			}
		}
		return rtn.toString();
	}

	/**
	 * 内部登録済みのすべてのデータを連結して出力します。
	 * 連結時には、改行コードを設定しています。
	 *
	 * @return  String 固定文字数に設定された文字列
	 */
	public String getAllFixData() {
		StringBuilder buf = new StringBuilder( 200 );

		int len = list.size();
		for( int i=0; i<len; i++ ) {
			buf.append( getFixData( i ) ).append( CR );
		}

		return buf.toString();
	}

	/**
	 * 固定文字列を作成するための種となるスペース文字列とゼロ文字列を作成します。
	 *
	 */
	private void makeSpace() {
		fill_X	= new String[size];
		fill_S	= new String[size];

		for( int i=0; i<size; i++ ) {
			char[] ch = new char[maxLen[i]+addLen[i]];
			switch( type[i] ) {
				case S:
						Arrays.fill( ch, '0' );
						fill_S[i] = String.valueOf( ch );
						break;
				case X:
				case K:
						Arrays.fill( ch, ' ' );
						fill_X[i] = String.valueOf( ch );
						break;
				default: // 基本的にありえない
						String errMsg = "不正な種別が指定されました[" + type[i] + "]" ;
						throw new RuntimeException( errMsg );
				//		break;
			}
		}
	}

	/**
	 * 内部変数のデータと、最大値のキャッシュをクリアします。
	 *
	 * それ以外の変数(size、encode、addLength、type)は、設定時のまま残っています。
	 *
	 */
	public void clear() {
		list.clear() ;
		maxLen	= new int[size];
		fill_X	= null;		// スペースの文字列
		fill_S	= null;		// ゼロの文字列
	}
}
