/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.ErrorMessage;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.util.ApplicationInfo;

/**
 * ステートメント文を execute() する事により,データベースを検索した結果を DBTableModel に
 * 割り当てるインターフェースです。
 * 実際にこのインターフェースを継承したクラスでは、データベース以外に
 * フラットファイルやXMLデータなどを読み込んで DBTableModel を作成させる
 * 様な実装クラスを作成することができます。
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface Query {

	/**
	 * Query オブジェクトを初期化します。
	 * これは、QueryFactory のプールから取り出すときに（または戻すとき）に
	 * 初期化するのに使用します。
	 *
	 */
	void init() ;

	/**
	 * ステートメント文字列をセットします。
	 *
	 * @param   statement String
	 *
	 */
	void setStatement( String statement ) ;

	/**
	 * ステートメント文字列を取り出します。
	 *
	 * @return   statement String
	 *
	 */
	String getStatement();

	/**
	 * クエリーを実行します。
	 * セットされているステートメント文字列とそのタイプが合っていない場合は,
	 * エラーになります。
	 * 実行結果は、DBTableModel にセットされます。
	 *
	 */
	void execute() ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery や
	 * { call xxxx( ?,?,? ) } などの CallableStatement の ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @param   args オブジェクトの引数配列
	 */
	void execute( String[] args ) ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @param   keys オブジェクトのキー配列
	 * @param   args オブジェクトの引数配列
	 */
	void execute( final String[] keys, final String[] args ) ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 引数をすべて受け取って実行するメソッドを標準メソッドとして追加
	 *
	 * @param names       String
	 * @param dbArrayType String
	 * @param sysArg    DBSysArg[]
	 * @param userArg   DBUserArg[]
	 */
	void execute( final String names,final String dbArrayType,
					final DBSysArg[] sysArg,final DBUserArg[] userArg ) ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * [カラム名] 部分の引数を、DBTableModelから順番にセットしていきます。
	 *
	 * @param   rowNo 選択された行番号配列(登録する対象行)
	 * @param   table DBTableModel オブジェクト(登録する元データ)
	 */
	void execute( final int[] rowNo, final DBTableModel table ) ;

	/**
	 * コミットを行います。
	 *
	 */
	void commit() ;

	/**
	 * ロールバックを行います。
	 *
	 */
	void rollback() ;

	/**
	 * 使用した Statementオブジェクトをクロースし、Connection オブジェクトを
	 * プールに返します。
	 *
	 */
	void close() ;

	/**
	 * Connection オブジェクトを実際にクローズ（破棄）します。
	 * プールからも削除します。
	 * 実行時エラー等が発生したときに、このメソッドを呼び出します。
	 *
	 */
	void realClose() ;

	/**
	 * クエリーの実行結果を返します。
	 * クエリーが失敗した場合や,CallableStatement の呼び出し等で実行件数が明確でない
	 * 場合は, -1 が返されます。
	 *
	 * @return  executeCount
	 */
	int getExecuteCount() ;

	/**
	 * 実行結果の DBTableModel を返します。
	 *
	 * @return  DBTableModel
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * データベースの最大検索件数を返します。
	 * データベース自体の検索は,指定されたSQLの全件を検索しますが,
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @return  最大検索件数
	 */
	int getMaxRowCount() ;

	/**
	 * データベースの最大検索件数をセットします。
	 * データベース自体の検索は,指定されたSQLの全件を検索しますが,
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @param   maxRowCount int
	 */
	void setMaxRowCount( int maxRowCount ) ;

	/**
	 * データベースの検索スキップ件数を返します。
	 * データベース自体の検索は,指定されたSQLの全件を検索しますが,
	 * DBTableModelのデータとしては、スキップ件数分は登録されません。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @return  最大検索件数
	 */
	int getSkipRowCount();

	/**
	 * データベースの検索スキップ件数をセットします。
	 * データベース自体の検索は,指定されたSQLの全件を検索しますが,
	 * DBTableModelのデータとしては、スキップ件数分は登録されません。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @param   skipRowCount int
	 */
	void setSkipRowCount( int skipRowCount );

	/**
	 * データベースの接続先IDをセットします。
	 * システムパラメータ ファイルに定義してある データベース識別IDによって、
	 * 接続先を切り替えます。
	 * この接続先IDを元に,Connection オブジェクトを作成します。
	 *
	 * @param   connectionID String
	 */
	void setConnectionID( String connectionID ) ;

	/**
	 * アップデートフラグを取得します。
	 * これは、Query で更新処理の SQL 文を実行したときに true にセットされます。
	 * 更新処理が実行：true / 検索処理のみ：false を取得できます。
	 *
	 * @og.rev 2.1.2.3 (2002/12/02) データベース更新時に、更新フラグをセットするように変更
	 * @og.rev 4.0.0 (2007/07/20) メソッド名変更( getUpdateFlag() ⇒ isUpdate() )
	 *
	 * @return	 アップデートされたかどうか（ 更新処理：true / 検索処理：false ）
	 */
//	boolean getUpdateFlag() ;
	boolean isUpdate() ;

	/**
	 * リソースマネージャーをセットします。
	 * これは、言語（ロケール）に応じた DBColumn をあらかじめ設定しておく為に
	 * 必要です。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、内部で DBColumn オブジェクトを作成します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @param  resource リソースマネージャー
	 */
	void setResourceManager( ResourceManager resource ) ;

	/**
	 * アクセスログ取得の為,ApplicationInfo オブジェクトを設定します。 <br />
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @param   appInfo ApplicationInfo
	 */
	void setApplicationInfo( ApplicationInfo appInfo ) ;

	/**
	 * エラーコード を取得します。
	 * エラーコード は、ErrorMessage クラスで規定されているコードです。
	 *
	 * @return   エラーコード
	 */
	int getErrorCode() ;

	/**
	 * エラーメッセージオブジェクト を取得します。
	 *
	 * @return   エラーメッセージオブジェクト
	 */
	ErrorMessage getErrorMessage() ;
}
