/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.filter;

import org.opengion.fukurou.util.Closer;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import java.util.zip.GZIPOutputStream;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;

/**
 * GZIPFilter で使用する、GZIP圧縮するServletOutputStreamクラスです。
 *
 * @og.group フィルター処理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class GZIPResponseStream extends ServletOutputStream {
	protected ByteArrayOutputStream baos = null;
	protected GZIPOutputStream gzipstream = null;
	protected boolean isClosed = false;
	protected HttpServletResponse response = null;
	protected ServletOutputStream output = null;

	/**
	 * コンストラクター
	 *
	 * @param response HttpServletResponse
	 * @throws IOException
	 */
	public GZIPResponseStream(final HttpServletResponse response) throws IOException {
		// 4.3.4.4 (2009/01/01)
//		super();
		isClosed = false;
		this.response = response;
		this.output = response.getOutputStream();
		baos = new ByteArrayOutputStream();
		gzipstream = new GZIPOutputStream(baos);
	}

	/**
	 * このストリームを閉じ、このストリームに関連するすべてのシステムリソースを解放します。
	 *
	 * close の汎用規約では、close は出力ストリームを閉じます。閉じられたストリームは
	 * 出力処理を実行できません。また、それを開き直すことはできません。
	 *
	 * @throws IOException
	 */
	public void close() throws IOException {
		if(isClosed) {
			throw new IOException("This output stream has already been closed");
		}
		try {
			gzipstream.finish();

			byte[] bytes = baos.toByteArray();

//			response.addHeader("Content-Length", Integer.toString(bytes.length));
			response.setContentLength( bytes.length );
// System.out.println( bytes.length );
			response.addHeader("Content-Encoding", "gzip");
			output.write(bytes);
			output.flush();
		}
		finally {
			isClosed = true;
			Closer.ioClose( output );
		}
	}

	/**
	 * この出力ストリームをフラッシュし、バッファに入っている出力バイトをすべて強制的書き込みますに。
	 *
	 * flush の汎用規約では、それまでに書き込まれたバイトが出力ストリームの
	 * 実装によってバッファに入れられている場合に flush を呼び出すと、それらのバイトは
	 * ただちにその目的の転送先に書き込まれます。
	 *
	 * @throws IOException
	 */
	public void flush() throws IOException {
		if(isClosed) {
			throw new IOException("Cannot flush a closed output stream");
		}
		gzipstream.flush();
	}

	/**
	 * この出力ストリームに指定されたバイトを書き込みます。
	 *
	 * write の汎用規約では、1 バイトが
	 * 出力ストリームに書き込まれます。書き込まれるバイトは、引数 b の下位 8 ビットです。
	 * b の上位 24 ビットは無視されます。
	 *
	 * @param bt int byteデータ
	 * @throws IOException
	 */
	public void write(final int bt) throws IOException {
		if(isClosed) {
			throw new IOException("Cannot write to a closed output stream");
		}
		gzipstream.write((byte)bt);
	}

	/**
	 * 指定されたバイト配列からこの出力ストリームに b.length バイトを書き込みます。
	 *
	 * write(b) の汎用規約では、write(b) の効果は write(b, 0, b.length) を呼び出す
	 * 場合とまったく同じです。
	 *
	 * @param bt byte[] データ
	 * @throws IOException
	 */
	public void write(final byte[] bt) throws IOException {
		write(bt, 0, bt.length);
	}

	/**
	 * オフセット off から始まる指定のバイト配列からこの出力ストリームに len バイトを書き込みます。
	 *
	 * write(b, off, len) の汎用規約では、配列 b 内の一定のバイトが出力ストリームに順番に
	 * 書き込まれます。この処理で最初に書き込まれるバイトは要素 b[off]、最後に書き込まれる
	 * バイトは要素 b[off+len-1] です。
	 *
	 * @param bt byte[]
	 * @param off int
	 * @param len int
	 * @throws IOException
	 */
	public void write(final byte bt[], final int off, final int len) throws IOException {
//		System.out.println("writing...");
		if(isClosed) {
			throw new IOException("Cannot write to a closed output stream");
		}
		gzipstream.write(bt, off, len);
	}

	/**
	 * すでにストリームが閉じられているかどうかを返します。
	 *
	 * @return boolean
	 */
	public boolean closed() {
		return isClosed;
	}
}
