/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.XHTMLTag;

import org.opengion.fukurou.util.StringUtil;

/**
 * 【入力】複数行の入力が可能なテキスト入力フィールドを作成するHTML拡張タグです。
 *
 * name 属性に ラベルリソース のキーを与えることで、 ロケールにあわせたリソースを使用して、
 * 画面に表示します。
 * ロケールは、ユーザー情報の lang 属性を初期値で使用し、 セットされていない場合は、
 * リクエスト情報のロケールから取得します。
 * ラベル部分と入力フィールド部分がテーブルタグの &lt;td&gt; により左右に分割されます。
 * HTML 表示時は、前後に&lt;tr&gt;タグで囲って、 整形できます。
 *
 * @og.formSample
 * ●形式：&lt;og:queryOption &gt;
 *            &lt;jsp:text &gt; SELECT文 &lt;/jsp:text&gt;
 *         &lt;/og:queryOption &gt;
 * ●body：あり
 *
 * ●使用例
 *     &lt;og:textarea name="BUN" /&gt;
 *
 *     &lt;og:textarea name="BUN"&gt;            初期値に値をセットしたいとき
 *         &lt;jsp:text&gt;test&lt;/jsp:text&gt;
 *     &lt;/og:textarea name="BUN"&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TextareaTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.0 (2003/03/28) body 属性に、 {&#064;XXXX} パース機能を付加する。
	 *
	 * @return  int
	 */
	@Override
	public int doAfterBody() {
		String str = getBodyString();

		if( str != null && str.length() > 0 ) {
			String label = StringUtil.htmlFilter( str );
			if( label != null && label.length() > 0 ) {
				set( "body",label );
			}
		}

		return(SKIP_BODY);
	}

	/**
	 * 検索条件の入力用のテキストフィールドを作成します。
	 *
	 * テーブルタグによりフィールドは分割されます。
	 * 使用時は、テーブルタグを前後に使用して下さい。
	 * 
	 * @og.rev 4.3.7.1 (2009/06/08) id=labelのclass化
	 *
	 * @return  入力用のテキストフィールドタグ
	 */
	protected String makeTag() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		if( getMsglbl()  == null ) { setLbl( get( "name" ) ); }

		// 4.3.7.1 (2009/06/08) id=labelのclass化
		// rtn.append( "<td id=\"label\">" );
		rtn.append( "<td class=\"label\">" );
		rtn.append( getLongLabel() );		// 4.0.0 (2005/01/31)
		rtn.append( "</td>" );
		rtn.append( "<td>" );
		rtn.append( XHTMLTag.textarea( getAttributes() ) );
		rtn.append( "</td>" );

		return rtn.toString() ;
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag 名前を指定します。
	 *
	 * @param   name String
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】高さ(行数)を指定します。
	 *
	 * @og.tag 高さ(行数)を指定します。
	 *
	 * @param   rows String
	 */
	public void setRows( final String rows ) {
		set( "rows",getRequestParameter( rows ) );
	}

	/**
	 * 【HTML】横幅(カラム数)を指定します。
	 *
	 * @og.tag 横幅を指定します。
	 *
	 * @param   cols String
	 */
	public void setCols( final String cols ) {
		set( "cols",getRequestParameter( cols ) );
	}
}
