/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.ErrorMessage;

/**
 * 旧OASで実装していた、半角文字(カナ含む)名称用記号を扱う為の、カラム属性を定義します。
 *
 * 使用可能文字は、半角英数大小文字,スペース,半角カタカナ,･,+,-,(,),*,#,/,.,% です。(,は含みません)
 * 半角カタカナのUnicode範囲は、(c &gt;= 0xff65 && c &lt;= 0xff9f ) を有効とします。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角文字＋半角カタカナ＋特殊名称チェック
 * ・文字パラメータの 正規表現チェック
 *
 * @og.rev 3.8.0.2 (2005/07/11) 新規作成
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_OASNM extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final String OAS_NAME = "+-()*#/.%" ;

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @param   key String
	 * @param   value String
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   param    dbType パラメータ
	 * @return  エラー内容
	 */
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String param ) {
		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		String check = DBTypeCheckUtil.byteLengthCheck( value,len );
		if( check != null ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,check,String.valueOf( len ) );
		}

		StringBuilder val = new StringBuilder();
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			char ch = value.charAt( i );
			boolean okFlag = (  (ch >= '0' && ch <= '9' ) || (ch >= 'A' && ch <= 'Z' ) ||
								(ch >= 'a' && ch <= 'z' ) || ( OAS_NAME.indexOf( ch ) >= 0 ) ||
								(ch >= 0xff65 && ch <= 0xff9f ) || ch == ' ' );

			if( okFlag ) {
				val.append( ch );
			}
			else {
				val.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,val.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,param );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
