/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;

import java.text.DecimalFormat;

/**
 * <p>DECIMAL エディターは、カラムのデータをDecimal（１０進数、少数）表示する場合に
 * 使用するクラスです。</p>
 *
 * 編集パラメータに与えられた文字列は、java.text.DecimalFormat を使用して
 * フォーマットされます。
 * フォーマット変換前に、カンマなどの数値変換時にエラーになる情報を削除しておきます。
 *
 * <p>フォーマットルールは、{@link java.text.DecimalFormat} を参照願います。</p>
 *
 * <p> カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。<br />
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。</p>
 *
 * @og.group データ編集
 * @og.rev 3.8.8.5 (2007/03/09) 新規作成
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_DECIMAL extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final int       minFraction;
	private final DecimalFormat format ;
	private final String	defValue ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_DECIMAL() {
		// 4.3.4.4 (2009/01/01)
//		super();
		minFraction = 0;
		format = null;
		defValue   = null;
	}

	/**
	 * コンストラクター。
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Editor_DECIMAL( final DBColumn clm ) {
		super( clm );
		minFraction = clm.getSizeY();

		String fm = clm.getRendererParam();
		if( fm == null || fm.length() == 0 || fm.equals( "_" ) ) {
			fm = "#,##0.#";
		}
		format = new DecimalFormat( fm );

		String defv = clm.getDefault();
		double dd = 0.0;
		if( defv != null && defv.length() != 0 && !defv.equals( "_" ) ) {
			dd  = StringUtil.parseDouble( defv );
		}

		defValue = format.format( dd );

		// -XXX,XXX,XXX.YYY 形式では、カンマ、小数点、マイナスなどの項目が入ります。
		int maxlength = (int)(clm.getSizeX() * 4.0/3.0) + minFraction + 2 ;	// +2 は、マイナスと少数点
		attributes.set( "maxlength"   ,String.valueOf( maxlength ) );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellEditor オブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DECIMAL( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @param   value String
	 * @return  データの編集用文字列
	 */
	public String getValue( final String value ) {
		return super.getValue( formatValue( value ) );
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param	row   int 行番号
	 * @param	value String
	 * @return	データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {
		return super.getValue( row,formatValue( value ) );
	}

	/**
	 * 内部的なフォーマット変換後の文字列を返します。
	 *
	 * @param	value String
	 * @return	データの表示用文字列
	 */
	private String formatValue( final String value ) {
		if( value == null || (value.trim()).length() == 0 ) {
			return defValue;
		}

		double dd = StringUtil.parseDouble( value );

		final String rtn ;
		synchronized( format ) {
			rtn = format.format( dd );
		}

		return rtn;
	}
}
