/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.StringUtil;

import java.text.DecimalFormat;

/**
 * DECIMAL レンデラーは、カラムのデータをDecimal（１０進数、少数）表示する場合に
 * 使用するクラスです。
 *
 * 表示パラメータに与えられた文字列は、java.text.DecimalFormat を
 * 使用してフォーマットされます。
 * フォーマット変換前に、カンマなどの数値変換時にエラーになる情報を削除しておきます。
 *
 * フォーマットルールは、{@link java.text.DecimalFormat} を参照願います。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.0.0 (2003/09/17) 新規作成
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_DECIMAL extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final DecimalFormat format ;
	private final String	defValue ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_DECIMAL() {
		format = null;
		defValue   = null;
	}

	/**
	 * 引数つきコンストラクター。
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Renderer_DECIMAL( final DBColumn clm ) {

		String fm = clm.getRendererParam();
		if( fm == null || fm.length() == 0 || fm.equals( "_" ) ) {
			fm = "#,##0.#";
		}
		format = new DecimalFormat( fm );

		String defv = clm.getDefault();
		double dd = StringUtil.parseDouble( defv );

		defValue = format.format( dd );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellRenderer オブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_DECIMAL( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.8.5.2 (2006/05/31) カンマ編集された数字の対応
	 *
	 * @param	value String
	 * @return	データの表示用文字列
	 */
	public String getValue( final String value ) {
		if( value == null || (value.trim()).length() == 0 ) {
			return defValue;
		}

		double dd = StringUtil.parseDouble( value );

		String rtn ;
		synchronized( format ) {
			rtn = format.format( dd );
		}

		if( dd < 0.0 ) {
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}

		return rtn;
	}
}
