/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * TEXTAREA レンデラは、カラムのデータをテキストエリアで表示する場合に
 * 使用するクラスです。
 * readonlyのテキストエリアでclass=renderer-textareaとして出力し、
 * name属性は付けません。（データは送信されません）
 * エリアの縦、横サイズはエディタのテキストエリアと同様にして算出されます。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 4.3.5.7 (2009/03/22) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class Renderer_TEXTAREA extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;
	
	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" ) ;	// 表示フィールドの大きさ
	// viewタグで表示する場合のカラムの大きさ
	private final int VIEW_COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_VIEW_COLUMNS_MAXSIZE" ) ;
	
	private final TagBuffer	tagBuffer = new TagBuffer();

	private  String		rows1;
	private  String		rows2;
	private  String		size1;
	private  String		size2;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Renderer_TEXTAREA() {
		// 何もありません。(super を呼び出しません)
	}

	/**
	 * コンストラクター
	 * textareaのサイズを決めるため、sizeとrowを決定する
	 * editorの計算を移植。
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Renderer_TEXTAREA( final DBColumn clm ) {
		String size = clm.getViewLength();
		int maxlength = clm.getTotalSize();

		if( size != null ) {
			if( size.indexOf( ',' ) >= 0 ) {
				size = size.substring( 0, size.indexOf( ',' ) );
			}
			size1 = size;
			size2 = size;
		}
		else {
			size1 = String.valueOf( clm.getFieldSize( maxlength, COLUMNS_MAXSIZE ) );
			size2 = String.valueOf( clm.getFieldSize( maxlength, VIEW_COLUMNS_MAXSIZE ) );
		}

		int r1 = ( clm.getTotalSize() / Integer.parseInt( size1 ) ) + 1;
		if( r1 > 5 ) {
			rows1 = "5";
		}
		else {
			rows1 = String.valueOf( r1 );
		}

		int r2 = ( clm.getTotalSize() / Integer.parseInt( size2 ) ) + 1;
		if( r2 > 5 ) {
			rows2 = "5";
		}
		else {
			rows2 = String.valueOf( r2 );
		}

		String param = StringUtil.nval( clm.getRendererParam(), clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			int st = param.indexOf( ',' );
			if( st > 0 ) {
				rows1 = param.substring( 0, st );
				rows2 = rows1;
				size1 = param.substring( st + 1 );
				size2 = size1;
			}
		}
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellEditor オブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		return new Renderer_TEXTAREA( clm );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @param   value String
	 * @return  データの表示用文字列
	 */
	public String getValue( final String value ) {

		TagBuffer tag = new TagBuffer( "textarea" );
		tag.add( "cols"    , size1 );
		tag.add( "rows"    , rows1 );
		tag.add( "readonly", "readonly" );
		tag.add( "class"   , "renderer-textarea" );
		tag.add( tagBuffer.makeTag() );
		tag.setBody( value );

		return tag.makeTag();
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @param   row   int 行番号
	 * @param   value String
	 * @return  データ表示用の文字列
	 */
	public String getValue( final int row,final String value ) {

		TagBuffer tag = new TagBuffer( "textarea" );
		tag.add( "cols"    , size2 );
		tag.add( "rows"    , rows2 );
		tag.add( "readonly", "readonly" );
		tag.add( "class"   , "renderer-textarea" );
		tag.add( tagBuffer.makeTag() );
		tag.setBody( value );

		return tag.makeTag( row,value );
	}
}
