/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.fukurou.util.Closer ;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.FileInputStream;
// import java.io.IOException;
import java.text.SimpleDateFormat;
import java.text.DateFormat ;
import java.text.NumberFormat ;
import java.text.DecimalFormat ;
import java.util.Locale;

/**
 * POI による、EXCELバイナリファイルを読み取る実装クラスです。
 *
 * ファイル名、シート名を指定して、データを読み取ることが可能です。
 * 第一カラムが # で始まる行は、コメント行なので、読み飛ばします。
 * カラム名の指定行で、カラム名が null の場合は、その列は読み飛ばします。
 * 
 * 入力形式は、openXML形式にも対応しています。
 * ファイルの内容に応じて、.xlsと.xlsxのどちらで読み取るかは、内部的に
 * 自動判定されます。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規作成
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableReader_Excel extends TableReader_Default {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String  sheetName		= null;		// 3.5.4.2 (2003/12/15)
	private String  filename		= null;		// 3.5.4.3 (2004/01/05)

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @see #isExcel()
	 */
	public void readDBTable() {
		InputStream  in = null;
		try {
			in = new FileInputStream(filename);

//			POIFSFileSystem fs = new POIFSFileSystem(in);
//			HSSFWorkbook wb = new HSSFWorkbook(fs);
//			HSSFSheet sheet ;
			Workbook wb = WorkbookFactory.create(in);
			Sheet sheet ;

			if( sheetName == null || sheetName.length() == 0 ) {
				sheet = wb.getSheetAt(0);
			}
			else {
				sheet = wb.getSheet( sheetName );
				if( sheet == null ) {
					String errMsg = "対応するシートが存在しません。 Sheet=[" + sheetName + "]" ;
					throw new HybsSystemException( errMsg );
				}
			}

			boolean  nameNoSet = true;
			table = DBTableModelUtil.newDBTable();

			int numberOfRows = 0;
			HeaderData data = new HeaderData();

			int nFirstRow = sheet.getFirstRowNum();
			int nLastRow  = sheet.getLastRowNum();
			for( int nIndexRow = nFirstRow; nIndexRow <= nLastRow; nIndexRow++) {
//				HSSFRow oRow = sheet.getRow(nIndexRow);
				Row oRow = sheet.getRow(nIndexRow);
				if( data.isSkip( oRow ) ) { continue; }
				if( nameNoSet ) {
					nameNoSet = false;
					table.init( data.getColumnSize() );
					setTableDBColumn( data.getNames() ) ;
				}

				if( numberOfRows < getMaxRowCount() ) {
					table.addColumnValues( data.row2Array( oRow ) );
					numberOfRows ++ ;
				}
				else {
					table.setOverflow( true );
				}
			}

			// 最後まで、#NAME が見つから無かった場合
			if( nameNoSet ) {
				String errMsg = "最後まで、#NAME が見つかりませんでした。"
								+ HybsSystem.CR
								+ "ファイルが空か、もしくは損傷している可能性があります。"
								+ HybsSystem.CR ;
				throw new HybsSystemException( errMsg );
			}
		}
//		catch (IOException e) {
		catch (Exception e) {
			String errMsg = "ファイル読込みエラー[" + filename + "]"  ;
			throw new HybsSystemException( errMsg,e );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.ioClose( in );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取る要に変更します。
	 * @og.rev 4.0.0 (2006/09/31) UnsupportedOperationException を発行します。
	 *
	 * @param   reader BufferedReader (使用していません)
	 */
	public void readDBTable( final BufferedReader reader ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * デフォルトは、第一シートです。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName String
	 */
	public void setSheetName( final String sheetName ) {
		this.sheetName = sheetName;
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return boolean
	 */
	public boolean isExcel() {
		return true;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = filename;
		if( filename == null ) {
			String errMsg = "ファイル名が指定されていません。" ;
			throw new HybsSystemException( errMsg );
		}
	}
}

/**
 * EXCEL ネイティブのデータを処理する ローカルクラスです。
 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
 * 行情報（Row）から、カラムの配列の取得などを行います。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規追加
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   儲
 * @since    JDK5.0,
 */
class HeaderData {
	private String[] names ;
//	private short[]  index;
	private int[]    index; // 4.3.4.0 (2008/12/01) POI3.2対応
	private int		 columnSize = 0;
	private boolean  nameNoSet = true;

	/**
	 * EXCEL ネイティブのデータを処理する ローカルクラスです。
	 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
	 * 行情報（Row）から、カラムの配列の取得などを行います。
	 * 
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 * @return true:コメント行/false:通常行
	 */
//	boolean isSkip( HSSFRow oRow ) {
	boolean isSkip( Row oRow ) {
		if( oRow == null ) { return true; }

//		short nFirstCell = oRow.getFirstCellNum();
		int nFirstCell = oRow.getFirstCellNum();
//		HSSFCell oCell = oRow.getCell(nFirstCell);
		Cell oCell = oRow.getCell(nFirstCell);
		String strText =  getValue( oCell );
		if( strText != null && strText.length() > 0 ) {
			if( nameNoSet ) {
				if( strText.equalsIgnoreCase( "#Name" ) ) {
					makeNames( oRow );
					nameNoSet = false;
					return true;
				}
				else if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。"
									+ HybsSystem.CR
									+ "可能性として、ファイルが、ネイティブExcelでない事が考えられます。"
									+ HybsSystem.CR ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
			}
		}

		return nameNoSet ;
	}

	/**
	 * EXCEL ネイティブの行情報（Row）からカラム名情報を取得します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) POI3.2対応
	 * 
	 * @param oRow Row EXCELの行オブジェクト
	 */
//	private void makeNames( final HSSFRow oRow ) {
	private void makeNames( final Row oRow ) {
		short nFirstCell = oRow.getFirstCellNum();
		short nLastCell  = oRow.getLastCellNum();

		int maxCnt = nLastCell - nFirstCell;
		String[] names2 = new String[maxCnt];
//		short[]  index2 = new short[maxCnt];
		int[]  index2 = new int[maxCnt];

		// 先頭カラムは、#NAME 属性行である。++ で、一つ進めている。
//		for( short nIndexCell = ++nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
		for( int nIndexCell = ++nFirstCell; nIndexCell <= nLastCell; nIndexCell++) {
//			HSSFCell oCell = oRow.getCell(nIndexCell);
			Cell oCell = oRow.getCell(nIndexCell);
			String strText = getValue( oCell );

			if( strText != null && strText.length() > 0 ) {
				names2[columnSize] = strText;
				index2[columnSize] = nIndexCell;
				columnSize++;
			}
		}

		if( maxCnt == columnSize ) {
			names = names2;
			index = index2;
		}
		else {
			names = new String[columnSize];
//			index = new short[columnSize];
			index = new int[columnSize];
			System.arraycopy(names2, 0, names, 0, columnSize);
			System.arraycopy(index2, 0, index, 0, columnSize);
		}
	}

	/**
	 * カラム名情報を返します。
	 * ここでは、内部配列をそのまま返します。
	 *
	 * @return String[] カラム列配列情報
	 */
	String[] getNames() {
		return names;
	}

	/**
	 * カラムサイズを返します。
	 *
	 * @return int カラムサイズ
	 */
	int getColumnSize() {
		return columnSize;
	}

	/**
	 * カラム名情報を返します。
	 *
	 * @param oRow Row EXCELの行オブジェクト
	 * @return String[] カラム列配列情報
	 */
//	String[] row2Array( final HSSFRow oRow ) {
	String[] row2Array( final Row oRow ) {
		if( nameNoSet ) {
			String errMsg = "#NAME が見つかる前にデータが見つかりました。";
			throw new HybsSystemException( errMsg );
		}

		String[] data = new String[columnSize];
		for( int i=0;i<columnSize; i++ ) {
//			HSSFCell oCell = oRow.getCell( index[i] );
			Cell oCell = oRow.getCell( index[i] );
			data[i] = getValue( oCell );
		}
		return data;
	}

	/**
	 * セルオブジェクト（Cell）から値を取り出します。<br />
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 取り出し方法を少し修正
	 *
	 * @param oCell Cell EXCELのセルオブジェクト
	 * @return String セルの値
	 */
//	private String getValue( final Cell oCell ) {
	private String getValue( final Cell oCell ) {
		if( oCell == null ) { return null; }

		String strText = "";
//		HSSFRichTextString richText;
		RichTextString richText;
		int nCellType = oCell.getCellType();
		switch(nCellType) {
//			case HSSFCell.CELL_TYPE_NUMERIC:
			case Cell.CELL_TYPE_NUMERIC:
					strText = getNumericTypeString( oCell );
					break;
//			case HSSFCell.CELL_TYPE_STRING:
			case Cell.CELL_TYPE_STRING:
	// POI3.0		strText = oCell.getStringCellValue();
					richText = oCell.getRichStringCellValue();
					if( richText != null ) {
						strText = richText.getString();
					}
					break;
//			case HSSFCell.CELL_TYPE_FORMULA:
			case Cell.CELL_TYPE_FORMULA:
	// POI3.0		strText = oCell.getStringCellValue();
					richText = oCell.getRichStringCellValue();
					if( richText != null ) {
						strText = richText.getString();
					}
					else {
						strText = getNumericTypeString( oCell );
					}
					break;
//			case HSSFCell.CELL_TYPE_BOOLEAN:
			case Cell.CELL_TYPE_BOOLEAN:
					strText = String.valueOf(oCell.getBooleanCellValue());
					break;
//			case HSSFCell.CELL_TYPE_BLANK :
//			case HSSFCell.CELL_TYPE_ERROR:
			case Cell.CELL_TYPE_BLANK :
			case Cell.CELL_TYPE_ERROR:
					break;
			default :
				break;
		}
		return strText.trim();
	}

	/**
	 * セル値が数字の場合に、数字か日付かを判断して、対応する文字列を返します。<br />
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param oCell Cell
	 * @return String 数字の場合は、文字列に変換した結果を、日付の場合は、"yyyyMMddHHmmss" 形式で返します。
	 *
	 */
//	private String getNumericTypeString( final HSSFCell oCell ) {
	private String getNumericTypeString( final Cell oCell ) {
		final String strText ;

		double dd = oCell.getNumericCellValue() ;
//		if( HSSFDateUtil.isCellDateFormatted( oCell ) ) {
		if( DateUtil.isCellDateFormatted( oCell ) ) {
			DateFormat dateFormat = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
//			strText = dateFormat.format( HSSFDateUtil.getJavaDate( dd ) );
			strText = dateFormat.format( DateUtil.getJavaDate( dd ) );
		}
		else {
			NumberFormat numFormat = NumberFormat.getInstance();
			if( numFormat instanceof DecimalFormat ) {
				((DecimalFormat)numFormat).applyPattern( "#.####" );
			}
			strText = numFormat.format( dd );
		}
		return strText ;
	}
}
