/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.ErrorMessage;

/**
 * データベースのカラム属性のうち、静的論理属性を規定する情報を保持しているオブジェクトです。
 * このインターフェースを実装したクラスは、カラムの名前、ラベル、桁数、
 * 種類（文字、数字、日付）やタイプ（全角、半角、大文字、小文字）の情報を
 * 持っています。さらに、そのカラムの表示方法（CellRenderer）や、
 * 編集方法（CellEditor）を静的に割り当てる事が可能です。
 * これらは，カラムの静的情報であって、そのカラムに対する値を保持したり
 * 各アプリケーション毎に変更される情報は持っていません。すべての画面で
 * 同一の情報を保持しています。
 * データを登録する場合の最低限の整合性チェックは、このクラスを用いて
 * 行います。
 * チェック方法は、種類やタイプで異なりますが，それらはこのインターフェース
 * を実装したサブクラス毎に異なります。
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface DBType {

// 4.1.1.2 (2008/02/28) fukurou.model.NativeTypeに移動
//	/** NATIVEの型 [int] の識別コード ({@value})  */
//	static final int NATIVE_INT  = 0 ;
//
//	/** NATIVEの型 [long] の識別コード ({@value})  */
//	static final int NATIVE_LONG  = 1 ;
//
//	/** NATIVEの型 [double] の識別コード ({@value})  */
//	static final int NATIVE_DOUBLE  = 2 ;
//
//	/** NATIVEの型 [String] の識別コード ({@value})  */
//	static final int NATIVE_STRING  = 3 ;
//
//	/** NATIVEの型 [Calendar] の識別コード ({@value})  */
//	static final int NATIVE_CALENDAR  = 4 ;

	/**
	 * NATIVEの型の識別コードを返します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	NativeType getNativeType() ;

	/**
	 * 文字タイプに応じた方法で固定長に変換した文字列を返します。
	 * 例えば，全角なら全角スペースで、数字タイプならゼロ埋めします。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	String valueFill( String value ,int sizeX ,int sizeY,String encode ) ;

	/**
	 * そのDBTypeの,デフォルトの値（物理的初期設定値）を返します。
	 *
	 * 一般に、文字列の場合は,ゼロストリング""  数字の場合は "0" です。
	 *
	 * @return  物理的初期設定値
	 */
	String getDefault() ;

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合（A,B,C など）は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。（桁上がりもあり）
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列（""）の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @param   value  String
	 * @return  String引数の文字列を＋１した文字列
	 */
	String valueAdd( String value ) ;

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと CellEditor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @param   value String （一般に編集データとして登録されたデータ）
	 * @return  修正後の文字列（一般にデータベースに登録するデータ）
	 */
	String valueSet( String value ) ;

	/**
	 * action で指定されたコマンドを実行して、値の変換を行います。
	 * oldValue(旧データ)は、元のDBTableModelに設定されていた値です。通常は、
	 * この値を使用してカラム毎に変換を行います。newValue(新データ)は、引数で
	 * 指定された新しい値です。この値には、パラメータを指定して変換方法を
	 * 制御することも可能です。
	 * 指定のアクションがカラムで処理できない場合は、エラーになります。
	 *
	 * @param   action アクションコマンド
	 * @param   oldValue 入力データ（旧データ）
	 * @param   newValue 入力データ（新データ）
	 * @return	実行後のデータ
	 */
	String valueAction( String action,String oldValue,String newValue ) ;

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 *
	 * @param   key   String
	 * @param   value String
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    少数部分の文字列の長さ
	 * @param   typeParam    dbType パラメータ
	 * @return  エラー内容
	 */
	ErrorMessage valueCheck( String key ,String value ,int sizeX ,int sizeY,String typeParam ) ;
}
