/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.report.DBTableReport;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Shell;

import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

// import javax.servlet.ServletException;
import java.io.File;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

/**
 * 検索結果の DBTableModel オブジェクトをレポート形式に変換するタグです。
 *
 * データ（DBTableModel)と、コントローラ（DBTableReport クラス)を与えて、
 * 外部からコントロールすることで、各種形式で データ（DBTableModel)を表示させることが
 * 可能です。
 *
 * @og.formSample
 * ●形式：&lt;og:report fileURL="[･･･]" listId="[･･･]" ･･･ /&gt;
 * ●body：なし
 *
 * ●使用例
 *      &lt;og:report
 *         fileURL             &quot;雛型のHTMLファイルの保存してある ディレクトリを指定します。&quot;
 *         listId              &quot;雛型のHTMLファイル名を指定します。(必須)&quot;
 *         outFileURL          &quot;出力HTMLファイルの保存してあるディレクトリを指定します。&quot;
 *         outFilename         &quot;ファイルを作成するときの出力ファイル名をセットします。&quot;
 *         programFile         &quot;HTMLファイルのEXCEL化を行うバッチファイルを指定します。&quot;
 *         headerKeys          &quot;固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します。&quot;
 *         reportClass         &quot;実際に書き出すクラス名の略称(DBTableReport_**** の ****)をセットします(初期値:HTML)。&quot;
 *         language            &quot;タグ内部で使用する言語コードを指定します。&quot;
 *         tableId             &quot;(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。&quot;
 *         scope               &quot;キャッシュする場合のスコープ(request,page,session,applicaton)を指定します(初期値:session)。&quot;
 *         debug               &quot;デバッグ情報を 出力するかどうか(true/false)を指定します(初期値:false)。&quot;
 *         headerVals          &quot;固定部のKEY に対応する値をCSV形式で複数指定します。&quot;
 *         footerKeys          &quot;繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します。&quot;
 *         footerVals          &quot;固定部のKEY に対応する値をCSV形式で複数指定します。&quot;
 *         pageEndCut          &quot;ボディー部（繰り返し部）がなくなったときに、それ以降を表示するかどうか(true/false)を指定します(初期値:true)。&quot;
 *      /&gt;
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ReportTableTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	// 印刷時に使用するテンポラリフォルダ名
	private final String REPORT_URL =
		nval( HybsSystem.sys( "REPORT_FILE_URL" ) ,
				 HybsSystem.sys( "FILE_URL" ) + "REPORT/" ) ;

	// 3.8.0.4 (2005/08/08) 印刷時に使用するシステムID
	private static final String SYSTEM_ID =HybsSystem.sys( "SYSTEM_ID" );

	// 3.8.0.4 (2005/08/08) 帳票出力に準拠した方式に変更
	private final int TIMEOUT = HybsSystem.sysInt( "REPORT_DAEMON_TIMEOUT" );

	private final String  BASE_URL	= HybsSystem.sys( "FILE_URL" );

	private String[]  headerKeys	= null;   // 固定部の{@KEY} の KEY 部分を指定する。カンマで複数指定できる。
	private String[]  headerVals	= null;   // 固定部のKEY に対応する値を指定する。 {@KEY} に置き換わる。
	private String[]  footerKeys	= null;   // 繰り返し部の終了後に表示する key 部分を指定する。カンマで複数指定できる。
	private String[]  footerVals	= null;   // 繰り返し部の終了後に表示する key に対する値を指定する。
	private boolean   pageEndCut	= true;   // ボディー部（繰り返し部）がなくなったときに、それ以降のページを出力するか指定する。
	private String    fileURL		= BASE_URL;		// 雛型のHTMLファイルの保存してある ディレクトリを指定します。
	private String    outFileURL	= BASE_URL;		// 出力HTMLファイルの保存してある ディレクトリを指定します。
	private String    outFilename	= null;			// 出力HTMLファイル名を指定します。 ディレクトリ名を含んでも構いません。
	private String    reportClass	= "HTML";

	private transient DBTableModel table	= null;
	private String  tableId			= HybsSystem.TBL_MDL_KEY ;
	// 3.8.0.4 (2005/08/08) 帳票出力に準拠した方式に変更
	private String  listId			= null ;	// 雛型のHTMLファイル名を指定します。
	private String  programFile		= null;		// HTMLファイルのEXCEL化を行うバッチファイルを指定します。
//	private boolean	direct			= false;
//	private static final String		disposition = "inline";	// 固定

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.0.4 (2005/08/08) 帳票出力に準拠した方式に変更
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();

//		int rtnCode = EVAL_PAGE;
		final int rtnCode;

		table = (DBTableModel)getObject( tableId );
		if( table == null || table.getRowCount() == 0 ) {
			rtnCode = SKIP_PAGE ;		// ページの残りの処理を行わない。
		}
		else {

//			try {

				synchronized( ReportTableTag.class ) {
					String reportDir  = HybsSystem.url2dir( REPORT_URL ) + SYSTEM_ID + HybsSystem.FS + listId ;
			//		String YKNO       = HybsSystem.getDate( "yyyyMMddHHmmss" );
					String YKNO       = String.valueOf( Math.round( Math.random() * 1000000 ) ) ;

					create( reportDir,YKNO ) ;

					// 処理クラス(reportClass)が HTML の場合は、Shell を起動して、VBS で EXCEL化が必要。
					if( programFile != null && "HTML".equalsIgnoreCase( reportClass ) ) {
						String htmlFile = reportDir + HybsSystem.FS + YKNO ;
						String cmd = makeShellCommand( htmlFile,listId );
						programRun( cmd );
					}

					if( outFilename != null ) {
						File xlsFile = new File( reportDir,YKNO + ".xls" );
						File outDir  = new File( HybsSystem.url2dir( outFileURL ) );
//						if( !outDir.exists() ) { outDir.mkdirs(); }
						if( !outDir.exists() && !outDir.mkdirs() ) {
							String errMsg = "所定のフォルダを作成できませんでした。[" + outDir + "]" ;
							throw new RuntimeException( errMsg );
						}
						File outFile = new File( outDir,outFilename );
		//				File outFile = new File( outFileURL,outFilename );
						FileUtil.copy(  xlsFile , outFile );

//						String msg = getResource().getMessage( "MSG0003" ) 		// MSG0003=ファイルの登録が完了しました。
						String msg = getResource().getLabel( "MSG0003" ) 		// MSG0003=ファイルの登録が完了しました。
									+ HybsSystem.BR
//									+ getResource().getMessage( "MSG0022" ) 	// MSG0022=ファイル名
									+ getResource().getLabel( "MSG0022" ) 	// MSG0022=ファイル名
									+ ":" + outFile.getAbsolutePath() ;
						jspPrint( msg );

//						if( direct ) {
//							directLoad( outFile.getAbsolutePath(),outFilename );
//							directLoad( xlsFile.getAbsolutePath(),outFilename );
//						}
					}
//					else {
//						String redirectUrl = StringUtil.urlAppend( sys( "CONTEXT_URL" ),
//												REPORT_URL + SYSTEM_ID + "/" +
//												listId + "/" + YKNO + ".xls" ) ;
//						HttpServletResponse response = (HttpServletResponse)pageContext.getResponse();
//						response.sendRedirect( response.encodeRedirectURL( redirectUrl ) );
//					}
				}
//			}
//			catch(IOException ex) {
//				String errMsg = "Error in ReportTableTag: " + toString();
//				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
//			}
			rtnCode = EVAL_PAGE ;
		}

		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.3.0 (2003/04/10) REPORT_ENCODE（帳票エンコーディング）を利用するように修正。
	 * @og.rev 3.8.0.4 (2005/08/08) filename 削除、listId ,programFile 追加 ,
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		headerKeys		= null;
		headerVals		= null;
		footerKeys		= null;
		footerVals		= null;
		pageEndCut		= true;
		fileURL			= BASE_URL;
		outFileURL		= BASE_URL;
		outFilename		= null;			// 出力ファイル名
		reportClass		= "HTML";
		table			= null;
		tableId			= HybsSystem.TBL_MDL_KEY ;
		listId			= null ;		// 3.8.0.4 (2005/08/08)
		programFile		= null ;		// 3.8.0.4 (2005/08/08)
//		direct			= false;
	}

	/**
	 * TableWriter の実オブジェクトを生成して，PrintWriter に書き込みます。
	 *
	 * @og.rev 3.1.3.0 (2003/04/10) REPORT_ENCODE（帳票エンコーディング）を利用するように修正。
	 * @og.rev 3.5.4.3 (2004/01/05) HTMLDBTableReport のクラス名変更。
	 * @og.rev 3.6.0.0 (2004/09/17) メソッド名の変更。setInputFile ⇒ setTemplateFile
	 * @og.rev 3.8.0.0 (2005/06/07) setTemplateFile メソッドの引数を String  ⇒ File に変更
	 * @og.rev 3.8.0.4 (2005/08/08) 帳票出力に準拠した方式に変更
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @param  reportDir String
	 * @param  YKNO String
	 */
	private void create( final String reportDir,final String YKNO )  {

		String className = "org.opengion.hayabusa.report.DBTableReport_" + reportClass  ;
		DBTableReport report = (DBTableReport)HybsSystem.newInstance( className );	// 3.5.5.3 (2004/04/09)

		String MODELDIR = HybsSystem.url2dir( fileURL ) ;

		File templateFile      = null;
		File firstTemplateFile = null;

		// 本来は、各クラス中で処理すべき。そのためのオブジェクト指向なのだから。
		if( "HTML".equalsIgnoreCase( reportClass ) ) {
			templateFile      = FileUtil.checkFile( MODELDIR, listId + ".html" , 1 );
			firstTemplateFile = FileUtil.checkFile( MODELDIR, listId + "_FIRST.html" ,1 );
		}
		else if( "Excel".equalsIgnoreCase( reportClass ) ) {
			templateFile      = FileUtil.checkFile( MODELDIR, listId + ".xls" , 1 );
		}

		FileUtil.copy( templateFile,new File( reportDir ) );

		report.setDBTableModel( table );
		report.setTemplateFile( templateFile );	// 3.6.0.0 (2004/09/17)
		report.setFirstTemplateFile( firstTemplateFile );	// 3.6.0.0 (2004/09/17)
		report.setOutputDir( reportDir );
		report.setOutputFileKey( YKNO );
		report.setHeaderKeys( headerKeys );
		report.setHeaderVals( headerVals );
		report.setFooterKeys( footerKeys );
		report.setFooterVals( footerVals );
		report.setPageEndCut( pageEndCut );
		report.setResourceManager( getResource() );	// 4.0.0 (2005/01/31)
		report.setListId( listId );		// 3.6.1.0 (2005/01/05)
		report.writeReport();
	}

	/**
	 * 【TAG】(通常使いません)sessionから所得する DBTableModel オブジェクトの ID。
	 *
	 * @og.tag
	 * 表示処理後に，（内部ポインタを書き換えた）DBTableModel オブジェクトを
	 * 同じキーで、sessionに登録します。
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @param   tableId sessionに登録する時の ID
	 */
	public void setTableId( final String tableId ) {
		this.tableId = nval( getRequestParameter( tableId ), this.tableId );
	}

	/**
	 * 【TAG】帳票IDを指定します。
	 *
	 * @og.tag
	 * 雛形ファイルは、帳票ID.html となります。また、ファーストページ対応の場合は、
	 * 帳票ID_FIRST.html になります。
	 * なお、filename 属性が指定された場合は、そちらが優先されます。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加
	 *
	 * @param   listId String 帳票ID
	 */
	public void setListId( final String listId ) {
		this.listId = nval( getRequestParameter( listId ), this.listId );
	}

	/**
	 * 【TAG】固定部の{&#064;KEY} の KEY 部分をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 *
	 * @param   hKeys 固定部の key
	 */
	public void setHeaderKeys( final String hKeys ) {
		headerKeys = getCSVParameter( hKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定で、リクエスト情報でも設定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 *
	 * @param   hVals 固定部の値
	 */
	public void setHeaderVals( final String hVals ) {
		headerVals = getCSVParameter( hVals );
	}

	/**
	 * 【TAG】繰り返し部の終了後に表示する key 部分をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 *
	 * @param   ftKeys 繰り返し部の終了後に表示する key
	 */
	public void setFooterKeys( final String ftKeys ) {
		footerKeys = getCSVParameter( ftKeys );
	}

	/**
	 * 【TAG】固定部のKEY に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * カンマで複数指定で、リクエスト情報でも設定できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.8.1.2 (2005/12/19) footer 関連の値とpageEndCut の関係を解除します。
	 *
	 * @param   ftVals 繰り返し部の終了後に表示する値
	 */
	public void setFooterVals( final String ftVals ) {
		footerVals = getCSVParameter( ftVals );
	}

	/**
	 * 【TAG】ボディー部（繰り返し部）がなくなったときに、それ以降を表示するかどうか(true/false)を指定します(初期値:true)。
	 *
	 * @og.tag
	 * true では、それ以降を出力しません。
	 * 初期値は "true" （なくなった時点で、出力しない。）です。
	 *
	 * @og.rev 3.8.1.2 (2005/12/19) footer 関連の値とpageEndCut の関係を解除します。
	 *
	 * @param   peCut 繰り返し部の終了後に継続処理するかどうか （true:処理しない/false:処理する）
	 */
	public void setPageEndCut( final String peCut ) {
		pageEndCut = nval( getRequestParameter( peCut ),pageEndCut );
	}

	/**
	 * 【TAG】雛型のHTMLファイルの保存してある ディレクトリを指定します。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを読み取ります。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこを操作します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 雛型のHTMLファイルのディレクトリ
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】HTMLファイルをEXCEL変換する場合に使用するBATファイルを指定します。
	 *
	 * @og.tag
	 * ファイルは、フルパスで指定してください。
	 * 指定がない場合は、変換処理は行いません。
	 * 通常は、dbdef2/def/Script/runExcelPrint.bat を呼び出してください。
	 * 初期値は、null(変換処理しない)です。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加
	 *
	 * @param   programFile プログラムファイル名
	 */
	public void setProgramFile( final String programFile ) {
		this.programFile = nval( getRequestParameter( programFile ),this.programFile );
	}

	/**
	 * 【TAG】出力HTMLファイルの保存してあるディレクトリを指定します。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリにファイルを出力します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこに出力します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 出力HTMLファイルのディレクトリ
	 */
	public void setOutFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			outFileURL = StringUtil.urlAppend( outFileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときの出力ファイル名をセットします。
	 *
	 * @og.tag ファイルを作成するときの出力ファイル名をセットします。
	 *
	 * @param   filename 出力ファイル名
	 */
	public void setOutFilename( final String filename ) {
		this.outFilename = nval( getRequestParameter( filename ),this.outFilename );
	}

	/**
	 * 【TAG】実際に書き出すクラス名の略称(DBTableReport_**** の ****)をセットします(初期値:HTML)。
	 *
	 * @og.tag
	 * これは、org.opengion.hayabusa.report 以下の DBTableReport_**** クラスの **** を
	 * 与えます。これらは、DBTableReport インターフェースを継承したサブクラスです。
	 * 初期値は、"HTML" です。
	 * 属性クラス定義の
	 * <a href="/ge/jsp/DOC03/index.jsp?command=NEW&SORT=VERSION+DESC%2CSEQ%2CATTKEY&VALUENAME=reportClass" target="CONTENTS">reportClass</a>
	 * を参照願います。
	 *
	 * @param   reportClass クラス名（の略称）
	 * @see		org.opengion.hayabusa.report.DBTableReport  DBTableReportのサブクラス
	 */
	public void setReportClass( final String reportClass ) {
		this.reportClass = nval( getRequestParameter( reportClass ),this.reportClass );
	}

	/**
	 * 【TAG】結果をダイレクトにEXCEL起動するかどうか(true/false)を指定します(初期値:false[ファイル])。
	 *
	 * @og.tag 結果をダイレクトに EXCEL ファイルとして出力するかどうかをセットします。
	 *
	 * @param  flag ダイレクト（true)／ ファイル（その他)
	 */
//	public void setDirect( final String flag ) {
//		direct = nval( getRequestParameter( flag ),direct );
//	}

	/**
	 * シェルコマンドの文字列を作成します。
	 *
	 * 処理クラス(reportClass)が HTML の場合は、Shell を起動して、VBS で EXCEL化が必要。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加
	 * @og.rev 3.8.0.8 (2005/10/03) デーモングループのデフォルト値設定
	 *
	 * @param htmlFile String パーサ済みのHTMLファイル(拡張子なし)
	 * @param listId String 雛形ファイル(帳票ID)
	 * @return シェルコマンドの文字列
	 */
	private String makeShellCommand( final String htmlFile,final String listId ) {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

		buf.append( programFile ).append( " " );		// 実行するコマンド
		buf.append( "\"" );
		buf.append( htmlFile ).append( "_*.html\" " );	// 入力HTMLファイル
		buf.append( "\"xls\" " );						// プリンタ名
		// ダミーファイルを指定する必要がある。
		buf.append( "\"" );
		buf.append( htmlFile ).append( ".xls\" " );		// ダミー出力ファイル
		buf.append( listId );							// モデルファイル名
		buf.append( " NULL" );							// ダミーデーモングループ

		return buf.toString();
	}

	/**
	 * 実際のレポート出力処理を行います。
	 *
	 * 処理クラス(reportClass)が HTML の場合は、Shell を起動して、VBS で EXCEL化が必要。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) 新規追加
	 *
	 * @param shellCmd String シェルを実行するコマンド文字列
	 */
	private void programRun( final String shellCmd ) {
		Shell shell = new Shell();
		shell.setCommand( shellCmd,true );		// BATCHプロセスで実行する
		shell.setWait( true );					// プロセスの終了を待つ
		shell.setTimeout( TIMEOUT );			// 3.6.1.0 (2005/01/05) Shell の タイムアウトを設定

		int rtnCode = shell.exec();				// 0 は正常終了を示す

		if( rtnCode != 0 ) {
			String errMsg = "Shell Command exequte Error." + HybsSystem.CR
						+ "=============================="
						+ shellCmd + HybsSystem.CR
						+ shell.getStdoutData() + HybsSystem.CR
						+ shell.getStderrData() + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}

	/**
	 * Excel ファイルをダイレクトで返す forward 処理を行います。
	 *
	 * @og.rev 4.0.0 (2007/06/11) 新規追加
	 *
	 * @param fwdUrl String 実際に転送するファイルアドレス
	 * @param fname  String 転送で返すファイル名
	 */
//	private void directLoad( final String fwdUrl,final String fname ) {
//		HttpServletResponse response = (HttpServletResponse)pageContext.getResponse();
//
//		try {
//			String url = response.encodeRedirectURL( fwdUrl );	// 3.5.4.9 (2004/02/25)
//			String filename = StringUtil.urlEncode( fname );
//
//			(response).setHeader( "Content-Disposition",disposition + "; filename=\"" + filename + "\"" );
//			response.sendRedirect( url );
//	//		pageContext.forward( url );
//		} catch(IOException ex) {
//			String errMsg = "フォワードでIOエラーが発生しました。" + toString();
//			throw new HybsSystemException( errMsg,ex );
////		} catch( ServletException ex) {
////			String errMsg = "フォワードでServletエラーが発生しました。" + toString();
////			throw new HybsSystemException( errMsg,ex );
//		}
//	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "report" ;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "headerKeys"		,headerKeys		)
				.println( "headerVals"		,headerVals		)
				.println( "footerKeys"		,footerKeys		)
				.println( "footerVals"		,footerVals		)
				.println( "pageEndCut"		,pageEndCut		)
				.println( "fileURL"			,fileURL		)
				.println( "outFileURL"		,outFileURL		)
				.println( "outFilename"		,outFilename	)
				.println( "reportClass"		,reportClass	)
				.println( "tableId"			,tableId		)
				.println( "listId"			,listId			)
				.println( "programFile"		,programFile	)
				.println( "REPORT_FILE_URL"	,REPORT_URL		)
				.println( "SYSTEM_ID"		,SYSTEM_ID		)
				.println( "TIMEOUT"			,TIMEOUT		)
				.println( "BASE_URL"		,BASE_URL		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
