/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.html.TabData;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.List;
import java.util.ArrayList;
import java.util.Locale ;

/**
 * 【廃止】タブペインで、項目を分割して表示するタブテーブルを作成します。
 *
 * 一つの大きなHTMLを、タブを使用することで複数の塊に分割表示できます。
 * 分割された各タブは、一つのHTMLのため、タブ間の移動による情報の消失はありません。
 * また、一つのHTMLのため、タブにまたがって入力した値は、すべて 一括送信することも
 * 可能です。（Formタグで、全てのタブをまとめて記述していれば）
 * タブテーブルは、orientation 属性で、横(horizontal)と縦(vertical)を指定できます。
 * これは、タブの位置が異なるだけで、機能は同一です。初期値は、横(horizontal)です。
 * 個々のタブの指定は、tab タグを使用します。
 * 必ず一つ以上のtab タグを BODY要素に記述する必要があります。
 * selectedIndex を指定すると、初期表示させるタブを指定できます。番号は０から始まります。
 *
 * @og.formSample
 * ●形式：&lt;og:tabTable
 *                    orientation = "[horizontal/vertical]"		タブの方向(横/縦)
 *                    height      = "[100px / 100%]"            テーブルの高さ(px OR %)
 *                    width       = "[100px / 100%]"            テーブルの幅(px OR %)
 *                    selectedIndex = "[0..]"                   初期表示するタブ番号
 *         &gt;
 *             &lt;og:tab lbl="･･･" &gt; ... &lt;/og:tab &gt;  タブそのもの
 *             &lt;og:tab lbl="･･･" &gt; ... &lt;/og:tab &gt;  タブそのもの
 *         &lt;/og:tabTable &gt;
 * ●body：あり
 *
 * ●使用例
 *  横型(horizontal)の場合。横型が初期値のため、無指定で使用できます。
 *  &lt;og:tabTable&gt;
 *      &lt;og:tab lbl="page 1"&gt;&lt;jsp:directive.include file="paGE1.jsp" /&gt;&lt;/og:tab&gt;
 *      &lt;og:tab lbl="page 2"&gt;&lt;jsp:directive.include file="page2.jsp" /&gt;&lt;/og:tab&gt;
 *      &lt;og:tab lbl="page 3"&gt;&lt;jsp:directive.include file="page3.jsp" /&gt;&lt;/og:tab&gt;
 *  &lt;/og:tabTable&gt;
 *
 *  縦型(vertical)の場合。テーブルの高さを指定する必要があります。(px OR %)
 *  &lt;og:tabTable orientation="vertical" height="100px" &gt;
 *      &lt;og:tab lbl="page 1"&gt;&lt;jsp:directive.include file="paGE1.jsp" /&gt;&lt;/og:tab&gt;
 *      &lt;og:tab lbl="page 2"&gt;&lt;jsp:directive.include file="page2.jsp" /&gt;&lt;/og:tab&gt;
 *      &lt;og:tab lbl="page 3"&gt;&lt;jsp:directive.include file="page3.jsp" /&gt;&lt;/og:tab&gt;
 *  &lt;/og:tabTable&gt;
 *
 * @og.rev 3.5.6.5 (2004/08/09) 新規作成
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 * @deprecated
 */
@Deprecated
public class TabTableTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String CR = HybsSystem.CR ;
//	private static final String JSV = "<script language=\"JavaScript\">ots.style.height = otab.offsetHeight ;</script>";
	private static final String JSV = "<script type=\"text/javascript\">ots.style.height = otab.offsetHeight ;</script>";

	private transient List<TabData> tabList = null;
	private boolean orientation = true;		// true:horizontal false:vertical

	// 3.5.6.6 (2004/08/23) height と width の初期値変更。
	private String	height	= null;
	private String	width	= null;
	private String	style	= null;	// 3.8.6.1 (2006/10/24)
	private int		selectedIndex = -1 ;	// 3.7.1.1 (2005/05/31) 初期選択されるページ番号
	private int		realIndex     = -1 ;	// 3.8.6.2 (2006/11/01) 実際の選択タブのページ番号
	private int		realTabCount  = -1 ;	// 3.8.6.2 (2006/11/01) 実際のタブのページ番号

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( tabList == null ) {
			String errMsg = "BODY部に TabTag が必ず必要です。";
			throw new HybsSystemException( errMsg );
		}

		jspPrint( makeTag() );
		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) height と width の初期値変更。
	 * @og.rev 3.8.6.1 (2006/10/24) style属性を追加
	 * @og.rev 3.8.6.2 (2006/11/01) selectedIndex の初期値変更(0 ⇒ -1)。
	 */
	@Override
	protected void release2() {
		super.release2();
		tabList			= null;
		orientation 	= true;		// true:horizontal false:vertical
		height			= null;
		width			= null;
		selectedIndex	= -1 ;	// 3.7.1.1 (2005/05/31) 初期選択されるページ番号
		realIndex		= -1 ;	// 3.8.6.2 (2006/11/01) 実際の選択タブのページ番号
		realTabCount	= -1 ;	// 3.8.6.2 (2006/11/01) 実際のタブのページ番号
		style			= null;
	}

	/**
	 * 出力するタグ文字列を作成します。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) メソッドを HorizontalとVerticalに分割。
	 *
	 * @return	String  タグ文字列
	 */
	private String makeTag() {
		if( orientation ) {
			if( height == null ) { height = "auto"; }
			if( width  == null ) { width  = "auto"; }
			return makeHorizontalTag();
		}
		else {
			if( height == null ) { height = "200px"; }
			if( width  == null ) { width  = "100%" ; }
			return makeVerticalTag();
		}
	}

	/**
	 * 出力する horizontal タグ文字列を作成します。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) メソッドを HorizontalとVerticalに分割。
	 * @og.rev 3.7.1.1 (2005/05/23) 初期表示するページ番号を指定
	 * @og.rev 3.8.6.0 (2006/08/23) IE7対応。mp:multipage の width:100%; を削除
	 * @og.rev 3.8.6.1 (2006/10/20) action属性を追加
	 * @og.rev 3.8.6.2 (2006/11/01) selectedIndex は、初めての OPENタブとします。
	 *
	 * @return	String  タグ文字列
	 */
	private String makeHorizontalTag() {
		// 注意：/**/ でマーカー付けされている行は、縦横で異なる記述が書かれています。
		if( realIndex < 0 ) { realIndex = 0; }	// 3.8.6.2 (2006/11/01)

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_LARGE );

		buf.append( "<table cellpadding=\"0px\" cellspacing=\"0px\"" ).append( CR );
		buf.append( " style=\"width:" ).append( width ).append( ";" );

		buf.append( "\">" ).append( CR );
		buf.append( "  <tr valign=\"top\">" ).append( CR );
		buf.append( "    <td id=\"otab\">" ).append( CR );
		buf.append( "      <ts:tabstrip id=\"ots\" targetid=\"omp\" style=\"height:100%\"" ).append( CR );
		buf.append( "                tabdefaultstyle=\"border:solid 1px black;padding:3px;\"" ).append( CR );
		buf.append( "                tabhoverstyle=\"color:blue;\"" ).append( CR );
/**/	buf.append( "                tabselectedstyle=\"border:solid 1px black;border-bottom:none\"" ).append( CR );
/**/	buf.append( "                sepdefaultstyle=\"border-bottom:solid 1px black;\"" ).append( CR );
		// 3.7.1.1 (2005/05/23) 追加
		buf.append( "                selectedIndex=\"" ).append( realIndex ).append( "\"" ).append( CR );
/**/	buf.append( "                orientation=\"horizontal\">" ).append( CR );

		int size = tabList.size();
		TabData tab ;
		for( int i=0; i<size; i++ ) {
			tab = tabList.get(i);
			if( tab.isOpen() ) {
				buf.append( tab.getTab( style ) ).append( CR );
			}
			else {
				buf.append( tab.getTab( null ) ).append( CR );
			}

			if( orientation && (i != size-1) ) {
				buf.append( "<ts:tabseparator />" ).append( CR );
			}
		}
		buf.append( "<ts:tabseparator defaultstyle=\"width:100%;height:100%\" />" ).append( CR );

		buf.append( "        </ts:tabstrip>" ).append( CR );
		buf.append( "    </td>" ).append( CR );

/**/	buf.append( "</tr><tr style=\"height:" ).append( height ).append( "\" >" ).append( CR );

		buf.append( "    <td width=\"100%\">" ).append( CR );
		buf.append( "      <mp:multipage id=\"omp\"" ).append( CR );
/**/ //	buf.append( "            style=\"border:solid 1px black;border-top:none;padding:5px;height:100%;width:100%;\">" ).append( CR );
/**/	buf.append( "            style=\"border:solid 1px black;border-top:none;padding:5px;height:100%;\">" ).append( CR );

		for( int i=0; i<size; i++ ) {
			tab = tabList.get(i);
			buf.append( tab.getTabBody() ).append( CR );
		}
		buf.append( "      </mp:multipage>" ).append( CR );
		buf.append( "    </td>" ).append( CR );
		buf.append( "  </tr>" ).append( CR );
		buf.append( "</table>" ).append( CR );

		return buf.toString();
	}

	/**
	 * 出力する vertical タグ文字列を作成します。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) メソッドを HorizontalとVerticalに分割。
	 * @og.rev 3.7.1.1 (2005/05/23) 初期表示するページ番号を指定
	 * @og.rev 3.8.6.0 (2006/08/23) IE7対応。mp:multipage の height:100%; を削除
	 * @og.rev 3.8.6.1 (2006/10/20) action属性を追加
	 * @og.rev 3.8.6.2 (2006/11/01) selectedIndex は、初めての OPENタブとします。
	 *
	 * @return	String  タグ文字列
	 */
	private String makeVerticalTag() {
		// 注意：/**/ でマーカー付けされている行は、縦横で異なる記述が書かれています。
		if( realIndex < 0 ) { realIndex = 0; }	// 3.8.6.2 (2006/11/01)

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_LARGE );

		buf.append( "<table cellpadding=\"0px\" cellspacing=\"0px\"" ).append( CR );
		buf.append( " style=\"width:" ).append( width ).append( ";" );
/**/	buf.append( "height:" ).append( height ).append( ";" );

		buf.append( "\">" ).append( CR );
		buf.append( "  <tr valign=\"top\">" ).append( CR );
		buf.append( "    <td id=\"otab\">" ).append( CR );
		buf.append( "      <ts:tabstrip id=\"ots\" targetid=\"omp\" style=\"height:100%\"" ).append( CR );
		buf.append( "                tabdefaultstyle=\"border:solid 1px black;padding:3px;\"" ).append( CR );
		buf.append( "                tabhoverstyle=\"color:blue;\"" ).append( CR );
/**/	buf.append( "                tabselectedstyle=\"border:solid 1px black;border-right:none\"" ).append( CR );
/**/	buf.append( "                sepdefaultstyle=\"border-right:solid 1px black;\"" ).append( CR );
		// 3.7.1.1 (2005/05/23) 追加
		buf.append( "                selectedIndex=\"" ).append( realIndex ).append( "\"" ).append( CR );
/**/	buf.append( "                orientation=\"vertical\">" ).append( CR );

		int size = tabList.size();
		TabData tab ;
		for( int i=0; i<size; i++ ) {
			tab = tabList.get(i);
			if( tab.isOpen() ) {
				buf.append( tab.getTab( style ) ).append( CR );
			}
			else {
				buf.append( tab.getTab( null ) ).append( CR );
			}

			if( orientation && (i != size-1) ) {
				buf.append( "<ts:tabseparator />" ).append( CR );
			}
		}
		buf.append( "<ts:tabseparator defaultstyle=\"width:100%;height:100%\" />" ).append( CR );

		buf.append( "        </ts:tabstrip>" ).append( CR );
		buf.append( "    </td>" ).append( CR );

		buf.append( "    <td style=\"width:100%;height:100%;\">" ).append( CR );
		buf.append( "      <mp:multipage id=\"omp\"" ).append( CR );
/**/ //	buf.append( "            style=\"border:solid 1px black;border-left:none;padding:5px;height:100%;width:100%;\">" ).append( CR );
/**/	buf.append( "            style=\"border:solid 1px black;border-left:none;padding:5px;height:100%;\">" ).append( CR );

		for( int i=0; i<size; i++ ) {
			tab = tabList.get(i);
			buf.append( tab.getTabBody() ).append( CR );
		}
		buf.append( "      </mp:multipage>" ).append( CR );
		buf.append( "    </td>" ).append( CR );
		buf.append( "  </tr>" ).append( CR );
		buf.append( "</table>" ).append( CR );
		buf.append( JSV ).append( CR );		// vertical 時に IE7でサイズの取り方が異なる為の対策。

		return buf.toString();
	}

	/**
	 * 設定する タブオブジェクトを、内部変数（List)に追加します。
	 *
	 * BODY 部に記述された タブオブジェクトを順番に追加します。
	 * タブペインのタグを出力する場合も、この順番で作成します。
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) action属性を追加
	 * @og.rev 3.8.6.2 (2006/11/01) realIndex は、初めての OPENタブとします。
	 *
	 * @param	data TabData タブオブジェクト
	 */
	protected void addTabData( final TabData data ) {
		if( tabList == null ) { tabList = new ArrayList<TabData>(); }
		tabList.add( data );

		// タブが選択されていれば、その値を選択番号とする。
		if( realIndex < 0 && data.isOpen() ) {
			realIndex = tabList.size()-1 ;
		}
	}

	/**
	 * selectedIndex で設定されたタブかどうかを判断して返します。
	 *
	 * <p>このメソッド呼び出しは、各タブから１回のみ有効とします。
	 * 呼び出すたびに、内部変数 realTabCount をカウントアップします。
	 * つまり、その数が、タブの個数に対応します。
	 * タブは、DELETE と判断されるケースがあるため、実際の数より少なく登録されます。
	 * そのときに、ここで自分自身が選択されていることを判断して、実際の選択タブを
	 * JavaScript に指定するときに使用します。</p>
	 *
	 * @og.rev 3.8.6.2 (2006/11/01) 新規作成
	 *
	 * @return	boolean 選択タブかどうか
	 */
	protected boolean isSelected() {
		realTabCount ++ ;
		return (selectedIndex == realTabCount) ;
	}

	/**
	 * 【TAG】タブの方向、横型(horizontal)か縦型(vertical)を指定します(初期値：横型)。
	 *
	 * @og.tag
	 * タブは、上にタブが並ぶ横型と左にタブが並ぶ縦型があります。
	 * この属性では、横型は、horizontal 、縦型は、vertical を指定します。
	 * 指定は、文字列の最初の一文字を見ているだけですので、HかVでも構いません。
	 * 初期値は、横型(horizontal) です。
	 *
	 * @param	ori タブの方向、横型(horizontal)か縦型(vertical)を指定
	 */
	public void setOrientation( final String ori ) {
		String ori2 = nval( getRequestParameter( ori ),null );
		if( ori2 != null && ori2.length() > 0 ) {
			char ch = ori2.toUpperCase(Locale.JAPAN).charAt( 0 );
			if( ch == 'H' ) { orientation = true; }
			else if( ch == 'V' ) { orientation = false; }
			else {
				String errMsg = "orientation の指定は、horizontal または、vertical です。";
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】タブの高さを、% 、px 、または "auto" で指定します
	 *
	 * @og.tag
	 * 縦型(orientation="vertical")の初期値は、"auto" です。
	 * 横型(orientation="horizontal")の初期値は、"200px"です。
	 * 横型の場合は、"auto" に設定すると、高さが "0" になってしまいます。
	 * 必ず、なにかの値（px）で指定する必要があります。
	 * 縦型 で "auto" に設定すると、各タブ毎に中の記述情報によって、タブの
	 * 大きさが替わります。タブを切り替えた時に、違和感がない様にするには、
	 * 高さを固定（px 指定）するとよいです。
	 *
	 * @param	ht String 高さ (% 、px 、または "auto" )
	 */
	public void setHeight( final String ht ) {
		height = nval( getRequestParameter( ht ),height );
	}

	/**
	 * 【TAG】タブの幅を % 、px 、または "auto" で指定します。
	 *
	 * @og.tag
	 * 縦型(orientation="vertical")の初期値は、"auto" です。
	 * 横型(orientation="horizontal")の初期値は、"100%"です。
	 * ※ 縦型の場合、幅に px で数字を設定しても、有効に作用しません。
	 *
	 * @param	wh String 幅 (% 、px 、または "auto" )
	 */
	public void setWidth( final String wh ) {
		width = nval( getRequestParameter( wh ),width );
	}

	/**
	 * 【TAG】初期表示するページ番号を指定します(初期値:0)。
	 *
	 * @og.tag
	 * タブテーブルには、複数のタブを含みます。初期表示時にどのタブを
	 * 表示するかを指定します。
	 * ページ番号は、0から始まる数字です。
	 * 初期値は、0です。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) 新規作成
	 *
	 * @param	no String 初期表示するページ番号(0..)
	 */
	public void setSelectedIndex( final String no ) {
		selectedIndex = nval( getRequestParameter( no ),selectedIndex );
	}

	/**
	 * 【TAG】初期表示時のタブに与える style 属性
	 *
	 * @og.tag
	 * <p>ts:tab 本体では、初期選択時のスタイルシートを、defaultStyle と
	 * selectedStyle で与える必要があります。これは、id 属性を設定して、
	 * 外部でスタイルシートを定義する形式で指定できません。</p>
	 * ここで指定した style 属性 は、個別の tabTag に与える style 属性 より優先度は
	 * 低くなります。
	 *
	 * @og.rev 3.8.6.1 (2006/10/24) 新規追加
	 *
	 * @param	st タブに与える 初期 style 属性
	 */
	public void setStyle( final String st ) {
		style = nval( getRequestParameter( st ),style );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "height"			,height			)
				.println( "width"			,width			)
				.println( "selectedIndex"	,selectedIndex	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
