/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.XHTMLTag;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.File;

/**
 * 【特殊】画面ＩＤと同じヘルプファイルがあればリンクを作成するタグです(通常は query.jsp に組込み)。
 *
 * ヘルプファイルは、システムパラメータ の HELP_URL で定義されているhelpフォルダに配置します。
 * このフォルダに、画面IDと同じファイル(例えば、GE0001.html など)があれば、リンクを作成します。
 * ファイルがなければ、リンクは表示されません。
 * メッセージの表示の制御は、viewMsg 属性で指定します。(false でファイルが存在した場合のみ表示）
 * ファイルの拡張子も指定できますが、一般に、html でヘルプファイルを作成するほうが
 * すばやく表示できます。
 *
 * @og.formSample
 * ●形式：一般ユーザーが直接組み込むことはありません。
 * ●body：なし
 *
 * ●使用例
 *     &lt;og:help guiInfoKey="{&#064;GUI.KEY}" msg="HELP" /&gt;
 *
 *     &lt;og:help
 *        guiInfoKey    = "GUIInfo のキーを指定します。(必須)"
 *        extension     = "拡張子を指定します(初期値:html)。"
 *        msg           = "メッセージリソースのメッセージIDを指定します。"
 *        target        = "TARGET 属性を指定します(初期値:_blank)。"
 *        viewMsg       = "メッセージを常時表示させるかどうか(true/false)を指定します(初期値:false)。"
 *     /&gt;
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableInOut extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	public static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private String	guiInfoKey	= null;
	private String	extension	= "html";
	private String  baseURL		= HybsSystem.sys( "HELP_URL" );
	private String	target		= "_blank";		// 3.6.0.7 (2004/11/06)
	private boolean	viewMsg		= false;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		jspPrint( makeTag() );

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加他
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.7 (2004/11/06) target 属性の初期値を _new から _blank に変更
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		guiInfoKey	= null;
		extension	= "html";
		baseURL		= HybsSystem.sys( "HELP_URL" );
		target		= "_blank";		// 3.6.0.7 (2004/11/06)
		viewMsg		= false;
	}

	/**
	 * リンクを作成します。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加
	 * @og.rev 3.0.1.0 (2003/03/03) viewMsg フラグの制御のバグ修正
	 *
	 * @return	リンクタグ文字列
	 */
	protected String makeTag() {
		String rtn = "";

		String url = baseURL + guiInfoKey + "." + extension;
		File  file = new File( HybsSystem.url2dir( url ) );

		// ファイルの存在チェック
		if( file.exists() ) {						// 3.5.6.0 (2004/06/18)
			Attributes attri = new Attributes();
			String path = getContextPath();
			attri.set( "href",path + "/" + url );
			attri.set( "body",getMsglbl() );
			attri.set( "target",target );

			rtn = XHTMLTag.link( attri ) ;
		}
		else if( viewMsg ) {
			rtn = getMsglbl() ;
		}

		return rtn;
	}

	/**
	 * 【TAG】GUIInfo のキーを指定します。
	 *
	 * @og.tag GUIInfo のキーを指定します。
	 *
	 * @param	key String
	 */
	public void setGuiInfoKey( final String key ) {
		guiInfoKey = getRequestParameter( key );
	}

	/**
	 * 【TAG】拡張子を指定します(初期値:html)。
	 *
	 * @og.tag
	 * なにも設定されていない場合は、"html" が初期値となります。
	 * ここでは、ピリオドは、含める必要はありません。
	 *
	 * @param	ext 拡張子
	 */
	public void setExtension( final String ext ) {
		extension = nval( getRequestParameter( ext ),extension );
	}

	/**
	 * 【TAG】TARGET 属性を指定します(初期値:_blank)。
	 *
	 * @og.tag
	 * 初期値は、 "_blank" として、新規に画面を立ち上げます。
	 * CONTENTS 等を指定すれば、コンテンツフレーム（メニューの右側全面）に、
	 * RESULT を指定すれば、リザルトフレーム（メニュー右下側）に表示します。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加
	 *
	 * @param	val TARGET 属性を指定します（初期値 "_blank"）
	 */
	public void setTarget( final String val ) {
		target = nval( getRequestParameter( val ),target );
	}

	/**
	 * 【TAG】メッセージを常時表示させるかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * "true"の場合は、常時表示させます。
	 * ファイルが、存在した場合は、リンクが張られ、存在しない場合は、リンクが
	 * 張られません。
	 * "false" の場合は、ファイルが、存在した場合は、リンクが張られ、存在しない場合は、
	 * なにも表示されません。
	 * 初期値は、 "false"(メッセージを常時表示しない)です。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) メッセージ表示属性の新規追加
	 *
	 * @param	flag メッセージを常時表示させるかどうかを指定( true:常時表示/false:非表示 ）
	 */
	public void setViewMsg( final String flag ) {
		viewMsg = nval( getRequestParameter( flag ),viewMsg );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "guiInfoKey"		,guiInfoKey	)
				.println( "extension"		,extension	)
				.println( "baseURL"			,baseURL	)
				.println( "target"			,target		)
				.println( "viewMsg"			,viewMsg	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
