/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.business;

import org.opengion.fukurou.model.DataModel;
import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;

/**
 * 業務ロジックを処理するためのテーブルモデルです。
 * 
 * このテーブルモデルでは、オブジェクト生成時に、カラム配列、値配列を元に、内部データを生成し、
 * その後は、行の追加や値の変更はできません。
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ロジック
 * 
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class ArrayTableModel implements DataModel<String> {
	private final String[] names;
	private final String[][] vals;
	private final String[] modTypes;

	/**
	 * 引数に名前配列、値配列を指定したコンストラクター
	 *
	 * @param   nms String[] 名前配列
	 * @param	vs String[][] 値配列
	 * @throws  IllegalArgumentException 引数の配列が不正な場合
	 */
	public ArrayTableModel( final String[] nms, final String[][] vs ) {
		this( nms, vs, null );
	}

	/**
	 * 引数に名前配列、値配列、変更区分配列を指定したコンストラクター
	 *
	 * @param   nms String[] 名前配列
	 * @param	vs String[][] 値配列
	 * @param	ms String[] 変更区分の配列
	 * @throws  IllegalArgumentException 引数の配列が不正な場合
	 */
	public ArrayTableModel( final String[] nms, final String[][] vs, final String[] ms ) {
		if( nms == null || nms.length == 0 ) {
			String errMsg = "引数の名前配列に、null は設定できません。";
			throw new IllegalArgumentException( errMsg );
		}
		if( vs == null ) {
			String errMsg = "引数の値配列に、null は設定できません。";
			throw new IllegalArgumentException( errMsg );
		}
		if( vs.length > 0 ) {
			if( vs[0] == null || vs[0].length == 0 || nms.length != vs[0].length ) {
				String errMsg = "名前配列を値配列のカラム数が異なります。";
				throw new IllegalArgumentException( errMsg );
			}
		}

		int cols = nms.length;
		names = new String[cols];
		System.arraycopy( nms, 0, names, 0, cols );

		int rows = vs.length;
		vals = new String[rows][cols];
		for( int i = 0; i < rows; i++ ) {
			System.arraycopy( vs[i], 0, vals[i], 0, cols );
		}

		if( ms != null && ms.length > 0 ) {
			if( vs.length != ms.length ) {
				String errMsg = "変更区分を指定する場合、値配列の行数と一致する必要があります。";
				throw new IllegalArgumentException( errMsg );
			}
			else {
				modTypes = new String[rows];
				System.arraycopy( ms, 0, modTypes, 0, rows );
			}
		}
		else {
			modTypes = null;
		}
	}

	/**
	 * rowで指定された行番号（インデックス番号）に行を追加します。
	 * (このクラスでは、このメソッドはサポートされていません。)
	 *
	 * @param   vals  配列値
	 * @param   row   追加する行番号
	 */
	public void setValues( final String[] vals, final int row ) {
		throw new RuntimeException( "このクラスでは、setValuesメソッドはサポートされていません" );
	}

	/**
	 * カラム名に対応する カラム番号を返します。
	 *
	 * 特殊なカラムが指定された場合は、負の値を返します。
	 * 例えば、[KEY.カラム名]、[I]、[ROW.ID] など、特定の負の値を返します。
	 * また、カラム名が元のデータモデルに存在しない場合も、負の値か、
	 * Exception を返します。負の値なのか、Exception なのかは、
	 * 実装に依存します。
	 *
	 * @param   columnName String 値が参照されるカラム名
	 * @return  指定されたセルのカラム番号。存在しなければ、-1
	 * @throws  IllegalArgumentException 引数のカラム名が null の場合
	 */
	public int getColumnNo( final String columnName ) {
		if( columnName == null ) {
			String errMsg = "引数のカラム名に、null は設定できません。";
			throw new IllegalArgumentException( errMsg );
		}

		int address = -1;
		for( int i = 0; i < names.length; i++ ) {
			if( columnName.equalsIgnoreCase( names[i] ) ) {
				address = i;
				break;
			}
		}

		return address;
	}

	/**
	 * カラム名配列に対応する カラム番号配列を返します。
	 *
	 * これは、#getColumnNo( String ) に対する 複数のカラム名を検索した
	 * 場合と同じです。
	 *
	 * @param   clmNms  String[] 値が参照されるカラム名配列
	 * @return  指定されたセルのカラム番号配列。
	 */
	public int[] getColumnNos( final String[] clmNms ) {
		if( clmNms == null ) { return new int[0]; }

		int[] clmNos = new int[clmNms.length];
		for( int j = 0; j < clmNms.length; j++ ) {
			int address = -1;
			for( int i = 0; i < names.length; i++ ) {
				if( clmNms[j].equalsIgnoreCase( names[i] ) ) {
					address = i;
					break;
				}
			}
			clmNos[j] = address;
		}

		return clmNos;
	}

	/**
	 * カラム名配列を返します。
	 *
	 * @return   nm String[]
	 */
	public String[] getNames() {
		return names.clone();
	}

	/**
	 * row にあるセルの属性値を配列で返します。
	 *
	 * @param   row     値が参照される行
	 * @return  指定されたセルの属性値 String[]
	 */
	public String[] getValues( final int row ) {
		return vals[row].clone();
	}

	/**
	 * row および clm にあるセルの属性値をStringに変換して返します。
	 *
	 * @param   row     値が参照される行
	 * @param   clm     値が参照される列
	 * @return  指定されたセルの値 String
	 *
	 */
	public String getValue( final int row, final int clm ) {
		return vals[row][clm];
	}

	/**
	 * row および clm にあるセルの属性値をStringに変換して返します。
	 *
	 * @param   row     値が参照される行
	 * @param   clm     値が参照される列(キー)
	 * @return  指定されたセルの値 String
	 *
	 */
	public String getValue( final int row, final String clm ) {
		return vals[row][getColumnNo( clm )];
	}

	/**
	 * データテーブル内の行の数を返します。
	 *
	 * @return  モデルの行数
	 *
	 */
	public int getRowCount() {
		return vals.length;
	}

	/**
	 * row 単位に変更されたタイプ（追加/変更/削除）を返します。
	 * タイプは始めに一度登録するとそれ以降に変更はかかりません。
	 * つまり、始めに 追加で作成したデータは、その後変更があっても追加のままです。
	 * なにも変更されていない場合は, ""（ゼロストリング）を返します。
	 *
	 * @param   row     値が参照される行
	 * @return  変更されたタイプの値 String
	 */
	public String getModifyType( final int row ) {
		return modTypes == null ? "" : modTypes[row];
	}

	/**
	 * clm のNativeタイプを返します。
	 * Nativeタイプはorg.opengion.fukurou.model.NativeTypeで定義されています。
	 * 
	 * @param  clm      値が参照される列
	 * @return Nativeタイプ
	 * @see org.opengion.fukurou.model.NativeType
	 */
	public NativeType getNativeType( final int clm ) {
		return StringUtil.getNativeType( vals[0][clm] );
	}
}
