/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.business;

import java.sql.Connection;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsLoader;

/**
 * 業務ロジックを処理するためのインターフェースです。
 * 
 * このインターフェースでは、業務ロジックを呼び出すのに
 * 必要な機能を定義しています。
 *
 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
 * @og.group 業務ロジック
 * 
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public interface BizLogic {

	/**
	 * DBのコネクションオブジェクトを指定します。
	 * 各実装クラスでは、コネクションのcommit,rollbackは行われません。
	 * (全てのDB処理は、1つのトランザクションとして処理されます。)
	 * このため、commit,rollbackは呼び出し元で行う必要があります。
	 * 
	 * @param cn DB接続
	 */
	void setConnection( final Connection cn );

	/**
	 * 業務ロジックのクラスをロードするためのクラスローダーをセットします。
	 * 
	 * @param ldr クラスローダー
	 */
	void setLoader( final HybsLoader ldr );

	/**
	 * テーブルモデルをセットします。
	 * 
	 * @param tbl 配列側テーブルモデル
	 */
	void setTable( final ArrayTableModel tbl );

	/**
	 * 固定値のキーをCSV形式で指定します。
	 * 
	 * @param ks キー
	 */
	void setKeys( final String[] ks );
	
	/**
	 * 固定値の値をCSV形式で指定します。
	 * 
	 * @param vs 値
	 */
	void setVals( final String[] vs );

	/**
	 * この処理の実行ユーザーIDを指定します。
	 * 
	 * @param id 実行ユーザーID
	 */
	void setUserId( final String id );

	/**
	 * 親(呼び出し)PGIDを指定します。
	 * 
	 * @param id 親PGID
	 */
	void setParentPgId( final String id );

	/**
	 * デバッグモードにします。
	 */
	void setDebug();

	/**
	 * デバッグメッセージを取得します。
	 * 
	 * @return デバッグメッセージ
	 */
	String getDebugMsg();

	/**
	 * 処理の方法は、各実装クラスにより定義されます。
	 * 実装クラスで発生した全ての例外は、Throwableオブジェクトとしてスローされます。
	 * 呼び出し元では、例外を確実にcatchして、commit,rollbackを行ってください。
	 * 
	 * @return 処理が正常終了したか
	 * @throws Throwable
	 */
	boolean exec() throws Throwable;

	/**
	 * 結果ステータスを返します。
	 * 
	 * @return 結果ステータス
	 */
	int getKekka();

	/**
	 * エラーメッセージオブジェクトを返します。
	 * 
	 * @return エラーメッセージ
	 */
	ErrorMessage getErrMsg();
	
	/**
	 * 業務ロジックの戻り値を返します。
	 * 
	 * @return 戻り値
	 */
	String getReturn();

	/**
	 * 業務ロジックを実行するために、テーブルモデルが外部からセットされる必要があるか
	 * を返します。
	 * 必須である場合、その業務ロジックは、子ロジックとして呼び出すことができません。
	 * これは、子ロジック呼び出し時は、テーブルモデルがセットされないためです。
	 * 
	 * @return テーブルモデルが外部からセットされる必要があるか
	 */
	boolean isRequireTable();
}
