/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.net.URL;
import java.util.Locale;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.io.File ;
import java.io.IOException ;

import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.NodeList ;
import org.w3c.dom.Node ;

//import javax.xml.bind.JAXBContext;
//import javax.xml.bind.JAXBException;
//import javax.xml.bind.Unmarshaller;
//import javax.xml.validation.SchemaFactory;

//import javax.xml.XMLConstants;

import org.xml.sax.SAXException;

//import org.opengion.fukurou.xml.jaxb.dbid.DbConfig;
//import org.opengion.fukurou.xml.jaxb.dbid.DbDriver;
//import org.opengion.fukurou.xml.jaxb.dbid.Dbid;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.xml.DomParser;

/**
 * DB設定XMLの内容をJAXBを利用してロードする
 * Driverをロードする
 * 上記２つの機能を備えたクラスです
 *
 * 外部からはgetDbidメソッドを利用してDB設定（ExpandedDbid型）を取得します。
 * DB設定情報が無い場合にXMLを読みにいきます。
 * このDBIDを決めるキーは、内部取り込み字に、大文字変換されますので、大文字・
 * 小文字の区別はありません。
 *
 * @og.rev 4.0.0 (2007/10/25) 新規作成
 * @og.rev 5.1.7.0 (2010/06/01) org.opengion.fukurou.xml.jaxb.dbid 関係 廃止
 * @og.group 初期化
 *
 * @version  4.0
 * @author 高橋正和
 * @since   JDK6.0,
 */
public class DatabaseConfig {
	// fukurou内で完結させるため、HybsDataからは読み込まずにここに書く
	private static final String DEFAULT_DRIVER	 = "oracle.jdbc.OracleDriver";
	// XMLファイル関連
//	private transient final String XmlFilename;
	private final String XmlFilename;
//	private final static String SCHEMA_FILENAME = "DBConfig.xsd";//xsdはfukurou.xml.jaxb.dbidパッケージに置く

//	private transient final Map<String, EDbid> dbidMap = new HashMap<String, EDbid>();
//	private transient List<String> driverList = new ArrayList<String>();
	private final Map<String, EDbid> dbidMap = new HashMap<String, EDbid>();
//	private List<String> driverList = new ArrayList<String>();

//	private static final String CR = System.getProperty( "line.separator" );

	/**
	 * 初期値を使ってXMLを読み込む
	 * XmlFilenameの初期値は../DBConfig.xml
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 自分のコンストラクターを呼ぶように修正
	 */
	public DatabaseConfig() {
		this( "../DBConfig.xml" );
	}

	/**
	 * XMLファイルの名前を指定して読み込む
	 *
	 * @param xmlfile String
	 */
	public DatabaseConfig( final String xmlfile ) {
//		XmlFilename = xmlfile;

		ClassLoader clsl	= getClass().getClassLoader();
		URL xmlURL			= clsl.getResource( xmlfile );
		XmlFilename			= xmlURL.getFile();
System.out.println( XmlFilename );
	}

	/**
	 * dbidKeyをキーにしてExpandedDbid型でマップの内容を返す。
	 * 存在しない場合はNULLを返します。
	 * キーが無い場合に初期化を行う。
	 *
	 * @og.rev 4.0.0.1 (2007/12/04) EDbid#clone() 廃止
	 *
	 * @param key String XMLで登録したdbidKey
	 * @return EDbid型オブジェクト
	 */
	public EDbid getDbid( final String key ) {
		synchronized ( dbidMap ) {
			if( dbidMap.isEmpty() ) {
				init();
			}

			return dbidMap.get( key.toUpperCase( Locale.JAPAN ) ) ;
		}
	}

	/**
	 * マップをクリアします。
	 * XMLファイルを再読み込みする場合に使用します。
	 */
	public void reload() {
		synchronized ( dbidMap ) {
			dbidMap.clear();
		}
	}

	/**
	 * 初期化処理
	 * 
	 * DB設定XMLファイル(DBConfig.xml)を読み込みます。
	 * このファイルから、ドライバーリストの取得、DBIDのオブジェクトマップの作成を
	 * 行います。
	 * EDbidオブジェクト は、環境変数で、共通の初期値を定義しておくことが可能です。
	 * 項目として、REALM_URL、REALM_NAME、REALM_PASSWORD が設定可能です。
	 *
	 * ドライバーリストの取得後、Class.forName で、ドライバの登録も行います。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) org.opengion.fukurou.xml.jaxb.dbid 関係 廃止
	 */
	private void init() {
		Document doc = DomParser.read( new File(XmlFilename) ) ;
		Element firstRoot = doc.getDocumentElement();

		List<String> driverList = getDriverList( firstRoot );
		for( String dr : driverList ) {
			try {
				Class.forName( dr );
			} catch ( ClassNotFoundException ex ) {
				String errMsg = "ドライバクラスが見つかりません。[" + dr + "]" ;
				LogWriter.log( errMsg );
				LogWriter.log( ex );
			}
		}

		EDbid defDdbid = new EDbid();		// 初期値

		defDdbid.setUrl(		System.getenv( "REALM_URL" ) ); 
		defDdbid.setUser(		System.getenv( "REALM_NAME" ) );
		defDdbid.setPassword(	System.getenv( "REALM_PASSWORD" ) );

		makeDbidMap( firstRoot,defDdbid );
	}

	/**
	 * ドライバーリストを取得します。
	 * 
	 * DB設定XMLファイル(DBConfig.xml)の、class タグを取り込みます。
	 * このファイルから、ドライバーリストを取得します。
	 *
	 * 内部的に３段階の処理が実行されます。
	 * 第１Step:DBConfig.xml から、ドライバーリストを取得
	 * 第２Step:ドライバーリストが存在しない場合、環境変数の REALM_DRIVER からドライバーを取得
	 * 第３Step:それでも存在しない場合、このクラスの DEFAULT_DRIVER 定数 からドライバーを取得
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) org.opengion.fukurou.xml.jaxb.dbid 関係 廃止
	 * 
	 * @param  element Element DB設定XMLファイルのElementオブジェクト
	 * @return List<String> ドライバーリスト
	 */
	private List<String> getDriverList( final Element element ) {
		List<String> driverList = new ArrayList<String>();

		NodeList list = element.getElementsByTagName( "class" ) ;
		int num = list.getLength();
		for (int i = 0; i < num; i++) {
			Element cls = (Element)list.item(i);
			driverList.add( cls.getTextContent() );
		}

		if( driverList.isEmpty() ) {
			String realmDriver = System.getenv( "REALM_DRIVER" );
			if( realmDriver != null && realmDriver.length() > 0 ) {
				driverList.add( realmDriver );
			}
		}

		if( driverList.isEmpty() ) { driverList.add( DEFAULT_DRIVER ); }

		return driverList ;
	}

	/**
	 * EDbidオブジェクトのマップを取得します。
	 * 
	 * DB設定XMLファイル(DBConfig.xml)の、dbid タグを取り込みます。
	 * このファイルから、EDbidオブジェクトの属性情報を取得し、オブジェクトを構築します。
	 *
	 * EDbidオブジェクト は、初期値をコピーして、作成していきます。
	 * EDbidオブジェクトをマップから取り出すキーとなる、dbidKey は、大文字化して設定します。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) org.opengion.fukurou.xml.jaxb.dbid 関係 廃止
	 * 
	 * @param  element Element DB設定XMLファイルのElementオブジェクト
	 * @param  defDdbid EDbid 初期情報の設定された、EDbidオブジェクト
	 */
	private void makeDbidMap( final Element element , EDbid defDdbid ) {

		NodeList list = element.getElementsByTagName( "dbid" ) ;
		int num = list.getLength();
		for (int i = 0; i < num; i++) {
			Element ele = (Element)list.item(i);
			NodeList childs = ele.getChildNodes();
			int numChild = childs.getLength();
//			EDbid dbid = new EDbid();
			EDbid dbid = defDdbid.clone();		// 初期値をコピーして、作成
			for (int j = 0; j < numChild; j++) {
				Node nd = childs.item(j);
				if( nd.getNodeType() == Node.ELEMENT_NODE ) {
					Element el = (Element)nd;
					String tag = el.getTagName();
					// dbidKey は、toUpperCase して、大文字のみとする。
					if( "dbidKey".equals( tag ) )	{
						String dbidKey = el.getTextContent();
						if( dbidKey != null && dbidKey.length() > 0 ) {
							dbid.setDbidKey( dbidKey.toUpperCase( Locale.JAPAN ) );
						}
					}
					else if( "url".equals( tag ) )		{ dbid.setUrl(		el.getTextContent() ); }
					else if( "user".equals( tag ) )		{ dbid.setUser(		el.getTextContent() ); }
					else if( "password".equals( tag ) ) { dbid.setPassword( el.getTextContent() ); }
					else if( "readonly".equals( tag ) ) { dbid.setReadonly( el.getTextContent() ); }
					else if( "mincount".equals( tag ) ) { dbid.setMincount( el.getTextContent() ); }
					else if( "maxcount".equals( tag ) ) { dbid.setMaxcount( el.getTextContent() ); }
					else if( "pooltime".equals( tag ) ) { dbid.setPooltime( el.getTextContent() ); }
					else if( "applicationInfo".equals( tag ) ) { dbid.setApplicationInfo( el.getTextContent() ); }
					else {
						System.err.println( "警告：dbid に新しい属性が、追加されています。" );
					}
				}
			}
			dbidMap.put( dbid.getDbidKey(), dbid );
		}
	}
}
