/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.LogWriter;

import java.util.Map ;
import java.util.HashMap ;
import java.util.LinkedHashMap ;

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Process_TableDiffは、ファイルから読み取った内容を、LineModel に設定後、
 * 下流に渡す、FirstProcess インターフェースの実装クラスです。
 *
 * DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、
 * 下流（プロセスチェインのデータは上流から下流に渡されます。）に渡します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_TableDiff -infile1=INFILE -infile2=INFILE2 -action=DIFF1 -encode=UTF-8 -columns=AA,BB,CC
 *
 *    -infile1=入力ファイル名1    ：入力ファイル名1
 *    -infile2=入力ファイル名2    ：入力ファイル名2
 *    -action=比較結果の方法      ：ONLY,DIFF,INTERSEC
 *   [-sep1=セパレータ文字      ] ：区切り文字1（初期値：タブ）
 *   [-sep2=セパレータ文字      ] ：区切り文字2（初期値：タブ）
 *   [-encode1=文字エンコード   ] ：入力ファイルのエンコードタイプ1
 *   [-encode2=文字エンコード   ] ：入力ファイルのエンコードタイプ2
 *   [-columns=読み取りカラム名 ] ：入力カラム名(カンマ区切り)
 *   [-keyClms=比較するカラム名 ] ：比較する列の基準カラム名(カンマ区切り)
 *   [-diffClms=比較するカラム名] ：比較するカラム名(カンマ区切り)
 *   [-display=false|true       ] ：結果を標準出力に表示する(true)かしない(false)か（初期値 false:表示しない)
 *   [-debug=false|true         ] ：デバッグ情報を標準出力に表示する(true)かしない(false)か（初期値 false:表示しない)
 *
 * @og.rev 4.2.3.0 (2008/05/26) 新規作成
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_TableDiff extends AbstractProcess implements FirstProcess {
	private final static String ENCODE = System.getProperty("file.encoding");

	private String		  	separator1	= TAB;	// 項目区切り文字
	private String		  	separator2	= TAB;	// 項目区切り文字
	private String			infile1		= null;
	private String			infile2		= null;
	private BufferedReader	reader1		= null;
//	private BufferedReader	reader2		= null;
	private LineModel		model		= null;
	private String			line		= null;
	private int[]			clmNos		= null;		// ファイルのヘッダーのカラム番号
	private int[]			keyClmNos	= null;		// 比較する列の基準カラム名のカラム番号
	private int[]			diffClmNos	= null;		// 比較するカラム名のカラム番号
	private String			action		= null;
	private boolean			display		= false;	// 表示しない
	private boolean			debug		= false;	// 表示しない
	private boolean			nameNull	= false;	// ０件データ時 true

	private final Map<String,String> file2Map = new HashMap<String,String>();	// 4.3.1.1 (2008/08/23) final化

	private int				inCount1	= 0;
	private int				inCount2	= 0;
	private int				outCount	= 0;

	private final static Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private final static Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "infile1",	"入力ファイル名1 (必須)" );
		mustProparty.put( "infile2",	"入力ファイル名2 (必須)" );
		mustProparty.put( "action",		"(必須)ONLY,DIFF,INTERSEC" );
		mustProparty.put( "keyClms",	"比較する列の基準カラム名(必須)(カンマ区切り)" );
		mustProparty.put( "diffClms",	"比較するカラム名(必須)(カンマ区切り)" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "sep1",			"区切り文字1（初期値：タブ）" );
		usableProparty.put( "sep2",			"区切り文字2（初期値：タブ）" );
		usableProparty.put( "encode1",		"入力ファイルのエンコードタイプ1" );
		usableProparty.put( "encode2",		"入力ファイルのエンコードタイプ2" );
		usableProparty.put( "columns",		"入力カラム名(カンマ区切り)" );
		usableProparty.put( "display",		"結果を標準出力に表示する(true)かしない(false)か" +
											CR + "（初期値 false:表示しない)" );
		usableProparty.put( "debug",		"デバッグ情報を標準出力に表示する(true)かしない(false)か" +
											CR + "（初期値 false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_TableDiff() {
		super( "org.opengion.fukurou.process.Process_TableDiff",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理（ファイルオープン、ＤＢオープン等）に使用します。
	 *
	 * @param   paramProcess ParamProcess
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		infile1				= arg.getProparty( "infile1" );
		infile2				= arg.getProparty( "infile2" );
		action				= arg.getProparty( "action"  );
		String  encode1		= arg.getProparty( "encode1",ENCODE );
		String  encode2		= arg.getProparty( "encode2",ENCODE );
		separator1			= arg.getProparty( "sep1",separator1 );
		separator2			= arg.getProparty( "sep2",separator2 );
		String  clms		= arg.getProparty( "columns"  );
		String  keyClms		= arg.getProparty( "keyClms"  );
		String  diffClms	= arg.getProparty( "diffClms" );
		display				= arg.getProparty( "display",display );
		debug				= arg.getProparty( "debug"  ,debug );

		if( infile1 == null || infile2 == null ) {
			String errMsg = "ファイル名が指定されていません。"
						+ "File1=[" + infile1 + "] , File2=[" + infile2 + "]" ;
			throw new RuntimeException( errMsg );
		}

		File file1 = new File( infile1 );
		File file2 = new File( infile2 );

		if( ! file1.exists() || ! file2.exists() ) {
			// 4.3.1.1 (2008/08/23) Avoid if (x != y) ..; else ..;
			String errMsg = "ファイルが存在しません。"
						+ ((file1.exists()) ? "" : "File1=[" + file1 + "] " )
						+ ((file2.exists()) ? "" : "File2=[" + file2 + "]" );
			throw new RuntimeException( errMsg );
		}

		if( ! file1.isFile() || ! file2.isFile() ) {
			// 4.3.1.1 (2008/08/23) Avoid if (x != y) ..; else ..;
			String errMsg = "フォルダは指定できません。ファイル名を指定してください。"
						+ ((file1.isFile()) ? "" : "File1=[" + file1 + "] " )
						+ ((file2.isFile()) ? "" : "File2=[" + file2 + "]" );
			throw new RuntimeException( errMsg );
		}

		reader1 = FileUtil.getBufferedReader( file1,encode1 );
//		reader2 = FileUtil.getBufferedReader( file2,encode2 );

		final String[] names ;
		if( clms != null ) {
			names = StringUtil.csv2Array( clms );	// 指定のカラム名配列
		}
		else {
			String[] clmNames = readName( reader1 );		// ファイルのカラム名配列
			if( clmNames == null || clmNames.length == 0 ) { nameNull = true; return ; }
			names = clmNames;
		}

		model = new LineModel();
		model.init( names );

		if( display ) { println( model.nameLine() ); }

		// 入力カラム名のカラム番号
		clmNos = new int[names.length];
		for( int i=0; i<names.length; i++ ) {
			clmNos[i] = i+1;						// 行番号分を＋１しておく。
//			int no = model.getColumnNo( names[i] );
//			if( no >= 0 ) { clmNos[no] = i+1; }		// 行番号分を＋１しておく。
		}

		// 比較する列の基準カラム名
		if( debug ) { println( "DEBUG:\tkeyClms=" + keyClms ); }
		final String[] keyClmNms = StringUtil.csv2Array( keyClms );
		keyClmNos = new int[keyClmNms.length];
		for( int i=0; i<keyClmNms.length; i++ ) {
			keyClmNos[i] = model.getColumnNo( keyClmNms[i] );
	//		if( debug ) { println( "DEBUG:" + keyClmNms[i] + ":[" + keyClmNos[i] + "]" ); }
	//		int no = model.getColumnNo( keyClmNms[i] );
	//		if( no >= 0 ) { keyClmNos[no] = i+1; }		// 行番号分を＋１しておく。
		}

		// 比較するカラム名
		if( debug ) { println( "DEBUG:\tdiffClms=" + diffClms ); }
		final String[] diffClmNms = StringUtil.csv2Array( diffClms );
		diffClmNos = new int[diffClmNms.length];
		for( int i=0; i<diffClmNms.length; i++ ) {
			diffClmNos[i] = model.getColumnNo( diffClmNms[i] );
	//		if( debug ) { println( "DEBUG:" + diffClmNms[i] + ":[" + diffClmNos[i] + "]" ); }
	//		int no = model.getColumnNo( diffClmNms[i] );
	//		if( no >= 0 ) { diffClmNos[no] = i+1; }		// 行番号分を＋１しておく。
		}

		readF2Data( file2,encode2 );
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理（ファイルクローズ、ＤＢクローズ等）に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか(true:成功/false:失敗）
	 */
	public void end( final boolean isOK ) {
		Closer.ioClose( reader1 );
		reader1 = null;
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @return  boolean 処理できる:true / 処理できない:false
	 */
	public boolean next() {
		if( nameNull ) { return false; }

		boolean flag = false;
		try {
			while((line = reader1.readLine()) != null) {
				inCount1++ ;
				if( line.length() == 0 || line.charAt( 0 ) == '#' ) { continue; }
				else {
					flag = true;
					break;
				}
			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + infile1 + "]:(" + inCount1 + ")"  ;
			throw new RuntimeException( errMsg,ex );
		}
		return flag;
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * ファイルより読み込んだ１行のデータを テーブルモデルに
	 * セットするように分割します
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 *
	 * @param   rowNo int 処理中の行番号
	 * @return  LineModel  処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		outCount++ ;
		String[] vals = StringUtil.csv2Array( line ,separator1.charAt(0) );

		int len = vals.length;
		for( int clmNo=0; clmNo<model.size(); clmNo++ ) {
			int no = clmNos[clmNo];
			if( len > no ) {
				model.setValue( clmNo,vals[no] );
			}
			else {
				// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
				model.setValue( clmNo,"" );
			}
		}
		model.setRowNo( rowNo ) ;

//		if( display ) { println( model.dataLine() ); }		// 5.1.2.0 (2010/01/01) display の条件変更

		return action( model );
	}

	/**
	 * キーと、DIFF設定値を比較し、action に応じた LineModel を返します。
	 * action には、ONLY,DIFF,INTERSEC が指定できます。
	 *   ONLY      inFile1 のみに存在する行の場合、inFile1 のレコードを返します。
	 *   DIFF      inFile1 と inFile2 に存在し、かつ、DIFF値が異なる、inFile1 のレコードを返します。
	 *   INTERSEC  inFile1 と inFile2 に存在し、かつ、DIFF値も同じ、inFile1 のレコードを返します。
	 * inFile2 側をキャッシュしますので、inFile2 側のデータ量が少ない様に選んでください。
	 *
	 * @param 	model LineModel
	 * @return	LineModel action 実行後のLineModel
	 */
	private LineModel action( final LineModel model ) {
		LineModel rtn = null;
		Object[] obj = model.getValues();

		// キーのカラムを合成します。
		StringBuilder keys = new StringBuilder();
		for( int i=0; i<keyClmNos.length; i++ ) {
			keys.append( obj[keyClmNos[i]] ).append( "," );
		}

		String data = file2Map.get( keys.toString() );
	//	if( debug ) { println( "DEBUG:" + keys.toString() + ":" + data ); }

		if( "ONLY".equalsIgnoreCase( action ) && data == null ) {
			if( debug ) { println( "DEBUG:ONLY\t" + keys.toString() ); }
			rtn = model;
		}
		else {
			// DIFF値のカラムを合成します。
			StringBuilder vals = new StringBuilder();
			for( int i=0; i<diffClmNos.length; i++ ) {
				vals.append( obj[diffClmNos[i]] ).append( "," );
			}

			boolean eq = ( vals.toString() ).equals( data );

			if( "DIFF".equalsIgnoreCase( action ) && ! eq ) {
				if( debug ) { println( "DEBUG:DIFF\t" + keys.toString() + "\t" + data + "\t" + vals.toString() ); }
				rtn = model;
			}
			else if( "INTERSEC".equalsIgnoreCase( action ) && eq ) {
				if( debug ) { println( "DEBUG:INTERSEC\t" + keys.toString() + "\t" + data ); }
				rtn = model;
			}
		}
		if( display && rtn != null ) { println( rtn.dataLine() ); }
		return rtn;
	}

	/**
	 * BufferedReader より、#NAME 行の項目名情報を読み取ります。
	 * データカラムより前に、項目名情報を示す "#Name" が存在する仮定で取り込みます。
	 * この行は、ファイルの形式に無関係に、TAB で区切られています。
	 *
	 * @param 	reader PrintWriter
	 * @return	String[] カラム名配列(存在しない場合は、サイズ０の配列)
	 */
	private String[] readName( final BufferedReader reader ) {
		try {
			// 4.0.0 (2005/01/31) line 変数名変更
			String line1;
			while((line1 = reader.readLine()) != null) {
				inCount1++ ;
				if( line1.length() == 0 ) { continue; }
				if( line1.charAt(0) == '#' ) {
					String key = line1.substring( 0,5 );
					if( key.equalsIgnoreCase( "#NAME" ) ) {
						// 超イレギュラー処理 最初の TAB 以前の文字は無視する。
						String line2 = line1.substring( line1.indexOf( TAB )+1 );
						return StringUtil.csv2Array( line2 ,TAB.charAt(0) );
					}
					else  { continue; }
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。";
					throw new RuntimeException( errMsg );
				}
			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + infile1 + "]:(" + inCount1 + ")"  ;
			throw new RuntimeException( errMsg,ex );
		}
		return new String[0];
	}

	/**
	 * inFile2 を読取り、キー情報を作成し、内部メモリマップにキャッシュします。
	 * このマップをもとに、inFile1 のデータを逐次読み取って、処理を進めます。
	 *
	 * @param 	file2 File
	 * @param 	encode2 String
	 */
	private void readF2Data( final File file2, final String encode2 ) {
		BufferedReader reader2 = null;
		try {
			if( debug ) { println( "DEBUG:\tFile2="+ file2 + " 初期処理" ); }
			reader2 = FileUtil.getBufferedReader( file2,encode2 );
			// 4.0.0 (2005/01/31) line 変数名変更
			String line1;
			char sep2 = separator2.charAt(0);
			while((line1 = reader2.readLine()) != null) {
				inCount2++ ;
				if( line1.length() == 0 ) { continue; }
				if( line1.charAt(0) == '#' ) { continue; }
				else {
					// 超イレギュラー処理 最初の TAB 以前の文字は無視する。
					String line2 = line1.substring( line1.indexOf( separator2 )+1 );
					Object[] obj = StringUtil.csv2Array( line2 , sep2 );

					// キーのカラムを合成します。
					StringBuilder keys = new StringBuilder();
					for( int i=0; i<keyClmNos.length; i++ ) {
						keys.append( obj[keyClmNos[i]] ).append( "," );
					}

					// DIFF値のカラムを合成します。
					StringBuilder vals = new StringBuilder();
					for( int i=0; i<diffClmNos.length; i++ ) {
						vals.append( obj[diffClmNos[i]] ).append( "," );
					}

					if( debug ) { println( "DEBUG:\t" + keys.toString() + "\t" + vals.toString() ); }

					file2Map.put( keys.toString(), vals.toString() );
				}
			}
			if( debug ) { println( "DEBUG:\t======初期処理終了======" ); }
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + infile2 + "]:(" + inCount2 + ")"  ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader2 );
		}
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Input  File1  : " + infile1	+ CR
				+ TAB + "Input  File2  : " + infile2	+ CR
				+ TAB + "Input  Count1 : " + inCount1	+ CR
				+ TAB + "Input  Count2 : " + inCount2	+ CR
				+ TAB + "Output Count  : " + outCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return  String
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_TableDiffは、ファイルから読み取った内容を、LineModel に設定後、" 	).append( CR );
		buf.append( "下流に渡す、FirstProcess インターフェースの実装クラスです。"					).append( CR );
		buf.append( CR );
		buf.append( "DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、"		).append( CR );
		buf.append( "下流（プロセスチェインのデータは上流から下流に渡されます。）に渡します。"		).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param   args String[]
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_TableDiff().usage() );
	}
}
