/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.LogWriter;

import java.util.Map ;
import java.util.LinkedHashMap ;

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Process_TableReaderは、ファイルから読み取った内容を、LineModel に設定後、
 * 下流に渡す、FirstProcess インターフェースの実装クラスです。
 *
 * DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、
 * 下流（プロセスチェインのデータは上流から下流に渡されます。）に渡します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * @og.formSample
 *  Process_TableReader -infile=INFILE -sep=, -encode=UTF-8 -columns=AA,BB,CC
 *
 *    -infile=入力ファイル名     ：入力ファイル名
 *   [-existCheck=存在確認     ] ：ファイルが存在しない場合エラーにする。(初期値：true)
 *   [-sep=セパレータ文字      ] ：区切り文字（初期値：タブ）
 *   [-encode=文字エンコード   ] ：入力ファイルのエンコードタイプ
 *   [-columns=読み取りカラム名] ：入力カラム名(カンマ区切り)
 *   [-display=false|true      ] ：結果を標準出力に表示する(true)かしない(false)か（初期値 false:表示しない)
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_TableReader extends AbstractProcess implements FirstProcess {
	private String		  	separator	= TAB;	// 項目区切り文字
	private String			infile		= null;
	private BufferedReader	reader		= null;
	private LineModel		model		= null;
	private String			line		= null;
	private int[]			clmNos		= null;		// ファイルのヘッダーのカラム番号
	private boolean			display		= false;	// 表示しない
	private boolean			nameNull	= false;	// ０件データ時 true

	private int				inCount		= 0;
	private int				outCount	= 0;

	private final static Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private final static Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "infile",	"入力ファイル名 (必須)" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "existCheck",	"ファイルが存在しない場合エラーにする。(初期値：true)" );
		usableProparty.put( "sep",			"区切り文字（初期値：タブ）" );
		usableProparty.put( "encode",		"入力ファイルのエンコードタイプ" );
		usableProparty.put( "columns",		"入力カラム名(カンマ区切り)" );
		usableProparty.put( "display",		"結果を標準出力に表示する(true)かしない(false)か" +
											CR + "（初期値 false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_TableReader() {
		super( "org.opengion.fukurou.process.Process_TableReader",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理（ファイルオープン、ＤＢオープン等）に使用します。
	 *
	 * @param   paramProcess ParamProcess
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		infile				= arg.getProparty("infile");
		boolean existCheck	= arg.getProparty("existCheck",true);
		String  encode		= arg.getProparty("encode",System.getProperty("file.encoding"));
		separator			= arg.getProparty("sep",separator );
		String  clms		= arg.getProparty("columns" );
		display				= arg.getProparty("display",display);

		if( infile == null ) {
			String errMsg = "ファイル名が指定されていません。" ;
			throw new RuntimeException( errMsg );
		}

		File file = new File( infile );

		if( ! file.exists() ) {
			if( existCheck ) {
				String errMsg = "ファイルが存在しません。File=[" + file + "]" ;
				throw new RuntimeException( errMsg );
			}
			else {
				nameNull = true; return ;
			}
		}

		if( ! file.isFile() ) {
			String errMsg = "ファイル名を指定してください。File=[" + file + "]" ;
			throw new RuntimeException( errMsg );
		}

		reader = FileUtil.getBufferedReader( file,encode );

		String[] clmNames = readName( reader );		// ファイルのカラム名配列
		if( clmNames == null || clmNames.length == 0 ) { nameNull = true; return ; }

		final String[] names ;
		if( clms != null ) {
			names = StringUtil.csv2Array( clms );	// 指定のカラム名配列
		}
		else {
			names = clmNames;
		}

		model = new LineModel();
		model.init( names );

		if( display ) { println( model.nameLine() ); }

		clmNos = new int[names.length];
		for( int i=0; i<clmNames.length; i++ ) {
			int no = model.getColumnNo( clmNames[i] );
			if( no >= 0 ) { clmNos[no] = i+1; }		// 行番号分を＋１しておく。
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理（ファイルクローズ、ＤＢクローズ等）に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか(true:成功/false:失敗）
	 */
	public void end( final boolean isOK ) {
		Closer.ioClose( reader );
		reader = null;
	}

	/**
	 * このデータの処理において、次の処理が出来るかどうかを問い合わせます。
	 * この呼び出し１回毎に、次のデータを取得する準備を行います。
	 *
	 * @return  boolean 処理できる:true / 処理できない:false
	 */
	public boolean next() {
		if( nameNull ) { return false; }

		boolean flag = false;
		try {
			while((line = reader.readLine()) != null) {
				inCount++ ;
				if( line.length() == 0 || line.charAt( 0 ) == '#' ) { continue; }
				else {
					flag = true;
					break;
				}
			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + reader.toString() + "]"  ;
			throw new RuntimeException( errMsg,ex );
		}
		return flag;
	}

	/**
	 * 最初に、 行データである LineModel を作成します
	 * FirstProcess は、次々と処理をチェインしていく最初の行データを
	 * 作成して、後続の ChainProcess クラスに処理データを渡します。
	 *
	 * ファイルより読み込んだ１行のデータを テーブルモデルに
	 * セットするように分割します
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 *
	 * @param   rowNo int 処理中の行番号
	 * @return  LineModel  処理変換後のLineModel
	 */
	public LineModel makeLineModel( final int rowNo ) {
		outCount++ ;
		String[] vals = StringUtil.csv2Array( line ,separator.charAt(0) );

		int len = vals.length;
		for( int clmNo=0; clmNo<model.size(); clmNo++ ) {
			int no = clmNos[clmNo];
			if( len > no ) {
				model.setValue( clmNo,vals[no] );
			}
			else {
				// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
				model.setValue( clmNo,"" );
			}
		}
		model.setRowNo( rowNo ) ;

		if( display ) { println( model.dataLine() ); }

		return model;
	}

	/**
	 * BufferedReader より、#NAME 行の項目名情報を読み取ります。
	 * データカラムより前に、項目名情報を示す "#Name" が存在する仮定で取り込みます。
	 * この行は、ファイルの形式に無関係に、TAB で区切られています。
	 *
	 * @param 	reader PrintWriter
	 * @return	String[] カラム名配列(存在しない場合は、サイズ０の配列)
	 */
	private String[] readName( final BufferedReader reader ) {
		try {
			// 4.0.0 (2005/01/31) line 変数名変更
			String line1;
			while((line1 = reader.readLine()) != null) {
				inCount++ ;
				if( line1.length() == 0 ) { continue; }
				if( line1.charAt(0) == '#' ) {
					String key = line1.substring( 0,5 );
					if( key.equalsIgnoreCase( "#NAME" ) ) {
						// 超イレギュラー処理 最初の TAB 以前の文字は無視する。
						String line2 = line1.substring( line1.indexOf( TAB )+1 );
						return StringUtil.csv2Array( line2 ,TAB.charAt(0) );
					}
					else  { continue; }
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。";
					throw new RuntimeException( errMsg );
				}
			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + reader.toString() + "]"  ;
			throw new RuntimeException( errMsg,ex );
		}
		return new String[0];
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "Input  File  : " + infile	+ CR
				+ TAB + "Input  Count : " + inCount	+ CR
				+ TAB + "Output Count : " + outCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return  String
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_TableReaderは、ファイルから読み取った内容を、LineModel に設定後、" 	).append( CR );
		buf.append( "下流に渡す、FirstProcess インターフェースの実装クラスです。"					).append( CR );
		buf.append( CR );
		buf.append( "DBTableModel 形式のファイルを読み取って、各行を LineModel にセットして、"		).append( CR );
		buf.append( "下流（プロセスチェインのデータは上流から下流に渡されます。）に渡します。"		).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param   args String[]
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_TableReader().usage() );
	}
}
