/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.CodeData;
import org.opengion.hayabusa.resource.ColumnData;
import org.opengion.hayabusa.resource.LabelData;
import org.opengion.hayabusa.resource.RoleMode;

/**
 * DBType インターフェースを継承した Abstractクラスです。
 * getRendererValue( String value ) 、getEditorValue( String value ) 、
 * isValueChack( String ) メソッドを、サブクラスで実装する必要があります。
 *
 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
 * @og.group テーブル管理
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBColumn {

	// 3.5.4.2 (2003/12/15) COLUMNS_MAXSIZE を定義しておきます。
	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" ) ;   // 表示フィールドの大きさ

	private final ColumnData columnData ;
	private final LabelData  labelData ;
	private final CodeData   codeData ;

	private final String		lang			;		// 言語
	private final boolean		writable		;		// カラムが書き込み可能かどうか
	private final CellRenderer	cellRenderer	;		// 表示用レンデラー
	private final CellEditor	cellEditor		;		// 編集用エディター
	private final DBType		dbType			;		// データのタイプ
	private final String		defValue		;		// データのデフォルト値
	private final Attributes	rendAttri		;		// 表示用レンデラー追加用属性
	private final Attributes	editAttri		;		// 編集用エディター追加用属性
	private final boolean		addNoValue 		;		// メニューに空の選択リストを追加するかどうか	// 3.5.5.7 (2004/05/10)
	private final boolean		official	 	;		// カラムリソースから作成されたかどうか		// 3.6.0.7 (2004/11/06)
	private final String		dbid			;		// データベース接続先ID

	private final int			checkLevel		;		// DBColumn の 整合性チェックを行うレベルを規定します。

	private final Selection	selection		;		// 4.0.0.0 (2007/11/07)

	private final String		eventColumn		;	// 4.3.6.0 (2009/04/01) イベントカラム
	private final String		rawEditParameter;	// 4.3.6.0 (2009/04/01) {@XXXX}の変換がされていない生パラメータ
	private final String		rawRendParameter;	// 5.1.7.0  (2009/04/01) {@XXXX}の変換がされていない生パラメータ
	private final String		eventURL		;	// 4.3.6.0 (2009/04/01) イベントカラムで利用するURL

	/**
	 * DBColumnConfig オブジェクトより作成されるコンストラクター
	 * すべての情報は、インスタンス作成時に設定します。
	 * このオブジェクトは、１度作成されると変更されることはありません。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.0.0.0 (2007/11/07) Selectionオブジェクトをキャッシュする
	 * @og.rev 4.3.6.0 (2009/04/01) eventColumnの対応
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   lang    String
	 * @param   clmData ColumnData
	 * @param   lblData LabelData
	 * @param   cdData  CodeData
	 */
	public DBColumn( final String 	  lang,
					 final ColumnData clmData ,
					 final LabelData  lblData ,
					 final CodeData   cdData ) {
		this.lang       = lang ;
		this.columnData = clmData ;
		this.labelData  = lblData ;
		this.codeData   = cdData  ;

		writable  = true ;

		try {
			dbType = DBTypeFactory.newInstance( columnData.getDbType() );
		}
		catch( RuntimeException e ) {
			String errMsg = "dbType の作成に失敗しました。"
						+ " name=[" + columnData.getName() + "]"
						+ " dbType=[" + columnData.getDbType() + "] "
						+ e.getMessage();
			throw new HybsSystemException( errMsg,e );
		}

		rendAttri	= null	;
		editAttri	= null	;
		addNoValue	= false	;
		official	= true	;		// 3.6.0.7 (2004/11/06)
		checkLevel	= -1;
		dbid		= null	;		// 標準から作成されるカラムオブジェクトは、DEFAULT 接続先を設定する。
		
		eventColumn  = null; 		// 4.3.6.0 (2009/04/01) 
		rawEditParameter = clmData.getEditorParam(); // 4.3.6.0 (2009/04/01) 
		rawRendParameter = clmData.getRendererParam(); // 5.1.7.0 (2010/06/01)
		eventURL	 = null; 		// 4.3.6.0 (2009/04/01) 

		String def = columnData.getDefault();
		if( def != null ) { defValue = def; }
		else {              defValue = dbType.getDefault() ; }

		// 4.0.0.0 (2007/11/07)
		selection = ( codeData == null ? null : new Selection_CODE( codeData ) );

		try {
			cellRenderer = DBCellFactory.newRenderer( columnData.getRenderer(),this );
			cellEditor   = DBCellFactory.newEditor( columnData.getEditor(),this );
		}
		catch( RuntimeException e ) {
			String errMsg = "Renderer,Editor の作成に失敗しました。"
						+ " name=[" + columnData.getName() + "]"
						+ " Renderer=[" + columnData.getRenderer() + "] "
						+ " Editor=[" + columnData.getEditor() + "] "
						+ e.getMessage();
			throw new HybsSystemException( errMsg,e );
		}
	}

	/**
	 * DBColumnConfig オブジェクトより作成されるコンストラクター
	 * すべての情報は、インスタンス作成時に設定します。
	 * このオブジェクトは、１度作成されると変更されることはありません。
	 *
	 * @og.rev 4.0.0.0 (2007/11/07) Selectionオブジェクトをキャッシュする
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   config DBColumnConfig
	 */
	public DBColumn( final DBColumnConfig config ) {
		lang      = config.getLang()		;

		columnData	= config.getColumnData();
		labelData	= config.getLabelData();
		codeData	= config.getCodeData();

		writable	= config.isWritable();
		dbType		= DBTypeFactory.newInstance( columnData.getDbType() );
		rendAttri	= config.getRendererAttributes();
		editAttri	= config.getEditorAttributes();
		addNoValue	= config.isAddNoValue();
		official	= config.isOfficial();		// 3.6.0.7 (2004/11/06)
		dbid		= config.getDbid();
		
		eventColumn  = config.getEventColumn();  // 4.3.6.0 (2009/04/01)
		rawEditParameter = config.getRawEditParameter(); // 4.3.6.0 (2009/04/01)
		rawRendParameter = config.getRawRendParameter(); // 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
		eventURL	 = config.getEventURL(); 	 // 4.3.6.0 (2009/04/01)

		// DBColumn の 整合性チェックを行うレベルを規定します。
		String CHECK_LEVEL = HybsSystem.sys( "DB_OFFICIAL_COLUMN_CHECK_LEVEL" );
		if( !official && ( CHECK_LEVEL != null && CHECK_LEVEL.length() > 0 ) ) {
			checkLevel = Integer.parseInt( CHECK_LEVEL );
		}
		else {
			checkLevel = -1;
		}

		String def = config.getDefValue();
		if( def != null ) { defValue = def; }
		else {              defValue = dbType.getDefault() ; }

		// 4.0.0.0 (2007/11/07)
		selection = ( codeData == null ? null : new Selection_CODE( codeData ) );

		cellRenderer = DBCellFactory.newRenderer( columnData.getRenderer(),this );
		cellEditor   = DBCellFactory.newEditor( columnData.getEditor(),this );
	}

	/**
	 * 言語を返します。
	 *
	 * @return  言語
	 */
	public String getLang() {
		return lang;
	}

	/**
	 * カラム名を返します。
	 *
	 * @return  カラム名
	 */
	public String getName() {
		return columnData.getName();
	}

	/**
	 * カラムのラベル名を返します。
	 *
	 * @return  カラムのラベル名
	 */
	public String getLabel() {
		return labelData.getLabel();
	}

	/**
	 * カラムのラベル名を返します。
	 *
	 * @return  カラムのラベル名(名前(短))
	 */
	public String getShortLabel() {
		return labelData.getShortLabel();
	}

	/**
	 * カラムのラベル名を返します。
	 *
	 * @return  カラムのラベル名(名前(長))
	 */
	public String getLongLabel() {
		return labelData.getLongLabel();
	}

	/**
	 * カラムのクラスを文字列にした名称を返します。
	 * これは，HTML上の各種タグに，データベース定義に応じたクラス名を
	 * 表します。(VARCHAR2、NUMBER など)
	 *
	 * これは、カラムのデータタイプ(X,S9など)と機能的に重複しますが､
	 * そのカラムが、大まかな分類で、文字列であるか、数字であるかを示します。
	 *
	 * 画面表示上の右寄せ、左寄せや、IMEのON/OFFなどの CSSファイルの指定は、
	 * カラムのデータタイプ(X,S9など)ですが、テーブルソート時のソート条件
	 * は、この クラス文字列(VARCHAR2、NUMBER)を参照します。
	 *
	 * @return  カラムのクラスを文字列にした名称(VARCHAR2、NUMBER など)
	 */
	public String getClassName() {
		return columnData.getClassName();
	}

	/**
	 * フィールドのデータ長を返します。
	 * 通常は、整数型の文字列とデータ長は同じですが,小数点を表すデータ長は
	 * x,y 形式の場合、x + 2 桁で表されます。(マイナス記号と小数点記号)
	 * 7,3 は、 xxxx,yyy のフォーマットで、データ長は、9 になります。
	 *
	 * @og.rev 2.1.1.2 (2002/11/21) 最大桁数入力時の桁数チェックの間違いを訂正。
	 * @og.rev 4.0.0 (2005/01/31) メソッド名変更 getMaxlength() → getTotalSize()
	 *
	 * @return  データ長定義文字列
	 */
	public int getTotalSize() {
		return columnData.getTotalSize();
	}

	/**
	 * フィールドの使用桁数を返します。
	 * 少数指定の場合は、"7,3" のようなカンマで整数部、小数部を区切った書式になります。
	 * 7,3 は、 xxxx,yyy のフォーマットで、整数部４桁、小数部３桁を意味します。
	 *
	 * @return  使用桁数
	 */
	public String getMaxlength() {
		return columnData.getMaxlength();
	}

	/**
	 * フィールドの整数部のデータ長を返します。
	 * 通常は、整数型の文字列とデータ長は同じですが,小数点を表すデータ長は
	 * x,y 形式の場合、x - y 桁で表されます。(マイナス記号含まず)
	 * 7,3 は、 xxxx,yyy のフォーマットで、データ長は、4 になります。
	 *
	 * @return  データ長定義文字列
	 */
	public int getSizeX() {
		return columnData.getSizeX() ;
	}

	/**
	 * フィールドの小数部のデータ長を返します。
	 * 通常は、整数型の文字列では、0 になりますが,小数点を表すデータ長は
	 * x,y 形式の場合、y 桁で表されます。
	 * 7,3 は、 xxxx,yyy のフォーマットで、データ長は、3 になります。
	 *
	 * @return  データ長定義文字列
	 */
	public int getSizeY() {
		return columnData.getSizeY() ;
	}

	/**
	 * カラムの表示桁数を返します。
	 * viewLength は、設定した場合のみ、使用できます。通常は、null が返ります。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 * @og.rev 4.0.0 (2005/01/31) メソッド名変更 getSize() → getViewLength()
	 *
	 * @return	カラムの文字桁数
	 */
	public String getViewLength() {
		return columnData.getViewLength();
	}

	/**
	 * カラムが書き込み可能かどうかを返します。
	 *
	 * @return  カラムが書き込み可能かどうか
	 */
	public boolean isWritable() {
		return writable;
	}

	/**
	 * データの値そのものではなく、その値のラベル文字を返します。
	 *
	 * @og.rev 3.5.5.4 (2004/04/15) value が null の場合は、""(ゼロストリング)にする。
	 *
	 * @param   value String
	 * @return  データ表示用の文字列
	 */
	public String getRendererValue( final String value ) {
		return cellRenderer.getValue( (value != null ) ? value : "" );
	}

	/**
	 * データ表示用のHTML文字列を作成します。
	 * 行番号毎に異なる値を返すことの出来る DBCellRenderer を使用することが出来ます。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 行番号に無関係に、値を返すように変更します。
	 * @og.rev 4.0.0 (2005/11/30) 行番号に対応した値を返すように変更します。
	 *
	 * @param   row   int
	 * @param   value String
	 * @return  データ編集用の文字列
	 */
	public String getRendererValue( final int row,final String value ) {
		return cellRenderer.getValue( row,(value != null ) ? value : "" );
	}

	/**
	 * データ編集用のHTML文字列を作成します。
	 *
	 * @og.rev 3.5.5.4 (2004/04/15) value が null の場合は、""(ゼロストリング)にする。
	 *
	 * @param   value String
	 * @return  データ編集用の文字列
	 */
	public String getEditorValue( final String value ) {
		return cellEditor.getValue( (value != null ) ? value : "" );
	}

	/**
	 * データ編集用のHTML文字列を作成します。
	 * 行番号付の編集データを作成します。名前_行番号 で登録する為、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param   row   int
	 * @param   value String
	 * @return  データ編集用の文字列
	 */
	public String getEditorValue( final int row,final String value ) {
		return cellEditor.getValue( row,(value != null ) ? value : "" );
	}

	/**
	 * データ出力用の固定長文字列を作成します。
	 * HOST送信用桁数がセットされていれば、そちらを優先します。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value String
	 * @param   encode   固定長で変換する文字エンコード
	 * @return  データ編集用の文字列
	 */
	public String getWriterValue( final String value,final String encode ) {
		return dbType.valueFill( value ,columnData.getSizeX(), columnData.getSizeY(), encode );
	}

	/**
	 * データの表示用レンデラーを返します。
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 *
	 * @return	データの表示用レンデラー
	 */
	public String getRenderer() {
		return columnData.getRenderer() ;
	}

	/**
	 * データの編集用エディターを返します。
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 *
	 * @return	データの編集用エディター
	 */
	public String getEditor() {
		return columnData.getEditor() ;
	}

	/**
	 * 文字種別名を返します。
	 * カラムの文字種別名名称を返します。
	 * これは，HTML上の各種タグに，データベース定義に応じたクラスを
	 * セットし,CSS（Cascading Style Sheet)の class="xxxxx" とする事により
	 * 各種スタイルを表現するのに使用します。
	 *
	 * ここでは, カラムリソースの DBTYPE 属性で指定の文字列（X,S9,KXなど）を返します。
	 *
	 * @return  データの文字種別(X,KX,S9 など）
	 */
	public String getDbType() {
		return columnData.getDbType() ;
	}

	/**
	 * データのNATIVEの型の識別コードを返します。<br />
	 * 
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @see org.opengion.fukurou.model.NativeType
	 */
	public NativeType getNativeType() {
		return dbType.getNativeType();
	}

	/**
	 * そのカラムの,デフォルト値の値を返します。
	 *
	 * カラムリソースに デフォルト情報が登録されている場合は,その値を返します。
	 * デフォルト値が設定されていない場合は, null を返します。
	 *
	 * @return  デフォルト値（無ければ null）
	 */
	public String getDefault() {
		return defValue;
	}

	/**
	 * 表示用レンデラーのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	表示用レンデラーのパラメータ
	 */
	public String getRendererParam() {
		return columnData.getRendererParam();
	}

	/**
	 * 編集用エディターのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	編集用エディターのパラメータ
	 */
	public String getEditorParam() {
		return columnData.getEditorParam();
	}

	/**
	 * データタイプのパラメータを取得します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 新規追加
	 *
	 * @return	データタイプのパラメータ
	 */
	public String getDbTypeParam() {
		return columnData.getDbTypeParam();
	}

	/**
	 * カラムロールを取得します。
	 *
	 * @og.rev 4.0.0 (2005/11/30) 新規追加
	 *
	 * @return	カラムロール
	 */
	public String getRoles() {
		return columnData.getRoles();
	}

	/**
	 * カラムオブジェクトのロールモードを返します。
	 *
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 *
	 * @return カラムオブジェクトのロールモード
	 */
	public RoleMode getRoleMode() {
		return columnData.getRoleMode();
	}

	/**
	 * 接続先IDを返します。
	 *
	 * @return  接続先ID
	 */
	public String getDbid() {
		return dbid;
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合（A,B,C など）は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。（桁上がりもあり）
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列（""）の場合は,引数を,そのまま返します。
	 *
	 * @param   value  String
	 * @return  引数の文字列を＋１した文字列
	 */
	public String valueAdd( final String value ) {
		return dbType.valueAdd( value );
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 3.6.0.7 (2004/11/06) official属性がfalseの場合、チェックレベルに対応したチェックを行う。
	 * @og.rev 3.8.0.8 (2005/10/03) 企画改善 61200-050921-02 ラベルからHTMLタグを削除する。
	 *
	 * @param   value String
	 * @return  エラー内容  正常時は null
	 */
	public ErrorMessage valueCheck( final String value ) {

		// 3.8.0.8 (2005/10/03) ラベルからHTMLタグを削除する。
		String lbl = labelData.getLabel() ;
		if( lbl.indexOf( '<' ) >= 0 ) {
			lbl = lbl.replaceAll( "<[^>]*>","" );
		}

		ErrorMessage errMsg = dbType.valueCheck( lbl,value,columnData.getSizeX(), columnData.getSizeY(),columnData.getDbTypeParam() );
		if( checkLevel >= 0 ) {
			// ERR0034:指定のカラムオブジェクトには、カラムリソースが存在しません。name={0} label={1}
			errMsg.addMessage( 0,checkLevel,"ERR0034", columnData.getName(),labelData.getLabel() );
		}
		return errMsg ;
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと DBCellEditor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @param   value String （一般に編集データとして登録されたデータ）
	 * @return  修正後の文字列（一般にデータベースに登録するデータ）
	 */
	public String valueSet( final String value ) {
		return dbType.valueSet( value );
	}

	/**
	 * action で指定されたコマンドを実行して、値の変換を行います。
	 * oldValue(旧データ)は、元のDBTableModelに設定されていた値です。通常は、
	 * この値を使用してカラム毎に変換を行います。newValue(新データ)は、引数で
	 * 指定された新しい値です。この値には、パラメータを指定して変換方法を
	 * 制御することも可能です。
	 * 指定のアクションがカラムで処理できない場合は、エラーになります。
	 *
	 * @param   action アクションコマンド
	 * @param   oldValue 入力データ（旧データ）
	 * @param   newValue 入力データ（新データ）
	 * @return	実行後のデータ
	 */
	public String valueAction( final String action,final String oldValue,final String newValue ) {
		return dbType.valueAction( action,oldValue,newValue );
	}

	/**
	 * 内部の設定情報オブジェクトを返します。
	 * このオブジェクトを ローカルで書き換えて、DBColumn を作るようにします。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig 関係の見直し。
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.5.5.5 (2004/04/23) size 属性の意味を変更、maxlength 属性を追加。
	 * @og.rev 3.5.5.8 (2004/05/20) codeName 属性を追加。
	 * @og.rev 3.6.0.7 (2004/11/06) カラムリソースから作成されたかどうか(official属性追加)
	 *
	 * @return  DBColumnConfig
	 */
	public DBColumnConfig getConfig() {
		// 互換性確保のため、DBColumnConfig の コンストラクタは修正していません。

		DBColumnConfig config =
			new DBColumnConfig(
						lang							,
						columnData.getName()			,
						labelData						,
						columnData.getClassName()		,
						columnData.getMaxlength()		,
						String.valueOf( writable )		,
						columnData.getRenderer()		,
						columnData.getEditor()			,
						codeData						,
						columnData.getDbType()			,
						defValue						,
						columnData.getRendererParam()	,
						columnData.getEditorParam()		,
						columnData.getDbTypeParam()		,
						columnData.getRoles()			,		// 4.0.0 (2005/11/30)
						official						,		// 3.6.0.7 (2004/11/06)
						dbid	 ) ;
		config.setViewLength( columnData.getViewLength() );

		return config ;
	}

	/**
	 * 表示用レンデラーの追加属性を返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig オブジェクトの内部キープを廃止
	 *
	 * @return   attri Attributes
	 */
	public Attributes getRendererAttributes() {
		return rendAttri ;
	}

	/**
	 * 編集用エディター用の追加属性を返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) DBColumnConfig オブジェクトの内部キープを廃止
	 *
	 * @return   attri Attributes
	 */
	public Attributes getEditorAttributes() {
		return editAttri ;
	}

	/**
	 * メニューに空の選択リストを追加するかどうかを取得します。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) 新規追加
	 *
	 * @return	addNoValue (true:追加する/false:追加しない)
	 */
	public boolean isAddNoValue() {
		return addNoValue ;
	}

	/**
	 * このカラムオブジェクトがカラムリソースより正式に作られたかどうかを取得します。
	 * カラムリソースが無い場合は、仮オブジェクトかデータベースメタデータより作成されます。
	 * その場合は、チェック機能が緩くなるため、正式なカラムオブジェクトと区別する為に
	 * この属性を持ちます。
	 *
	 * @og.rev 3.6.0.7 (2004/11/06) 新規追加
	 *
	 * @return	official (true:正式/false:暫定)
	 */
	public boolean isOfficial() {
		return official ;
	}

	/**
	 * カラムのラベルデータオブジェクトを返します。
	 *
	 * @return  カラムのラベルデータオブジェクト
	 */
	public LabelData getLabelData() {
		return labelData;
	}

	/**
	 * カラムのコードデータオブジェクトを返します。
	 *
	 * @return  カラムのコードデータオブジェクト
	 */
	public CodeData getCodeData() {
		if( codeData == null ) {
			String errMsg = "コードリソースが定義されていません。" + HybsSystem.CR
						+ "name=[" + getName() + "],label=[" + getLabel() + "]" ;
			throw new HybsSystemException( errMsg );
		}
		return codeData;
	}

	/**
	 * 最大入力サイズ(maxlength)より、実際のフィールドのサイズを求めます。
	 *
	 * maxlength : colums_maxsize ～  ⇒ size="maxlength"  変換しません。
	 *
	 * ただし、maxlength が、colums_maxsize を超える場合は、colums_maxsize とします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
	 * @og.rev 4.0.0 (2007/02/05) ADJUSTMENT 処理を廃止します。(CSSにて対応)
	 *
	 * @param maxlength	  実際の最大桁数
	 * @param colums_maxsize 表示上の最大桁数。
	 * @return 表示すべきサイズ
	 * @see #getFieldSize( int )
	 */
	public int getFieldSize( final int maxlength,final int colums_maxsize ) {
		int size = maxlength;

		if( size <= 0 || size > colums_maxsize ) {
			size = colums_maxsize;
		}

		return size;
	}

	/**
	 * 最大入力サイズ(maxlength)より、実際のフィールドのサイズを求めます。
	 *
	 * maxlength : colums_maxsize ～  ⇒ size="maxlength"  変換しません。
	 *
	 * ただし、maxlength が、COLUMNS_MAXSIZE を超える場合は、COLUMNS_MAXSIZE とします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
	 * @og.rev 4.0.0 (2007/02/05) ADJUSTMENT 処理を廃止します。(CSSにて対応)
	 *
	 * @param maxlength	  実際の最大桁数
	 * @return 表示すべきサイズ
	 * @see #getFieldSize( int ,int )
	 */
	public int getFieldSize( final int maxlength ) {
		return getFieldSize( maxlength,COLUMNS_MAXSIZE );
	}

	/**
	 * コードリソースSelection オブジェクトを返します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/02) 新規追加
	 *
	 * @return コードSelectionオブジェクト
	 */
	public Selection getSelection() {
		if( selection == null ) {
			String errMsg = "コードリソースが定義されていません。" + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}
		return selection ;
	}

	/**
	 * イベントカラム（親カラム）
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return イベントカラム
	 */
	public String getEventColumn() {
		return eventColumn ;
	}

//	/**
//	 * {&#064;xxx}を変換していない状態の編集パラメータを返す
//	 *
//	 * @og.rev 4.3.6.0 (2009/04/01)
//	 *
//	 * @return 生編集パラメータ
//	 */
//	public String getRawParam() {
//		return rawEditParameter ;
//	}

	/**
	 * {&#064;xxx}を変換していない状態の編集パラメータを返します。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return 生編集パラメータ
	 */
	public String getRawEditParam() {
		return rawEditParameter ;
	}

	/**
	 * {&#064;xxx}を変換していない状態の編集パラメータを返します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @return 生表示パラメータ
	 */
	public String getRawRendParam() {
		return rawRendParameter ;
	}

	/**
	 * eventColumn利用時にJSで利用するURL
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 *
	 * @return イベントURL
	 */
	public String getEventURL() {
		return eventURL ;
	}

	/**
	 * 引数のタグ文字列に対して、イベントカラムの実行に必要なspanタグを付加した
	 * 文字列を返します。
	 * 
	 * @param tag タグ文字列
	 * @param initVal 子カラムの初期値
	 * @param writable タグの要素が書き込み可能かどうか
	 * @return spanタグを付加したタグ文字列
	 */
	public String getEventColumnTag( final String tag, final String initVal, final boolean writable ) {
		return getEventColumnTag( tag, initVal, -1, writable );
	}

	/**
	 * 引数のタグ文字列に対して、イベントカラムの実行に必要なspanタグを付加した
	 * 文字列を返します。
	 * 
	 * @param tag タグ文字列
	 * @param initVal 子カラムの初期値
	 * @param row 行番号
	 * @param writable タグの要素が書き込み可能かどうか
	 * @return spanタグを付加したタグ文字列
	 */
	public String getEventColumnTag( final String tag, final String initVal, final int row, final boolean writable ) {
		TagBuffer span = new TagBuffer( "span" );
		span.add( "class", HybsSystem.EVENT_COLUMN_CLASS );
		span.add( HybsSystem.EVENT_COLUMN_ID,  columnData.getName() + ( row < 0 ? "" : HybsSystem.JOINT_STRING + String.valueOf( row ) ) );
		span.add( HybsSystem.EVENT_COLUMN_INITVAL, initVal );
		span.add( HybsSystem.EVENT_COLUMN_WRITABLE, String.valueOf( writable ) );
		span.setBody( tag );
		return span.makeTag();
	}

	/**
	 * セットされている表示パラメータ、編集パラメータに"{&#64;XXXX}"が含まれているか(パラメーターのパースが必要か)を
	 * 返します。
	 * 
	 * @return "{&#64;XXXX}"が含まれているか(含まれている場合true)
	 */
	public boolean isNeedsParamParse() {
		return	( columnData.getRendererParam() != null && columnData.getRendererParam().indexOf( "{@" ) >= 0 )
			||	( columnData.getEditorParam() != null && columnData.getEditorParam().indexOf( "{@" ) >= 0 );
	}
}
