/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * 指定された value 値の{&#064;XXXX} 変数が設定された場合のみ表示するタグです。
 *
 * value 値に、{&#064;XXXX} 変数を使用して、リクエスト値が設定された場合のみ
 * その値を表示します。これは、{&#064;XXXX} 変数と固定値を組み合わせた場合でも
 * 同様に、値が設定されていない場合は、使用されません。
 * defaultVal が設定されており、リクエスト値が設定されていない場合、defaultVal 値が
 * 使用されます。
 * このタグが使用されるケースの代表は、SQLでの order by 句です。一般のリクエスト時には、
 * order by 句をリクエストで設定して、ユーザーが検索する時に、切り替えができるように
 * します。別画面から、リンク等で画面を呼び出す場合は、通常 order by 句の条件まで、
 * 指定しません。そのような場合に備えて、og:appear タグでdefaultVal 値を設定しておき、
 * 未指定時の検索順を予め指定しておきます。
 *
 * @og.formSample
 * ●形式：&lt;og:appear startKey="[order by|…]" value="…" defaultVal="[…]" /&gt;
 * ●body：なし
 *
 * ●使用例
 *     &lt;og:query command="{&#064;command}" &gt;
 *         &lt;jsp:text&gt;
 *             SELECT  YOBI,NMEN,NMJP,SIYO1,SIYO2,SIYO3,SIYO4,SIYO5,SIYO6
 *             FROM    RK18
 *             WHERE   FGJ IN ('0','1')
 *         &lt;/jsp:text&gt;
 *         &lt;og:where startKey="AND"&gt;
 *             &lt;og:and value="YOBI LIKE '{&#064;YOBI}%'" /&gt;
 *             &lt;og:and value="NMEN LIKE '{&#064;NMEN}%'" /&gt;
 *             &lt;og:and value="NMJP LIKE '{&#064;NMJP}%'" /&gt;
 *         &lt;/og:where&gt;
 *         &lt;og:appear startKey="order by" value="{&#064;ORDER_BY}" defaultVal="YOBI" /&gt;
 *     &lt;/og:query&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class AppearTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private String   startKey    = "";
	private String   value       = null;
	private String   defaultVal  = null;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		String output = getRequestParameter( value );
		if( isNull() ) {
			output = defaultVal;
		}

		if( output != null ) {
			jspPrint( startKey + " " + output );
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 *
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		startKey    = "";
		value       = null;
		defaultVal  = null;
	}

	/**
	 * 【TAG】固定値の開始文字列を設定します(初期値="")。
	 *
	 * @og.tag
	 * このキーは、バリューと接続される場合に空白文字を一つ挿入します。
	 * <del>この値は,固定値のみ設定可能です。{&#064;xxx}文字は使用できません。</del>
	 *
	 * @param	val String
	 */
	public void setStartKey( final String val ) {
		startKey = nval( getRequestParameter( val ),startKey );
	}

	/**
	 * 【TAG】値をセットします(指定された値が 設定されている場合のみ表示されます)。
	 *
	 * @og.tag
	 * 指定された値が 設定されている場合のみ表示されます。
	 * これは、一般にvalue値が変動する場合に、defaultVal 等に重複する値を
	 * 設定したくない場合に使用します。{&#064;xxx}文字が使用できます。
	 *
	 * @param	val String
	 */
	public void setValue( final String val ) {
		value = val;
	}

	/**
	 * 【TAG】初期値を設定します(value値がNULLの場合に、この初期値が表示)。
	 *
	 * @og.tag
	 * value値がNULLの場合に、この初期値が表示されます。
	 *
	 * @param	val String
	 */
	public void setDefaultVal( final String val ) {
		defaultVal = nval( getRequestParameter( val ),defaultVal );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "startKey"	,startKey	)
				.println( "value"		,value		)
				.println( "defaultVal"	,defaultVal	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
