/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.tagext.BodyContent;
import javax.servlet.jsp.tagext.BodyTagSupport;
import javax.servlet.jsp.tagext.TryCatchFinally;

import org.opengion.fukurou.db.DBFunctionName;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBEventColumn;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.LabelInterface;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.UserInfo;

/**
 * TagSupport から継承されたサブクラスです。
 *
 * 汎用属性 のsetterメソッドと、Attributes オブジェクトを持っています。
 * それ以外に、{&#064;XXXX} 変数の対応と、lang属性のメソッドも用意しています。
 *
 * ロケールは、ユーザー情報の lang 属性をデフォルトで使用し、
 * セットされていない場合は、リクエスト情報のロケールから取得します。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に
 * 割り当てます。つまり、このXXXXをキーにリクエストすれば、
 * この変数に値をセットすることができます。
 *
 * http://localhost/query.jsp?KEY1=VLA1&amp;KEY2=VAL2
 *
 * のようなリクエストで、{&#064;KEY1} とすれば、 VAL1 がセットされます。
 *
 * このタグは、ラベル部分と入力フィールド部分がテーブルタグの&lt;td&gt;
 * により左右に分割されます。HTML 表示時は、前後に&lt;tr&gt;タグで囲って,
 * 整形できます。
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
class CommonTagSupport extends BodyTagSupport implements TryCatchFinally {
	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private transient Attributes		attri		= new Attributes();
	private transient ResourceManager	resource	= null;
	private transient UserInfo			userInfo	= null;
	private transient GUIInfo			guiInfo 	= null;
	private transient HttpSession		session 	= null;
	private transient ServletRequest	request 	= null;
	private transient Map<String,String[]>	requestCache = null;		// 3.5.6.2 (2004/07/05)
	private transient LabelInterface	msglbl      = null;			// 4.0.0 (2005/01/31)
	private String				language	= null;
	private boolean				debugFlag	= false;		// 3.5.5.3 (2004/04/09)
	private boolean				isReqNull	= false;
	private boolean				quotCheck	= false;		// 4.0.0 (2005/08/31)
	private String				scope		= "session";	// "request","page","session","applicaton"
	// 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	private boolean             useValue    = true;
	private Long				startTransaction = null;	// 3.6.0.8 (2004/11/19)
	private	int[]				rowNo		= null;			// 4.0.0 (2005/01/31)
	private Calendar			rightNow	= null;				// 3.8.0.2 (2005/07/11)
	private boolean				xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" ); // 5.0.0.2 (2009/09/15)
	private boolean				useMainTrans	= false;		// 5.1.6.0 (2010/05/01)

	private static final Map<String,String>	DATE_FORMAT = new HashMap<String,String>();	// 3.8.0.2 (2005/07/11)
	static {
		DATE_FORMAT.put( "YMD"    ,"yyyyMMdd"             );
		DATE_FORMAT.put( "Y2MD"   ,"yyMMdd"               );
		DATE_FORMAT.put( "YM"     ,"yyyyMM"               );
		DATE_FORMAT.put( "HMS"    ,"HHmmss"               );
		DATE_FORMAT.put( "YMDHMS" ,"yyyyMMddHHmmss"       );
		DATE_FORMAT.put( "YMDF"   ,"yyyy/MM/dd"           );
		DATE_FORMAT.put( "Y2MDF"  ,"yy/MM/dd"             );
		DATE_FORMAT.put( "YMF"    ,"yyyy/MM"              );
		DATE_FORMAT.put( "HMSF"   ,"HH:mm:ss"             );
		DATE_FORMAT.put( "YMDHMSF","yyyy/MM/dd/ HH:mm:ss" );
		DATE_FORMAT.put( "EEE"    ,"EEE"                  );
	}

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		return(SKIP_BODY);				// Body を評価しない
	//	return( EVAL_BODY_INCLUDE );	// Body インクルード（ extends TagSupport 時）
	//	return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doAfterBody() {
		return(SKIP_BODY);				// Body を評価しない
	//	return( EVAL_BODY_AGAIN );		// ボディーを再評価（ extends TagSupport 時）
	//	return( EVAL_BODY_BUFFERED );	// ボディーを再評価（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		return(EVAL_PAGE);		// ページの残りを評価する。
	//	return(SKIP_PAGE);		// ページの残りの処理を行わない。
	}

	/**
	 * タグの処理中（セッターメソッドを除く）の例外を全て受け取ります。
	 *
	 * タグの中のボディ部の評価中、または Tag.doStartTag(), Tag.doEndTag(),
	 * IterationTag.doAfterBody(), BodyTag.doInitBody() のいずれもの
	 * メソッドの中で、Throwableが投げられたときに呼び出されます。
	 *
	 * このメソッドはセッターメソッドの中でThrowableが起きた場合は呼び出されません。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) TryCatchFinally インターフェースを適用。
	 *
	 * @param  th Throwable このタグを通過してきたThrowableな例外
	 */
	public void doCatch(final Throwable th) throws Throwable {
		throw th;
	}

	/**
	 * タグの処理毎の、doEndTag()の後で呼び出されます。
	 *
	 * Tag,IterationTag,BodyTagを実装した全てのクラスの doEndTag()の
	 * 後で呼び出されます。 このメソッドはタグのボディ部や Tag.doStartTag(),
	 * Tag.doEndTag(), IterationTag.doAfterBody() ,BodyTag.doInitBody()の
	 * すべてのメソッドで例外が発生した後でも呼び出されます。
	 *
	 * このメソッドはセッターメソッドの中でThrowableが起きた場合は呼び出されません。
	 *
	 * このメソッドからは例外を投げるべきではありません。
	 * このメソッドは呼び出し毎のデータの整合性をとることとリソース管理の
	 * 動作をさせることを意図しています。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) TryCatchFinally インターフェースを適用。
	 *
	 */
	public void doFinally() {
		release2();
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。HybsRequestWrapper 廃止。直接 Mapでキャッシュする。
	 * @og.rev 3.1.3.0 (2003/04/10) エンコード情報の取得を廃止する。
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	 * @og.rev 3.5.5.3 (2004/04/09) debugFlag を、String ではなく、boolean 型に変更
	 * @og.rev 3.6.0.8 (2004/11/19) startTransaction 属性を追加
	 * @og.rev 3.8.0.2 (2005/07/11) rightNow 属性を追加
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSQL周りの実装見直し
	 *
	 */
	protected void release2() {
		language	= null;
		attri		= new Attributes();
		resource	= null;
		debugFlag	= false;		// 3.5.5.3 (2004/04/09)
		userInfo	= null;
		guiInfo 	= null;
		session 	= null;
		request 	= null;
		isReqNull	= false;
		scope		= "session";	// "request","page","session","applicaton"
		requestCache = null;
		useValue	= true;
		startTransaction = null;	// 3.6.0.8 (2004/11/19)
		rowNo		= null;			// 4.0.0 (2005/01/31)
		msglbl		= null;			// 4.0.0 (2005/01/31)
		rightNow	= null;			// 3.8.0.2 (2005/07/11)
		quotCheck	= false;		// 4.0.0 (2005/08/31)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" ); // 5.0.0.2 (2009/09/15)
		useMainTrans = false;			 // 5.1.6.0 (2010/05/01)
	}

	/**
	 * 【TAG】タグ内部で使用する言語コードを指定します。
	 *
	 * @og.tag
	 * この言語コードに基づいて、表示のラベルをリソースから作成します。
	 *
	 * @param   lang 言語コード
	 */
	public void setLanguage( final String lang ) {
		language = getRequestParameter( lang );
	}

	/**
	 * 言語コード を取得します。
	 *
	 * 言語コードが、セットされた場合は，そちらを、セットされていない
	 * 場合は，session 登録項目の HybsSystem#LANG_KEY をデフォルトで返します。
	 *
	 * @og.rev 2.1.1.0 (2002/11/08) セッション情報から取得できない場合に、クライアントの
	 *                      リクエスト情報のロケールから取得する処理を追加
	 * @og.rev 2.2.0.0 (2002/12/17) セッション情報から取得するのではなく、ユーザー情報より
	 *                      取得するように変更。そこにない場合は、リクエスト情報の
	 *                      ロケールから取得する
	 *
	 * @return   言語コード
	 */
	protected String getLanguage() {
		if( language == null ) {
			language = getUser().getLang();
			if( language == null ) {
				language = getRequest().getLocale().getLanguage();
			}
		}

		if( language != null ) {
			return language ;
		}

		String errMsg = "言語コードがセットされていません。" ;
		throw new HybsSystemException( errMsg );
	}

	/**
	 * 【TAG】ラベルリソースのラベルIDを指定します。
	 *
	 * @og.tag
	 * ラベルを変更するときに、lbl属性を使います。
	 *
	 * ラベルID は、所定の language に基づく ResourceManager の
	 * getLabel( id ) を呼び出し、その結果を msglbl に登録します。
	 * getMsglbl() で取り出せます。
	 * ラベルIDとメッセージIDは同時には登録できません。
	 *
	 * @og.rev 4.0.0 (2005/01/31) label 変数は、生データを保管するように変更。
	 *
	 * @param   lbl ラベルID
	 */
	public void setLbl( final String lbl ) {
		if( msglbl != null ) {
			String errMsg = "メッセージIDとラベルIDは同時には登録できません。: " + HybsSystem.CR
						+ "msg=[" + msglbl + "] lbl=[" + lbl + "]";
			throw new HybsSystemException( errMsg );
		}
		msglbl = (LabelInterface)getResource().getLabelData( getRequestParameter( lbl ) ) ;		// 4.0.0 (2005/01/31)
	}

	/**
	 * 【TAG】メッセージリソースのメッセージIDを指定します。
	 *
	 * @og.tag
	 * メッセージID は、所定の language に基づく ResourceManager の
	 * getMessage( id ) を呼び出し、その結果を msglbl に登録します。
	 * getMsglbl() で取り出せます。
	 * ラベルIDとメッセージIDは同時には登録できません。
	 *
	 * @og.rev 4.0.0 (2007/10/17) メッセージリソース統合に伴い、廃止
	 *
	 * @param   msg メッセージID
	 */
//	public void setMsg( final String msg ) {
//		setLbl( msg );
//		if( msglbl != null ) {
//			String errMsg = "メッセージIDとラベルIDは同時には登録できません。: " + HybsSystem.CR
//						+ "msg=[" + msg + "] lbl=[" + msglbl + "]";
//			throw new HybsSystemException( errMsg );
//		}
//		msglbl = (LabelInterface)getResource().getMessageData( getRequestParameter( msg ) ) ;		// 4.0.0 (2005/01/31)
//	}

	/**
	 * 【TAG】デバッグ情報を 出力するかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag デバッグ情報を 出力する／しないを指定します
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) debugFlag を、String ではなく、boolean 型に変更
	 *
	 * @param   flag 出力する ("true")／しない (それ以外)
	 */
	public void setDebug( final String flag ) {
		debugFlag = nval( getRequestParameter( flag ),debugFlag );
	}

	/**
	 * 【TAG】キャッシュする場合のスコープ(request,page,session,applicaton)を指定します(初期値:session)。
	 *
	 * @og.tag
	 * "request","page","session","applicaton" が指定できます。
	 *
	 * @param  scp スコープ
	 */
	public void setScope( final String scp ) {
		scope = nval( getRequestParameter( scp ),scope );
	}

	/**
	 * 【TAG】キャッシュする場合のスコープ(request,page,session,applicaton)を返します。
	 *
	 * @og.tag
	 * "request","page","session","applicaton" があります。
	 *
	 * @og.rev 3.5.5.8 (2004/05/20) 新規追加
	 *
	 * @return   スコープ
	 */
	public String getScope() {
		return scope ;
	}

	/**
	 * (通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 *
	 * @og.tag
	 * (通常使いません)タグで処理される処理が、メインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 * 
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 * 
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクションかどうか
	 */
	protected void useMainTrans( final boolean flag ) {
		useMainTrans = flag;
	}

	/**
	 * メッセージラベル（msglbl)を取得します。
	 *
	 * メッセージラベルは、meg属性か、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 * meg属性 と lbl属性は、同時登録できません。
	 *
	 * @og.rev 4.0.0 (2005/01/31) msglbl は、LabelInterface オブジェクトを利用
	 *
	 * @return   メッセージラベル
	 */
	protected String getMsglbl() {
		String rtn = null;

		if( msglbl != null ) { rtn = msglbl.getLabel(); }

		return rtn ;
	}

	/**
	 * メッセージラベル（msglbl)のチップス表記を取得します。
	 *
	 * メッセージラベルは、meg属性か、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 * 別途、title属性を指定している場合は、置き換えます。
	 * meg属性 と lbl属性は、同時登録できません。
	 *
	 * @og.rev 4.0.0 (2005/01/31) msglbl は、LabelInterface オブジェクトを利用
	 *
	 * @return   メッセージラベル
	 */
	protected String getLongLabel() {
		String rtn = null;

		if( msglbl != null ) {
			rtn = msglbl.getLongLabel( get( "title" ) );
		}

		return rtn ;
	}

	/**
	 * メッセージラベル（msglbl)を取得します。
	 *
	 * メッセージラベルは、meg属性か、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 * meg属性 と lbl属性は、同時登録できません。
	 *
	 * @return   メッセージラベル
	 */
	protected LabelInterface getLabelInterface() {
		return msglbl ;
	}

	/**
	 * メッセージリソース情報を処理します。
	 *
	 * これは、{&#064;MSG.XXXX AA &#064;BB #CC} 引数処理をおこなうための、サポートメソッドです。
	 * 引数は、"XXXX AA &#064;BB #CC" という状態で受け取ります。(MSG. ははずした形)
	 * "AA @BB #CC" は、スペースで区切られたメッセージリソースの引数文字です。
	 * この、引数文字には、通常文字(AA)、リクエスト文字(@BB)、ラベル文字(#CC)が指定できます。
	 * ・通常文字(AA)：そのまま、メッセージリソースの引数にセットされます。
	 * ・リクエスト文字(&#064;BB)：リクエスト引数に置き換えます。通常、{&#064;BB} 指定しますが、
	 *   メッセージパラメータの引数、{&#064;XXXX ･･･ {&#064;BB} ･･･} と、入れ子で指定できないため、
	 *   {&#064;XXXX ･･･ &#064;BB ･･･} と、省略形で指定します。よって、引数として、通常文字の
	 *   先頭に、"&#064;" を使用することは出来ません。
	 * ・ラベル文字(#CC)：ラベルキーを指定されたと解釈して、処理します。これは、"CC"を
	 *   キーに、ラベルリソースを検索し、その言語に応じた文字に変換後、メッセージリソースの
	 *   引数に指定します。PL/SQL等で、SET_ERRMSGS プロシージャの引数に、{#CC} 文字を
	 *   指定して、ラベルリソースを利用できますが、{&#064;XXXX ･･･ {#CC} ･･･} と、入れ子で
	 *   指定できないため、{&#064;XXXX ･･･ #CC ･･･} と、省略形で指定します。
	 *   よって、引数として、通常文字の先頭に、"#" を使用することは出来ません。(エスケープ未採用)
	 * メッセージキーそのものをパラメータ化することが出来ます。
	 * これは、{&#064;MSG.&#064;XXXX}という形式になります。引数は、先の説明と同じです。
	 * この場合は、XXXX をキーにリクエスト引数の値が、メッセージリソースのキーになります。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.0.0 (2007/10/17) メッセージリソース統合に伴い、廃止
	 *
	 * @param   msg String パラメータ
	 * @return   メッセージ情報
	 */
//	protected String getMessage( final String msg ) {
//		String   key     = msg;
//		String[] msgVals = null;
//		int spc = msg.indexOf( ' ' );		// スペースがあるかどうか
//		if( spc > 0 ) {
//			key     = msg.substring( 0,spc );
//			msgVals = StringUtil.csv2Array( msg.substring( spc+1 ),' ' );
//			for( int i=0; i<msgVals.length; i++ ) {
//				// リクエスト文字パラメータ時の処理。その他は、ラベル文字は処理不要。
//				if( msgVals[i].startsWith( "@" ) ) {
//					msgVals[i] = getRequestValue( msgVals[i].substring( 1 ) );
//				}
//			}
//		}
//		if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }
//
//		return getResource().getMessage( key,msgVals );
//	}

	/**
	 * メッセージラベル（msglbl)を設定します。
	 *
	 * メッセージラベルは、meg属性か、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 * meg属性 と lbl属性は、同時登録できません。
	 * ここで登録された値が、最優先されます。
	 * もちろん、protected メソッドなので、TagLib属性での登録はできません。
	 *
	 * @param lbl String メッセージラベル
	 */
	protected void setMsglbl( final String lbl ) {
		if( msglbl != null ) {
			String errMsg = "メッセージIDとラベルIDは同時には登録できません。: " + HybsSystem.CR
						+ "label=[" + lbl + "] msglbl=[" + msglbl + "]";
			throw new HybsSystemException( errMsg );
		}
		msglbl = (LabelInterface)getResource().getLabelData( getRequestParameter( lbl ) ) ;		// 4.0.0 (2005/01/31)
	}

	/**
	 * ResourceManager を取得します。
	 *
	 * ページスコープ にセットされた ResourceManager があれば、それを、
	 * なければ、language 属性よりデフォルト ResourceManager を構築します。
	 * LOCAL_RES_KEY で管理される ResourceManager は、LocalResourceTag で
	 * 登録されたリソースです。これは、ローカルリソース情報として、使用されます。
	 *
	 * @return   ResourceManager オブジェクト
	 */
	protected ResourceManager getResource() {
		if( resource == null ) {
			resource = (ResourceManager)pageContext.getAttribute( HybsSystem.LOCAL_RES_KEY );
			if( resource == null ) {
				resource = ResourceFactory.newInstance( getLanguage() );
			}
		}
		return resource;
	}

	/**
	 * デバッグ状態 を取得します。
	 *
	 * setDebug( String )で登録します。
	 * 初期値は、false です。
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) getDebug() から、メソッド名変更
	 *
	 * @return   true(デバッグ状態）/ false（通常）
	 */
	protected boolean isDebug() {
		return debugFlag ;
	}

	/**
	 * ラベル文字列を返します。
	 *
	 * これは、{&#064;LBL.XXXX Y} 引数処理をおこなうための、サポートメソッドです。
	 * 引数は、"XXXX Y" という状態で受け取ります。(LBL. ははずした形)
	 * ラベルには、通常のラベル(Label)以外に、Short,Tips,Description の情報を持っています。
	 * {&#064;LBL.XXXX Y} の Y に、先のLabel,Short,Tips,Description の頭文字('L','S','T','D')を
	 * 指定することで、それぞれの状態を取得することが可能になります。
	 * Y を指定しない場合({&#064;LBL.XXXX}) は、'L' が指定された場合と同じ効果があります。
	 * Y は、先頭文字１文字で判断していますので、{&#064;LBL.XXXX Short}と記述できます。
	 * Y 引数には、&#064;で始まるリクエスト引数が指定できます。例えば、{&#064;LBL.XXXX &#064;ZZ}
	 * とすると、ZZ のリクエスト引数の値が Y に適用されます。
	 * ラベルキーそのものをパラメータ化することが出来ます。
	 * これは、{&#064;LBL.&#064;XXXX}という形式になります。引数は、先の説明と同じです。
	 * この場合は、XXXX をキーにリクエスト引数の値が、ラベルリソースのキーになります。
	 *
	 * @og.rev 4.0.0 (2007/10/17) メッセージリソース統合に伴い、{&#064;LBL.XXXX Y}⇒{&#064;LBL.XXXX %Y}
	 *
	 * @param    lbl ラベルのキー
	 * @return   ラベル文字列
	 */
	protected String getLabel( final String lbl ) {

		String key = lbl ;
		String val = null;

		int spc = lbl.indexOf( ' ' );		// " " があるかどうか
		if( spc > 0 ) {
			key = lbl.substring( 0,spc );
			if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }

			// リクエスト引数が指定された場合
			char ch  = lbl.length() > spc+1 ? lbl.toUpperCase( Locale.JAPAN ).charAt( spc+1 ) : ' ';	// Label,Short,Tips,Description
			char ch2 = lbl.length() > spc+2 ? lbl.toUpperCase( Locale.JAPAN ).charAt( spc+2 ) : ' ';	// Label,Short,Tips,Description
			if( ch == '@' ) {
				String tmp = getRequestValue( lbl.substring( spc+2 ) );
				if( tmp != null && tmp.length() > 0 ) {
					ch  = tmp.toUpperCase( Locale.JAPAN ).charAt( 0 );
					ch2 = tmp.length() > 1 ? tmp.toUpperCase( Locale.JAPAN ).charAt( 1 ) : ' ';
				}
			}
			// 4.0.0.0 (2007/10/19)
			if( ch == '%' ) {
				switch( ch2 ) {
					case 'L': val = getResource().getLabel( key ); break;
					case 'S': val = getResource().getLabelData( key ).getShortLabel(); break;
					case 'T': val = getResource().getLabelData( key ).getLongLabel(); break;
					case 'D': val = getResource().getLabelData( key ).getDescription(); break;
					default : break;
				}
			}
			else if( ch != ' ' ) {
				String[] msgVals = StringUtil.csv2Array( lbl.substring( spc+1 ),' ' );
				for( int i=0; i<msgVals.length; i++ ) {
					// リクエスト文字パラメータ時の処理。その他は、ラベル文字は処理不要。
					if( msgVals[i].startsWith( "@" ) ) {
						msgVals[i] = getRequestValue( msgVals[i].substring( 1 ) );
					}
				}
				val = getResource().getLabel( key,msgVals );
			}
		}
		else {
			if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }
		}

		if( val == null ) { val = getResource().getLabel( key ); }
		return val;
	}

	/**
	 * DBColumn オブジェクトを返します。
	 *
	 * これは、キーを元に DBColumnオブジェクトをカラムリソースの
	 * 定義ファイルより取得して、リソースマネージャで管理します。
	 *
	 * @param    key DBColumn オブジェクトのキー
	 * @return   DBColumnオブジェクト
	 */
	protected DBColumn getDBColumn( final String key ) {
		return getResource().makeDBColumn( key ) ;
	}

	/**
	 * 属性値セット。
	 *
	 * @param   key   String
	 * @param   value String
	 */
	protected void set( final String key, final String value ) {
		attri.set( key,value );
	}

	/**
	 * 内部の Attributes オブジェクトに、属性値を追加します。
	 *
	 * @og.rev 4.0.0 (2007/05/18) 新規追加
	 *
	 * @param   key   String
	 * @param   value String
	 */
	protected void add( final String key, final String value ) {
		attri.add( key,value );
	}

	/**
	 * 内部の Attributes オブジェクトに、属性値を追加します。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) 新規追加
	 * @og.rev 3.5.5.9 (2004/06/07) セパレータ引数付きのメソッドに変更
	 *
	 * @param   key   String
	 * @param   value String
	 * @param   sepa  String
	 */
	protected void add( final String key, final String value, final String sepa ) {
		attri.add( key,value,sepa );
	}

	/**
	 * 属性値の取得。
	 *
	 * @param    key   String
	 * @return   value
	 */
	protected String get( final String key ) {
		return attri.get( key );
	}

	/**
	 * 属性オブジェクトの取得。
	 *
	 * Attributes オブジェクトを取得します。
	 *
	 * @return   Attributes オブジェクト
	 */
	protected Attributes getAttributes() {
		return attri;
	}

	/**
	 * {&#064;XXXX} 形式の文字列から XXXX をキーとして ServletRequest から
	 * getParameter で値を取り出します。
	 *
	 * 他の文字列に混在した {&#064;XXXX} 文字を変換可能です。
	 * ただし、処理の簡素化のため、上記形式以外は変換いたしません。
	 * エラー例）× { &#064;XXXX }、{&#064; XXXX }、{&#064;XXXX&#064;yyyy}、{&#064;XXXX{&#064;yyyy}}
	 * また、"{&#064;" を通常の記述で使うことは無いと考え、エスケープも用意して
	 * いません。よって、"{&#064;" のパターンが見つかって,"}" で閉じられていない
	 * 場合は,エラーとして、HybsSystemException を throw します。
	 *
	 * @og.rev 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
	 *
	 * @param   key リクエストのキー
	 * @return  リクエストの値
	 *
	 */
	protected String getRequestParameter( final String key ) {
		isReqNull = false;

		if( key == null ) { isReqNull = true; return ""; }
		int index = key.indexOf( "{@" );
		if( index < 0 ) { return key; }

		// 変数が "{@XXXX}" の場合を優先的に検索。
		// これにより多くのパターンで、StringTokenizer による
		// 文字列操作を行う必要がなくなります。
		if( index == 0 &&
			( key.indexOf( '}' ) == key.lastIndexOf( '}' ) ) &&
			( key.charAt(key.length()-1) == '}' ) ) {
				return getRequestValue( key.substring( 2,key.length()-1 ) );
		}

		// 3.8.0.4 (2005/08/08) {} の処理方法見直し。連続処理、単体処理を可能にします。
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_SMALL );
		int start = 0;
		while( index >= 0 ) {
			int end = key.indexOf( '}',index );
			if( end < 0 ) {
				String errMsg = "{@ と } との対応関係がずれています。" + HybsSystem.CR
							+ "key=[" + key + "] : index=" + index ;
				throw new HybsSystemException( errMsg );
			}

			// {@ より前方の文字列を追加
			if( index > 0 ) { rtn.append( key.substring( start,index ) ); }

			// {@XXXX} の XXXX部分を処理
			String val = getRequestValue( key.substring( index+2,end ) );
			if( val != null ) { rtn.append( val ); }

			start = end+1 ;
			index = key.indexOf( "{@",start );
		}
		rtn.append( key.substring( start ) );

		return rtn.toString();
	}

	/**
	 * {&#064;XXXX} 形式の文字列から XXXX をキーとして ServletRequest から
	 * getParameterValues で値を取り出します。
	 *
	 * これは、複数（配列）でリクエストを取り出すことが可能です。
	 * そのため､他の文字列に混在させて変換することができません。
	 * "{&#064;XXXX}" 形式 からのみの変換となります。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) キーがnull のときにnullではなく長さ０の配列を返します。
	 *
	 * @param   key リクエストのキー
	 * @return  リクエストの値
	 *
	 */
	protected String[] getRequestParameterValues( final String key ) {
		if( key == null ) { return new String[0]; }		// 3.6.0.0 (2004/09/22)
		int index = key.indexOf( "{@" );
		if( index < 0 ) { return StringUtil.csv2Array( key ); }

		if( index == 0 && ( key.charAt( key.length()-1 ) == '}' )) {
			return getRequestValues( key.substring( 2,key.length()-1 ) );
		}

		String errMsg = "引数の形式が異なります。 [" + key + "]" ;
		throw new HybsSystemException( errMsg );
	}

	/**
	 * 引数 in が、引数 check の文字列の中に存在すれば､ true を、
	 * 存在しなければ､false を返します。
	 *
	 * check は、 非null のString を、in は、null でも構いません。
	 *
	 * @param    in    チェックする文字列
	 * @param    check チェック用の基本文字列
	 * @return   存在する true ／  存在しない false
	 */
	protected boolean check( final String in, final String check ) {
		if( in == null ) { return false; }
		return check.indexOf( in ) >= 0 ;
	}

	/**
	 * 引数 in が、引数 check の文字列配列の中に存在すれば､ true を、
	 * 存在しなければ､false を返します。
	 *
	 * check は、 String配列 を、in は、null でも構いません。
	 *
	 * @og.rev 2.1.0.3 (2002/11/08) 文字列配列を引数に取るメソッドを追加
	 *
	 * @param    in    チェックする文字列
	 * @param    check チェック用の基本文字列配列
	 * @return   存在する true ／  存在しない false
	 */
	protected boolean check( final String in, final String[] check ) {
		if( in == null || check == null ) { return false; }
		for( int i=0; i<check.length; i++ ) {
			if( in.equals( check[i] ) ) { return true; }
		}
		return false ;
	}

	/**
	 * ユーザーオブジェクトが持っている内部情報を取得します。
	 *
	 * これは、UserInfo#getAttribute( String ) で取得される情報です。
	 * ユーザーパラメータとは異なります。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 getUser() を使用するように変更
	 *
	 * @param    user ユーザー情報を取り出すキー
	 * @return   ユーザー情報文字列
	 */
	protected String getUserInfo( final String user ) {
		if( user == null ) { return null; }

		String key = user ;
		if( key.charAt(0) == '@' ) {
			key = getRequestValue( key.substring( 1 ) );
		}

		return getUser().getAttribute( key );
	}

	/**
	 * ユーザー情報を設定します。
	 *
	 * 初めての場合は、session 登録項目 の HybsSystem#USERINFO_KEY キー の値で
	 * 取得します。
	 *
	 * @og.rev 2.1.1.4 (2002/11/25) ユーザー情報をセットするメソッドを追加
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 getUser() を使用するように変更
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 *
	 * @param    key ユーザー情報をセットするキー
	 * @param    value ユーザー情報文字列
	 * @param    save GE20(ユーザー定数)に情報を保存するか
	 */
//	protected void setUserInfo( final String key,final String value ) {
	protected void setUserInfo( final String key,final String value, final boolean save ) {
		if( key != null ) {
//			getUser().setAttribute( key, value );
			getUser().setAttribute( key, value, save );
		}
	}

	/**
	 * ユーザー情報オブジェクトを取得します。
	 *
	 * 初めての場合は、session 登録項目 の HybsSystem#USERINFO_KEY キー の値で
	 * 取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 新規追加
	 * @og.rev 3.6.0.0 (2004/09/17) private ⇒ protected 化します。
	 *
	 * @return   ユーザー情報オブジェクト
	 */
	protected UserInfo getUser() {
		if( userInfo == null ) {
			userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
		}
		if( userInfo == null ) {
			String errMsg = "ユーザーオブジェクトが存在しません。";
			throw new HybsSystemException( errMsg );
		}
		return userInfo ;
	}

	/**
	 * 画面情報（GUIInfo）を取得します。
	 *
	 * これは、session 登録項目 の HybsSystem#GUIMAP_KEY キー の値で
	 * 登録された MAP を取り出し、そこから取得します。
	 * 画面情報は、ログインユーザー毎に個別に持っています。
	 *
	 * @og.rev 4.0.0 (2005/01/31) GUIInfo が存在しない場合も処理を続けます。
	 *
	 * @param    gamenId 画面ID
	 * @return   画面情報（GUIInfo）
	 */
	protected GUIInfo getGUIInfo( final String gamenId ) {
		return getUser().getGUIInfo( gamenId );
	}

	/**
	 * 画面情報（GUIInfo）の属性値を取得します。
	 *
	 * これは、{&#064;GUI.XXXX ID} 引数処理をおこなうための、サポートメソッドです。
	 * 引数は、"XXXX ID" という状態で受け取ります。(GUI. ははずした形)
	 * XXXXには、画面情報（GUIInfo）の属性キーを指定します。IDが指定されない場合は、
	 * 実行中の自分自身の画面が指定されたことになります。
	 * これは、session 登録項目 の HybsSystem#GUIINFO_KEY キー の値で取得します。
	 * この値は,画面が呼び出される毎に毎回設定されており、リクエスト毎に
	 * 所得し直す必要があります。
	 *
	 * ID に、画面IDを指定した場合は、&#064; 指定によるリクエスト引数の値を適用できます。
	 * {&#064;GUI.&#064;XXXX ID} や、{&#064;GUI.XXXX &#064;ID} です。(同時指定も可能)
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) GUIInfo が存在しない場合も処理を続けます。
	 * @og.rev 4.0.0 (2004/11/30) 画面ID引数や、リクエスト引数の使用を可能にします。
	 *
	 * @param    attkey 画面情報を取り出すキー
	 * @return   画面情報文字列
	 */
	protected String getGUIInfoAttri( final String attkey ) {
		if( attkey == null ) { return null; }

		String  key = attkey ;
		final GUIInfo gui ;

		int spc = key.indexOf( ' ' );		// " " があるかどうか
		if( spc > 0 ) {
			key = attkey.substring( 0,spc );
			String id = attkey.substring( spc+1 );
			if( id.charAt(0) == '@' ) { id = getRequestValue( id.substring( 1 ) ); }

			gui = getUser().getGUIInfo( id );
		}
		else {
			if( guiInfo == null ) {
				guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			}
			gui = guiInfo;
		}
		if( gui == null ) { return "Not Found[" + attkey + "]"; }
		if( key.charAt(0) == '@' ) { key = getRequestValue( key.substring( 1 ) ); }

		return gui.getAttribute( key );
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 * 
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策
	 *
	 * @param    key キー
	 * @return   リクエスト情報の文字列
	 */
	protected String getRequestValue( final String key ) {
		return getRequestValue( key, xssCheck);
	}
	
	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.0.0.0 (2002/12/25) ValueTag追加の為、指定の scope の Attributeより取得
	 * @og.rev 3.1.0.1 (2003/03/26) Valueタグの値と、request情報の値の所得優先順位を、request が優先されるように変更。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。HybsRequestWrapper 廃止。直接 Mapでキャッシュする。
	 * @og.rev 3.1.5.0 (2003/04/22) SYS.XXXX で、システムパラメータ の値を取得できるように修正。
	 * @og.rev 3.1.7.0 (2003/05/02) リクエスト情報の取得順序を、Request、キャッシュ、Value の順に変更。
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	 * @og.rev 3.4.0.3 (2003/09/10) MEM.XXXX で、REQUEST_CACHE の値を取得できるように修正。
	 * @og.rev 3.5.4.7 (2004/02/06) getRequestCacheData を使用するように修正
	 * @og.rev 3.5.5.3 (2004/04/09) {&#064;SESSION.XXXX} で、session.getAttribute( "XXXX" ) の値を取得するように修正
	 * @og.rev 3.5.6.6 (2004/08/23) SYS.XXXX の処理を getSystemParameter( String key ) メソッドへ移動
	 * @og.rev 3.8.0.1 (2005/06/17) NVAR.XXXX で、getUnicodeEscape 変換() を行います。
	 * @og.rev 3.8.0.2 (2005/07/11) MSG.XXXX , LBL.XXXX の処理を追加
	 * @og.rev 3.8.0.2 (2005/07/11) チェックボックス対応で、重複リクエストに対応させます。
	 * @og.rev 3.8.8.8 (2007/05/11) 重複リクエスト処理の場所を移動。リクエストのみ対象とする。
	 * @og.rev 4.0.0 (2005/08/31) quotCheck によるＳＱＬインジェクション対策
	 * @og.rev 4.0.0 (2005/08/31) getSystemParameter を sys に名称変更
	 * @og.rev 4.0.0 (2007/04/02) Valueタグの値と、キャッシュでは、Valueタグの値を優先するように変更
	 * @og.rev 4.0.0.0 (2007/11/16) "."付きのパラメータのエラー処理をなくし、getRequestAttributeで取得する。
	 * @og.rev 4.3.0.0 (2008/07/04) DB.XXXX は、必ずStringオブジェクトとし、String.valueOf しない。
	 * @og.rev 4.3.6.0 (2009/04/01) メールモジュール用の予約語MAIL.XXXXの取得対応
	 * @og.rev 4.4.0.0 (2009/08/02) データロール対応(SEC.xxxの取得対応)
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策用にメソッドにフラグを追加
	 *
	 * @param    key キー
	 * @return   リクエスト情報の文字列
	 */
	protected String getRequestValue( final String key, final boolean xssCheckFlg ) {
		if( key == null ) { isReqNull = true; return ""; }		// 3.4.0.3 (2003/09/10)

		String rtn ;
		int adrs = key.indexOf( '.' );

		if( adrs > 0 ) {
			String subKey = key.substring( adrs+1 );
			if( key.startsWith( "USER." ) ) {
				rtn = getUserInfo( subKey );
			}
			else if( key.startsWith( "GUI." ) ) {
				rtn = getGUIInfoAttri( subKey );	// 4.0.0 (2005/01/31)
			}
			else if( key.startsWith( "SYS." ) ) {
				rtn = sys( subKey );		// 3.5.6.6 (2004/08/23)
			}
			else if( key.startsWith( "SESSION." ) ) {		// 3.5.5.3 (2004/04/09)
				rtn = String.valueOf( getSessionAttribute( subKey ) );
			}
			// 3.4.0.3 (2003/09/10) MEM.XXXX で、REQUEST_CACHE の値を取得できるように修正。
			else if( key.startsWith( "MEM." ) ) {
				// 3.5.4.7 (2004/02/06) getRequestCacheData を使用するように修正
				rtn = getRequestCacheData( subKey );
			}
			// 3.8.0.2 (2005/07/11) MSG.XXXX で、メッセージリソースの値を取得できるように追加。
//			else if( key.startsWith( "MSG." ) ) {
//				rtn = getMessage( subKey );
//			}
			// 3.8.0.2 (2005/07/11) LBL.XXXX で、ラベルリソースの値を取得できるように追加。
			else if( key.startsWith( "LBL." ) ) {
				rtn = getLabel( subKey );
			}
			// 3.8.0.2 (2005/07/11) DATE.XXXX で、日付関係の値を取得できるように追加。
			else if( key.startsWith( "DATE." ) ) {
				rtn = getDateFormat( subKey );
			}
			// 3.8.0.1 (2005/06/17) NVAR.XXXX で、getUnicodeEscape 変換() を行います。
			// NVAR. を取り除いた XXXX で再度、リクエスト値を取得し、それを Escape変換します。
			else if( key.startsWith( "NVAR." ) ) {
				rtn = StringUtil.getUnicodeEscape( getRequestValue( subKey ) );
			}
			// 4.3.6.0 (2009/04/01) メールモジュール用の予約語
			else if( key.startsWith( "MAIL." ) ) {
				rtn = ( String )getSessionAttribute( key );
			}
			// 4.3.7.0 (2009/06/01) DB関数名の取得
			else if( key.startsWith( "DBF." ) ) {
				rtn = getDBFunctionName( subKey );
			}
			// 4.4.0.0 (2009/08/02) データロールに基づく条件式の取得
			else if( key.startsWith( "SEC." ) ) {
				rtn = getDataCondition( subKey );
			}
			// 4.0.0 (2007/06/12) DB.XXXX は、直接取り出すように変更します。
//			else if( key.startsWith( "DB." ) ) {
			else { // 4.0.0.0 (2007/11/16)
				// DB.XXXX は、必ずStringオブジェクトとし、String.valueOf しない。
//				rtn = String.valueOf( getRequestAttribute( key ) );		// ※ 取り出しは、key で
				rtn = (String)getRequestAttribute( key );		// ※ 取り出しは、key で
			}
//			else {
//				String errMsg = "想定外のキーが渡されました。key=[{@" + key + "}]" ;
//				throw new HybsSystemException( errMsg );
//			}
		}
		else {
			rtn = getRequest().getParameter( key );
			
			// 5.0.0.2 (2009/09/15) tagCheck によるthan signチェック Parameterのみにかけるためこの位置
			if( rtn != null && rtn.length() > 0 && xssCheckFlg && ( rtn.indexOf( "<" ) >= 0 || rtn.indexOf( ">" ) >= 0 ) ) {
				String errMsg = "リクエスト引数に Less/Greater than sign(<,>)を含むことは出来ません。" + HybsSystem.CR
							+ " 処理が正しい場合は、xssCheck 属性を false にセットしてください。" + HybsSystem.CR
							+ " key=[" + key + "]"
							+ " val=[" + rtn + "]"
							+ " tag=[" + getTagName() + "]" ;
				throw new HybsSystemException( errMsg );
			}

			// 3.8.8.8 (2007/05/11) 重複リクエスト処理の場所を移動。リクエストのみ対象とする。
			// 3.8.0.2 (2005/07/11) チェックボックス対応で、重複リクエストに対応させます。
			// {@XXXX} で、値が"0"の場合、複数リクエストを確認して、"1"が含まれていれば、"1"とします。
			if( "0".equals(rtn) ) {
				boolean backFlag = isReqNull ;
				String[] vals = getRequestValues( key );
				if( vals != null && vals.length > 1 ) {
					for( int i=0; i<vals.length; i++ ) {
						if( "1".equals( vals[i] ) ) { rtn = "1"; break; }
					}
				}
				isReqNull = backFlag;	// 3.8.8.8 (2007/05/11) getRequestValues での NULLセット解除
			}

			// 3.1.0.1 (2003/03/26) Valueタグの値と、request情報の値の取得優先順位を、
			// request が優先されるように変更。
			if( ( rtn == null || rtn.length() == 0 ) && requestCache != null ) {
				String[] str = requestCache.get( key );
				if( str != null && str.length > 0 ) {
					rtn = str[0];
				}
			}
			if( ( rtn == null || rtn.length() == 0 ) && useValue ) {
				Object obj = pageContext.findAttribute( key );
				if( obj != null ) {
					rtn = obj.toString();
				}
			}
		}
		if( rtn == null || rtn.length() == 0 ) {
			isReqNull = true;
			rtn    = "";
		}
		// 4.0.0 (2005/08/31) quotCheck によるＳＱＬインジェクション対策
		else if( quotCheck && rtn.indexOf( "'" ) >= 0 && !key.startsWith( "SEC." ) ) {
			String errMsg = "リクエスト引数に、クォーティション(')を含むことは出来ません。" + HybsSystem.CR
						+ " 処理が正しい場合は、quotCheck 属性を false にセットしてください。" + HybsSystem.CR
						+ " key=[" + key + "]"
						+ " val=[" + rtn + "]"
						+ " tag=[" + getTagName() + "]" ;
			throw new HybsSystemException( errMsg );
		}

		// 3.8.0.2 (2005/07/11) チェックボックス対応で、重複リクエストに対応させます。
		// {@XXXX} で、値が"0"の場合、複数リクエストを確認して、"1"が含まれていれば、"1"とします。
//		if( "0".equals(rtn) ) {
//			String[] vals = getRequestValues( key );
//			if( vals != null ) {
//				for( int i=0; i<vals.length; i++ ) {
//					if( "1".equals( vals[i] ) ) { rtn = "1"; break; }
//				}
//			}
//		}

		return rtn ;
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.8.0 (2003/05/16) RequestCache データをリクエスト配列情報にも適用する。
	 *
	 * @param    key キー
	 * @return   リクエスト情報の文字列
	 */
	protected String[] getRequestValues( final String key ) {
		String[] rtn = getRequest().getParameterValues( key );

		// 3.1.8.0 (2003/05/16) RequestCache データをリクエスト配列情報にも適用する。
		if( ( rtn == null || rtn.length == 0 ) && requestCache != null ) {
			rtn =requestCache.get( key );
		}

		if( rtn == null || rtn.length == 0 ) { isReqNull = true; }
		return rtn ;
	}

	/**
	 * リクエスト情報の文字列のキー集合を取得します。
	 *
	 * @return   リクエスト情報の文字列のキー集合
	 */
	protected Enumeration<?> getParameterNames() {		// 4.3.3.6 (2008/11/15) Generics警告対応
		return getRequest().getParameterNames();
	}

	/**
	 * リクエスト情報の文字列に NULL が存在していたかどうかを取得します。
	 *
	 * これは、getRequestParameter( String ) の呼出し毎に設定されます。
	 * つまり、上記メソッドの実行直後の値を取り出す必要があります。
	 * NULL が含まれていた（true)／含まれていなかった。(false)
	 *
	 * @return  boolean NULL が含まれていた（true)／含まれていなかった。(false)
	 */
	protected boolean isNull() {
		return isReqNull;
	}

	/**
	 * リクエスト情報の文字列値取得時に、value値の使用可否を指定します。
	 *
	 * value 値は、通常、ValueTag等によりセットされますが、ValueTag自身がリクエスト
	 * 情報から値を取得する場合に、リクエスト情報が無い場合に、自分自身の値を
	 * 取得してしまい、予想しない動きをすることを避けるため、使用します。
	 * 初期値は、true(使用する）です。
	 *
	 * @og.rev 3.1.7.0 (2003/05/02) value値の使用可否を指定する、useValue 属性を追加。
	 *
	 * @param  flag value値の使用可否（true:使用化/false:使用不可）
	 */
	protected void setUseValue( final boolean flag ) {
		useValue = flag;
	}

	/**
	 * セッションに登録されているオブジェクトを取得します。
	 *
	 * @param   key キー
	 * @return   セッションに登録されているオブジェクト
	 */
	protected Object getSessionAttribute( final String key ) {
		if( session == null ) { session = pageContext.getSession(); }
		return session.getAttribute( key );
	}

	/**
	 * セッションに 指定のキーでオブジェクトをセットします。
	 *
	 * @param   key キー
	 * @param   object セッションに登録するオブジェクト
	 */
	protected void setSessionAttribute( final String key ,final Object object ) {
		if( session == null ) { session = pageContext.getSession(); }
		session.setAttribute( key,object );
	}

	/**
	 * セッションに指定のキーで登録されているオブジェクトを 削除します。
	 *
	 * @param   key キー
	 */
	protected void removeSessionAttribute( final String key ) {
		if( session == null ) { session = pageContext.getSession(); }
		session.removeAttribute( key );
	}

	/**
	 * リクエストに登録されているオブジェクトを取得します。
	 *
	 * @param   key キー
	 * @return   リクエストンに登録されているオブジェクト
	 */
	protected Object getRequestAttribute( final String key ) {
		return getRequest().getAttribute( key );
	}

	/**
	 * リクエストに 指定のキーでオブジェクトをセットします。
	 *
	 * @param   key キー
	 * @param   object リクエストに登録するオブジェクト
	 */
	protected void setRequestAttribute( final String key ,final Object object ) {
		getRequest().setAttribute( key,object );
	}

	/**
	 * リクエストに指定のキーで登録されているオブジェクトを 削除します。
	 *
	 * @param   key キー
	 */
	protected void removeRequestAttribute( final String key ) {
		getRequest().removeAttribute( key );
	}

	/**
	 * コンテキスト（applicaton）に登録されているオブジェクトを取得します。
	 *
	 * scope属性に、"applicaton" が指定された場合に、実行されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 * @return   コンテキスト（applicaton）に登録されているオブジェクト
	 */
	protected Object getContextAttribute( final String key ) {
		ServletContext applicaton = pageContext.getServletContext();
		return applicaton.getAttribute( key );
	}

	/**
	 * コンテキスト（applicaton）指定のキーでオブジェクトをセットします。
	 *
	 * scope属性に、"applicaton" が指定された場合に、実行されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 * @param   object コンテキスト（applicaton）に登録するオブジェクト
	 */
	protected void setContextAttribute( final String key ,final Object object ) {
		ServletContext applicaton = pageContext.getServletContext();
		applicaton.setAttribute( key,object );
	}

	/**
	 * コンテキスト（applicaton）指定のキーで登録されているオブジェクトを 削除します。
	 *
	 * scope属性に、"applicaton" が指定された場合に、実行されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 */
	protected void removeContextAttribute( final String key ) {
		ServletContext applicaton = pageContext.getServletContext();
		applicaton.removeAttribute( key );
	}

	/**
	 * アプリケーションサーバーのコンテキストパスのURLを返します。
	 *
	 * @return   コンテキストパス
	 */
	protected String getContextPath() {
		return ((HttpServletRequest)getRequest()).getContextPath();
	}

	/**
	 * スコープに応じて登録されているオブジェクトを取得します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 * @return   スコープに応じて登録されているオブジェクト
	 */
	protected Object getObject( final String key ) {
		if( "session".equals( scope ) ) { return getSessionAttribute( key ); }
		else if( "request".equals( scope ) ) { return getRequestAttribute( key ); }
		else if( "applicaton".equals( scope ) ) { return getContextAttribute( key ); }
		else {
			String errMsg = "このスコープはサポートされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * スコープに応じて登録されているオブジェクトを指定のキーでセットします。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 * @param   object リクエストに登録するオブジェクト
	 */
	protected void setObject( final String key ,final Object object ) {
		if( "session".equals( scope ) ) { setSessionAttribute( key,object ); }
		else if( "request".equals( scope ) ) { setRequestAttribute( key,object ); }
		else if( "applicaton".equals( scope ) ) { setContextAttribute( key,object ); }
		else {
			String errMsg = "このスコープはサポートされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * スコープに応じて登録されているオブジェクトを指定のキーで削除します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) scope="applicaton" 指定の追加
	 *
	 * @param   key キー
	 */
	protected void removeObject( final String key ) {
		if( "session".equals( scope ) ) { removeSessionAttribute( key ); }
		else if( "request".equals( scope ) ) { removeRequestAttribute( key ); }
		else if( "applicaton".equals( scope ) ) { removeContextAttribute( key ); }
		else {
			String errMsg = "このスコープはサポートされていません。[" + scope + "]";
			throw new IllegalArgumentException( errMsg );
		}
	}

	/**
	 * リクエストオブジェクトを取得します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 2.2.0.0 (2002/12/17) 文字化け対策 setCharacterEncoding が効いていないので削除
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。HybsRequestWrapper 廃止。直接 Mapでキャッシュする。
	 * @og.rev 3.4.0.3 (2003/09/10) 冗長な個所や、無意味な個所を、等価な内容のロジックに置き換える。
	 * @og.rev 3.5.5.0 (2004/03/12) command=RESET 時にも、キャッシュを取り出すように変更します。
	 *
	 * @return   リクエストオブジェクト
	 */
	@SuppressWarnings(value={"unchecked"})
	protected ServletRequest getRequest() {
		if( request == null ) {
			request = pageContext.getRequest();
			// リクエストキャッシュ機能
			String cmd =request.getParameter( "command" );
			if( "RENEW".equals( cmd ) || "RESET".equals( cmd ) ) {	// 3.5.5.0
				requestCache = (Map<String,String[]>)getSessionAttribute( HybsSystem.REQ_CACHE_KEY );
			}
		}
		return request;
	}

	/**
	 * BodyContent オブジェクトを取得して、ボディの内容を取得します。
	 *
	 * 処理としては、getRequestParameter() によるパラメータ処理も含みます。
	 * このメソッドは、必ず doAfterBody() から呼び出してください。それ以外（例えば、
	 * doEndTag()等）では、すでに Body情報が破棄/再利用されている可能性があり、
	 * 正常に動作しなくなる可能性があります。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) BodyContent オブジェクトを取得して、ボディの内容を取得する処理を追加
	 *
	 * @return   ボディ文字列
	 */
	protected String getBodyString() {
		BodyContent body = getBodyContent();
		return getRequestParameter( body.getString() );
	}

	/**
	 * BodyContent オブジェクトを取得して、ボディの内容を取得します。
	 *
	 * {&#064;xxx}を変換しない生のBODY文を返します
	 *
	 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
	 * 
	 * @return   ボディ文字列
	 */
	protected String getBodyRawString() {
		BodyContent body = getBodyContent();
		return body.getString();
	}

	/**
	 * BodyContent オブジェクトを取得して、ボディの内容を取得します。
	 *
	 * {&#064;xxx}の変換を行いますが、その変換結果に、"["が含まれる場合は、
	 * "\\]\\"に変換して、フォーマット処理されないようにサニタイズします。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * 
	 * @return   ボディ文字列
	 */
	protected String getSanitizedBodyString() {
		BodyContent body = getBodyContent();
		String rawBodyString = body.getString();
		if( rawBodyString.indexOf( "{@" ) >= 0 ) {
			SystemParameter sysParam = new SystemParameter( rawBodyString );
			String[] clms = sysParam.getColumns();
			if( clms != null ) {
				for( int i=0; i<clms.length; i++ ) {
					String key = "{@" + clms[i] + "}";
					String parsedVal = getRequestParameter( key );
					if( parsedVal != null && parsedVal.indexOf( "[" ) >= 0 ) {
						rawBodyString = rawBodyString.replace( key, parsedVal.replace( "[", "\\]\\" ) );
					}
				}
			}
		}
		return getRequestParameter( rawBodyString );
	}

	/**
	 * JspWriter を使用した画面出力です。
	 *
	 * @param msg 画面に出力する文字列
	 */
	protected void jspPrint( final String msg ) {
		if( msg == null ) { return ; }
		try {
			JspWriter out = pageContext.getOut();
			out.print( msg );
		} catch(IOException ex) {
			String errMsg = "画面出力時の PageContext の取得時にエラーが発生しました。";
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}

	/**
	 * デバッグ用の文字列を JspWriter を使用して画面に出力します。
	 * このメソッドは、debugFlag=true の場合のみ動作します。
	 *
	 * 出力内容は，各オブジェクトの toString() 文字列です。
	 *
	 * @og.rev 4.0.0 (2005/02/28) debugFlag の条件式を追加。
	 * @og.rev 4.0.0 (2005/02/28) 簡易リファレンスへのリンクを追加。
	 */
	protected void debugPrint() {
		if( debugFlag ) {
			try {
				JspWriter out = pageContext.getOut();
				out.println( getDocumentLink() );		// 4.0.0 (2005/02/28)
				out.println( "<pre>" );
				out.println( toString() );
				out.println( "</pre>" );
			} catch(IOException ex) {
			String errMsg = "デバッグ画面出力時の PageContext の取得時にエラーが発生しました。";
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
		}
	}

	/**
	 * GAMENID付のリクエストキャッシュ情報を取り出します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 *
	 * @param key リクエストキャッシュのキー情報
	 * @return リクエスト情報（存在しない場合は、null)
	 */
	protected String getRequestCacheData( final String key ) {
		String rtn = null;

		String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
//		Map mem = (Map)getSessionAttribute( memKey );
		Map<?,?> mem = (Map<?,?>)getSessionAttribute( memKey );		// 4.3.3.6 (2008/11/15) Generics警告対応

		if( mem != null ) {
			String[] vals = (String[])mem.get( key );
			if( vals != null && vals.length > 0 ) {
				rtn = vals[0];
			}
		}
		return rtn ;
	}

	/**
	 * GAMENID付のリクエストキャッシュ情報を取り出します。
	 *
	 * @og.rev 3.5.4.7 (2004/02/06) 新規作成
	 *
	 * @param key  リクエストキャッシュのキー情報
	 * @param value リクエストキャッシュに登録する値
	 */
	@SuppressWarnings(value={"unchecked"})
	protected void setRequestCacheData( final String key,final String value ) {
		String[] vals = new String[] { value } ;

		String memKey = HybsSystem.REQ_CACHE_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		Map<String,String[]> mem = (Map<String,String[]>)getSessionAttribute( memKey );
		if( mem != null ) {
			mem.put( key,vals );
		}
	}

	/**
	 * カンマ区切り引数(CSV引数)を配列に分解して返します。
	 *
	 * カンマ区切り引数(CSV引数)で複数指定されたリクエストパラメータを
	 * 文字列配列に分解して、返します。
	 * 引数は、{&#064;XXXX} 変数も使用できます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
	 *
	 * @param	csvKey カンマ区切り引数(CSV引数)
	 * @return	配列に分解されたリクエストパラメータ値
	 */
	protected String[] getCSVParameter( final String csvKey ) {
		String[] keys = StringUtil.csv2Array( csvKey );
		String[] vals = new String[keys.length];
		for( int i=0; i<keys.length; i++ ) {
			vals[i] = getRequestParameter( keys[i] ) ;
		}
		return vals ;
	}

	/**
	 * リクエスト変数 {&#064;SYS.XXXX} に対する値の取得を行います。
	 *
	 * 本来は、システムパラメータ の値を取得できますが、
	 * システム的な共有値も取得できるように機能追加しています。
	 * また、ユーザー個別にシステムパラメータを変更できます。この取得は、
	 * システムパラメータとして取得します。(値はユーザー個別値を返します。)
	 * ここで、引数に、&#064;変数が使用できます。具体的には、{&#064;SYS.&#064;XXXX}
	 * で、&#064;XXXX をリクエスト変数から取得した値を利用します。
	 * この中で、&#064;GUIID だけが、さらに特殊で、実行中の画面IDを割り当てます。
	 * この &#064;GUIID は、ここまでの文字列を画面IDに置き換えるとともに、それ以降の
	 * 文字列を、画面IDに連結させます。
	 * {&#064;SYS.&#064;GUIID_XXXX} ⇒ 画面ID_XXXX 文字列で、システムパラメータ の値を取得します。
	 *
	 *	SERVER_NAME     このTomcatが実行しているサーバー名             localhost 等
	 *	SERVER_URL      Portも含むURLアドレス                          http://localhost:8823/
	 *	CONTEXT_URL     実行しているコンテキストのURLアドレス          http://localhost:8823/dbdef2/
	 *	REAL_PATH       / ルートに対応する物理ディレクトリ             d:/webapps/dbdef2/ 等
	 *	CONTEXT_NAME    コンテキスト名(webapps 直下の仮想フォルダ名)   dbdef 等
	 *	DATE            YMDH とほぼ同じですが、'yyyy/MM/dd HH:mm:ss' の形式で取得できます。
	 *	任意            ユーザーパラメータ(GE16) の値/システムパラメータ(GE12)の値を取得
	 *	  &#064;GUIID_XXXX  既存の画面IDに、_XXXX を追加した文字列
	 *	  &#064;XXXX        XXXX でリクエスト変数から取得した文字列
	 *	  XXXX              XXXX の文字列
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 新規作成
	 * @og.rev 3.7.0.3 (2005/03/01) クッキー取得機能を追加
	 * @og.rev 4.0.0.0 (2005/11/30) ユーザーパラメータは、システムパラメータとして取得します。
	 * @og.rev 5.1.6.0 (2010/05/01) システムパラメータに、&#064;GUIID という特殊パラメータが使用できるように対応します。
	 *
	 * @param    key String {&#064;SYS.XXXX} の XXXX 部分の文字列（キー）
	 * @return   キーに対する値。なければ、null
	 */
	protected String sys( final String key ) {
		final String rtn ;

		if( key.startsWith( "COOKIE." ) ) {		// 3.7.0.3 (2005/03/01)
			rtn = getCookie( key.substring( "COOKIE.".length() ) );
		}
		else if( key.startsWith( "DATE" ) ) {
			int idx = key.indexOf( ' ' );
			if( idx >= 0 ) {
				rtn = HybsSystem.getDate( key.substring( idx+1 ) );
			}
			else {
				rtn = HybsSystem.getDate();
			}
		}
		// 5.1.6.0 (2010/05/01) {@SYS.@GUIID_XXXX} パラメータ対応
		else if( key.startsWith( "@GUIID" ) ) {
			String key2 = getGUIInfoAttri( "ID" ) +  key.substring( "@GUIID".length() );
			rtn = getUser().getParameter( key2 );
		}
		// 5.1.6.0 (2010/05/01) {@SYS.@XXXX} パラメータ対応
		else if( key.startsWith( "@" ) ) {
			String key2 = getRequestValue( key.substring( 1 ) );
			rtn = getUser().getParameter( key2 );
		}
		else {
			rtn = getUser().getParameter( key );
		}

		return rtn ;
	}

	/**
	 * システムパラメータの値を、boolean 型に変換して返します。
	 *
	 * 本来は、システムパラメータ の値を取得できますが、
	 * システム的な共有値も取得できるように機能追加しています。
	 * また、ユーザー個別にシステムパラメータを変更できます。この取得は、
	 * システムパラメータとして取得します。(値はユーザー個別値を返します。)
	 *
	 * @og.rev 4.0.0 (2005/11/30) 新規追加
	 *
	 * @param	key システム設定 キー
	 * @return	システム設定値（boolean型)
	 */
	protected boolean sysBool( final String key ) {
		return Boolean.valueOf( sys( key ) ).booleanValue();
	}

	/**
	 * システムパラメータの値を、int 型に変換して返します。
	 *
	 * 本来は、システムパラメータ の値を取得できますが、
	 * システム的な共有値も取得できるように機能追加しています。
	 * また、ユーザー個別にシステムパラメータを変更できます。この取得は、
	 * システムパラメータとして取得します。(値はユーザー個別値を返します。)
	 *
	 * @og.rev 4.0.0 (2005/11/30) 新規追加
	 *
	 * @param	key システム設定 キー
	 * @return	システム設定値（int型)
	 * @throws	NumberFormatException,IllegalArgumentException
	 */
	protected int sysInt( final String key ) {
		String tmp = null;
		int rtn ;
		try {
			tmp = sys( key );
			rtn = Integer.parseInt( tmp );
		}
		catch( NumberFormatException ex ) {
			String errMsg = "システムパラメータの値が数字ではありません。" + HybsSystem.CR
					+ "  Resource key=[" + key + "] val=[" + tmp + "]"  ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		catch( IllegalArgumentException ex ) {
			String errMsg = "キーの値が null です。key=[" + key + "] val=[" + tmp + "]";
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		return rtn;
	}

	/**
	 * session に、処理開始時刻を設定します。
	 * これは、DBTableModel を登録する場合に、一連の処理が連続であるかどうかを
	 * 判断する時に使用します。
	 * 処理が一連でない（start 時のタイムスタンプが書き換えられている）場合は、
	 * DBTableModel の登録処理を行いません。
	 * なお、判断処理を行うのは、scope が session の場合のみです。
	 * 判定は、commitTableObject( String ,DBTableModel ) で行います。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) 新規追加
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @param   tableId キー
	 * @see     #commitTableObject( String ,DBTableModel )
	 */
	protected void startQueryTransaction( final String tableId ) {
		if( "session".equals( scope ) ) {
			startTransaction = Long.valueOf( System.currentTimeMillis() );
			setSessionAttribute( tableId+"_TRANSACTION", startTransaction );
		}

		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		if( useMainTrans ) {
			// 4.3.0.0 (2008/07/04) fileUD 対応
			removeSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );	// 無条件削除
		}
	}

	/**
	 * スコープに応じて登録されている DBTableModel を指定のキーでセットします。
	 * これは、startQueryTransaction( String ) でセッションに登録した処理開始時刻と、
	 * このオブジェクト自身が持っている（セッションに登録した開始時刻そのもの）を
	 * 比較し、異なる場合は、DBTableModel の登録を行いません。
	 * これにより、検索処理の開始順にしか登録しないようなロジックを入れています。
	 * 検索処理時間が掛かるSQLを実行した場合、先に検索した結果があとから登録される
	 * ケースがあるためです。
	 * また、判断処理を行うのは、scope が session の場合のみです。
	 *
	 * @og.rev 3.6.0.8 (2004/11/19) 新規追加
	 * @og.rev 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
	 * @og.rev 4.3.0.0 (2008/07/04) fileUD 対応。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @param   tableId キー
	 * @param   table   登録するDBTableModelオブジェクト
	 * @return  正常に登録(true) / 一連でないため登録していない(false)
	 * @see     #startQueryTransaction( String )
	 */
	protected boolean commitTableObject( final String tableId ,final DBTableModel table ) {
		// 登録しないケースをピックアップします。
		if( "session".equals( scope ) ) {
			String key = tableId+"_TRANSACTION";
			Long endTime = (Long)getSessionAttribute( key );
			removeSessionAttribute( key );
			if( endTime == null ||
				startTransaction == null ||
				endTime.compareTo( startTransaction ) != 0 ) {
					String msg = "CommonTagSupport Query処理が割り込まれました。DBTableModel は登録しません。"
								+ "[" + getUser().getUserID() + "],"
								+ "[" + getGUIInfoAttri( "KEY" ) + "]" 	// 4.0.0 (2005/01/31)
								+ "[" + startTransaction + "]" 				// 4.0.0 (2005/01/31)
								+ "[" + endTime + "]"; 						// 4.0.0 (2005/01/31)
					System.out.println( msg );
					return false;
			}
			// 3.8.1.1 (2005/11/21) ExcelOut の整合性を取る為の仕掛け
			if( table != null && HybsSystem.TBL_MDL_KEY.equals( tableId ) ) {
				String consisKey = table.getConsistencyKey();
				setSessionAttribute( HybsSystem.TBL_MDL_CONKEY,consisKey );
			}
		}

		// 4.3.0.0 (2008/07/04) fileUD 対応
		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		if( useMainTrans && table != null ) {
			String guikey = getGUIInfoAttri( "KEY" );
			DBLastSql lastSql = new DBLastSql( scope,guikey,table.isOverflow(),tableId );
			setSessionAttribute( HybsSystem.DB_LAST_SQL_KEY,lastSql );
		}

		setObject( tableId,table );
		return true;
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行番号の
	 * 配列を返します。
	 * 配列情報は、行番号でソートされて返されます。
	 * なにも選ばれていない場合は、サイズ０の配列を返します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * return int[] (選ばれていない場合は、サイズ０の配列を返す)
	 */
	protected int[] getParameterRows() {
		if( rowNo != null ) { return rowNo; }

		rowNo = (int[])getRequestAttribute( HybsSystem.ROW_SEL_KEY );
		if( rowNo != null ) { return rowNo; }

		String[] selected = getRequestValues( HybsSystem.ROW_SEL_KEY ) ;
		if( selected != null && selected.length != 0 ) {
			rowNo = new int[ selected.length ];
			for( int i=0; i<selected.length; i++ ) {
				if( selected[i] != null && selected[i].length() > 0 ) {
					rowNo[i] = Integer.parseInt( selected[i] );
				}
			}
			Arrays.sort( rowNo );
		}
		else {
			rowNo = new int[0];
		}

		return rowNo;
	}

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY に対して、選ばれた 行番号の
	 * 配列を設定します。
	 * ここで設定した選択配列は、getParameterRows() メソッドで取得する場合、優先されます。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @param rowNo int[]
	 */
	protected void setParameterRows( final int[] rowNo ) {
		setRequestAttribute(  HybsSystem.ROW_SEL_KEY , rowNo );
	}

	/**
	 * 指定のクッキーをセットします。
	 * これは、従来，各Taglibパッケージで使用していました、ErrorMessage オブジェクトを、
	 * HTMLテーブル形式で表示する為に、DBUtilクラスや、ViewFormクラスなど、複数のクラスを
	 * 複雑に組み合わせて使用していましたが、一つの static メソッドにまとめたものです。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規登録
	 *
	 * @param   key  String クッキーのキー
	 * @param   value String クッキーの設定値
	 * @param   maxage int 最長存続期間を秒単位で設定(負の値は Cookie を保存しない、 0 なら Cookie を削除する)
	 */
	protected void setCookie( final String key,final String value,final int maxage ) {
		HttpServletResponse res = (HttpServletResponse)pageContext.getResponse();
		Cookie ck = new Cookie( key, value );
		ck.setMaxAge( maxage );		// 有効秒
		res.addCookie( ck );
	}

	/**
	 * 指定のクッキーを取得します。
	 * 見つからない場合は、null を返します。
	 *
	 * @og.rev 3.7.0.3 (2005/03/01) 新規登録
	 *
	 * @param   key  String クッキーのキー
	 * @return  String クッキーの設定値
	 */
	protected String getCookie( final String key ) {
		HttpServletRequest req = (HttpServletRequest)pageContext.getRequest();
		Cookie[] cks = req.getCookies();

		String val = null;
		for( int i=0; i<cks.length; i++ ) {
			Cookie ck = cks[i];
			if( ck.getName().equals( key ) ) {
				val = ck.getValue();
				break;
			}
		}
		return val ;
	}

	/**
	 * リクエスト情報の クォーティション(') 存在チェックを実施するかどうか(true/false)を設定します(初期値:USE_SQL_INJECTION_CHECK)。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * 初期値は、SystemData#USE_SQL_INJECTION_CHECK です。
	 *
	 * @og.rev 4.0.0 (2005/08/31) 新規追加
	 *
	 * @param flag boolean クォーティションチェックする (true)／しない (false)
	 */
	protected void useQuotCheck( final boolean flag ) {
		quotCheck = flag;
	}

	/**
	 * リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか(true/false)を設定します(初期値:USE_XSS_CHECK)。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * 現在の実装としてはリクエストパラメータのみチェックして、attributesに対しては行いません。
	 * 初期値はSystemData#USE_XSS_CHECKで設定可能です。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 新規追加
	 *
	 * @param flag boolean XSSチェックする (true)／しない (false)
	 */
	protected void useXssCheck( final boolean flag ) {
		xssCheck = flag;
	}

	/**
	 * 日付関係の情報を簡易的に取り出す処理を行います。
	 *
	 * これは、{&#064;DATE.XXXX AA BB} 引数処理をおこなうための、サポートメソッドです。
	 * 引数は、"XXXX AA BB" という状態で受け取ります。(DATE. ははずした形)
	 * "XXXX" は、日付処理を行うキー文字列で予約語になっています。
	 * ・YMD　　：８文字の日付データ(200X年XX月XX日)を扱います。
	 * ・Y2MD　 ：６文字の日付データ(0X年XX月XX日)を扱います。
	 * ・YM　　 ：６文字の日付データ(200X年XX月)を扱います。
	 * ・HMS　　：６文字の時刻データ(XX時XX分XX秒)を扱います。
	 * ・YMDHMS ：１４文字の日付データ(200X年XX月XX日XX時XX分XX秒)を扱います。
	 * ・EEE　　：曜日をデフォルトロケールで表示します。
	 *
	 * F付きは、フォーマットされた日付を返します。
	 * ・YMDF　 ：８文字の日付データ(200X年XX月XX日)を扱います。
	 * ・Y2MDF　：６文字の日付データ(0X年XX月XX日)を扱います。
	 * ・YMF　　：６文字の日付データ(200X年XX月)を扱います。
	 * ・HMSF　 ：６文字の時刻データ(XX時XX分XX秒)を扱います。
	 * ・YMDHMSF：１４文字の日付データ(200X年XX月XX日XX時XX分XX秒)を扱います。
	 *
	 * AA 引数は、基準となる日付を、YYYYMMDD形式で指定します。無指定の場合は、
	 * 処理時刻を基準にした、同一タグ内での同一日付(年月日時分秒)を使用して値を取得します。
	 * 指定できる日付は、必ず、YYYYMMDD形式とし、時分秒は ゼロにリセットされた状態になります。
	 * AA には、数字で始まる(20050701など)実日付と@で始まるパラメータが使用できます。
	 * &#064;ZZ と記述することで、{&#064;ZZ}で指定する場合と同様のリクエストパラメータが使用できます。
	 * このパラメータの値は、YYYYMMDD形式の数字か、null(またはゼロストリング)です。nullの
	 * 場合は、なにも指定されていないと判断して、処理時刻を使用します。
	 * 第２引数は、(数字か、@)以外の場合は、省略されたと判断されます。
	 *
	 * BB 引数は、日付についての加減算処理を行います。
	 * 省略すると、なにも加減算処理を行いません。
	 * ・SD ：当月の最初の日付にセットします。(当月１日)
	 * ・ED ：当月の最後の日付にセットします。(当月月末)
	 * ・SM ：本年の最初の月にセットします。(本年１月)
	 * ・EM ：本年の最後の月にセットします。(本年１２月)
	 * ・SW ：日付処理の週初め(月曜日)にセットします。日付は当日より前に移動します。
	 * ・EW ：日付処理の週末(日曜日)にセットします。日付は当日より後ろに移動します。
	 * ・D1 ～ DXXX ：日を指定の分だけ進めます。D1なら翌日、D200 なら200日後
	 * ・M1 ～ MXXX ：月を指定の分だけ進めます。M1なら翌月、M6 なら半年後
	 *
	 * @og.rev 3.8.0.2 (2005/07/11) 新規追加
	 *
	 * @param   value String パラメータ
	 * @return   メッセージ情報
	 */
	protected String getDateFormat( final String value ) {
		if( rightNow == null ) { rightNow = Calendar.getInstance(); }
		Calendar now = (Calendar)rightNow.clone();

		String[] vals = StringUtil.csv2Array( value,' ' );

		String key = vals[0] ;

		String prmB = null;
		if( vals.length == 3 ) { prmB = vals[2]; }

		if( vals.length > 1 ) {
			String prmA = vals[1];
			if( prmA.startsWith( "@" ) ) {
				prmA = getRequestValue( prmA.substring(1) );
			}

			if( prmA != null && prmA.length() > 0 ) {
				if( prmA.length() == 8 && prmA.charAt(0) >= '0' && prmA.charAt(0) <= '9' ) {
					int yy  = Integer.parseInt( prmA.substring( 0,4 ) );
					int mm  = Integer.parseInt( prmA.substring( 4,6 ) ) - 1;
					int dd  = Integer.parseInt( prmA.substring( 6,8 ) );
					now.clear();
					now.set( yy,mm,dd );
				}
				else {
					prmB = prmA;
				}
			}
		}

		if( prmB != null ) {
			if( "SD".equals( prmB ) ) { now.set( Calendar.DATE,1 ); }
			else if( "ED".equals( prmB ) ) { now.set( Calendar.DATE,now.getActualMaximum( Calendar.DATE ) ); }
			else if( "SM".equals( prmB ) ) { now.set( Calendar.MONTH,Calendar.JANUARY  ); }
			else if( "EM".equals( prmB ) ) { now.set( Calendar.MONTH,Calendar.DECEMBER  ); }
			else if( "SW".equals( prmB ) ) {
				// 日付型文字列入力データの開始日を月曜日にセットします。
				// SUNDAY=1 , MONDAY=2 になります。月曜日との差だけ、前に戻します。
				// 指定日が日曜日の場合は、月曜日まで戻します。
				int shu = now.get( Calendar.DAY_OF_WEEK ) - Calendar.MONDAY ;

				if(      shu > 0 ) { now.add( Calendar.DATE, -shu ); }
				else if( shu < 0 ) { now.add( Calendar.DATE, -6 );   }	// 日曜日の処理

			}
			else if( "EW".equals( prmB ) ) {
				// 日付型文字列入力データの終了日を日曜日にセットします。
				// SUNDAY=1 , MONDAY=2 になります。日曜日になるように、先に進めます。
				int shu = now.get( Calendar.DAY_OF_WEEK ) ;

				if( shu != Calendar.SUNDAY ) { now.add( Calendar.DATE, 8-shu ); }
			}
			else if( prmB.startsWith( "D" ) && prmB.length() > 1 ) {
				int day = Integer.parseInt( prmB.substring( 1 ) );
				now.add( Calendar.DATE, day );
			}
			else if( prmB.startsWith( "M" ) && prmB.length() > 1 ) {
				int month = Integer.parseInt( prmB.substring( 1 ) );
				now.add( Calendar.MONTH , month );
			}
			else {
				String errMsg = "日付変数パラメータに、不正な値が指定されました。以下の中から指定しなおしてください。"
							+ "指定可能：[SD,ED,SM,EM,SW,EW,D1～DXXX,M1～MXXX]"
							+ " value=[" + prmB + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}

		String format = DATE_FORMAT.get( key );
		if( format == null ) {
			String errMsg = "日付変数キーに、不正な値が指定されました。以下の中から指定しなおしてください。"
						+ "指定可能：[YMD,Y2MD,YM,HMS,YMDHMS]"
						+ " key=[" + value + "]" ;
			throw new HybsSystemException( errMsg );
		}

		DateFormat formatter = new SimpleDateFormat( format,Locale.JAPAN );

		return formatter.format( now.getTime() );
	}

	/**
	 * debug や エラー時に参考にする、簡易リファレンスへのリンクを作成します。
	 * リンクを行うタグの名称は、getTagName() メソッドより取得します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 * @og.rev 4.2.1.0 (2008/04/11) URLを相対パスに変更
	 *
	 * @return  簡易リファレンスへのリンク
	 * @see  #getTagName()
	 */
	protected String getDocumentLink() {
		String name    = getTagName();

	//	try {
	//		Field fld = getClass().getDeclaredField( "VERSION" ) ;
	//		version = (String)fld.get( null );
	//	}
	//	catch( Exception ex ) {
	//		version = ex.toString();
	//	}

		// org.opengion.hayabusa.taglib.AbcdTag というクラス名より、abcd を取り出す。
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );
		buf.append( "<br /><a href=\"../common/documents.html#VAL_" );
		buf.append( name );
		buf.append( "\" target=\"_brank\" >簡易リファレンス[" );
		buf.append( name );
		buf.append( "] " );
	//	buf.append( version );
		buf.append( "</a>" ).append( HybsSystem.BR );

		return buf.toString();
	}

	/**
	 * タグの名称を、返します。
	 * これは、debug や エラー時に参考にする、簡易リファレンスへのリンクを作成する場合に
	 * 使用します。
	 * 通常は、org.opengion.hayabusa.taglib.AbcdTag という自分自身のクラス名より、
	 * abcd の部分を取り出し、返します。
	 * クラス名とタグ名が、上記変換ルールと異なる場合は、このメソッドを
	 * 使用して、直接 abcd の部分に相当する文字列を返すようにしてください。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @see  #getDocumentLink()
	 */
	protected String getTagName() {
		String name = getClass().getName();
		int adrs = name.lastIndexOf('.');

		// org.opengion.hayabusa.taglib.AbcdTag というクラス名より、abcd を取り出す。
		return name.substring( adrs+1,adrs+2 ).toLowerCase(Locale.JAPAN)
				+ name.substring( adrs+2,name.length()-3 ) ;
	}

	/**
	 * リクエストに対して、画面遷移なしモードを有効にします。
	 * この情報は画面IDをキーにセッションに保存されるため、
	 * 各タグで共有することができます。
	 * 
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) リクエストからセッションに変更（名称も変更)
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 * 
	 * @param flg boolean 画面遷移なしモードを使用するか true/false
	 */
	protected void setNoTransitionRequest( final boolean flg ) {
//	protected void setNoTransitionSession( final boolean flg ) {
		setRequestAttribute( HybsSystem.NO_TRANSITION_MODE_KEY, String.valueOf( flg ));
//		setSessionAttribute( HybsSystem.NO_TRANSITION_MODE_KEY +"_"+ getGUIInfoAttri( "KEY" ), String.valueOf( flg ) );
	}

	/**
	 * リクエストで画面遷移なしモードが有効になっているかを返します。
	 * この情報はセッションから画面IDをキーに取得します。
	 * セッションに情報が設定されていない(#setNoTransition()が呼ばれていない)場合は、
	 * falseを返します。
	 * 
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.8.0 (2009/08/01) リクエストからセッションに変更
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 * 
	 * @return 画面遷移なしモードが有効
	 */
	protected boolean isNoTransitionRequest() {
//	protected boolean isNoTransitionSession() {
		return StringUtil.nval( (String)getRequestAttribute( HybsSystem.NO_TRANSITION_MODE_KEY ), false );
//		return StringUtil.nval( (String)getSessionAttribute( HybsSystem.NO_TRANSITION_MODE_KEY +"_"+ getGUIInfoAttri( "KEY" ) ), false );
	}
	
	/**
	 * リクエストに対して、AjaxSubmitモードを有効にします。
	 * この情報は画面IDをキーにセッションに保存されるため、
	 * 各タグで共有することができます。
	 * 
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 * 
	 * @param flg boolean AjaxSubmitモードを使用するか true/false
	 */
	protected void setAjaxSubmitRequest( final boolean flg ) {
//	protected void setAjaxSubmitSession( final boolean flg ) {
		setRequestAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY, String.valueOf( flg ));
//		setSessionAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY +"_"+ getGUIInfoAttri( "KEY" ), String.valueOf( flg ) );
	}

	/**
	 * リクエストでAjaxSubmitモードが有効になっているかを返します。
	 * この情報はセッションから画面IDをキーに取得します。
	 * セッションに情報が設定されていない(#setNoTransition()が呼ばれていない)場合は、
	 * falseを返します。
	 * 
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition、ajaxSubmitのコントロールは、requestで行う。
	 * 
	 * @return AjaxSubmitモードが有効
	 */
	protected boolean isAjaxSubmitRequest() {
//	protected boolean isAjaxSubmitSession() {
		return StringUtil.nval( (String)getRequestAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY ), false );
//		return StringUtil.nval( (String)getSessionAttribute( HybsSystem.USE_AJAX_SUBMIT_KEY +"_"+ getGUIInfoAttri( "KEY" ) ), false );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
		attri = new Attributes();
	}

	/**
	 * アクセスログ取得の為,ApplicationInfo オブジェクトを返します。<br />
	 * 見つからない場合は、null が返ります。(暫定対応)
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @return  ApplicationInfo
	 */
	protected ApplicationInfo getApplicationInfo() {
		String gamenId = getGUIInfoAttri( "KEY" );
		String jspId   = (String)getSessionAttribute( "JSPID" );

		return getUser().getApplicationInfo( gamenId,jspId );
	}

	/**
	 * イベントカラムの実行に必要なカラム情報をマップに登録します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * 
	 * @param col DBカラム
	 */
	protected void addEventColumn( final DBColumn col ) {
		addEventColumn( col.getName(), col.getEventColumn(), col.getEventURL()
						, col.getRenderer(), col.getEditor(), col.getRawRendParam(), col.getRawEditParam() );
	}

	/**
	 * イベントカラムの実行に必要なカラム情報をマップに登録します。
	 *
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param name カラム名
	 * @param evCol イベントカラム名
	 * @param url イベントURL
	 * @param renderer カラムのレンデラー
	 * @param editor カラムのエディター
	 * @param rendParam カラムの表示パラメーター
	 * @param editParam カラムの編集パラメーター
	 */
	@SuppressWarnings(value={"unchecked"})
	protected void addEventColumn( final String name, final String evCol, final String url, final String renderer, final String editor, final String rendParam, final String editParam ) {
		if( evCol == null || evCol.length() == 0 ){ return; }

		String key = (String)getRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY );
		if( key == null || key.length() == 0 ) {
			key = HybsSystem.USE_EVENT_COLUMN_KEY + String.valueOf( System.currentTimeMillis() );
			setRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY, key );
		}

		Map<String, DBEventColumn> evColMap = (Map<String, DBEventColumn>)( getSessionAttribute( key ) );
		if( evColMap == null ){
			evColMap = new HashMap<String,DBEventColumn>();
		}
		if( evColMap.get( name ) == null ) {
			evColMap.put( name, new DBEventColumn( name, evCol, url, renderer, editor, rendParam, editParam ) );
		}
		setSessionAttribute( key, evColMap );
	}

//	/**
//	 * キーを画面ID_xxx.jspでnameとparamのマップを追加します。
//	 * 又、リクエスト変数のUSE_EVENT_COLUMN_KEYをtrueにします。
//	 * 一応複数セッションの対応としてsynchronizedする。
//	 *
//	 * @og.rev 4.3.6.0 (2009/04/01) 新規追加
//	 *
//	 */
//	@SuppressWarnings(value={"unchecked"})
//	protected void addEventColumnSQL( final String name, final String param ) {
//		if( session == null ) { session = pageContext.getSession(); }
//		synchronized( session.toString().intern() ){
//			Map<String, String> ecSql = (Map<String, String>)( getSessionAttribute( getGUIInfoAttri( "KEY" ) + "_" + (String) getSessionAttribute( "JSPID" ) ) );
//			if( ecSql == null ){ ecSql = new HashMap<String,String>(); }
//			ecSql.put( name, param );
//			setSessionAttribute( getGUIInfoAttri( "KEY" ) + "_" + (String) getSessionAttribute( "JSPID" ), ecSql );
//		}
//		setRequestAttribute( HybsSystem.USE_EVENT_COLUMN_KEY, "true" ); // 利用フラグを立てる
//	}

	/**
	 * 各データベースに対応するファンクション名を返します。
	 * 
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * @og.rev 5.1.4.0 (2010/03/01) データベース名 でなく、DBID 名で検索するようにします。
	 *
	 * @param   key ファンクション名(定義文字)
	 * @return  実ファンクション名
	 */
	protected String getDBFunctionName( final String key ) {
		int idx = key.indexOf( " " );
		String func = null;
		String dbid = null;
		if( idx >= 0 ) {
			func = key.substring( 0, idx );
			dbid = key.substring( idx+1, key.length() );
			if( dbid.charAt(0) == '@' ) { dbid = getRequestValue( dbid.substring( 1 ) ); }
		}
		else {
			func = key;
		}
//		return DBFunctionName.getFunctionName( func, ConnectionFactory.getDBFullName( dbid ) );
		// 5.1.4.0 (2010/03/01) データベース名 でなく、DBID 名で検索する。
		return DBFunctionName.getFunctionName( func, dbid );
	}

	/**
	 * データロールの設定に基づき、キー(カラム名)に対応する条件式を返します。
	 * 
	 * @og.rev 4.4.0.0 (2009/08/02) 新規作成
	 *
	 * @param   key カラム名
	 * @return  条件式
	 */
	protected String getDataCondition( final String key ) {
		return getUser().getDataCondition( key );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
