/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.ArrayList;
import java.util.Locale ;

import java.sql.ResultSetMetaData ;
import java.sql.DatabaseMetaData ;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

/**
 * DatabaseMetaData の情報を検索するタグです。
 *
 * データベースに関する包括的な情報を提供する、DatabaseMetaData の内容を
 * 表示する、タグです。テスト的に使用します。
 *
 * @og.formSample
 * ●形式：
 *       ・&lt;og:databaseMetaData /&gt;
 * ●body：なし
 *
 * ●使用例
 *       &lt;og:databaseMetaData /&gt;
 *
 * @og.group ＤＢ検索
 *
 * @version  0.9.0	2000/10/17
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DatabaseMetaDataTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/11/30)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/11/30)

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	dbid			= "DEFAULT";
	private String	dbid			= null ;
	private String	catalog			= null ;
	private String	schema			= "GE" ;
	private String	attributeName	= null ;
	private String	columnName		= null ;
	private String	procedureName	= null ;
	private String	tableName		= null ;
	private String	typeName		= null ;
	private int		rowNo			= 0;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 *
	 * @return	int
	 */
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		Connection conn = null;
		boolean errFlag = true;
		try {
			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );	// 3.8.7.0 (2006/12/15)
			DatabaseMetaData metaData = conn.getMetaData() ;

			callPrintMessage( metaData );
			errFlag = false;		// エラーではない
		}
		catch (Throwable ex) {
			String errMsg = "データベース処理を実行できませんでした。"
						+ ex.getMessage()  + HybsSystem.CR ;
//			errMsg = errMsg + HybsSystem.CR ;
//			errMsg = errMsg + ex.getMessage() ;

//			ConnectionFactory.remove( conn,dbid );
//			conn = null;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			// 以下のコーディングの見直しを考える必要有り。
			// Connection はプールしている為、close() しても再利用されるだけ。
//			ConnectionFactory.close( conn,dbid );
			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
//			conn = null;
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 */
	protected void release2() {
		super.release2();
//		dbid			= "DEFAULT";
		dbid			= null ;
		catalog			= null ;
		schema			= null ;
		attributeName	= null ;
		columnName		= null ;
		procedureName	= null ;
		tableName		= null ;
		typeName		= null ;
		rowNo			= 0;
	}

	/**
	 * 画面にテーブルタグの（TR)として、情報を出力します。
	 *
	 * @param in boolean
	 * @param lbl String
	 * @param msg String
	 */
	private void printMessage( final boolean in, final String lbl, final String msg ) {
		String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
				+ lbl + "</td><td>" + in + "</td><td>" + msg + "</td></tr>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 画面にテーブルタグの（TR)として、情報を出力します。
	 *
	 * @param in int
	 * @param lbl String
	 * @param msg String
	 */
	private void printMessage( final int in, final String lbl, final String msg ) {
		String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
				 + lbl + "</td><td>" + in + "</td><td>" + msg + "</td></tr>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 画面にテーブルタグの（TR)として、情報を出力します。
	 *
	 * @param in String
	 * @param lbl String
	 * @param msg String
	 */
	private void printMessage( final String in, final String lbl, final String msg ) {
		String str = "<tr class=\"row_" + (rowNo++ % 2) + "\"><td>"
				 + lbl + "</td><td>" + in + "</td><td>" + msg + "</td></tr>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 画面にテーブルとして、情報を出力します。
	 *
	 * @param resultSet ResultSet
	 * @param msg String
	 */
	private void printMessage( final ResultSet resultSet, final String msg ) {
		jspPrint( msg + HybsSystem.BR );
		jspPrint( makeTableTag( resultToString( resultSet ) ) );
	}

	/**
	 * 画面にテーブルとして、情報を出力します。
	 *
	 * @param lbl String
	 * @param msg String
	 */
	private void printMessage( final String lbl, final String msg ) {
		jspPrint( msg + HybsSystem.BR );
		String str = "<table><tr><td>" + lbl + "</td></tr></table>" + HybsSystem.CR ;
		jspPrint( str );
	}

	/**
	 * 指定の文字列配列をテーブル上に表示する為の、テーブルを作成します。
	 *
	 * @param data String[][]
	 * @return String
	 */
	private String makeTableTag( final String[][] data ) {
		if( data == null || data.length == 0 ) { return "" ; }

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_LARGE );
		buf.append( "<table>" ).append( HybsSystem.CR );

		int maxRow = data.length ;
		int maxClm = data[0].length ;

		buf.append( "<tr class=\"row_h\">" );
		for( int clm=0; clm<maxClm; clm++ ) {
			buf.append( "<th>" ).append( data[0][clm] ).append( "</th>" );
		}
		buf.append( "</tr>" ).append( HybsSystem.CR );
		for( int row=1; row<maxRow; row++ ) {
			buf.append( "<tr class=\"row_" ).append( (row-1) % 2 ).append( "\">" );
			for( int clm=0; clm<maxClm; clm++ ) {
				buf.append( "<td>" ).append( data[row][clm] ).append( "</td>" );
			}
			buf.append( "</tr>" ).append( HybsSystem.CR );
		}
		buf.append( "</table>" ).append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * ステートメントと引数により、Queryを実行します。
	 * これは、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 *
	 * @param   resultSet ResultSet
	 * @return  ResultSet の検索結果配列
	 */
	private String[][] resultToString( final ResultSet resultSet ) {
		ArrayList<String[]> data = new ArrayList<String[]>();
		try {
			ResultSetMetaData metaData  = resultSet.getMetaData();
			int numberOfColumns =  metaData.getColumnCount();

			String[] columnNames = new String[numberOfColumns];
			for(int column = 0; column < numberOfColumns; column++) {
				columnNames[column]	 = (metaData.getColumnLabel(column+1)).toUpperCase(Locale.JAPAN) ;
			}
			data.add( columnNames );

			while( resultSet.next() ) {
				String[] columnValues = new String[numberOfColumns];
				for(int i = 1; i <= numberOfColumns; i++) {
					Object obj = resultSet.getObject(i);
					if( obj == null ) {
						columnValues[i-1] = "";
					}
					else {
						columnValues[i-1] = String.valueOf( obj );
					}
				}
				data.add( columnValues );
			}
		}
		catch (SQLException ex) {
			String errMsg = "処理結果を実行できませんでした。";
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		int size = data.size();
		String[][] rtn = new String[size][];
		for( int i=0; i<size; i++ ) {
			rtn[i] = data.get(i);
		}

		return rtn;
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	public void setCatalog( final String val ) {
		catalog = nval( getRequestParameter( val ),catalog );
	}

	public void setSchema( final String val ) {
		schema = nval( getRequestParameter( val ),schema );
	}

	public void setAttributeName( final String val ) {
		attributeName = nval( getRequestParameter( val ),attributeName );
	}

	public void setColumnName( final String val ) {
		columnName = nval( getRequestParameter( val ),columnName );
	}

	public void setProcedureName( final String val ) {
		procedureName = nval( getRequestParameter( val ),procedureName );
	}

	public void setTableName( final String val ) {
		tableName = nval( getRequestParameter( val ),tableName );
	}

	public void setTypeName( final String val ) {
		typeName = nval( getRequestParameter( val ),typeName );
	}

	/**
	 * DatabaseMetaData のフィールドで定義されている値に対応する メッセージを返します。
	 *
	 * @param	code short のコード番号
	 */
//	private String dbMessage( final short code ) {
//		String msg = null ;
//		switch( code ) {
//			case DatabaseMetaData.attributeNoNulls :
//					msg = "NULL 値が許可されない可能性があります。" ;
//					break ;
//			case DatabaseMetaData.attributeNullable :
//					msg = "NULL 値が必ず許可されます。" ;
//					break ;
//			case DatabaseMetaData.attributeNullableUnknown :
//					msg = "NULL 値が許可されるかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.tableIndexClustered :
//					msg = "このテーブルインデックスがクラスタ化されたインデックスです。" ;
//					break ;
//			case DatabaseMetaData.tableIndexHashed :
//					msg = "このテーブルインデックスがハッシュ化されたインデックスです。" ;
//					break ;
//			case DatabaseMetaData.tableIndexOther :
//					msg = "このテーブルインデックスがクラスタ化されたインデックスでも、ハッシュ化されたインデックスでも、テーブルの統計情報でもありません。" ;
//					break ;
//			case DatabaseMetaData.tableIndexStatistic :
//					msg = "この列がテーブルのインデックスの記述と連動して返されるテーブルの統計情報を格納します。" ;
//					break ;
//			default :
//					msg = "対象のメッセージがありません。" ;
//					break ;
//		}
//
//		return msg ;
//	}

	/**
	 * DatabaseMetaData のフィールドで定義されている値に対応する メッセージを返します。
	 *
	 * @param	code short のコード番号
	 */
//	private String dbMessage( final int code ) {
//		String msg = null ;
//		switch( code ) {
//			case DatabaseMetaData.bestRowNotPseudo :
//					msg = "最適な行識別子が擬似列ではありません。" ;
//					break ;
//			case DatabaseMetaData.bestRowPseudo :
//					msg = "最適な行識別子が擬似列です。" ;
//					break ;
//			case DatabaseMetaData.bestRowSession :
//					msg = "最適な行識別子のスコープが現在のセッションの残りの部分です。" ;
//					break ;
//			case DatabaseMetaData.bestRowTemporary :
//					msg = "最適な行識別子のスコープが一時的 (行が使用されている間だけ) です。" ;
//					break ;
//			case DatabaseMetaData.bestRowTransaction :
//					msg = "最適な行識別子のスコープが現在のトランザクションの残りの部分です。" ;
//					break ;
//			case DatabaseMetaData.bestRowUnknown :
//					msg = "最適な行識別子が擬似列である場合とそうでない場合があります。" ;
//					break ;
//			case DatabaseMetaData.columnNoNulls :
//					msg = "列が NULL 値を許可しない場合があります。" ;
//					break ;
//			case DatabaseMetaData.columnNullable :
//					msg = "列が必ず NULL 値を許可されます。" ;
//					break ;
//			case DatabaseMetaData.columnNullableUnknown :
//					msg = "列が NULL 値を許可するかどうか不明です。" ;
//					break ;
//			case DatabaseMetaData.importedKeyCascade :
//					msg = "列 UPDATE_RULE の場合は、主キーが更新されるときに、外部キー (インポートされたキー) がそれに合致するように変更されます。" ;
//					break ;
//			case DatabaseMetaData.importedKeyInitiallyDeferred :
//					msg = "遅延可能性を示します。" ;
//					break ;
//			case DatabaseMetaData.importedKeyInitiallyImmediate :
//					msg = "遅延可能性を示します。" ;
//					break ;
//			case DatabaseMetaData.importedKeyNoAction :
//					msg = "列 UPDATE_RULE および DELETE_RULE の場合は、主キーがインポートされている場合、それが更新および削除ができません。" ;
//					break ;
//			case DatabaseMetaData.importedKeyNotDeferrable :
//					msg = "遅延可能性を示します。" ;
//					break ;
//			case DatabaseMetaData.importedKeyRestrict :
//					msg = "列 UPDATE_RULE の場合は、別のテーブルによって外部キーとしてインポートされている場合に、主キーは更新できません。" ;
//					break ;
//			case DatabaseMetaData.importedKeySetDefault :
//					msg = "列 UPDATE_RULE および DELETE_RULE の場合は、主キーが更新または削除された場合に、外部キー (インポートされたキー) がデフォルト値に変更されます。" ;
//					break ;
//			case DatabaseMetaData.importedKeySetNull :
//					msg = "列 UPDATE_RULE および DELETE_RULE の場合、主キーが更新または削除されたときに、外部キー (インポートされたキー) が NULL に変更されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnIn :
//					msg = "列が IN パラメータを格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnInOut :
//					msg = "列が INOUT パラメータを格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnOut :
//					msg = "列が OUT パラメータを格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnResult :
//					msg = "列が結果を格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnReturn :
//					msg = "列が戻り値を格納されます。" ;
//					break ;
//			case DatabaseMetaData.procedureColumnUnknown :
//					msg = "列の型が不明です。" ;
//					break ;
//			case DatabaseMetaData.procedureNoNulls :
//					msg = "NULL 値が許可されません。" ;
//					break ;
//			case DatabaseMetaData.procedureNoResult :
//					msg = "プロシージャが結果を返しません。" ;
//					break ;
//			case DatabaseMetaData.procedureNullable :
//					msg = "NULL 値が許可されます。" ;
//					break ;
//			case DatabaseMetaData.procedureNullableUnknown :
//					msg = "NULL 値が許可されるかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.procedureResultUnknown :
//					msg = "プロシージャが結果を返すかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.procedureReturnsResult :
//					msg = "プロシージャが結果を返すことを示します。" ;
//					break ;
//			case DatabaseMetaData.sqlStateSQL99 :
//					msg = "値が SQL99 SQLSTATE 値です。" ;
//					break ;
//			case DatabaseMetaData.sqlStateXOpen :
//					msg = "値が X/Open (現在は Open Group) の SQL CLI SQLSTATE 値です。" ;
//					break ;
//			case DatabaseMetaData.typeNoNulls :
//					msg = "このデータの型で NULL 値が許可されません。" ;
//					break ;
//			case DatabaseMetaData.typeNullable :
//					msg = "このデータの型で NULL 値が許可されます。" ;
//					break ;
//			case DatabaseMetaData.typeNullableUnknown :
//					msg = "このデータの型で NULL 値が許可されるかどうかが不明です。" ;
//					break ;
//			case DatabaseMetaData.typePredBasic :
//					msg = "WHERE . . .LIKE を除くすべての WHERE 検索の節がこのデータの型に基づくことができます。" ;
//					break ;
//			case DatabaseMetaData.typePredChar :
//					msg = "この型に基づくことができる唯一の WHERE 検索の節が WHERE . . .LIKE です。" ;
//					break ;
//			case DatabaseMetaData.typePredNone :
//					msg = "この型で WHERE 検索の節がサポートされていません。" ;
//					break ;
//			case DatabaseMetaData.typeSearchable :
//					msg = "すべての WHERE 検索の節がこの型に基づくことができます。" ;
//					break ;
//			case DatabaseMetaData.versionColumnNotPseudo :
//					msg = "このバージョンの列が擬似列でありません。" ;
//					break ;
//			case DatabaseMetaData.versionColumnPseudo :
//					msg = "このバージョンの列が擬似列です。" ;
//					break ;
//			case DatabaseMetaData.versionColumnUnknown :
//					msg = "このバージョンの列が擬似列である場合とそうでない場合があります。" ;
//					break ;
//			default :
//					msg = "対象のメッセージがありません。" ;
//					break ;
//		}
//
//		return msg ;
//	}

	/**
	 * 各メソッド情報を画面に出力していきます。
	 *
	 * @param metaData DatabaseMetaData
	 * @throws Throwable
	 */
	private void callPrintMessage( final DatabaseMetaData metaData ) throws Throwable {
		// 単体のリターン値を持つ メタデータ

		int type = ResultSet.TYPE_FORWARD_ONLY ;
		int concurrency = ResultSet.CONCUR_READ_ONLY;
		int holdability = ResultSet.CLOSE_CURSORS_AT_COMMIT;

		jspPrint( "<table>" );
		jspPrint( "<tr class=\"row_h\"><th>キー</th><th>値</th><th>説明</th></tr>" );

		try { printMessage( metaData.getCatalogSeparator() ,"getCatalogSeparator",getCatalogSeparator); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getCatalogSeparator",getCatalogSeparator); }
		try { printMessage( metaData.getCatalogTerm() ,"getCatalogTerm",getCatalogTerm); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getCatalogTerm",getCatalogTerm); }
		try { printMessage( metaData.getDatabaseProductName() ,"getDatabaseProductName",getDatabaseProductName); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDatabaseProductName",getDatabaseProductName); }
		try { printMessage( metaData.getDatabaseProductVersion() ,"getDatabaseProductVersion",getDatabaseProductVersion); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDatabaseProductVersion",getDatabaseProductVersion); }
		try { printMessage( metaData.getDriverMajorVersion() ,"getDriverMajorVersion",getDriverMajorVersion); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverMajorVersion",getDriverMajorVersion); }
		try { printMessage( metaData.getDriverMinorVersion() ,"getDriverMinorVersion",getDriverMinorVersion); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverMinorVersion",getDriverMinorVersion); }
		try { printMessage( metaData.getDriverName() ,"getDriverName",getDriverName); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverName",getDriverName); }
		try { printMessage( metaData.getDriverVersion() ,"getDriverVersion",getDriverVersion); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDriverVersion",getDriverVersion); }
		try { printMessage( metaData.getExtraNameCharacters() ,"getExtraNameCharacters",getExtraNameCharacters); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getExtraNameCharacters",getExtraNameCharacters); }
		try { printMessage( metaData.getIdentifierQuoteString() ,"getIdentifierQuoteString",getIdentifierQuoteString); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getIdentifierQuoteString",getIdentifierQuoteString); }
		try { printMessage( metaData.getNumericFunctions() ,"getNumericFunctions",getNumericFunctions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getNumericFunctions",getNumericFunctions); }
		try { printMessage( metaData.getProcedureTerm() ,"getProcedureTerm",getProcedureTerm); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getProcedureTerm",getProcedureTerm); }
		try { printMessage( metaData.getSchemaTerm() ,"getSchemaTerm",getSchemaTerm); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getSchemaTerm",getSchemaTerm); }
		try { printMessage( metaData.getSearchStringEscape() ,"getSearchStringEscape",getSearchStringEscape); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getSearchStringEscape",getSearchStringEscape); }
		try { printMessage( metaData.getSQLKeywords() ,"getSQLKeywords",getSQLKeywords); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getSQLKeywords",getSQLKeywords); }
		try { printMessage( metaData.getStringFunctions() ,"getStringFunctions",getStringFunctions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getStringFunctions",getStringFunctions); }
		try { printMessage( metaData.getSystemFunctions() ,"getSystemFunctions",getSystemFunctions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getSystemFunctions",getSystemFunctions); }
		try { printMessage( metaData.getTimeDateFunctions() ,"getTimeDateFunctions",getTimeDateFunctions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getTimeDateFunctions",getTimeDateFunctions); }
		try { printMessage( metaData.getURL() ,"getURL",getURL); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getURL",getURL); }
		try { printMessage( metaData.getUserName() ,"getUserName",getUserName); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getUserName",getUserName); }
		try { printMessage( metaData.getDefaultTransactionIsolation() ,"getDefaultTransactionIsolation",getDefaultTransactionIsolation); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getDefaultTransactionIsolation",getDefaultTransactionIsolation); }
		try { printMessage( metaData.getSQLStateType() ,"getSQLStateType",getSQLStateType); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getSQLStateType",getSQLStateType); }
		try { printMessage( metaData.getResultSetHoldability() ,"getResultSetHoldability",getResultSetHoldability); }
			catch( Throwable e ) { printMessage( e.getMessage(),"getResultSetHoldability",getResultSetHoldability); }
		try { printMessage( metaData.allProceduresAreCallable() ,"allProceduresAreCallable",allProceduresAreCallable); }
			catch( Throwable e ) { printMessage( e.getMessage(),"allProceduresAreCallable",allProceduresAreCallable); }
		try { printMessage( metaData.allTablesAreSelectable() ,"allTablesAreSelectable",allTablesAreSelectable); }
			catch( Throwable e ) { printMessage( e.getMessage(),"allTablesAreSelectable",allTablesAreSelectable); }
		try { printMessage( metaData.dataDefinitionCausesTransactionCommit() ,"dataDefinitionCausesTransactionCommit",dataDefinitionCausesTransactionCommit); }
			catch( Throwable e ) { printMessage( e.getMessage(),"dataDefinitionCausesTransactionCommit",dataDefinitionCausesTransactionCommit); }
		try { printMessage( metaData.dataDefinitionIgnoredInTransactions() ,"dataDefinitionIgnoredInTransactions",dataDefinitionIgnoredInTransactions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"dataDefinitionIgnoredInTransactions",dataDefinitionIgnoredInTransactions); }
		try { printMessage( metaData.deletesAreDetected(type) ,"deletesAreDetected",deletesAreDetected); }
			catch( Throwable e ) { printMessage( e.getMessage(),"deletesAreDetected",deletesAreDetected); }
		try { printMessage( metaData.doesMaxRowSizeIncludeBlobs() ,"doesMaxRowSizeIncludeBlobs",doesMaxRowSizeIncludeBlobs); }
			catch( Throwable e ) { printMessage( e.getMessage(),"doesMaxRowSizeIncludeBlobs",doesMaxRowSizeIncludeBlobs); }
		try { printMessage( metaData.insertsAreDetected(type) ,"insertsAreDetected",insertsAreDetected); }
			catch( Throwable e ) { printMessage( e.getMessage(),"insertsAreDetected",insertsAreDetected); }
		try { printMessage( metaData.isCatalogAtStart() ,"isCatalogAtStart",isCatalogAtStart); }
			catch( Throwable e ) { printMessage( e.getMessage(),"isCatalogAtStart",isCatalogAtStart); }
		try { printMessage( metaData.isReadOnly() ,"isReadOnly",isReadOnly); }
			catch( Throwable e ) { printMessage( e.getMessage(),"isReadOnly",isReadOnly); }
		try { printMessage( metaData.locatorsUpdateCopy() ,"locatorsUpdateCopy",locatorsUpdateCopy); }
			catch( Throwable e ) { printMessage( e.getMessage(),"locatorsUpdateCopy",locatorsUpdateCopy); }
		try { printMessage( metaData.nullPlusNonNullIsNull() ,"nullPlusNonNullIsNull",nullPlusNonNullIsNull); }
			catch( Throwable e ) { printMessage( e.getMessage(),"nullPlusNonNullIsNull",nullPlusNonNullIsNull); }
		try { printMessage( metaData.nullsAreSortedAtEnd() ,"nullsAreSortedAtEnd",nullsAreSortedAtEnd); }
			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedAtEnd",nullsAreSortedAtEnd); }
		try { printMessage( metaData.nullsAreSortedAtStart() ,"nullsAreSortedAtStart",nullsAreSortedAtStart); }
			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedAtStart",nullsAreSortedAtStart); }
		try { printMessage( metaData.nullsAreSortedHigh() ,"nullsAreSortedHigh",nullsAreSortedHigh); }
			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedHigh",nullsAreSortedHigh); }
		try { printMessage( metaData.nullsAreSortedLow() ,"nullsAreSortedLow",nullsAreSortedLow); }
			catch( Throwable e ) { printMessage( e.getMessage(),"nullsAreSortedLow",nullsAreSortedLow); }
		try { printMessage( metaData.othersDeletesAreVisible(type) ,"othersDeletesAreVisible",othersDeletesAreVisible); }
			catch( Throwable e ) { printMessage( e.getMessage(),"othersDeletesAreVisible",othersDeletesAreVisible); }
		try { printMessage( metaData.othersInsertsAreVisible(type) ,"othersInsertsAreVisible",othersInsertsAreVisible); }
			catch( Throwable e ) { printMessage( e.getMessage(),"othersInsertsAreVisible",othersInsertsAreVisible); }
		try { printMessage( metaData.othersUpdatesAreVisible(type) ,"othersUpdatesAreVisible",othersUpdatesAreVisible); }
			catch( Throwable e ) { printMessage( e.getMessage(),"othersUpdatesAreVisible",othersUpdatesAreVisible); }
		try { printMessage( metaData.ownDeletesAreVisible(type) ,"ownDeletesAreVisible",ownDeletesAreVisible); }
			catch( Throwable e ) { printMessage( e.getMessage(),"ownDeletesAreVisible",ownDeletesAreVisible); }
		try { printMessage( metaData.ownInsertsAreVisible(type) ,"ownInsertsAreVisible",ownInsertsAreVisible); }
			catch( Throwable e ) { printMessage( e.getMessage(),"ownInsertsAreVisible",ownInsertsAreVisible); }
		try { printMessage( metaData.ownUpdatesAreVisible(type) ,"ownUpdatesAreVisible",ownUpdatesAreVisible); }
			catch( Throwable e ) { printMessage( e.getMessage(),"ownUpdatesAreVisible",ownUpdatesAreVisible); }
		try { printMessage( metaData.storesLowerCaseIdentifiers() ,"storesLowerCaseIdentifiers",storesLowerCaseIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"storesLowerCaseIdentifiers",storesLowerCaseIdentifiers); }
		try { printMessage( metaData.storesLowerCaseQuotedIdentifiers() ,"storesLowerCaseQuotedIdentifiers",storesLowerCaseQuotedIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"storesLowerCaseQuotedIdentifiers",storesLowerCaseQuotedIdentifiers); }
		try { printMessage( metaData.storesMixedCaseIdentifiers() ,"storesMixedCaseIdentifiers",storesMixedCaseIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"storesMixedCaseIdentifiers",storesMixedCaseIdentifiers); }
		try { printMessage( metaData.storesMixedCaseQuotedIdentifiers() ,"storesMixedCaseQuotedIdentifiers",storesMixedCaseQuotedIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"storesMixedCaseQuotedIdentifiers",storesMixedCaseQuotedIdentifiers); }
		try { printMessage( metaData.storesUpperCaseIdentifiers() ,"storesUpperCaseIdentifiers",storesUpperCaseIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"storesUpperCaseIdentifiers",storesUpperCaseIdentifiers); }
		try { printMessage( metaData.storesUpperCaseQuotedIdentifiers() ,"storesUpperCaseQuotedIdentifiers",storesUpperCaseQuotedIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"storesUpperCaseQuotedIdentifiers",storesUpperCaseQuotedIdentifiers); }
		try { printMessage( metaData.supportsAlterTableWithAddColumn() ,"supportsAlterTableWithAddColumn",supportsAlterTableWithAddColumn); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsAlterTableWithAddColumn",supportsAlterTableWithAddColumn); }
		try { printMessage( metaData.supportsAlterTableWithDropColumn() ,"supportsAlterTableWithDropColumn",supportsAlterTableWithDropColumn); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsAlterTableWithDropColumn",supportsAlterTableWithDropColumn); }
		try { printMessage( metaData.supportsANSI92EntryLevelSQL() ,"supportsANSI92EntryLevelSQL",supportsANSI92EntryLevelSQL); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsANSI92EntryLevelSQL",supportsANSI92EntryLevelSQL); }
		try { printMessage( metaData.supportsANSI92FullSQL() ,"supportsANSI92FullSQL",supportsANSI92FullSQL); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsANSI92FullSQL",supportsANSI92FullSQL); }
		try { printMessage( metaData.supportsANSI92IntermediateSQL() ,"supportsANSI92IntermediateSQL",supportsANSI92IntermediateSQL); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsANSI92IntermediateSQL",supportsANSI92IntermediateSQL); }
		try { printMessage( metaData.supportsBatchUpdates() ,"supportsBatchUpdates",supportsBatchUpdates); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsBatchUpdates",supportsBatchUpdates); }
		try { printMessage( metaData.supportsCatalogsInDataManipulation() ,"supportsCatalogsInDataManipulation",supportsCatalogsInDataManipulation); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInDataManipulation",supportsCatalogsInDataManipulation); }
		try { printMessage( metaData.supportsCatalogsInIndexDefinitions() ,"supportsCatalogsInIndexDefinitions",supportsCatalogsInIndexDefinitions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInIndexDefinitions",supportsCatalogsInIndexDefinitions); }
		try { printMessage( metaData.supportsCatalogsInPrivilegeDefinitions() ,"supportsCatalogsInPrivilegeDefinitions",supportsCatalogsInPrivilegeDefinitions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInPrivilegeDefinitions",supportsCatalogsInPrivilegeDefinitions); }
		try { printMessage( metaData.supportsCatalogsInProcedureCalls() ,"supportsCatalogsInProcedureCalls",supportsCatalogsInProcedureCalls); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInProcedureCalls",supportsCatalogsInProcedureCalls); }
		try { printMessage( metaData.supportsCatalogsInTableDefinitions() ,"supportsCatalogsInTableDefinitions",supportsCatalogsInTableDefinitions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCatalogsInTableDefinitions",supportsCatalogsInTableDefinitions); }
		try { printMessage( metaData.supportsColumnAliasing() ,"supportsColumnAliasing",supportsColumnAliasing); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsColumnAliasing",supportsColumnAliasing); }
		try { printMessage( metaData.supportsConvert() ,"supportsConvert",supportsConvertFunc); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsConvert",supportsConvertFunc); }
		try { printMessage( metaData.supportsCoreSQLGrammar() ,"supportsCoreSQLGrammar",supportsCoreSQLGrammar); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCoreSQLGrammar",supportsCoreSQLGrammar); }
		try { printMessage( metaData.supportsCorrelatedSubqueries() ,"supportsCorrelatedSubqueries",supportsCorrelatedSubqueries); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsCorrelatedSubqueries",supportsCorrelatedSubqueries); }
		try { printMessage( metaData.supportsDataDefinitionAndDataManipulationTransactions() ,"supportsDataDefinitionAndDataManipulationTransactions",supportsDataDefinitionAndDataManipulationTransactions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsDataDefinitionAndDataManipulationTransactions",supportsDataDefinitionAndDataManipulationTransactions); }
		try { printMessage( metaData.supportsDataManipulationTransactionsOnly() ,"supportsDataManipulationTransactionsOnly",supportsDataManipulationTransactionsOnly); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsDataManipulationTransactionsOnly",supportsDataManipulationTransactionsOnly); }
		try { printMessage( metaData.supportsDifferentTableCorrelationNames() ,"supportsDifferentTableCorrelationNames",supportsDifferentTableCorrelationNames); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsDifferentTableCorrelationNames",supportsDifferentTableCorrelationNames); }
		try { printMessage( metaData.supportsExpressionsInOrderBy() ,"supportsExpressionsInOrderBy",supportsExpressionsInOrderBy); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsExpressionsInOrderBy",supportsExpressionsInOrderBy); }
		try { printMessage( metaData.supportsExtendedSQLGrammar() ,"supportsExtendedSQLGrammar",supportsExtendedSQLGrammar); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsExtendedSQLGrammar",supportsExtendedSQLGrammar); }
		try { printMessage( metaData.supportsFullOuterJoins() ,"supportsFullOuterJoins",supportsFullOuterJoins); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsFullOuterJoins",supportsFullOuterJoins); }
		try { printMessage( metaData.supportsGetGeneratedKeys() ,"supportsGetGeneratedKeys",supportsGetGeneratedKeys); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGetGeneratedKeys",supportsGetGeneratedKeys); }
		try { printMessage( metaData.supportsGroupBy() ,"supportsGroupBy",supportsGroupBy); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGroupBy",supportsGroupBy); }
		try { printMessage( metaData.supportsGroupByBeyondSelect() ,"supportsGroupByBeyondSelect",supportsGroupByBeyondSelect); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGroupByBeyondSelect",supportsGroupByBeyondSelect); }
		try { printMessage( metaData.supportsGroupByUnrelated() ,"supportsGroupByUnrelated",supportsGroupByUnrelated); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsGroupByUnrelated",supportsGroupByUnrelated); }
		try { printMessage( metaData.supportsIntegrityEnhancementFacility() ,"supportsIntegrityEnhancementFacility",supportsIntegrityEnhancementFacility); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsIntegrityEnhancementFacility",supportsIntegrityEnhancementFacility); }
		try { printMessage( metaData.supportsLikeEscapeClause() ,"supportsLikeEscapeClause",supportsLikeEscapeClause); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsLikeEscapeClause",supportsLikeEscapeClause); }
		try { printMessage( metaData.supportsLimitedOuterJoins() ,"supportsLimitedOuterJoins",supportsLimitedOuterJoins); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsLimitedOuterJoins",supportsLimitedOuterJoins); }
		try { printMessage( metaData.supportsMinimumSQLGrammar() ,"supportsMinimumSQLGrammar",supportsMinimumSQLGrammar); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMinimumSQLGrammar",supportsMinimumSQLGrammar); }
		try { printMessage( metaData.supportsMixedCaseIdentifiers() ,"supportsMixedCaseIdentifiers",supportsMixedCaseIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMixedCaseIdentifiers",supportsMixedCaseIdentifiers); }
		try { printMessage( metaData.supportsMixedCaseQuotedIdentifiers() ,"supportsMixedCaseQuotedIdentifiers",supportsMixedCaseQuotedIdentifiers); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMixedCaseQuotedIdentifiers",supportsMixedCaseQuotedIdentifiers); }
		try { printMessage( metaData.supportsMultipleOpenResults() ,"supportsMultipleOpenResults",supportsMultipleOpenResults); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMultipleOpenResults",supportsMultipleOpenResults); }
		try { printMessage( metaData.supportsMultipleResultSets() ,"supportsMultipleResultSets",supportsMultipleResultSets); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMultipleResultSets",supportsMultipleResultSets); }
		try { printMessage( metaData.supportsMultipleTransactions() ,"supportsMultipleTransactions",supportsMultipleTransactions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsMultipleTransactions",supportsMultipleTransactions); }
		try { printMessage( metaData.supportsNamedParameters() ,"supportsNamedParameters",supportsNamedParameters); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsNamedParameters",supportsNamedParameters); }
		try { printMessage( metaData.supportsNonNullableColumns() ,"supportsNonNullableColumns",supportsNonNullableColumns); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsNonNullableColumns",supportsNonNullableColumns); }
		try { printMessage( metaData.supportsOpenCursorsAcrossCommit() ,"supportsOpenCursorsAcrossCommit",supportsOpenCursorsAcrossCommit); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenCursorsAcrossCommit",supportsOpenCursorsAcrossCommit); }
		try { printMessage( metaData.supportsOpenCursorsAcrossRollback() ,"supportsOpenCursorsAcrossRollback",supportsOpenCursorsAcrossRollback); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenCursorsAcrossRollback",supportsOpenCursorsAcrossRollback); }
		try { printMessage( metaData.supportsOpenStatementsAcrossCommit() ,"supportsOpenStatementsAcrossCommit",supportsOpenStatementsAcrossCommit); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenStatementsAcrossCommit",supportsOpenStatementsAcrossCommit); }
		try { printMessage( metaData.supportsOpenStatementsAcrossRollback() ,"supportsOpenStatementsAcrossRollback",supportsOpenStatementsAcrossRollback); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOpenStatementsAcrossRollback",supportsOpenStatementsAcrossRollback); }
		try { printMessage( metaData.supportsOrderByUnrelated() ,"supportsOrderByUnrelated",supportsOrderByUnrelated); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOrderByUnrelated",supportsOrderByUnrelated); }
		try { printMessage( metaData.supportsOuterJoins() ,"supportsOuterJoins",supportsOuterJoins); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsOuterJoins",supportsOuterJoins); }
		try { printMessage( metaData.supportsPositionedDelete() ,"supportsPositionedDelete",supportsPositionedDelete); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsPositionedDelete",supportsPositionedDelete); }
		try { printMessage( metaData.supportsPositionedUpdate() ,"supportsPositionedUpdate",supportsPositionedUpdate); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsPositionedUpdate",supportsPositionedUpdate); }
		try { printMessage( metaData.supportsResultSetConcurrency(type, concurrency) ,"supportsResultSetConcurrency",supportsResultSetConcurrency); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetConcurrency",supportsResultSetConcurrency); }
		try { printMessage( metaData.supportsResultSetHoldability(holdability) ,"supportsResultSetHoldability",supportsResultSetHoldability); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetHoldability",supportsResultSetHoldability); }
		try { printMessage( metaData.supportsResultSetType(ResultSet.TYPE_FORWARD_ONLY) ,"supportsResultSetType[TYPE_FORWARD_ONLY]",supportsResultSetType); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetType",supportsResultSetType); }
		try { printMessage( metaData.supportsResultSetType(ResultSet.TYPE_SCROLL_INSENSITIVE) ,"supportsResultSetType[TYPE_SCROLL_INSENSITIVE]", supportsResultSetType); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetType",supportsResultSetType); }
		try { printMessage( metaData.supportsResultSetType(ResultSet.TYPE_SCROLL_SENSITIVE) ,"supportsResultSetType[TYPE_SCROLL_SENSITIVE]", supportsResultSetType); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsResultSetType",supportsResultSetType); }
		try { printMessage( metaData.supportsSavepoints() ,"supportsSavepoints",supportsSavepoints); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSavepoints",supportsSavepoints); }
		try { printMessage( metaData.supportsSchemasInDataManipulation() ,"supportsSchemasInDataManipulation",supportsSchemasInDataManipulation); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInDataManipulation",supportsSchemasInDataManipulation); }
		try { printMessage( metaData.supportsSchemasInIndexDefinitions() ,"supportsSchemasInIndexDefinitions",supportsSchemasInIndexDefinitions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInIndexDefinitions",supportsSchemasInIndexDefinitions); }
		try { printMessage( metaData.supportsSchemasInPrivilegeDefinitions() ,"supportsSchemasInPrivilegeDefinitions",supportsSchemasInPrivilegeDefinitions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInPrivilegeDefinitions",supportsSchemasInPrivilegeDefinitions); }
		try { printMessage( metaData.supportsSchemasInProcedureCalls() ,"supportsSchemasInProcedureCalls",supportsSchemasInProcedureCalls); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInProcedureCalls",supportsSchemasInProcedureCalls); }
		try { printMessage( metaData.supportsSchemasInTableDefinitions() ,"supportsSchemasInTableDefinitions",supportsSchemasInTableDefinitions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSchemasInTableDefinitions",supportsSchemasInTableDefinitions); }
		try { printMessage( metaData.supportsSelectForUpdate() ,"supportsSelectForUpdate",supportsSelectForUpdate); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSelectForUpdate",supportsSelectForUpdate); }
		try { printMessage( metaData.supportsStatementPooling() ,"supportsStatementPooling",supportsStatementPooling); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsStatementPooling",supportsStatementPooling); }
		try { printMessage( metaData.supportsStoredProcedures() ,"supportsStoredProcedures",supportsStoredProcedures); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsStoredProcedures",supportsStoredProcedures); }
		try { printMessage( metaData.supportsSubqueriesInComparisons() ,"supportsSubqueriesInComparisons",supportsSubqueriesInComparisons); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInComparisons",supportsSubqueriesInComparisons); }
		try { printMessage( metaData.supportsSubqueriesInExists() ,"supportsSubqueriesInExists",supportsSubqueriesInExists); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInExists",supportsSubqueriesInExists); }
		try { printMessage( metaData.supportsSubqueriesInIns() ,"supportsSubqueriesInIns",supportsSubqueriesInIns); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInIns",supportsSubqueriesInIns); }
		try { printMessage( metaData.supportsSubqueriesInQuantifieds() ,"supportsSubqueriesInQuantifieds",supportsSubqueriesInQuantifieds); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsSubqueriesInQuantifieds",supportsSubqueriesInQuantifieds); }
		try { printMessage( metaData.supportsTableCorrelationNames() ,"supportsTableCorrelationNames",supportsTableCorrelationNames); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTableCorrelationNames",supportsTableCorrelationNames); }
		try { printMessage( metaData.supportsTransactions() ,"supportsTransactions",supportsTransactions); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactions",supportsTransactions); }
		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_READ_COMMITTED) ,"supportsTransactionIsolationLevel[TRANSACTION_READ_COMMITTED]",supportsTransactionIsolationLevel); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_READ_UNCOMMITTED) ,"supportsTransactionIsolationLevel[TRANSACTION_READ_UNCOMMITTED]",supportsTransactionIsolationLevel); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_REPEATABLE_READ) ,"supportsTransactionIsolationLevel[TRANSACTION_REPEATABLE_READ]",supportsTransactionIsolationLevel); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
		try { printMessage( metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_SERIALIZABLE) ,"supportsTransactionIsolationLevel[TRANSACTION_SERIALIZABLE]",supportsTransactionIsolationLevel); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsTransactionIsolationLevel",supportsTransactionIsolationLevel); }
		try { printMessage( metaData.supportsUnion() ,"supportsUnion",supportsUnion); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsUnion",supportsUnion); }
		try { printMessage( metaData.supportsUnionAll() ,"supportsUnionAll",supportsUnionAll); }
			catch( Throwable e ) { printMessage( e.getMessage(),"supportsUnionAll",supportsUnionAll); }
		try { printMessage( metaData.updatesAreDetected(type) ,"updatesAreDetected",updatesAreDetected); }
			catch( Throwable e ) { printMessage( e.getMessage(),"updatesAreDetected",updatesAreDetected); }
		try { printMessage( metaData.usesLocalFilePerTable() ,"usesLocalFilePerTable",usesLocalFilePerTable); }
			catch( Throwable e ) { printMessage( e.getMessage(),"usesLocalFilePerTable",usesLocalFilePerTable); }
		try { printMessage( metaData.usesLocalFiles() ,"usesLocalFiles",usesLocalFiles); }
			catch( Throwable e ) { printMessage( e.getMessage(),"usesLocalFiles",usesLocalFiles); }

		jspPrint( "</table>" );

		if( metaData.supportsConvert() ) {
			supportsConvert( metaData );
		}

		// ResultSet のリターン値を持つ メタデータ
		boolean nullable = true;
		int scope = 0;

		try { printMessage( metaData.getAttributes(catalog, schema, typeName, attributeName) ,getAttributes); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getAttributes); }
		try { printMessage( metaData.getBestRowIdentifier(catalog, schema, tableName, scope, nullable) ,getBestRowIdentifier); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getBestRowIdentifier); }
		try { printMessage( metaData.getCatalogs() ,getCatalogs); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getCatalogs); }
		try { printMessage( metaData.getColumnPrivileges(catalog, schema, tableName, columnName) ,getColumnPrivileges); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getColumnPrivileges); }
		try { printMessage( metaData.getColumns(catalog, schema, tableName, columnName) ,getColumns); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getColumns); }
		try { printMessage( metaData.getCrossReference(catalog, schema, tableName, catalog, schema, tableName) ,getCrossReference); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getCrossReference); }
		try { printMessage( metaData.getExportedKeys(catalog, schema, tableName) ,getExportedKeys); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getExportedKeys); }
		try { printMessage( metaData.getImportedKeys(catalog, schema, tableName) ,getImportedKeys); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getImportedKeys); }
		try { printMessage( metaData.getIndexInfo(catalog, schema, tableName, false, false) ,getIndexInfo); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getIndexInfo); }
		try { printMessage( metaData.getPrimaryKeys(catalog, schema, tableName) ,getPrimaryKeys); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getPrimaryKeys); }
		try { printMessage( metaData.getProcedureColumns(catalog, schema, procedureName, columnName) ,getProcedureColumns); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getProcedureColumns); }
		try { printMessage( metaData.getProcedures(catalog, schema, procedureName) ,getProcedures); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getProcedures); }
		try { printMessage( metaData.getSchemas() ,getSchemas); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getSchemas); }
		try { printMessage( metaData.getSuperTables(catalog, schema, tableName) ,getSuperTables); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getSuperTables); }
		try { printMessage( metaData.getSuperTypes(catalog, schema, typeName) ,getSuperTypes); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getSuperTypes); }
		try { printMessage( metaData.getTablePrivileges(catalog, schema, tableName) ,getTablePrivileges); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getTablePrivileges); }
		try { printMessage( metaData.getTables(catalog, schema, tableName, null) ,getTables); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getTables); }
		try { printMessage( metaData.getTableTypes() ,getTableTypes); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getTableTypes); }
		try { printMessage( metaData.getTypeInfo() ,getTypeInfo); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getTypeInfo); }
		try { printMessage( metaData.getUDTs(catalog, schema, typeName, null) ,getUDTs); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getUDTs); }
		try { printMessage( metaData.getVersionColumns(catalog, schema, tableName) ,getVersionColumns); }
			catch( Throwable e ) { printMessage( e.getMessage() ,getVersionColumns); }

	}

	/**
	 * データベースの Types 定数を文字列に変換したテーブルを画面に出力します。
	 *
	 * @param metaData DatabaseMetaData
	 * @throws Throwable
	 * @see java.sql.Types
	 */
	private void supportsConvert( final DatabaseMetaData metaData ) throws Throwable {

		int cnt = 30;
		int[]    key = new int[cnt] ;
		String[] msg = new String[cnt] ;

		int i = 0;
		key[i] = Types.ARRAY ;			msg[i++] = "ARRAY" ;			// 0
		key[i] = Types.BIGINT ;			msg[i++] = "BIGINT" ;			// 1
		key[i] = Types.BINARY ;			msg[i++] = "BINARY" ;			// 2
		key[i] = Types.BIT ;			msg[i++] = "BIT" ;				// 3
		key[i] = Types.BLOB ;			msg[i++] = "BLOB" ;				// 4
		key[i] = Types.BOOLEAN ;		msg[i++] = "BOOLEAN" ;			// 5
		key[i] = Types.CHAR ;			msg[i++] = "CHAR" ;				// 6
		key[i] = Types.CLOB ;			msg[i++] = "CLOB" ;				// 7
		key[i] = Types.DATALINK ;		msg[i++] = "DATALINK" ;			// 8
		key[i] = Types.DATE ;			msg[i++] = "DATE" ;				// 9
		key[i] = Types.DECIMAL ;		msg[i++] = "DECIMAL" ;			// 10
		key[i] = Types.DISTINCT ;		msg[i++] = "DISTINCT" ;			// 11
		key[i] = Types.DOUBLE ;			msg[i++] = "DOUBLE" ;			// 12
		key[i] = Types.FLOAT ;			msg[i++] = "FLOAT" ;			// 13
		key[i] = Types.INTEGER ;		msg[i++] = "INTEGER" ;			// 14
		key[i] = Types.JAVA_OBJECT ;	msg[i++] = "JAVA_OBJECT" ;		// 15
		key[i] = Types.LONGVARBINARY ;	msg[i++] = "LONGVARBINARY" ;	// 16
		key[i] = Types.LONGVARCHAR ;	msg[i++] = "LONGVARCHAR" ;		// 17
		key[i] = Types.NULL ;			msg[i++] = "NULL" ;				// 18
		key[i] = Types.NUMERIC ;		msg[i++] = "NUMERIC" ;			// 19
		key[i] = Types.OTHER ;			msg[i++] = "OTHER" ;			// 20
		key[i] = Types.REAL ;			msg[i++] = "REAL" ;				// 21
		key[i] = Types.REF ;			msg[i++] = "REF" ;				// 22
		key[i] = Types.SMALLINT ;		msg[i++] = "SMALLINT" ;			// 23
		key[i] = Types.STRUCT ;			msg[i++] = "STRUCT" ;			// 24
		key[i] = Types.TIME ;			msg[i++] = "TIME" ;				// 25
		key[i] = Types.TIMESTAMP ;		msg[i++] = "TIMESTAMP" ;		// 26
		key[i] = Types.TINYINT ;		msg[i++] = "TINYINT" ;			// 27
		key[i] = Types.VARBINARY ;		msg[i++] = "VARBINARY" ;		// 28
		key[i] = Types.VARCHAR ;		msg[i  ] = "VARCHAR" ;			// 29

		// それぞれ、Types 名称用のフィールドを追加
		String[][] table = new String[cnt+1][cnt+1];		// ヘッダー行を追加
		for( int hd=0; hd<cnt; hd++ ) {
			table[0][hd+1] = msg[hd];		// 行のヘッダー
			table[hd+1][0] = msg[hd];		// 列のヘッダー
		}

		for( int row=0; row<cnt; row++ ) {
			for( int clm=0; clm<cnt; clm++ ) {
				if( metaData.supportsConvert( key[row], key[clm]) ) {
					table[row+1][clm+1] = "○" ;
				}
				else {
					table[row+1][clm+1] = "×" ;
				}
			}
		}

		jspPrint( supportsConvertFunc + HybsSystem.BR );
		jspPrint( makeTableTag( table ) );
	}

	private static final String getCatalogSeparator = "このデータベースがカタログ名とテーブル名のセパレータとして使用する String" ;
	private static final String getCatalogTerm = "「catalog」に対するデータベースベンダーの推奨用語を取得します。 " ;
	private static final String getDatabaseProductName = "このデータベース製品の名前を取得します。 " ;
	private static final String getDatabaseProductVersion = "このデータベース製品のバージョンを取得します。 " ;
	private static final String getDriverMajorVersion = "この JDBC ドライバのメジャーバージョンを取得します。 " ;
	private static final String getDriverMinorVersion = "この JDBC ドライバのマイナーバージョンを取得します。 " ;
	private static final String getDriverName = "この JDBC ドライバの名前を取得します。 " ;
	private static final String getDriverVersion = "この JDBC ドライバのバージョンを String" ;
	private static final String getExtraNameCharacters = "引用符で囲まれていない識別名に使用できるすべての「特殊」文字 (a-z、A-Z、0-9、および _ 以外) を取得します。 " ;
	private static final String getIdentifierQuoteString = "SQL 識別子を引用するのに使用する文字列を取得します。 " ;
	private static final String getNumericFunctions = "このデータベースで使用可能なコンマで区切った数学関数のリストを取得します。 " ;
	private static final String getProcedureTerm = "「procedure」に対するデータベースベンダーの推奨用語を取得します。 " ;
	private static final String getSchemaTerm = "「schema」に対するデータベースベンダーの推奨用語を取得します。 " ;
	private static final String getSearchStringEscape = "ワイルドカード文字をエスケープするのに使用できる文字列を取得します。 " ;
	private static final String getSQLKeywords = "このデータベースの SQL キーワードであって、SQL92 のキーワードではない、すべてのキーワードをコンマで区切ったリストを取得します。 " ;
	private static final String getStringFunctions = "このデータベースで使用可能なコンマで区切った文字列関数のリストを取得します。 " ;
	private static final String getSystemFunctions = "このデータベースで使用可能なコンマで区切ったシステム関数のリストを取得します。 " ;
	private static final String getTimeDateFunctions = "このデータベースで使用可能な時間関数と日付関数をコンマで区切ったリストを取得します。 " ;
	private static final String getURL = "この DBMS の URL を取得します。 " ;
	private static final String getUserName = "このデータベースに記録されているユーザ名を取得します。 " ;

	private static final String getDefaultTransactionIsolation = "このデータベースのデフォルトのトランザクション遮断レベルを取得します。 " ;
	private static final String getSQLStateType = "Throwable.getSQLState によって返される SQLSTATE が X/Open (現在は Open Group) の SQL CLI であるか SQL99 であるかを示します。 " ;
	private static final String getResultSetHoldability = "この ResultSet オブジェクトのデフォルトの保持機能を取得します。 " ;

	private static final String allProceduresAreCallable = "getProcedures メソッドによって返されるすべてのプロシージャが、現在のユーザから呼び出せるかどうかを取得します。 " ;
	private static final String allTablesAreSelectable = "getTables メソッドによって返されるすべてのテーブルが、現在のユーザによって使用できるかどうかを取得します。 " ;
	private static final String dataDefinitionCausesTransactionCommit = "トランザクションのデータ定義文が、トランザクションを強制的にコミットさせるかどうかを取得します。 " ;
	private static final String dataDefinitionIgnoredInTransactions = "このデータベースによって、トランザクションでデータ定義文が無視されるかどうかを取得します。 " ;
	private static final String deletesAreDetected = "ResultSet.rowDeleted メソッドを呼び出すことによって可視の行が削除されたことを検出できるかどうかを取得します。 " ;
	private static final String doesMaxRowSizeIncludeBlobs = "getMaxRowSize メソッドの戻り値が SQL データの型の LONGVARCHAR および LONGVARBINARY を含むかどうかを取得します。 " ;
	private static final String insertsAreDetected = "ResultSet.rowInserted メソッドを呼び出すことによって可視の行が挿入されたことを検出できるかどうかを取得します。 " ;
	private static final String isCatalogAtStart = "完全修飾されたテーブル名の開始部分 (または終了部分) にカタログが現れるかどうかを取得します。 " ;
	private static final String isReadOnly = "このデータベースが読み込み専用モードかどうかを取得します。 " ;
	private static final String locatorsUpdateCopy = "LOB への変更が、コピーに対して行われたのか、LOB に直接行われたのかを示します。 " ;
	private static final String nullPlusNonNullIsNull = "このデータベースが、NULL 値と非 NULL 値の連結を NULL とするかどうかを取得します。 " ;
	private static final String nullsAreSortedAtEnd = "NULL 値が、終了時にソート順にかかわらずソートされるかどうかを取得します。 " ;
	private static final String nullsAreSortedAtStart = "NULL 値が、開始時にソート順にかかわらずソートされるかどうかを取得します。 " ;
	private static final String nullsAreSortedHigh = "NULL 値が高位にソートされるかどうかを取得します。 " ;
	private static final String nullsAreSortedLow = "NULL 値が下位にソートされるかどうかを取得します。 " ;
	private static final String othersDeletesAreVisible = "他で行われた削除が可視かどうかを取得します。 " ;
	private static final String othersInsertsAreVisible = "他で行われた挿入が可視かどうかを取得します。 " ;
	private static final String othersUpdatesAreVisible = "他で行われた更新が可視かどうかを取得します。 " ;
	private static final String ownDeletesAreVisible = "結果セット自身の削除が可視かどうかを取得します。 " ;
	private static final String ownInsertsAreVisible = "結果セット自身の挿入が可視かどうかを取得します。 " ;
	private static final String ownUpdatesAreVisible = "指定された ResultSet オブジェクトについて、結果セット自身の更新が可視かどうかを取得します。 " ;
	private static final String storesLowerCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 " ;
	private static final String storesLowerCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、小文字で格納するかどうかを取得します。 " ;
	private static final String storesMixedCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 " ;
	private static final String storesMixedCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字小文字混在で格納するかどうかを取得します。 " ;
	private static final String storesUpperCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 " ;
	private static final String storesUpperCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別しないで処理し、大文字で格納するかどうかを取得します。 " ;
	private static final String supportsAlterTableWithAddColumn = "このデータベースによって、追加列のある ALTER TABLE がサポートされるかどうかを取得します。 " ;
	private static final String supportsAlterTableWithDropColumn = "このデータベースによって、ドロップ列のある ALTER TABLE がサポートされるかどうかを取得します。 " ;
	private static final String supportsANSI92EntryLevelSQL = "このデータベースによって、ANSI92 エントリレベルの SQL 文法がサポートされるかどうかを取得します。 " ;
	private static final String supportsANSI92FullSQL = "このデータベースによって、ANSI92 完全レベルの SQL 文法がサポートされるかどうかを取得します。 " ;
	private static final String supportsANSI92IntermediateSQL = "このデータベースによって、ANSI92 中間レベルの SQL 文法がサポートされるかどうかを取得します。 " ;
	private static final String supportsBatchUpdates = "このデータベースによってバッチ更新がサポートされるかどうかを取得します。 " ;
	private static final String supportsCatalogsInDataManipulation = "データ操作文でカタログ名を使用できるかどうかを取得します。 " ;
	private static final String supportsCatalogsInIndexDefinitions = "インデックス定義文でカタログ名を使用できるかどうかを取得します。 " ;
	private static final String supportsCatalogsInPrivilegeDefinitions = "特権定義文でカタログ名を使用できるかどうかを取得します。 " ;
	private static final String supportsCatalogsInProcedureCalls = "プロシージャ呼び出し文でカタログ名を使用できるかどうかを取得します。 " ;
	private static final String supportsCatalogsInTableDefinitions = "テーブル定義文でカタログ名を使用できるかどうかを取得します。 " ;
	private static final String supportsColumnAliasing = "このデータベースによって列の別名がサポートされるかどうかを取得します。  " ;
	private static final String supportsConvertFunc = "このデータベースによって、SQL の型間の CONVERT 関数がサポートされるかどうかを取得します。 " ;
	private static final String supportsCoreSQLGrammar = "このデータベースによって、ODBC Core SQL 文法がサポートされるかどうかを取得します。 " ;
	private static final String supportsCorrelatedSubqueries = "このデータベースによって照合関係サブクエリーがサポートされるかどうかを取得します。 " ;
	private static final String supportsDataDefinitionAndDataManipulationTransactions = "このデータベースによって、トランザクションで、データ定義文とデータ操作文の両方がサポートされるかどうかを取得します。 " ;
	private static final String supportsDataManipulationTransactionsOnly = "このデータベースによって、トランザクションでデータ操作文だけがサポートされるかどうかを取得します。 " ;
	private static final String supportsDifferentTableCorrelationNames = "テーブル相互関係名がサポートされる場合、テーブルの名前と異なる名前であるという制限を付けるかどうかを取得します。 " ;
	private static final String supportsExpressionsInOrderBy = "このデータベースによって、ORDER BY リスト中で式がサポートされるかどうかを取得します。 " ;
	private static final String supportsExtendedSQLGrammar = "このデータベースによって、ODBC Extended SQL 文法がサポートされるかどうかを取得します。 " ;
	private static final String supportsFullOuterJoins = "このデータベースによって、完全入れ子の外部結合がサポートされるかどうかを取得します。 " ;
	private static final String supportsGetGeneratedKeys = "文が実行されたあとに自動生成キーを取得できるかどうかを取得します。 " ;
	private static final String supportsGroupBy = "このデータベースによって、GROUP BY 節のフォームがサポートされるかどうかを取得します。 " ;
	private static final String supportsGroupByBeyondSelect = "SELECT 文中のすべての列が GROUP BY 節に含まれるという条件で、このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " ;
	private static final String supportsGroupByUnrelated = "このデータベースによって、GROUP BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " ;
	private static final String supportsIntegrityEnhancementFacility = "このデータベースによって、SQL Integrity Enhancement Facility がサポートされるかどうかを取得します。 " ;
	private static final String supportsLikeEscapeClause = "このデータベースによって、LIKE エスケープ節の指定がサポートされるかどうかを取得します。 " ;
	private static final String supportsLimitedOuterJoins = "このデータベースによって、外部結合に関し、制限されたサポートが提供されるかどうかを取得します (supportsFullOuterJoins メソッドが true を返す場合は true)。 " ;
	private static final String supportsMinimumSQLGrammar = "このデータベースによって、ODBC Minimum SQL 文法がサポートされるかどうかを取得します。 " ;
	private static final String supportsMixedCaseIdentifiers = "このデータベースが、大文字小文字が混在する引用符なしの SQL 識別子を、大文字小文字を区別して処理し、大文字小文字混在で格納するかどうかを取得します。 " ;
	private static final String supportsMixedCaseQuotedIdentifiers = "このデータベースが、大文字小文字が混在する引用符付きの SQL 識別子を、大文字小文字を区別して処理し、結果として大文字小文字混在で格納するかどうかを取得します。 " ;
	private static final String supportsMultipleOpenResults = "CallableStatement オブジェクトから同時に返された複数の ResultSet オブジェクトを持つことが可能かどうかを取得します。 " ;
	private static final String supportsMultipleResultSets = "このデータベースによって、execute メソッドの単一の呼び出しからの複数の ResultSet オブジェクトの取得がサポートされるかどうかを取得します。 " ;
	private static final String supportsMultipleTransactions = "このデータベースが一度に複数のトランザクションを (異なった接続で) オープンできるかどうかを取得します。 " ;
	private static final String supportsNamedParameters = "このデータベースによって、callable 文への名前付きパラメータがサポートされるかどうかを取得します。 " ;
	private static final String supportsNonNullableColumns = "このデータベースの列を非 null として定義できるかどうかを取得します。 " ;
	private static final String supportsOpenCursorsAcrossCommit = "このデータベースによって、コミット間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 " ;
	private static final String supportsOpenCursorsAcrossRollback = "このデータベースによって、ロールバック間でカーソルがオープンされたままの状態がサポートされるかどうかを取得します。 " ;
	private static final String supportsOpenStatementsAcrossCommit = "このデータベースによって、コミット間で文がオープンされたままの状態がサポートされるかどうかを取得します。 " ;
	private static final String supportsOpenStatementsAcrossRollback = "このデータベースによって、ロールバック間で文がオープンされたままの状態がサポートされるかどうかを取得します。 " ;
	private static final String supportsOrderByUnrelated = "このデータベースによって、ORDER BY 節で SELECT 文中にない列の使用がサポートされるかどうかを取得します。 " ;
	private static final String supportsOuterJoins = "このデータベースによって、外部結合のなんらかの形式がサポートされるかどうかを取得します。 " ;
	private static final String supportsPositionedDelete = "このデータベースによって、位置指定された DELETE 文がサポートされるかどうかを取得します。 " ;
	private static final String supportsPositionedUpdate = "このデータベースによって、位置指定された UPDATE 文がサポートされるかどうかを取得します。 " ;
	private static final String supportsResultSetConcurrency = "このデータベースが、指定された結果セットの型と与えられた並行処理の種類の組み合わせをサポートするかどうかを取得します。 " ;
	private static final String supportsResultSetHoldability = "このデータベースが、指定された結果セットの保持機能をサポートするかどうかを取得します。 " ;
	private static final String supportsResultSetType = "このデータベースが、指定された結果セットの型をサポートするかどうかを取得します。 " ;
	private static final String supportsSavepoints = "このデータベースによってセーブポイントがサポートされるかどうかを取得します。 " ;
	private static final String supportsSchemasInDataManipulation = "データ操作文でスキーマ名を使用できるかどうかを取得します。 " ;
	private static final String supportsSchemasInIndexDefinitions = "インデックス定義文でスキーマ名を使用できるかどうかを取得します。 " ;
	private static final String supportsSchemasInPrivilegeDefinitions = "特権定義文でスキーマ名を使用できるかどうかを取得します。 " ;
	private static final String supportsSchemasInProcedureCalls = "プロシージャ呼び出し文でスキーマ名を使用できるかどうかを取得します。 " ;
	private static final String supportsSchemasInTableDefinitions = "テーブル定義文でスキーマ名を使用できるかどうかを取得します。 " ;
	private static final String supportsSelectForUpdate = "このデータベースによって SELECT FOR UPDATE 文がサポートされるかどうかを取得します。 " ;
	private static final String supportsStatementPooling = "このデータベースによって文のプールがサポートされるかどうかを取得します。 " ;
	private static final String supportsStoredProcedures = "このデータベースによって、ストアドプロシージャエスケープ構文を使用するストアドプロシージャコールがサポートされるかどうかを判定します。 " ;
	private static final String supportsSubqueriesInComparisons = "このデータベースによって、比較式中でサブクエリーがサポートされるかどうかを取得します。 " ;
	private static final String supportsSubqueriesInExists = "このデータベースによって、EXISTS 式中でサブクエリーがサポートされるかどうかを取得します。 " ;
	private static final String supportsSubqueriesInIns = "このデータベースによって、IN 文中でサブクエリーがサポートされるかどうかを取得します。 " ;
	private static final String supportsSubqueriesInQuantifieds = "このデータベースによって、定量化された式中でサブクエリーがサポートされるかどうかを取得します。 " ;
	private static final String supportsTableCorrelationNames = "このデータベースによってテーブル相互関係名がサポートされるかどうかを取得します。 " ;
	private static final String supportsTransactionIsolationLevel = "このデータベースが、指定されたトランザクション遮断レベルをサポートするかどうかを取得します。 " ;
	private static final String supportsTransactions = "このデータベースによってトランザクションがサポートされるかどうかを取得します。 " ;
	private static final String supportsUnion = "このデータベースによって SQL UNION がサポートされるかどうかを取得します。 " ;
	private static final String supportsUnionAll = "このデータベースによって SQL UNION ALL がサポートされるかどうかを取得します。 " ;
	private static final String updatesAreDetected = "ResultSet.rowUpdated メソッドを呼び出すことによって可視の行が更新されたことを検出できるかどうかを取得します。 " ;
	private static final String usesLocalFilePerTable = "このデータベースが、各テーブルにファイルを使用するかどうかを取得します。 " ;
	private static final String usesLocalFiles = "このデータベースが、ローカルファイルにテーブルを格納するかどうかを取得します。 " ;

	private static final String getAttributes = "指定されたスキーマおよびカタログで使用可能なユーザ定義の型 (UDT) のための指定された型の指定された属性に関する記述を取得します。  " ;
	private static final String getBestRowIdentifier = "行を一意に識別するテーブルの最適な列セットに関する記述を取得します。 " ;
	private static final String getCatalogs = "このデータベースで使用可能なカタログ名を取得します。 " ;
	private static final String getColumnPrivileges = "テーブルの列へのアクセス権に関する記述を取得します。  " ;
	private static final String getColumns = "指定されたカタログで使用可能なテーブル列の記述を取得します。  " ;
	private static final String getCrossReference = "指定された主キーテーブルの主キー列を参照する指定された外部のキーテーブル中の、外部のキー列に関する記述 (テーブルが別のキーをインポートする方法を記述) を取得します。 " ;
	private static final String getExportedKeys = "指定されたテーブルの主キー列 (テーブルによってエクスポートされた外部キー) を参照する外部キー列に関する記述を取得します。 " ;
	private static final String getImportedKeys = "テーブルの外部キー列 (テーブルによってインポートされる主キー) を参照する主キー列に関する記述を取得します。 " ;
	private static final String getIndexInfo = "指定されたテーブルのインデックスと統計情報に関する記述を取得します。 " ;
	private static final String getPrimaryKeys = "指定されたテーブルの主キー列の記述を取得します。 " ;
	private static final String getProcedureColumns = "指定されたカタログのストアドプロシージャパラメータと結果列に関する記述を取得します。  " ;
	private static final String getProcedures = "指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。  " ;
	private static final String getSchemas = "このデータベースで使用可能なスキーマ名を取得します。 " ;
	private static final String getSuperTables = "このデータベースの特定のスキーマで定義されているテーブル階層の説明を取得します。  " ;
	private static final String getSuperTypes = "このデータベースの特定のスキーマで定義されているユーザ定義型 (UDT) 階層の説明を取得します。 " ;
	private static final String getTablePrivileges = "カタログで使用可能な各テーブルに対するアクセス権に関する記述を取得します。 " ;
	private static final String getTables = "指定されたカタログで使用可能なテーブルに関する記述を取得します。 " ;
	private static final String getTableTypes = "このデータベースで使用可能なテーブルの型を取得します。 " ;
	private static final String getTypeInfo = "このデータベースでサポートされているすべての標準 SQL の型に関する記述を取得します。 " ;
	private static final String getUDTs = "特定のスキーマで定義されているユーザ定義型 (UDT) の説明を取得します。 " ;
	private static final String getVersionColumns = "行の任意の値が変更された場合に、自動的に更新されるテーブルの列に関する記述を取得します。 " ;

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "dbid"			,dbid			)
				.println( "catalog"			,catalog		)
				.println( "schema"			,schema			)
				.println( "attributeName"	,attributeName	)
				.println( "columnName"		,columnName		)
				.println( "procedureName"	,procedureName	)
				.println( "tableName"		,tableName		)
				.println( "typeName"		,typeName		)
				.println( "rowNo"			,rowNo			)
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
