/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.report2.DocConverter_OOO;

/**
 * ドキュメントの変換・マージを行うタグです。
 *
 * 変換を行うことのできる入出力のフォーマット以下の通りです。
 * 
 * [対応フォーマット]
 *  入力[Calc(ods)   ,Excel(xls)     ] -> 出力[Calc(ods)   ,Excel(xls)     ,PDF]
 *  入力[Writer(odt) ,Word(doc)      ] -> 出力[Writer(odt) ,Word(doc)      ,PDF]
 *  入力[Impress(odp),PowerPoint(ppt)] -> 出力[Impress(odp),PowerPoint(ppt),PDF]
 *  
 * 入出力のフォーマットは、入出力ファイルの拡張子により自動判別されます。
 * また、入出力ファイル名が同じ場合は何も処理されません。
 * 
 * 入力ファイルを、カンマ区切りで複数指定することで、複数の入力ファイルをマージして
 * 出力することもできます。
 * 
 * ※2 現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
 * ※1 この機能を利用するには、OpenOfficeが正しく設定されている必要があります。
 *
 * @og.formSample
 * ●形式：&lt;og:docConvert fileURL="…" inputFile="…" outputFile="…" /&gt;
 * ●body：なし
 *
 * ●使用例
 *    ・Calc(ods)ファイルをPDFに変換
 *        &lt;og:docConvert inputFile="temp.ods" outputFile="out.pdf" /&gt;
 *
 *    ・Excel(xls)ファイルを連結
 *        &lt;og:docConvert inputFile="temp1.xls,temp2.xls" outputFile="out.xls" /&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class DocConvertTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.6.0 (2010/05/06)" ;

	private static final long serialVersionUID = 5160 ;

	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String[]inputFile	= null;
	private String	outputFile	= null;
	private boolean	delInput	= false;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		return ( SKIP_BODY );
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		// 出力ファイルで拡張子だけが指定された場合は、入力ファイル名を付加する
		if( outputFile.indexOf( '.' ) < 0 ) {
			int inSufIdx = inputFile[0].lastIndexOf( '.' );
			if( inSufIdx >= 0 ) {
				outputFile = inputFile[0].substring( 0, inSufIdx ) + "." + outputFile;
			}
			
		}

		String directory = HybsSystem.url2dir( fileURL );
		String[] input = new String[inputFile.length];
		for( int i=0; i<input.length; i++ ) {
			input[i] = StringUtil.urlAppend( directory,inputFile[i] );
		}
		String output = StringUtil.urlAppend( directory,outputFile );

		// ドキュメント変換
		DocConverter_OOO.convert( input, output );

		if( delInput ) {
			for( int i=0; i<input.length; i++ ) {
				if( !( new File( input[i] ) ).delete() ) {
					System.err.println( "入力ファイルの削除に失敗しました。file=[" + input + "]" );
				}
			}
		}
		
		return( EVAL_PAGE );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		fileURL 	= HybsSystem.sys( "FILE_URL" );
		inputFile	= null;
		outputFile	= null;
		delInput	= false;
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します(初期値:システムパラメータのFILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下を、使用します。
	 *
	 * @param	url ファイルURL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】入力ファイル名を指定します。
	 *
	 * @og.tag
	 * 入力ファイル名を指定します。
	 * 
	 * 入力ファイルは、カンマ区切りで複数指定することができます。
	 * この場合、複数の入力ファイルをマージして出力を行います。
	 * 
	 * ※現状は、ファイルのマージは、入力ファイルがExcelまたはCalcの場合のみ対応しています。
	 *   また、マージを行う場合、印刷範囲など、ドキュメントに関連する情報は、1つ目のファイルの
	 *   情報が使用されます。
	 *
	 * @param	fname 入力ファイル名
	 */
	public void setInputFile( final String fname ) {
		inputFile = getCSVParameter( fname );
	}

	/**
	 * 【TAG】出力ファイル名を指定します。
	 *
	 * @og.tag
	 * 出力ファイル名を指定します。
	 * 出力ファイルには、拡張子のみ(xls,ods等)を指定することもできます。
	 * この場合、出力ファイル名は、入力ファイル名と同じになります。(拡張子のみが変換される)
	 *
	 * @param	fname 出力ファイル名
	 */
	public void setOutputFile( final String fname ) {
		outputFile = nval( getRequestParameter( fname ),outputFile );
	}

	/**
	 * 【TAG】ドキュメント変換後、元のファイルを削除するかどうかを指定します。
	 *
	 * @og.tag
	 * ドキュメント変換後、元のファイルを削除するかどうかを指定します。
	 * (初期値:false(削除しない))
	 *
	 * @param	flg 出力元のファイルを削除するかどうか名
	 */
	public void setDelInput( final String flg ) {
		delInput = nval( getRequestParameter( flg ),delInput );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "fileURL"		,fileURL	)
				.println( "inputFile"	,inputFile		)
				.println( "outputFile"	,outputFile		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
