/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.XHTMLTag;

// import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * プルダウンメニューの選択項目を作成するHTML拡張タグです。
 *
 * name 属性は、ラベルリソース のキーを与えることで、使用する上位のタグの
 * ロケールにあわせたリソースを使用して、画面に表示します。
 * 従って、このタグでは ロケールは指定できません。
 * selected属性は、そのタグが選ばれている場合を、"true" で指定します。 初期値は、"false" です。
 *
 * @og.formSample
 * ●形式：&lt;og:option value="…" lbl ="…" selected="…" /&gt;
 * ●body：なし
 *
 * ●使用例
 *     プルダウンで選択する 値(value) に対して ラベル(lbl) を付けられます。
 *     &lt;og:select name="ORDER_BY" msg="ORDER_BY"&gt;
 *         &lt;og:option value="SYSTEM_ID,CLM,LANG" lbl ="CLM" selected="selected" /&gt;
 *     &lt;/og:select&gt;
 *
 *     ラベルが複数ある場合は、lbls 属性を使用します。
 *     &lt;og:select name="ORDER_BY" msg="ORDER_BY"&gt;
 *         &lt;og:option value="SYSTEM_ID,CLM,LANG" lbls="LANG,NAME_JA" /&gt;
 *     &lt;/og:select&gt;
 *
 *     MessageResource プロパティの値を使用したいとききはmsg属性を使います。
 *     &lt;og:select name="CDC"&gt;
 *         &lt;og:option msg="MSG0001" /&gt;
 *     &lt;/og:select&gt;
 *
 *     LabelResource プロパティの値を使用したいとききはlbl属性を使います。
 *     &lt;og:select name="CDC"&gt;
 *         &lt;og:option lbl="CDC1" /&gt;
 *     &lt;/og:select&gt;
 *
 *     選択肢の中から複数選択できるようにするときはmultiple属性を使います。
 *     &lt;og:select name="CDC" multiple="multiple"&gt;
 *         &lt;og:option value="AAA" /&gt;
 *     &lt;/og:select&gt;
 *
 *     選択不可にするときはdisabled属性を使います。
 *     &lt;og:select name="CDC" disabled="disabled"&gt;
 *         &lt;og:option value="AAA" /&gt;
 *     &lt;/og:select&gt;
 *
 *     選択肢をSELECT文の結果から作成したいときはqueryOptionタグと組み合わせて使います。
 *     &lt;og:select name="CDC"&gt;
 *         &lt;og:queryOption&gt;
 *         &lt;jsp:text&gt;
 *                     select NOSYN,NOSYN,':',NMSYN from DB01 ORDER BY 1
 *         &lt;/jsp:text&gt;
 *         &lt;/og:queryOption&gt;
 *     &lt;/og:select&gt;
 *
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class OptionTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 3562 ;	// 3.5.6.2 (2004/07/05)

//	private String code = null;

	// 3.5.4.0 (2003/11/25) selVal 属性を追加。
	private String selVal = null;
	private boolean multipleAll	= false;	// 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.0 (2003/11/25) selVal 属性を追加。
	 * @og.rev 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。
	 * @og.rev 4.3.0.0 (2008/07/04) ロールモードマルチ対応
	 * @og.rev 5.0.2.0 (2009/11/01) 複数パラメーターの選択に対応
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 4.0.0 (2005/11/30) 指定のロールが、アクセス許可の場合は、出力します。
		String roles = get( "roles" );
//		if( roles != null && ! getUser().isAccess( roles ) ) {
//			return(EVAL_PAGE);
//		}
		// 4.3.0.0 (2008/07/04) ロールモードマルチ対応
		if( ! getUser().isAccess( roles ) ) {
			return(EVAL_PAGE);
		}

		SelectTag select = (SelectTag)findAncestorWithClass( this,SelectTag.class );
		if( select == null ) {
			String errMsg = "このタグは、SelectTag のBODY に記述する必要があります。";
			throw new HybsSystemException( errMsg );
		}
//		selVal = select.getValue();			// 3.5.4.0 (2003/11/25)
		selVal = "|" + select.getValue() + "|";	// 5.0.2.0 (2009/11/01)
		multipleAll	= select.isMultipleAll();	// 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。
		select.addOption( makeTag() );

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.0 (2003/11/25) selVal 属性を追加。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
//		code		= null;
		selVal		= null;
		multipleAll	= false;	// 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。
	}

	/**
	 * ラベルを作成します。
	 *
	 * lbl 属性／msg 属性でセットされた場合は，そちらを優先します。
	 * セットされていない場合は，value 属性をキーに、
	 * LabelResource プロパティの値をセットします。
	 * value 属性に、カンマ(,)で区切られた複数の Label を
	 * セットできます。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) selVal 属性を追加。
	 * @og.rev 3.5.5.7 (2004/05/10) DBColumn.getOption( String ) メソッド廃止
	 * @og.rev 3.8.0.9 (2005/10/17) 複数選択可能時に全選択を設定する。
	 * @og.rev 5.0.2.0 (2009/11/01) 複数パラメーターの選択に対応
	 *
	 * @return  オプションタグのラベル
	 */
	protected String makeTag() {
//		if( code != null ) {
//			String val = nval( get( "value" ),selVal );
//
//			// 次期バージョンでは、削除したいと思います。
//			String tag = getDBColumn( code ).getEditorValue( val );
//			int st  = tag.indexOf( '>' );
//			int end = tag.lastIndexOf( "</select" );
//			return tag.substring( st+1,end );
//		}

		// 3.5.4.0 (2003/11/25) selVal 属性を追加。
//		if( ( selVal != null && selVal.length() > 0 && selVal.equals( get( "value" ) ) ) || multipleAll ) {
		// 5.0.2.0 (2009/11/01)
		if( ( selVal != null && selVal.length() > 0 && selVal.indexOf( "|" + get( "value" ) + "|" ) >= 0 ) || multipleAll ) {
			set( "selected","selected" );
		}

		String msglbl = getMsglbl();
		if( msglbl != null ) {
			set( "body", msglbl );
		}

		return XHTMLTag.option( getAttributes() );
	}

	/**
	 * 【TAG】値を指定します。
	 *
	 * @og.tag
	 * ここで指定した値がプルダウンメニュー中に存在する場合、選択状態になります。
	 *
	 * @param   val 値を指定
	 */
	public void setValue( final String val ) {
		set( "value",getRequestParameter( val ) );
	}

	/**
	 * 【TAG】optionタグのラベルを指定します。
	 *
	 * @og.tag
	 * ここでのラベルは、optionタグのラベルです。(lbl属性は、異なります。)
	 * これは、optgroup とともに使用される階層化メニュー時に使用されます。
	 *
	 * @param   label String
	 */
	public void setLabel( final String label ) {
		set( "label",getRequestParameter( label ) );
	}

	/**
	 * 【TAG】オプションを選択済みの状態(selected)にセットします(初期値:未選択)。
	 *
	 * @og.tag
	 * selected="selected" または selected="true" 以外の値はセットできないように
	 * 制限をかけます。
	 *
	 * @param   sel 選択済み "selected"  ／未選択（初期値） それ以外
	 */
	public void setSelected( final String sel ) {
		String select = getRequestParameter( sel );
		if( "selected".equalsIgnoreCase( select ) || "true".equalsIgnoreCase( select ) ) {
			set( "selected","selected" );
		}
	}

//	/**
//	 * 【廃止】コードリソースのコードID を指定します。
//	 *
//	 * @og.tag
//	 * コードリソースで定義されている内容を再利用します。基本的には、カラムリソースで、
//	 * コードカラム属性を定義して使用してください。
//	 *
//	 * @og.rev 4.0.0 (2007/06/18) 廃止
//	 *
//	 * @param   code String
//	 * @deprecated
//	 */
//	@Deprecated public void setCode( final String code ) {
//		this.code = getRequestParameter( code ) ;
//	}

	/**
	 * 【TAG】ラベルをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * シングルラベルの lbl 属性との違いは,ここではラベルを複数 カンマ区切りで
	 * 渡すことが可能であることです。これにより、"A,B,C" という値に対して、
	 * "Aのラベル表示,Bのラベル表示,Cのラベル表示" という具合に文字列を
	 * 再合成します。
	 * これは、SQL文のOrdr By 句で、ソート順を指定する場合などに便利です。
	 * &lt;og:option lbls="MKNMJP,MKCD,MKNMEN" /&gt;
	 *
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 *
	 * @param   lbls 複数ラベルID（カンマ区切り）
	 */
	public void setLbls( final String lbls ) {

		String[] keys = getCSVParameter( lbls );
		if( keys == null || keys.length == 0 ) { return ; }

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		buf.append( getLabel( keys[0] ) );
		for( int i=1; i<keys.length; i++ ) {
			buf.append( "," );
			buf.append( getLabel( keys[i] ) );
		}
		setMsglbl( buf.toString() );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
//				.println( "code"		,code		)
				.println( "selVal"		,selVal		)
				.println( "multipleAll"	,multipleAll)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
