/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

import java.util.StringTokenizer ;

/**
 * AUTOAREA エディターは、カラムのデータをテキストエリアで編集する場合に
 * 使用するクラスです。
 *
 * エリアの大きさは、表示する文字列によって、自動的に変更されます。
 * 初期値や、文字数が小さい場合のサイズは、デフォルト値を使用するか、
 * 編集パラメータに、x,y形式で指定された値を使います。
 * 列方向での最大桁数を、指定することが可能です。システムパラメータで
 * HTML_AUTOAREA_MAX_COL_SIZE を指定することで、折り返し列数の調整も
 * 同時に行われます。0 が指定された場合は、無制限になります。
 * HTML_AUTOAREA_MAX_ROW_SIZE を指定することで、行数の最大値を
 * 指定することが可能です。0 が指定された場合は、無制限になります。
 * 編集パラメータは、『行,列』指定可能です。例えば、5,10 とすると、５行１０列の
 * テキストエリアを最小範囲として設定できます。
 * デフォルトは、HTML_COLUMS_MAXSIZE で指定の列数と、定義されているデータサイズ
 * 割る HTML_COLUMS_MAXSIZE で、５を超えない値を、行数としています。
 * 編集パラメータの、『行,列』指定で、同時に、最大行列数の指定も可能です。
 * 5-10,15-20 とすると、最小５行－最大１０行で、最小１５列－最大２０列の
 * テキストエリアを指定できます。編集パラメータでの最大値指定は、
 * システムパラメータでの最大値指定より、優先されます。
 *
 * このエディターでは、カラムの内容に応じて、書込み禁止属性を強制的に付与する
 * 事も可能です。
 * value (検索結果）の先頭１文字目が、アンダーバー（_) の場合は、
 * 編集モードになりません。（読取専用）
 * データベースに書き込むときには、通常のアンダーバー無しの文字列に変換して登録します。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.8.0.2 (2005/06/30) 新規追加。
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_AUTOAREA extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final int COL = 0 ;
	private static final int ROW = 1 ;
	private static final String CODE = "Windows-31J";

	private	int	cols1	= 0 ;
	private	int	cols2	= 0 ;
	private	int	rows1	= 0 ;
	private	int	rows2	= 0 ;
	private int maxColSize = HybsSystem.sysInt( "HTML_AUTOAREA_MAX_COL_SIZE" );
	private int maxRowSize = HybsSystem.sysInt( "HTML_AUTOAREA_MAX_ROW_SIZE" );

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_AUTOAREA() {
		// 4.3.4.4 (2009/01/01)
//		super();
	}

	/**
	 * コンストラクター。
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Editor_AUTOAREA( final DBColumn clm ) {
		super( clm );
		String  disabled = clm.isWritable() ? null : "disabled" ;

		int r1 = (clm.getTotalSize()/Integer.parseInt(size1)) + 1;	// 4.0.0 (2005/01/31) メソッド名変更
		if( r1 > 5 ) { rows1 = 5; }
		else { rows1 = r1; }

		int r2 = (clm.getTotalSize()/Integer.parseInt(size2)) + 1;	// 4.0.0 (2005/01/31) メソッド名変更
		if( r2 > 5 ) { rows2 = 5; }
		else { rows2 = r2; }

		// 3.8.0.2 (2005/07/11) size に、"rows-maxRow,cols-maxCols" を指定
		String param = StringUtil.nval( clm.getEditorParam(),clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			int prmAdrs = param.indexOf( ',' );
			if( prmAdrs > 0 ) {
				String rowStr = param.substring( 0,prmAdrs );
				String colStr = param.substring( prmAdrs+1 );

				int rowAdrs = rowStr.indexOf( '-' );	// rows-maxRow 設定時 '-' がなければ、ただのrows
				if( rowAdrs > 0 ) {
					rows1 = Integer.parseInt( rowStr.substring( 0,rowAdrs ) );
					maxRowSize = Integer.parseInt( rowStr.substring( rowAdrs+1 ) );
				}
				else {
					rows1 = Integer.parseInt( rowStr );
				}
				rows2 = rows1 ;

				int colAdrs = colStr.indexOf( '-' );	// cols-maxCols 設定時 '-' がなければ、ただのcols
				if( colAdrs > 0 ) {
					cols1 = Integer.parseInt( colStr.substring( 0,colAdrs ) );
					maxColSize = Integer.parseInt( colStr.substring( colAdrs+1 ) );
				}
				else {
					cols1 = Integer.parseInt( colStr );
				}
				cols2 = cols1;
			}
		}

		attributes = new Attributes();
		attributes.set( "disabled" ,disabled );

		attributes.addAttributes( clm.getEditorAttributes() );
		attributes.add( "class"  ,clm.getDbType() );		// 4.0.0 (2005/01/31)

		optAttr = attributes.get( "optionAttributes" );
		tagBuffer.add( XHTMLTag.textareaAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellEditor オブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_AUTOAREA( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 * 
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   value String
	 * @return  データの編集用文字列
	 */
	public String getValue( final String value ) {
//		if( value != null && value.length() >= 1 && value.charAt(0) == '_' ) {
//			return value.substring( 1 );
//		}

		int[] rowcol = getRowsCols( value,cols1,rows1 );

		TagBuffer tag = new TagBuffer( "textarea" );
		tag.add( "name"    , name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , name ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( "cols"    , String.valueOf( rowcol[COL] ) );
		tag.add( "rows"    , String.valueOf( rowcol[ROW] ) );
		if( maxRowSize > 0 ) {
			tag.add( "onKeyup" ,"autoArea( this," + maxRowSize + " );" );
		}
		else {
			tag.add( "onKeyup" ,"autoArea( this );" );
		}
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)
		tag.setBody( value );
		
		return tag.makeTag();
//		return tag.makeTag() + createEventColumnJS( name, editor, -1, eventURL ); // 4.3.6.0 (2009/04/01)
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 * 
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   row   int 行番号
	 * @param   value String
	 * @return  データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {
//		if( value != null && value.length() >= 1 && value.charAt(0) == '_' ) {
//			return value.substring( 1 );
//		}

		int[] rowcol = getRowsCols( value,cols2,rows2 );

		TagBuffer tag = new TagBuffer( "textarea" );
		String newName = name + HybsSystem.JOINT_STRING + row;
		// tag.add( "name"    , name + HybsSystem.JOINT_STRING + row );
		tag.add( "name"     , newName );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"    , newName ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( "cols"    , String.valueOf( rowcol[COL] ) );
		tag.add( "rows"    , String.valueOf( rowcol[ROW] ) );
		if( maxRowSize > 0 ) {
			tag.add( "onKeyup" ,"autoArea( this," + maxRowSize + " );" );
		}
		else {
			tag.add( "onKeyup" ,"autoArea( this );" );
		}
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)
		tag.setBody( value );

		return tag.makeTag( row,value );
//		return tag.makeTag( row,value ) + createEventColumnJS( name, editor, row, eventURL ); //4 3.6.0 (2009/04/01)
	}

	/**
	 * 自動表示する行列の数を求めます。
	 * 行数は、引数の文字列中に含まれる 改行コードの個数を求めます。
	 * 列数は、各行数のなかの最大桁数より求めます。これには半角、全角が含まれる為、
	 * 半角換算での文字数ということになります。
	 * 行数と列数が、初期設定の行数と列数より小さい場合は、初期設定値が使用されます。
	 *
	 * @param   cols int 最小カラム数
	 * @param   rows int 最小行数
	 * @param   value String 表示文字列
	 * @return  自動計算した行列の配列
	 */
	private int[] getRowsCols( final String value,final int cols, final int rows ) {
		if( value == null ) {
			return new int[] { rows,cols };
		}

		StringTokenizer token = new StringTokenizer( value, "\n", true );

		int cntRow = 1;
		int maxCol = 0;
		while ( token.hasMoreTokens() ) {
			String val = token.nextToken();
			if( "\n".equals( val ) ) { cntRow++; }
			else {
				byte[] byteValue = StringUtil.makeByte( val,CODE );	// 3.5.5.3 (2004/04/09)
				int byteSize = byteValue.length;
				if( maxColSize > 0 && byteSize > maxColSize ) {		// 最大列数
					cntRow += (byteSize / maxColSize);
					maxCol = maxColSize ;
				}
				else if( byteSize > maxCol ) { maxCol = byteSize; }
			}
			if( maxRowSize > 0 && cntRow >= maxRowSize ) {		// 最大行数
				cntRow = maxRowSize;
				break;
			}
		}

		maxCol += 2;	// マージン。フォントや画面サイズに影響する為、比率のほうがよい？

		int[] rtn = new int[2];
		rtn[ROW] = (rows<cntRow) ? cntRow : rows ;
		rtn[COL] = (cols<maxCol) ? maxCol : cols ;

		return rtn ;
	}
}
