/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * TEXTAREA エディターは、カラムのデータをテキストエリアで編集する場合に
 * 使用するクラスです。
 *
 * 従来との違いは、cols 属性の最大値を、検索時(query画面)では、HTML_COLUMNS_MAXSIZE を、
 * 登録時(result画面)では、HTML_VIEW_COLUMNS_MAXSIZE を使用します。
 * エリアの大きさは、デフォルト値を使用するか、編集パラメータに、x,y形式で
 * 指定します。
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 * なお、このクラスには、初めの文字が _(アンダーバー)の場合は書込み禁止になる仕様が
 * 標準的に導入されています。
 *
 * @og.rev 4.0.0 (2005/01/31) 新規作成
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_TEXTAREA extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private  String		  rows1		 ;
	private  String		  rows2		 ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 *
	 */
	public Editor_TEXTAREA() {
		// 4.3.4.4 (2009/01/01)
//		super();
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.2.4.0 (2003/06/12) パラメータより、行列（row,column）情報があれば、その値を利用する。
	 * @og.rev 3.3.1.1 (2003/07/03) name , attributes 属性を final にする。
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.5.5.0 (2004/03/12) 漢字入力（IMEモード）をONにするのを、"K" と、"KX" のみとします。
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 TEXTAREA_KEY を隠蔽します。
	 * @og.rev 4.0.0 (2005/01/31) DBColumn の getClassName() を getDbType() に変更
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Editor_TEXTAREA( final DBColumn clm ) {
		super( clm );
		String  disabled = clm.isWritable() ? null : "disabled" ;

		int r1 = (clm.getTotalSize()/Integer.parseInt(size1)) + 1;	// // 4.0.0 (2005/01/31) メソッド名変更
		if( r1 > 5 ) { rows1 = "5"; }
		else { rows1 = String.valueOf( r1 ); }

		int r2 = (clm.getTotalSize()/Integer.parseInt(size2)) + 1;	// // 4.0.0 (2005/01/31) メソッド名変更
		if( r2 > 5 ) { rows2 = "5"; }
		else { rows2 = String.valueOf( r2 ); }

		// 3.7.0.4 (2005/03/14) size に、"rows,cols" を指定できるように変更
		String param = StringUtil.nval( clm.getEditorParam(),clm.getViewLength() );
		if( param != null && param.length() != 0 ) {
			int st = param.indexOf( ',' );
			if( st > 0 ) {
				rows1 = param.substring( 0,st );
				rows2 = rows1 ;
				size1 = param.substring( st+1 );
				size2 = size1;
			}
		}

		attributes = new Attributes();
		attributes.set( "disabled" ,disabled );

		attributes.addAttributes( clm.getEditorAttributes() );
		attributes.add( "class" ,clm.getDbType() );		// 4.0.0 (2005/01/31)

		optAttr = attributes.get( "optionAttributes" );
		tagBuffer.add( XHTMLTag.textareaAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellEditor オブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_TEXTAREA( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 * 
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   value String
	 * @return  データの編集用文字列
	 */
	public String getValue( final String value ) {
//		if( value != null && value.length() >= 1 && value.charAt(0) == '_' ) {
//			return value.substring( 1 );
//		}

		TagBuffer tag = new TagBuffer( "textarea" );
		tag.add( "name"    , name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , name ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( "cols"    , size1 );
		tag.add( "rows"    , rows1 );
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)
		tag.setBody( value );

		return tag.makeTag();
//		return tag.makeTag() + createEventColumnJS( name, editor, -1, eventURL ); // 4.3.6.0 (2009/04/01)
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 名前と行番号の区切り記号を "^" から "__" に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.2.0 (2010/01/01) 先頭の'_'による書き込み制御を行わない。(他のクラスとの実装の共通化)
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param   row   int 行番号
	 * @param   value String
	 * @return  データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {
//		if( value != null && value.length() >= 1 && value.charAt(0) == '_' ) {
//			return value.substring( 1 );
//		}

		TagBuffer tag = new TagBuffer( "textarea" );
		String newName = name + HybsSystem.JOINT_STRING + row;
		// tag.add( "name"    , name + HybsSystem.JOINT_STRING + row );
		tag.add( "name"    , newName );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , newName );
		}
		tag.add( "cols"    , size2 );
		tag.add( "rows"    , rows2 );
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)
		tag.setBody( value );

		return tag.makeTag( row,value );
//		return tag.makeTag( row,value ) + createEventColumnJS( name, editor, row, eventURL ); // 4.3.6.0 (2009/04/01)
	}
}
