/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * NUMBER レンデラーは、カラムのデータを数字表示する場合に使用するクラスです。
 *
 * フォーマットには、java.text.NumberFormat を使用せずに、独自クラスを使用しており
 * double 以上の精度をもつ値でも正確に変換できます。
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_NUMBER extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final String	defValue ;
	private final int		minFraction;

	private static final CellRenderer[] dbCell = {
							new Renderer_NUMBER(),
							new Renderer_NUMBER("0",1),
							new Renderer_NUMBER("0",2)
	};

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。
	 * @og.rev 4.0.0.0 (2007/11/29) 初期値を""に変更。
	 *
	 */
	public Renderer_NUMBER() {
		defValue   = ""; // 4.0.0.0 (2007/11/29)
		minFraction = 0;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。NumberFormatクラスは、廃止します。
	 *
	 * @param   defval String 初期値
	 * @param   size   int    小数点
	 */
	private Renderer_NUMBER( final String defval,final int size ) {
		defValue    = defval;
		minFraction = size;
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellRenderer オブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		String defval = clm.getDefault();
		int    size     = clm.getSizeY();
		if( ( defval == null || defval.length() == 0 || defval.equals( "0" ) )
			&& size < dbCell.length ) {
				return dbCell[size];
		}

		return new Renderer_NUMBER( defval,size );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 内部に、DBColumn オブジェクトをキープしないように変更
	 * @og.rev 3.3.0.0 (2003/06/23) NumberFormatクラスは、廃止します。
	 *
	 * @param	value String
	 * @return	データの表示用文字列
	 */
	public String getValue( final String value ) {
		String rtn ;
		if( value == null || (value.trim()).length() == 0 ) {
			rtn = defValue;
		}
		else {
			rtn = value;
		}

		rtn = StringUtil.numberFormat( rtn,minFraction );
		if( rtn != null && (rtn.trim()).length() > 0 && rtn.charAt( 0 ) == '-' ) {
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}
		return rtn;
	}
}
