/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import org.opengion.hayabusa.io.TableReader;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.CSVTokenizer;

import java.io.BufferedReader;
import java.io.IOException;

/**
 * 指定の区切り記号（初期値：タブ区切り）ファイルの読み取りクラスです。
 *
 * 名前，データの入力部のみオーバーライドすれば，各種入力フォーマットに合わせた
 * サブクラスを実現する事が可能です。
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableReader_Default implements TableReader {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String	separator	= TAB_SEPARATOR;		// 項目区切り文字
	private ResourceManager resource = null;			// 4.0.0 (2005/01/31)
	private int		maxRowCount	= HybsSystem.sysInt( "DB_MAX_ROW_COUNT" ) ;

	protected DBTableModel	table		= null;
	protected DBColumn[]	dbColumn	= null;

	// 3.5.4.5 (2004/01/23) カラム名の外部指定を出来る様にする。
//	private String	  columns	= null;	 // 外部指定のカラム名
	protected String  columns	= null;	 // 外部指定のカラム名 ( 4.3.4.7 (2009/01/22) protectedに変更 )
	private String	  encode	= null;
	private boolean	  useNumber	= true;		// 3.7.0.5 (2005/04/11)

	private int		skipRowCount	= 0;	// 5.1.6.0 (2010/05/01) データの読み飛ばし設定

	/**
	 * デフォルトコンストラクター
	 *
	 */

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @see #isExcel()
	 */
	public void readDBTable() {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.4.2 (2003/12/15) writer の null チェックを廃止します。
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取ル要に変更します。
	 * @og.rev 3.5.4.5 (2004/01/23) カラム名の外部指定を優先して使用する。
	 * @og.rev 5.1.6.0 (2010/05/01) readDBTableのエラーチェック強化
	 * @og.rev 5.1.6.0 (2010/05/01) skipRowCountの追加
	 *
	 * @param   reader BufferedReader
	 */
	public void readDBTable( final BufferedReader reader ) {
		try {
			String line;
			String[] names = null;
			int numberOfRows = 0;
			char  sepa = separator.charAt( 0 );

			boolean nameNoSet = true;
			table = DBTableModelUtil.newDBTable();

			// 3.5.4.5 (2004/01/23) カラム名の外部指定を優先して使用する。
			if( columns != null && columns.length() > 0 ) {
				names = StringUtil.csv2Array( columns );
				table.init( names.length );
				setTableDBColumn( names ) ;
				nameNoSet = false;
			}

			int skip = skipRowCount;						// 5.1.6.0 (2010/05/01)
			while((line = reader.readLine()) != null) {
				if( skip > 0 ) { skip--; continue; }		// 5.1.6.0 (2010/05/01)
				if( line.length() == 0 ) { continue; }
				if( line.charAt( 0 ) == '#' ) {
					String key = line.substring( 0,5 );
					if( nameNoSet && ( key.equalsIgnoreCase( "#NAME" ) )) {
						// 超イレギュラー処理 最初の TAB_SEPARATOR 以前の文字は無視する。
						String line2 = line.substring( line.indexOf( sepa )+1 );
						names = StringUtil.csv2Array( line2 ,sepa );
						table.init( names.length );
						setTableDBColumn( names ) ;
						nameNoSet = false;
					}
					else  { continue; }
				}
				else {
					if( nameNoSet ) {
						String errMsg = "#NAME が見つかる前にデータが見つかりました。";
						throw new HybsSystemException( errMsg );
					}
					if( numberOfRows < getMaxRowCount() ) {
						table.addColumnValues( readData( line,names.length ) );
						numberOfRows ++ ;
					}
					else {
						table.setOverflow( true );
					}
				}
			}

			// 5.1.6.0 (2010/05/01) readDBTableのエラーチェック強化
			if( nameNoSet ) {
				String errMsg = "ファイルから有効なデータが見つかりませんでした。";
				throw new HybsSystemException( errMsg );
			}
		}
		catch ( IOException ex ) {
			String errMsg = "ファイル読込みエラー[" + reader + "]"  ;
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
	}

	/**
	 * リソースマネージャーをセットします。
	 * これは、言語（ロケール）に応じた DBColumn をあらかじめ設定しておく為に
	 * 必要です。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、内部で DBColumn オブジェクトを作成します。
	 *
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @param  resource リソースマネージャー
	 */
	public void setResourceManager( final ResourceManager resource ) {
		this.resource = resource;
	}

	/**
	 * DBColumn オブジェクトをDBTable に設定します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) private を protected に変更。
	 * @og.rev 3.5.4.5 (2004/01/23) DBColumn 配列に値をセットします。
	 *
	 * @param names String[]
	 */
	protected void setTableDBColumn( final String[] names ) {
		dbColumn = new DBColumn[names.length] ;	 // 3.5.4.5 追加
		for( int i=0; i<names.length; i++ ) {
			DBColumn clm = resource.makeDBColumn( names[i] );
			table.setDBColumn( i,clm );
			dbColumn[i] = clm;		  // 3.5.4.5 追加
		}
	}

	/**
	 * BufferedReader より読み込んだ１行のデータを テーブルモデルに
	 * セットするように分割します
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 *
	 * @og.rev 3.3.3.1 (2003/07/18) ファイルリード/ライト時に後ろスペースの除去を行います。
	 * @og.rev 3.7.0.5 (2005/04/11) useNumber 属性を考慮します。
	 *
	 * @param   data String
	 * @param   clmSize int
	 * @return  String[]
	 */
	protected String[] readData( final String data,final int clmSize ) {
		String[] rtnData = new String[ clmSize ];
		CSVTokenizer token = new CSVTokenizer( data,separator.charAt(0) );
		// 超イレギュラー処理 最初の separator 以前の文字は無視する。
		// 3.7.0.5 (2005/04/11)
		if( useNumber ) { token.nextToken(); }	  // 先頭は行番号のため無視する。

		int clmNo = 0;
		while( token.hasMoreTokens() ) {
			String val = StringUtil.csvOutQuote( token.nextToken() );
			if( val != null && val.startsWith( "'0" ) ) {
				rtnData[clmNo++] = StringUtil.rTrim( val.substring( 1 ) );
			}
			else {
				rtnData[clmNo++] = StringUtil.rTrim( val );
			}
			if( clmNo >= clmSize ) { break; }	// 3.7.0.5 (2005/04/11) 多い場合は、以降を無視する。
		}
		// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
		for( int i=clmNo; i<clmSize; i++ ) {
			rtnData[i] = "";
		}

		return rtnData;
	}

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return  DBTableModel
	 */
	public DBTableModel getDBTableModel() {
		return table;
	}

	/**
	 * データを読み込む場合の,区切り文字をセットします。
	 *
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param   sep 区切り文字
	 */
	public void setSeparator( final String sep ) {
		if( sep != null ) { this.separator = sep; }
	}

	/**
	 * データを書き込む場合の,区切り文字を返します。
	 *
	 * @return  separator 区切り文字
	 */
	public String getSeparator() {
		return separator;
	}

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @return  最大検索件数
	 */
	public int getMaxRowCount() {
		return maxRowCount;
	}

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 * @param   maxRowCount int
	 */
	public void setMaxRowCount( final int maxRowCount ) {
		this.maxRowCount = maxRowCount;
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * デフォルトは、第一シートです。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName String
	 */
	public void setSheetName( final String sheetName ) {
		String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return boolean
	 */
	public boolean isExcel() {
		return false;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	public void setFilename( final String filename ) {
		String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 読み取り元ファイルのカラム列を、外部（タグ）より指定します。
	 * ファイルに記述された #NAME より優先して使用されます。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   clms 読み取り元ファイルのカラム列（カンマ区切り文字）
	 */
	public void setColumns( final String clms ) {
		columns = clms ;
	}

	/**
	 * 読み取り元ファイルのエンコード文字列を指定します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。（例えば、半角文字は、Shift_JIS では、１バイト）
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   enc ファイルのエンコード文字列
	 */
	public void setEncode( final String enc ) {
		encode = enc;
	}

	/**
	 * 読み取り元ファイルのエンコード文字列を取得します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。（例えば、半角文字は、Shift_JIS では、１バイト）
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @return String ファイルのエンコード文字列
	 */
	protected String getEncode() {
		return encode;
	}

	/**
	 * 行番号情報を、使用している(true)/していない(false)を指定します。
	 *
	 * @og.tag
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等）では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * デフォルトは、true（使用する） です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 *
	 * @param useNumber boolean 行番号情報を、使用している(true)/していない(false)を指定
	 */
	public void setUseNumber( final boolean useNumber ) {
		this.useNumber = useNumber ;
	}

	/**
	 * データの読み始めの初期値を取得します。
	 *
	 * @og.tag
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。（１と指定すると、１件読み飛ばし、２行目から読み込みます。）
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @return	int	skipRowCount 読み始めの初期値
	 */
	public int getSkipRowCount() {
		return skipRowCount ;
	}

	/**
	 * データの読み飛ばし件数を設定します。
	 *
	 * @og.tag
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。（１と指定すると、１件読み飛ばし、２行目から読み込みます。）
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	count 読み始めの初期値
	 */
	public void setSkipRowCount( final int count ) {
		skipRowCount = count;
	}

	/**
	 * 行番号情報を、使用している(true)/していない(false)を返します。
	 *
	 * @og.tag
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等）では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * デフォルトは、true（使用する） です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 * @og.rev 4.0.0 (2007/07/20) メソッド名変更(getUseNumber() ⇒  isUseNumber())
	 *
	 * @return   useNo 行番号情報を、使用している(true)/していない(false)を指定
	 */
	protected boolean isUseNumber() {
		return useNumber ;
	}
}
