/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;

/**
 * ネイティブEXCELファイル(表示種別(RENDERER)適用)の書き出しクラスです。
 * 
 * TableWriter_Excelを継承しており、
 * データ作成部分のみオーバーライドしています。
 * 
 * 出力形式は、openXML形式にも対応しています。
 * 出力ファイルの拡張子が、.xlsならExcel2003のバイナリ形式、.xlsxならExcel2007の
 * openXML形式で出力されます。
 *
 * @og.group ファイル出力
 *
 * @og.rev 4.3.4.3 (2008/12/22) 追加
 * @og.rev 4.3.6.7 (2009/05/22) ooxml形式対応
 * 
 * @version  4.0
 * @author	 Takahashi Masakazu
 * @since    JDK5.0,
 */
public class TableWriter_ExcelRenderer extends TableWriter_Excel {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション（"）で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 5.1.4.0 (2010/03/01) columns 対応
	 * @og.rev 5.1.4.0 (2010/03/01) useNumber属性対応
	 *
	 * @param	table DBTableModel
	 * @param	writer PrintWriter
	 */
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		int numberOfRows =	table.getRowCount();

		short nColIndex;
		Row  oRow;
		boolean useNumber = isUseNumber();

		boolean[] nvar = new boolean[numberOfColumns];
		int[] cellType = new int[numberOfColumns];
		for( int i=0; i<numberOfColumns; i++ ) {
			int clm = clmNo[i];
			NativeType nativeType = dbColumn[clm].getNativeType();
			switch( nativeType ) {
				case INT    :
				case LONG   :
				case DOUBLE :
					cellType[i] = Cell.CELL_TYPE_NUMERIC ;
						break;
				case STRING :
				case CALENDAR :
				default :
						cellType[i] = Cell.CELL_TYPE_STRING ;
						break;
			}
			nvar[i] = "NVAR".equals( dbColumn[clm].getDbType()) ;
		}

		for( int row=0; row<numberOfRows; row++ ) {
			nColIndex = 0;
			oRow = setFirstCellValue( nRowIndex++, nColIndex++, String.valueOf( row+1 ) );

			// 5.1.4.0 (2010/03/01) useNumber=false の場合は、nColIndex を一つ戻す。
			if( !useNumber ) {
				nColIndex-- ;
			}

			for( int i=0; i<numberOfColumns; i++ ) {
				int clm = clmNo[i];
				String val = table.getValue(row,clm);
				if( nvar[i] ) {
					val = StringUtil.getReplaceEscape( val );
				}
				else {
					val = StringUtil.spanCut( dbColumn[clm].getRendererValue( val ) );
				}

				setRowCellValue( oRow, nColIndex++, val,cellType[i] );
			}

			// 余計なセルを削除
			removeRowCell( oRow, nColIndex );
		}
	}
}
