/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
// import java.util.StringTokenizer;

// import org.opengion.hayabusa.common.HybsSystem;
// import org.opengion.hayabusa.common.HybsSystemException;

/**
 * SystemParameter は、{&#064;XXXX} 文字列を処理するクラスです。
 * このクラスでは、{&#064;XXXX} 文字列を別の文字列と置き換えることや、
 * 予め予約されている予約語 {&#064;SYS.XXXX} 文字列を置き換えます。
 * 通常の {&#064;XXXX} 文字列の置き換えは、キーと値のペアを、HybsEntry オブジェクトに
 * セットして、その配列を受け取って処理します。
 *
 * 以下の値はあらかじめ、動的に作成されます。
 * ・SYS.YMD       ８byte の今日のシステム日付(yyyyMMdd)
 * ・SYS.YMDH    １４byte の今日のシステム日時(yyyyMMddHHmmss)
 * ・SYS.HMS       ６byte の今日のシステム時間(HHmmss)
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class SystemParameter {

//	private final Map<String,String> sysMap = new HashMap<String,String>();
	/** 改行コード */
	public final static String CR = System.getProperty("line.separator");	// 5.1.9.0 (2010/08/01) 追加

	private final String original ;

	private final String[] clms;
	private final String[] formats;

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 * 
	 * @og.rev 5.1.8.0 (2010/07/01) パース方法見直し(StringTokenizerでは、{@xxx}が連続してある場合に対応できない)
	 * @og.rev 5.3.2.0 (2011/02/01) original データを、パース結果を利用するように修正する。
	 *
	 * @param orig String  変換する文字列(オリジナル)
	 */
	public SystemParameter( final String orig ) {
//		original = orig;			// 5.3.2.0 (2011/02/01) original は、パース後に作成します。

		if( orig == null || orig.length() == 0 || orig.indexOf( "{@" ) < 0 ) {
			clms = null;
			formats = null;
			original = orig;		// 5.3.2.0 (2011/02/01)
		}
		else {
			StringBuilder buf = new StringBuilder(orig.length());		// 5.3.2.0 (2011/02/01)

			DateFormat formatter = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
			String ymdh = formatter.format( new Date() );

			ArrayList<String> formatList = new ArrayList<String>();
			ArrayList<String> clmList = new ArrayList<String>();

			// 5.1.8.0 (2010/07/01) パース方法見直し
			int start = 0;
			int index = orig.indexOf( "{@" );
			while( index >= 0 ) {
				buf.append( orig.substring( start, index ) );		// 5.3.2.0 (2011/02/01)
				formatList.add( orig.substring( start, index ) );
				int end = orig.indexOf( '}',index );
				if( end < 0 ) {
//					String errMsg = "{@ と } との対応関係がずれています。" + HybsSystem.CR
					String errMsg = "{@ と } との対応関係がずれています。" + CR
								+ "str=[" + orig + "] : index=" + index ;
					throw new RuntimeException( errMsg );
				}
				String param = orig.substring( index+2,end );
//				if( param.startsWith( "SYS.YMD" ) ) {
				if( param.equalsIgnoreCase( "SYS.YMD" ) ) {
					clmList.add( ymdh.substring( 0,8 ) );
					buf.append( ymdh.substring( 0,8 ) );		// 5.3.2.0 (2011/02/01)
				}
//				else if( param.startsWith( "SYS.YMDH" ) ) {
				else if( param.equalsIgnoreCase( "SYS.YMDH" ) ) {
					clmList.add( ymdh );
					buf.append( ymdh );		// 5.3.2.0 (2011/02/01)
				}
//				else if( param.startsWith( "SYS.HMS" ) ) {
				else if( param.equalsIgnoreCase( "SYS.HMS" ) ) {
					clmList.add( ymdh.substring( 8 ) );
					buf.append( ymdh.substring( 8 ) );		// 5.3.2.0 (2011/02/01)
				}
				else {
					clmList.add( param );
					buf.append( param );		// 5.3.2.0 (2011/02/01)
				}
				start = end+1;
				index = orig.indexOf( "{@",start );
			}
			formatList.add( orig.substring( start, orig.length() ) );
			buf.append(orig.substring( start, orig.length() ) );		// 5.3.2.0 (2011/02/01)

			clms	= clmList.toArray( new String[clmList.size()] );
			formats = formatList.toArray( new String[formatList.size()] );
			original = buf.toString();		// 5.3.2.0 (2011/02/01)
		}
	}

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @param  entry   HybsEntry 置換文字列のキーと値のペアを管理しているEntryオブジェクトの配列
	 * @return String  置換後の文字列
	 */
	public String replace( final HybsEntry[] entry ) {
//		if( orginal == null ) { return null; }
//		int index = orginal.indexOf( "{@" );
//		if( index < 0 ) { return orginal; }
		if( formats == null || clms == null ) { return original; }
		if( entry == null || entry.length == 0 ) { return original; }

		// HybsEntry[] データを、Mapにマッピングします。
		Map<String, String> sysMap = new HashMap<String, String>();
//		if( entry != null ) {
			int size = entry.length;
			for( int i=0; i<size; i++ ) {
				sysMap.put( entry[i].getKey(),entry[i].getValue() );
			}
//		}
		return replace( sysMap );
	}

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @param  map  置換文字列のキーと値のペアを管理しているMapオブジェクト
	 * @return String  置換後の文字列
	 */
	public String replace( final Map<String,String> map ) {
		if( formats == null || clms == null ) { return original; }
//		if( map == null || map.size() == 0 ) { return original; }
		if( map == null || map.isEmpty() ) { return original; }

		StringBuilder sb = new StringBuilder();
		for( int i=0; i<formats.length; i++ ) {
			sb.append( formats[i] );
			if( i < clms.length ) {
				sb.append(  StringUtil.nval( map.get( clms[i] ), "" ) );
			}
		}

		return sb.toString();
	}

	/**
	 * フォーマットをパースした結果から、カラム一覧を配列形式で返します。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * 
	 * @return カラム配列
	 */
	public String[] getColumns() {
		if( clms == null ) { return new String[0]; }
		return clms.clone();
	}

	/**
	 * フォーマットをパースした結果から、フォーマット一覧を配列形式で返します。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * 
	 * @return フォーマット配列
	 */
	public String[] getFormats() {
		if( formats == null ) { return new String[0]; }
		return formats.clone();
	}
}
