/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.common;

import java.io.File;

/**
 * リソースファイルの代わりとなるシステムパラメータをクラス化したデータクラスです。
 *
 * それぞれの設定値は、データベースにコメントとともに設定されます。
 * そのとき、og.paramLevel タグにより、属性を設定しています。
 *
 * 0:設定必須
 * 1:設定確認
 * 2:使用時設定
 * 3:標準推奨
 * 4:個人設定可
 * 5:追加設定
 * 7:要再起動
 * 8:設定不可
 * 9:自動設定
 *
 * og.cryptography タグは、指定の属性値を暗号化するかどうかを指定します。
 * 暗号化は、org.opengion.hayabusa.security.HybsCryptography クラスで行います。将来的な拡張機能として、
 * 各システム毎に暗号化キーを変えることで、セキュリティ強化が可能です。
 *
 * エンジンパラメータデータは、エンジンとともに配布されるマスタデータになります。
 * よって、配布形式はどのような形でもよく、今回は、JavaDoc にて各パラメータの
 * 意味やデフォルト設定値が判断できるように、クラスの public static final 変数として
 * 定義しました。
 * 従来は、resource.システムパラメータ の一般設定用の システムパラメータ ファイルと
 * エンジン内部で持っている org/opengion/hayabusa/resource/properties の
 * システムパラメータ ファイルにより維持されていましたが、エンジンパラメータ
 * 定義テーブル(GE12)と、commom/SystemData.java クラスによる管理に
 * 変更されました。
 * システムパラメータは、ＤＢへのアクセスを決定するため、初期設定値を定義する必要があります。
 * これは、、アプリケーション・ディスクリプタ(WEB-INF/web.xml)に、context-param として、
 * キーと値のセットで、初期アクセス用の情報を渡します。
 * システムパラメータ定義テーブル(GE12)には、SYSTEM_IDとして、通常のシステムIDと、
 * エンジンパラメータがあります。エンジンパラメータは、SYSTEM_ID='**'として、登録
 * されています。
 *
 * SystemData クラスのフィールドは、その記述した順番でセットしますので、
 * 設定順(フィールドの記述順)を考慮する必要があります。
 * また、データベースへの登録は、このクラスをJavaDocのカスタムドックレットにより
 * 一旦 XML ファイルに変換後、オラクルXDK形式のXMLファイルにXSLTで変換し、
 * このパッケージ(org.opengion.hayabusa.common)に配置します。
 * このデータの変換時にコメント情報を付加したXMLファイルを作成し、DB登録(GE12)しますので、
 * このデータを参考に、システム個別に書き換えを行えます。
 *
 * @og.rev 4.0.0 (2005/01/31) 新規作成
 * @og.group 初期化
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class SystemData {

	/**
	 * インスタンスを作らないので、コンストラクタは、private に設定します。
	 */
	private SystemData() {}

	/*
	 * ======================================================================
	 * これは，本システム全般に渡って,使用されるリソースです。
	 * システムとしての初期値や，設定値などは，すべてここで登録されます。
	 * このリソースを修正する場合は,注意して下さい。
	 * 不要な個所(値)を記入したり，記入漏れが発生すると,システムが
	 * 正常に起動しない場合があり得ますので,ご注意願います。
	 * ======================================================================
	 */

	/*
	 * ======================================================================
	 * システム毎に設定する必要のある変数
	 * ======================================================================
	 */

	/**
	 * アドレス(/)に対する、実ディレクトリパス。
	 * REAL_PATH = "G:\webapps\ge\" ;
	 * @og.paramLevel 9:自動設定
	 */
	public static final String REAL_PATH = "" ;

	/**
	 * アクセス先の仮想フォルダ名(URLのコンテキスト名)。
	 * CONTEXT_NAME = "ge" ;
	 * @og.paramLevel 9:自動設定
	 */
	public static final String CONTEXT_NAME = "" ;

	/**
	 * アクセス先のJSPフォルダ名(/URLのコンテキスト名/jsp)。
	 * JSP = "/ge/jsp" ;
	 * @og.paramLevel 9:自動設定
	 */
	public static final String JSP = "" ;

	/**
	 * 共通メールサーバーＵＲＬ。
	 * COMMON_MAIL_SERVER = "localhost" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String COMMON_MAIL_SERVER = "" ;

	/**
	 * メールデーモンデフォルトユーザー。
	 * MAIL_DAEMON_DEFAULT_USER = "opengion" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String MAIL_DAEMON_DEFAULT_USER = "opengion" ;

	/**
	 * メールデーモンデフォルトパスワード。
	 * MAIL_DAEMON_DEFAULT_PASS = "****" ;
	 * @og.paramLevel 3:標準推奨
	 * @og.cryptography 1:暗号化
	 */
	public static final String MAIL_DAEMON_DEFAULT_PASS = "password" ;

	/**
	 * メールデーモンで実際に処理する 受信リスナー。
	 * org.opengion.hayabusa.mail.MailReceiveListener を継承している必要があります。
	 * MAIL_RECEIVE_LISTENER = "org.opengion.hayabusa.report.ExcelInsertReceiveListener" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String MAIL_RECEIVE_LISTENER = "org.opengion.hayabusa.report.ExcelInsertReceiveListener" ;

	/**
	 * システムエラー時 メール発信元。
	 * エラーメールを返信する場合に使用する FROM アドレス。
	 * 未設定の場合は、エラーメールを送信しません。
	 * ERROR_MAIL_FROM_USER = "opengion@users.sourceforge.jp" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String ERROR_MAIL_FROM_USER = "opengion@users.sourceforge.jp" ;

	/**
	 * システムエラー時 メール送信者。
	 * 複数存在する場合は、カンマで区切って指定します。
	 * 未設定の場合は、送信しません。
	 * ERROR_MAIL_TO_USERS = "opengion@users.sourceforge.jp" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String ERROR_MAIL_TO_USERS = "" ;

	/**
	 * ファイル出力 基準ＵＲＬ(ファイル出力する実フォルダ)。
	 * FILE_URL = "filetemp/" ;
	 * FILE_URL = "G:/webapps/webdav/" ;
	 * FILE_URL = "\\\\hnXXXX/ge/filetemp/" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String FILE_URL = "filetemp/" ;

	/**
	 * ファイル出力 基準ＵＲＬ(ExcelOut でのリターンURL)。
	 * FILE_OUT_URL = "../../filetemp/" ;
	 * FILE_OUT_URL = "G:/webapps/webdav/" ;
	 * FILE_OUT_URL = "http://hnXXXX:8823/ge/filetemp/" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String FILE_OUT_URL = "../../filetemp/" ;

	/**
	 * デフォルトファイル名。
	 * FILE_FILENAME = "file.xls" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String FILE_FILENAME = "file.xls" ;

	/**
	 * 本番環境のＵＲＬ。
	 * WebDAV での接続や、リソース移行時などに利用します。
	 * HONBAN_URL = "" ;
	 * @og.paramLevel 1:設定確認
	 */
	public static final String HONBAN_URL = "" ;

	/**
	 * RDC_CODE(設計変更連絡番号)の先頭文字。
	 * リソース管理で使用する RDC_CODE は、分散環境(本社、中国、フィリピンなど)
	 * でそれぞれ採番された後に、1箇所で共通に使用されます。それぞれの拠点で
	 * 採番ルールを変えることで、重複を防ぎ、発生元を明確にします。
	 * RDC_PREFIX = "RDC" ;
	 * @og.paramLevel 1:設定確認
	 */
	public static final String RDC_PREFIX = "RDC" ;

	/**
	 * ヘルプファイル 基準ＵＲＬ。
	 * HELP_URL = "help/" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HELP_URL = "help/" ;

	/**
	 * エラーメッセージ表示画面。
	 * ERR_MSG_FILENAME = "jsp/common/errorMessage.jsp" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String ERR_MSG_FILENAME = "jsp/common/errorMessage.jsp" ;

	/**
	 * エラーメッセージ表示レベル。
	 * 設定により、以下のようにエラーメッセージの表示内容を変更することができます。
	 * (※ここでのエラーメッセージとは、例外がスローされた場合のエラーメッセージの表示方法です
	 *    columnCheckタグ等で表示されるアプリケーションエラーの内容は、この設定と関係ありません)
	 * 
	 * 0:エラー内容は何も表示されません。エラー発生時刻のみが表示されます。
	 * 1:例外名とエラー内容のメッセージのみが表示されます。スタックとレースは表示されません。
	 * 2:スタックトレースも含めてエラー内容を表示します。
	 * 
	 * ERR_DISP_LEVEL = "1" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String ERR_DISP_LEVEL = "1" ;

	/**
	 * 帳票ファイル出力 基準ＵＲＬ。
	 * 設定されていない場合は、FILE_URL + /REPORT/ に設定されます。
	 * REPORT_FILE_URL = "filetemp/REPORT/" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_FILE_URL = "filetemp/REPORT/" ;

	/**
	 * EXCEL取込時に使用するテンポラリ基準ＵＲＬ。
	 * 設定されていない場合は、FILE_URL + /EXCELIN/ に設定されます。
	 * EXCEL_IN_FILE_URL = "filetemp/EXCELIN/" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String EXCEL_IN_FILE_URL = "filetemp/EXCELIN/" ;

	/**
	 * 帳票エンコーディング。
	 * 設定されていない場合は、UTF-8 に設定されます。
	 * この値は、固定値ですので、変更しないでください。
	 * REPORT_ENCODE = "UTF-8" ;
	 * @og.paramLevel 8:設定不可
	 */
	//	public static final String REPORT_ENCODE = "UTF-8" ;

	/**
	 * 帳票デーモンの Shell 実行時のタイムアウト時間 。
	 * 帳票デーモンで起動する Shell が、何らかの原因で無応答になる場合があります。
	 * この場合、それ以降の処理がすべて停止してしまうため、タイムアウト時間を指定します。
	 * 単位は、秒です。
	 * 初期値は、３００秒です。
	 * REPORT_DAEMON_TIMEOUT = "300" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_DAEMON_TIMEOUT = "300" ;

	/**
	 * 帳票デーモンに、エラー時にスリープして待機する時間(秒) 。
	 * この設定時間だけスリープし、再び起動を試みます。そして失敗すれば、
	 * 再び、スリープして待機します。これを、無限に繰り返します。
	 * スリープ時間(秒)に、０ をセットすると、エラー時即終了となります。
	 * 初期値は、３０秒です。
	 * REPORT_DAEMON_ERROR_SLEEP_TIME = "30" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_DAEMON_ERROR_SLEEP_TIME = "30" ;

	/**
	 * HTMLからEXCELへ変換するときに要求番号でオープンするように指定します。
	 * Office 2003のEXCELより、XXXX.html と XXXX.files の XXXX 部分が異なる場合でも
	 * 画像ファイルを読み込むようになりました。従来は、この XXXX を同一にする為、
	 * 帳票ID でファイルをオープンしていましたが、EXCELを同時に起動できない為、
	 * 排他制御をエンジン側で行っていました。
	 * このフラグを true に設定すると、XXXX を要求番号にして、EXCELの同時起動を
	 * 行えるようにします。同時に、帳票IDでのロックを行いません。
	 * 初期値は、false (互換性の関係)です。
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_EXCEL_NAME_USE_YKNO = "false" ;

	/**
	 * アプリケーションのURLアクセスに関してのセキュリティ制限レベルを指定します。
	 * URLHashFilter フィルターを使用する設定(web.xml)を行う必要があります。
	 * 
	 * レベル０：なにも制限はありません。
	 * レベル１：Referer チェックを行います。つまり、URLを直接入力しても動作しません。
	 * レベル２：URLのハッシュ化/暗号化処理を、外部URLに対してのみ行います。(チェックは、レベル１と同等)
	 * レベル３：URLのパラメータがハッシュ化/暗号化されている必要があります。
	 * それ以外：アクセスを停止します。
	 *
	 * @og.rev 5.2.3.0 (2010/12/01) 新規追加
	 *
	 * URL_ACCESS_SECURITY_LEVEL = "0" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String URL_ACCESS_SECURITY_LEVEL = "0" ;

	/*
	 * ======================================================================
	 * これは，データベース接続先情報に関するリソースです。
	 * ドライバー、ＵＲＬ，ユーザー、パスワードのほかに、キャッシュする
	 * コネクション数、最大同時接続数、コネクション取得待機時間等を設定
	 * します。
	 * ======================================================================
	 */

	/**
	 * データベースエンコーディング。
	 * オラクルのエンコーディング名ではなく、Javaのエンコーディング名で指定します。
	 * Javaとオラクル間の文字コード変換は、JDBCが自動で行います。
	 * ここでの設定は、データベース登録時の文字バイト数チェックに利用しています。
	 *
	 * DB_ENCODE = "Shift_JIS" ;
	 * DB_ENCODE = "MS932" ;
	 * DB_ENCODE = "Windows-31J" ;
	 * DB_ENCODE = "UTF-8" ;
	 *
	 * @og.paramLevel 1:設定確認
	 */
	public static final String DB_ENCODE = "UTF-8" ;

	/**
	 * 接続データベースのドライバー。
	 * 複数登録する場合は,キーを "DB_DRIVER" + "_番号" でユニークにします。
	 *
	 * DB_DRIVER  = "oracle.jdbc.OracleDriver" ;
	 * DB_DRIVER3 = "com.microsoft.sqlserver.jdbc.SQLServerDriver" ;
	 * DB_DRIVER4 = "org.postgresql.Driver" ;
	 * DB_DRIVER5 = "com.mysql.jdbc.Driver" ;
	 * DB_DRIVER6 = "org.firebirdsql.jdbc.FBDriver" ;
	 * DB_DRIVER7 = "org.hsqldb.jdbcDriver" ;
	 *
	 * @og.rev 4.0.0.0 コネクションの設定値はfukurou/db/EDbid内です
	 *
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String DB_DRIVER = "oracle.jdbc.OracleDriver" ;
//	public static final String DB_DRIVER = "";// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * 接続データベースの設定  DEFAULT dbid。
	 * DBIDキーワード + 各設定キーワードで検索します。
	 * 設定キーワード は、URL,USER,PASSWD,最大プール数、初期接続数 です。
	 *
	 * DEFAULT_DB_URL = "jdbc:oracle:thin:@localhost:1521:HYBS" ;
	 * DEFAULT_DB_USER = "GE" ;
	 * DEFAULT_DB_PASSWD = "GE" ;
	 *
	 */

	/**
	 * データベース接続時のＵＲＬ。
	 * JDBCで接続にいく場合の、JDBC接続先のURLを指定します。
	 * web.xml で初期値指定する場合は、値がセットされます。
	 * デフォルト以外のDBIDで指定する場合の接続先は、XX_DB_URL で設定します。
	 * (DEFAULTの箇所に、DBIDで使用する文字、通常接続先システムIDを使用します。)
	 *
	 * DEFAULT_DB_URL = "jdbc:oracle:oci:@HYBS" ;
	 * DEFAULT_DB_URL = "jdbc:oracle:thin:@localhost:1521:HYBS" ;
	 * DEFAULT_DB_URL = "jdbc:postgresql://localhost:5432/hybs" ;
	 * DEFAULT_DB_URL = "jdbc:mysql://localhost/DBGE?useUnicode=true&characterEncoding=SJIS" ;
	 * DEFAULT_DB_URL  = jdbc:hsqldb:hsql://localhost
	 *
	 * @og.paramLevel 0:設定必須
	 */
//	 public static final String DEFAULT_DB_URL = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * データベース接続時のユーザー。
	 * データベースに接続するときのログインユーザー名を指定します。
	 * デフォルト以外のDBIDで指定する場合の接続先は、XX_DB_USER で設定します。
	 * (DEFAULTの箇所に、DBIDで使用する文字、通常接続先システムIDを使用します。)
	 * DEFAULT_DB_USER = "GE" ;
	 * @og.paramLevel 0:設定必須
	 */
//	 public static final String DEFAULT_DB_USER = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * データベース接続時のパスワード。
	 * データベースに接続するときのパスワードを指定します。
	 * デフォルト以外のDBIDで指定する場合の接続先は、XX_DB_PASSWD で設定します。
	 * (DEFAULTの箇所に、DBIDで使用する文字、通常接続先システムIDを使用します。)
	 * DEFAULT_DB_PASSWD = "****" ;
	 * @og.paramLevel 0:設定必須
	 * @og.cryptography 1:暗号化
	 */
//	 public static final String DEFAULT_DB_PASSWD = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * コネクションプールの最小数。
	 * コネクションプールの最小数 は、予めコネクションを準備しておくコネクションの数です。
	 * データベースとの接続を、プールしておき、必要つど貸し出します。
	 * ある程度の接続人数が判っている場合は、最少数を設定しておきます。
	 * デフォルト以外のDBIDで指定する場合の最小数は、XX_DB_MINCOUNT で設定します。
	 * (DEFAULTの箇所に、DBIDで使用する文字、通常接続先システムIDを使用します。)
	 * DEFAULT_DB_MINCOUNT = "3" ;
	 *
	 * @og.rev 4.0.0.0 コネクションの設定値はfukurou/db/EDbid内です
	 *
	 * @og.paramLevel 1:設定確認
	 */
//	 public static final String DEFAULT_DB_MINCOUNT = "3" ;
//	 public static final String DEFAULT_DB_MINCOUNT = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * コネクションプールの最大数。
	 * コネクションプールの最小数 は、同時接続要求に対する上限のコネクション数です。
	 * 同時接続がこの数を超えると、MissingResourceException が、投げられますので、
	 * 後程再接続する必要があります。(自動的に空きができれば再接続を試みます。)
	 * 同時接続数がある程度わかっており、データベースに余裕がある場合は、この値を
	 * 大きくしておいても構いません。接続は、DB_MAX_CONNECTION_POOL_TIME で指定された
	 * 時間が経過したコネクションは、クローズしていきます。(次回接続時のタイミングで)
	 * デフォルト以外のDBIDで指定する場合の接続先は、XX_DB_MAXCOUNT で設定します。
	 * (DEFAULTの箇所に、DBIDで使用する文字、通常接続先システムIDを使用します。)
	 * DEFAULT_DB_MAXCOUNT = "30" ;
	 *
	 * @og.rev 4.0.0.0 コネクションの設定値はfukurou/db/EDbid内です
	 *
	 * @og.paramLevel 1:設定確認
	 */
//	 public static final String DEFAULT_DB_MAXCOUNT = "30" ;
//	 public static final String DEFAULT_DB_MAXCOUNT = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * DEFAULTでないコネクションプールの初期最小数。
	 * デフォルト以外のDBIDで指定する場合の最小数は、XX_DB_MINCOUNT で設定します。
	 * この指定が省略された場合の初期値を指定します。
	 * DB_MINCOUNT = "1" ;
	 *
	 * @og.rev 4.0.0.0 コネクションの設定値はfukurou/db/EDbid内です
	 *
	 * @og.paramLevel 3:標準推奨
	 */
//	 public static final String DB_MINCOUNT = "1" ;
//	 public static final String DB_MINCOUNT = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * DEFAULTでないコネクションプールの初期最大数。
	 * デフォルト以外のDBIDで指定する場合の最大数は、XX_DB_MAXCOUNT で設定します。
	 * この指定が省略された場合の初期値を指定します。
	 * DB_MAXCOUNT = "5" ;
	 *
	 * @og.rev 4.0.0.0 コネクションの設定値はfukurou/db/EDbid内です
	 *
	 * @og.paramLevel 3:標準推奨
	 */
//	 public static final String DB_MAXCOUNT = "5" ;
//	 public static final String DB_MAXCOUNT = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * データベース製品の名前。
	 * 接続情報(メタデータ)より、データベース製品の名前を取得します。
	 * この接続情報は、DEFAULT接続に関する情報です。
	 * DatabaseMetaData#getDatabaseProductName() より、自動取得されます。
	 *
	 * @og.paramLevel 9:自動設定
	 */
//	 public static final String DATABASE_PRODUCT_NAME = "" ;	// 4.0.0.0 (2007/10/10)
//	 public static final String DB_PRODUCT_NAME = "" ;

	/**
	 * データベース製品のバージョン。
	 * 接続情報(メタデータ)より、データベース製品のバージョンを取得します。
	 * この接続情報は、DEFAULT接続に関する情報です。
	 * DatabaseMetaData#getDatabaseProductVersion() より、自動取得されます。
	 *
	 * @og.paramLevel 9:自動設定
	 */
//	 public static final String DATABASE_PRODUCT_VERSION = "" ;	// 4.0.0.0 (2007/10/10)
//	 public static final String DB_PRODUCT_VERSION = "" ;

	/**
	 * JDBC ドライバの名前。
	 * 接続情報(メタデータ)より、JDBC ドライバの名前を取得します。
	 * この接続情報は、DEFAULT接続に関する情報です。
	 * DatabaseMetaData#getDriverName() より、自動取得されます。
	 *
	 * @og.paramLevel 9:自動設定
	 */
//	 public static final String JDBC_DRIVER_NAME = "" ;

	/**
	 * JDBC ドライバのバージョン。
	 * 接続情報(メタデータ)より、JDBC ドライバのバージョンを取得します。
	 * この接続情報は、DEFAULT接続に関する情報です。
	 * DatabaseMetaData#getDriverVersion() より、自動取得されます。
	 *
	 * @og.paramLevel 9:自動設定
	 */
//	 public static final String JDBC_DRIVER_VERSION = "" ;

	/*
	 * ======================================================================
	 * Ver 3.5.2.0 (2003/10/20)  内部オブジェクトタイプ名を定義します。
	 * デフォルトスキーマ以外のオブジェクトを引数で渡せるようにします。
	 * 例： ARG_ARRAY = "GE.ARG_ARRAY"
	 * ======================================================================
	 */

	/**
	 * PL/SQL 接続用の内部アレイタイプ名。
	 * PL/SQL 接続用に使用する、ORACLE ARRAYタイプのENTRY系引数を定義します。
	 * このARG_ARRAYは、HYBS.sql で定義しています。
	 * ARG_ARRAY = "ARG_ARRAY" ;
	 * @og.paramLevel 7:要再起動
	 */
//	public static final String ARG_ARRAY = "ARG_ARRAY" ;

	/**
	 * PL/SQL 接続用の内部システムオブジェクトタイプ名。
	 * PL/SQL 接続用に使用する、ORACLE OBJECTタイプのシステムで使用する引数を定義します。
	 * このSYSARGは、HYBS.sql で定義しています。
	 * SYSARG = "SYSARG" ;
	 * @og.paramLevel 7:要再起動
	 */
//	public static final String SYSARG = "SYSARG" ;

	/**
	 * PL/SQL 接続用の内部システムアレイタイプ名。
	 * PL/SQL 接続用に使用する、ORACLE ARRAYタイプのシステムで使用する引数を定義します。
	 * このSYSARGは、HYBS.sql で定義しています。
	 * SYSARG_ARRAY = "SYSARG_ARRAY" ;
	 * @og.paramLevel 7:要再起動
	 */
//	public static final String SYSARG_ARRAY = "SYSARG_ARRAY" ;

	/**
	 * PL/SQL 接続用の内部エラーオブジェクトタイプ名。
	 * PL/SQL 接続用に使用する、ORACLE OBJECTタイプのエラーで使用する引数を定義します。
	 * このSYSARGは、HYBS.sql で定義しています。
	 * ERR_MSG = "ERR_MSG" ;
	 * @og.paramLevel 7:要再起動
	 */
//	public static final String ERR_MSG = "ERR_MSG" ;

	/**
	 * PL/SQL 接続用の内部エラーアレイタイプ名。
	 * PL/SQL 接続用に使用する、ORACLE ARRAYタイプのエラーで使用する引数を定義します。
	 * このSYSARGは、HYBS.sql で定義しています。
	 * ERR_MSG_ARRAY = "ERR_MSG_ARRAY" ;
	 * @og.paramLevel 7:要再起動
	 */
//	public static final String ERR_MSG_ARRAY = "ERR_MSG_ARRAY" ;

	/*
	 * ======================================================================
	 * これは，ＬＤＡＰ接続先情報に関するリソースです。
	 * initctx、service、entrydn、searchbase を指定できます。
	 * もちろん、使用時に指定することも可能です。
	 * ======================================================================
	 */

	/**
	 * LDAP 初期コンテキストファクトリ。
	 * LDAP で使用する初期コンテキストファクトリを指定します。
	 * LDAP_INITIAL_CONTEXT_FACTORY = "com.sun.jndi.ldap.LdapCtxFactory";
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String LDAP_INITIAL_CONTEXT_FACTORY = "com.sun.jndi.ldap.LdapCtxFactory" ;

	/**
	 * LDAP サービスプロバイダの構成情報。
	 * LDAP で使用するサービスプロバイダの構成情報を指定します。
	 * LDAP_PROVIDER_URL = "ldap://ldap.opengion.org:389";
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String LDAP_PROVIDER_URL = "ldap://ldap.opengion.org:389" ;

	/**
	 * LDAP 属性の取得元のオブジェクトの名前。
	 * LDAP で使用する属性の取得元のオブジェクトの名前
	 * LDAP_ENTRYDN = "cn=inquiry-sys,o=opengion,c=JP";
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String LDAP_ENTRYDN = "cn=inquiry-sys,o=opengion,c=JP" ;

	/**
	 * LDAP 属性の取得元のパスワード。
	 * LDAP で使用する属性の取得元のパスワード
	 * LDAP_PASSWORD = "****" ;
	 * @og.paramLevel 2:使用時設定
	 * @og.cryptography 1:暗号化
	 */
	public static final String LDAP_PASSWORD = "password" ;

	/**
	 * 検索するコンテキストまたはオブジェクトのベース名。
	 * LDAP で使用する検索するコンテキストまたはオブジェクトのベース名
	 * LDAP_SEARCH_BASE = "soouid=employeeuser,o=opengion,c=JP";
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String LDAP_SEARCH_BASE = "soouid=employeeuser,o=opengion,c=JP" ;

	/**
	 * LDAP 検索範囲。
	 * LDAP で使用する検索範囲を設定します。
	 * 『OBJECT』、『ONELEVEL』、『SUBTREE』のどれか 1 つ
	 * 指定の無い場合は、SearchControls の初期値で設定されます。
	 * LDAP_SEARCH_SCOPE = "SUBTREE" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String LDAP_SEARCH_SCOPE = "SUBTREE" ;

	/*
	 * ======================================================================
	 * ログファイルを出力するクラスの設定項目です。
	 * initctx、service、entrydn、searchbase を指定できます。
	 * もちろん、使用時に指定することも可能です。
	 * ======================================================================
	 */

	/**
	 * ログファイルのフォルダ/ファイル名。
	 * ログファイルを出力するディレクトリ/ファイル名.サフィックスを定義します。
	 * SYS_LOG_URL = "log/log_$(yyyyMM).txt" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String SYS_LOG_URL = "log/syslog_$(yyyyMM).txt" ;

	/*
	 * ======================================================================
	 * JSP で使用している変数
	 * ======================================================================
	 */

	/**
	 * セッションタイムアウト(秒)。
	 * Servlet/JSP でのセッションタイムアウトを秒で指定します。
	 * 最後にセッションに対してアクセスを行ってから、この秒間の
	 * アイドル時間が経過したセッションは、接続が切断されます。
	 * エンジンでは、セッション切断後、検索処理の場合は、自動的に新しい
	 * セッションでシームレスに接続しますが、登録処理は、実行できません。
	 * MAX_INACTIVE_INTERVAL = "1800" ;
	 * @og.paramLevel 1:設定確認
	 */
	public static final String MAX_INACTIVE_INTERVAL = "1800" ;

	/*
	 * ======================================================================
	 * org.opengion.hayabusa.common.* で使用している変数
	 * ======================================================================
	 */

	/**
	 * 時刻のフォーマット。
	 * 時刻のフォーマット方法を、指定します。 例( 1998/04/17 15:48:22.005 )
	 *
	 * SYS_TIME = "yyyy/MM/dd HH:mm:ss" ;
	 * SYS_TIME = "yyyy/MM/dd HH:mm:ss.SSS" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String SYS_TIME = "yyyy/MM/dd HH:mm:ss" ;

	/*
	 * ======================================================================
	 * org.opengion.hayabusa.io.* で使用している変数
	 * ======================================================================
	 */

	/**
	 * ファイルエンコーディング。
	 * ファイルエンコーディングは、EXCEL出力等のファイルをセーブする場合に
	 * 使用する、エンコーディング名です。
	 * 多言語対応のため、初期値は、UnicodeLittle に設定されています。
	 * (UTF-8 ではないので、ご注意ください。)
	 *
	 * FILE_ENCODE = "UnicodeLittle" ;
	 * FILE_ENCODE = "Shift_JIS" ;
	 * FILE_ENCODE = "MS932" ;
	 * FILE_ENCODE = "Windows-31J" ;
	 * FILE_ENCODE = "UTF-8" ;
	 * FILE_ENCODE = "ISO-8859-1" ;
	 * FILE_ENCODE = "GB2312" ;
	 * FILE_ENCODE = "GB18030" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String FILE_ENCODE = "UnicodeLittle" ;

	/**
	 * DBTableModel ファイル出力ヘッダー情報。
	 * DBTableWriter で、DBTableModel を出力する場合のヘッダー部に指定する
	 * 情報を文字列で指定します。指定の順番どおりに出力されます。
	 * なお、必ず "D" は必要です。
	 * N: Name(名前)属性。ファイルを読み取る(DBTableReader)場合には必要です。
	 * L: Label(ラベル)属性。指定のカラムの名称を表示します。
	 * S: Size(サイズ)属性。指定のカラムの桁数を表示します。
	 * C: Class(クラス)属性。指定のカラムのクラス(文字、数字など)を表示します。
	 * -: ----(区切り記号)属性。必要な箇所に入れることで、ヘッダー部とデータ部を分離します。
	 * D: Data(データ)属性。実際のデータを繰り返し表示します。
	 *
	 * WRITER_HEADER_SEQUENCE = "NL-D" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String WRITER_HEADER_SEQUENCE = "NL-D" ;

	/**
	 * EXCEL出力で、初期出力ファイル形式。
	 * EXCEL出力で、出力ファイル形式を指定します。
	 * 指定できる値は、DBTableWriter のサブクラス名になります。
	 * writeTable タグの writeClass 属性の設定値の初期値になります。
	 *
	 * WRITER_HEADER_SEQUENCE = "Default" ;
	 * WRITER_HEADER_SEQUENCE = "Excel" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String TABLE_WRITER_DEFAULT_CLASS = "Default" ;

	/**
	 * EXCEL出力で、初期入力ファイル形式。
	 * EXCEL入力で、入力ファイル形式を指定します。
	 * readTable タグの readClass 属性の設定値の初期値になります。
	 * 指定できる値は、DBTableReader のサブクラス名になります。
	 * また、AutoReader を指定すると、Excel と Default(テキスト) を
	 * 順番に試します。
	 * 初期値は、AutoReader です。
	 *
	 * TABLE_READER_DEFAULT_CLASS = "Default" ;
	 * TABLE_READER_DEFAULT_CLASS = "Excel" ;
	 * TABLE_READER_DEFAULT_CLASS = "AutoReader" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String TABLE_READER_DEFAULT_CLASS = "AutoReader" ;

	/**
	 * EXCEL出力時のデフォルトフォント名。
	 * EXCEL出力時のデフォルトフォント名を設定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。
	 * 内部的に、org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )に設定されます。
	 * デフォルトは、"ＭＳ Ｐゴシック" です。(無指定時 POI のデフォルトは、Arial)
	 *
	 * TABLE_WRITER_DEFAULT_FONT_NAME = "ＭＳ Ｐゴシック" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String TABLE_WRITER_DEFAULT_FONT_NAME = "ＭＳ Ｐゴシック" ;

	/**
	 * EXCEL出力時のデフォルトフォントポイント数。
	 * EXCEL出力時のデフォルトフォントポイント数を設定します。
	 * 内部的に、org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )に設定されます。
	 * デフォルトは、"11" です。(無指定時 POI のデフォルトは、10)
	 *
	 * TABLE_WRITER_DEFAULT_FONT_POINTS = "11" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String TABLE_WRITER_DEFAULT_FONT_POINTS = "11" ;

	/**
	 * 書込処理でコードリソースのラベル変換を行うかどうか。
	 * 
	 * コードリソースをラベルに変換して出力するかどうかを指定します。
	 * 当然、コードはユニークですが、ラベルはユニークになるかどうか保障はされていませんので
	 * TableReader 系で読み込む場合には、リスクが発生します。
	 * また、TableReader 系で読み込む場合にも、ラベルからコードを求める逆変換を行うように、
	 * setUseRenderer メソッドで指定する必要があります。
	 *
	 * USE_TABLE_WRITER_RENDERER = "false" ;
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規追加
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_TABLE_WRITER_RENDERER = "false" ;

	/**
	 * 読取処理でラベルをコードリソースに逆変換を行うかどうか。
	 * 
	 * ラベルをコードリソースに変換して入力するかどうかを指定します。
	 * TableWriter_Renderer 系のクラスで出力した場合は、コードリソースがラベルで出力されます。
	 * そのファイルを読み取ると、当然、エラーになります。
	 * ここでは、コードリソースのカラムに対して、ラベルからコードを求める逆変換を行うことで、
	 * Renderer 系で出力したファイルを取り込むことができるようにします。
	 * 当然、コードはユニークですが、ラベルはユニークになるかどうか保障はされていませんので
	 * TableReader 系で読み込む場合には、リスクが発生します。
	 *
	 * USE_TABLE_READER_RENDERER = "false" ;
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規追加
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_TABLE_READER_RENDERER = "false" ;

	/*
	 * ======================================================================
	 * org.opengion.hayabusa.db.* で使用している変数
	 * ======================================================================
	 */

	/**
	 * コネクションを取得するのに,何回リトライするか。
	 * DB_RETRY_COUNT = "10" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String DB_RETRY_COUNT = "10" ;

	/**
	 * コネクションを取得するリトライの間隔(ミリ秒)。
	 * DB_RETRY_TIME = "1000" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String DB_RETRY_TIME = "1000" ;

	/**
	 * データ検索時の最大件数。
	 * この件数以上のデータは、物理的に取得できなくなります。
	 * 個人設定可能ですが、帳票デーモン等は、システムパラメータでの
	 * 設定値が使用されます。
	 * DB_MAX_ROW_COUNT = "1000" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String DB_MAX_ROW_COUNT = "1000" ;

	/**
	 * データ検索時の最大処理制限時間。
	 * この時間(秒数)以上 SQLが終了しない場合は, 実行中の処理を途中で中断します。
	 * PL/SQL等で、異常に長い場合や、ループ状態になっている場合の対応です。
	 * はじめから、処理時間が長いことが判っている場合は、調整してください。
	 * 初期値は、３００秒です。
	 * DB_MAX_QUERY_TIMEOUT = "300" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String DB_MAX_QUERY_TIMEOUT = "300" ;

	/**
	 * Connection をプールしておく最大時間(秒)。
	 * この時間(秒数)以上プールされているConnection は物理的に接続を終了させて
	 * 新しくConnection を作成します。
	 * 初期値 3000秒(50分)に設定します。
	 * DB_MAX_CONNECTION_POOL_TIME = "3000" ;
	 *
	 * @og.rev 4.0.0.0 コネクションの設定値はfukurou/db/EDbid内です
	 *
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String DB_MAX_CONNECTION_POOL_TIME = "3000" ;
//	public static final String DB_MAX_CONNECTION_POOL_TIME = "" ;// 4.0.0.0 (2007/10/31) XML化につき不要

	/**
	 * Selection_DB オブジェクトをキャッシュしておく最大時間(ミリ秒)。
	 * この時間(ミリ秒)以上プールされているSelection_DB はキャッシュを使用しません。
	 * Selection_DB オブジェクトは、DB検索結果のプルダウンメニュー情報です。
	 * 行ごとのプルダウンメニューをコードリソースではなく、DBMENU等のデータベース
	 * より作成する場合、毎回検索していたのではサーバー負荷や実行時間が増えてしまいます。
	 * この値をキャッシュしておき、一覧表示状態では、キャッシュを利用するようにすると、
	 * パフォーマンスが向上します。
	 * 設定の目安として、一回の検索で連続してＤＢプルダウンメニュー検索を行う処理時間
	 * を確保しておけばよいでしょう。それより短いと、キャッシュとのやり取りの分、
	 * キャッシュを使用しない方が早くなります。また、長すぎると、取得先のＤＢの
	 * 値が反映されない為、登録した項目がプルダウンメニューより選べなくなります。
	 * 初期値 10秒(10000 ms)に設定します。
	 * DB_CACHE_TIME = "10000" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String DB_CACHE_TIME = "10000" ;

	/**
	 * DBColumn オブジェクトが正式で無い場合のチェックレベル。
	 * DBColumn オブジェクトは、通常はカラムリソースから作成されます。それ以外に
	 * 定義が無い場合の初期値値(size 20のテキストフィールド)や、データベース
	 * メタデータより作成される場合もあります。(使いすてカラムや検索カラム)
	 * しかし、カラムリソースより作成していない場合は、チェック項目が緩く、
	 * そのままデータベース登録すると問題になるケースがあります。
	 * そこで、正式なカラムオブジェクトで無い場合のチェックレベルを指定することで
	 * 何もしない, 正常(0), 警告(1), 異常(2) のレベルを登録します。
	 * DB_OFFICIAL_COLUMN_CHECK_LEVEL = "1" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String DB_OFFICIAL_COLUMN_CHECK_LEVEL = "1" ;

	/**
	 * アクセスログ取得の為,アプリケーション情報を追記するかどうか指定します。
	 *
	 * 内部情報とは、接続ユーザーのクライアント情報と、実行状況のアプリケーション情報があります。
	 *
	 * クライアント情報とは、"i=192.168.51.81,h=null,u=C12345" 的な文字列で、
	 * i=の後ろに IPアドレス、h=の後ろにホスト名、u=の後ろにユーザー名をセットして、
	 * DBMS_APPLICATION_INFO.SET_CLIENT_INFO( "i=192.168.51.81,h=null,u=C12345" ) を
	 * CALL します。
	 *
	 * アプリケーション情報とは、"o=SELECT,p=GEXXXX" 的な文字列で、o=の後ろに 操作、
	 * p=の後ろにプログラムIDをセットして、
	 * DBMS_APPLICATION_INFO.SET_MODULE( "GE0010","o=操作,p=プログラムID" ) を
	 * CALL します。
	 *
	 * このPL/SQL を使用するコネクションについて実行すると、アクセスログ記録を行う為の
	 * 情報として取り出すことが可能になります。
	 *
	 * USE_DB_APPLICATION_INFO = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_DB_APPLICATION_INFO = "true" ;

	/*
	 * ======================================================================
	 * org.opengion.hayabusa.html.* で使用している変数
	 * ======================================================================
	 */

	/**
	 * 画面上に一度に表示されるデータ件数。
	 * ただし，HTMLTextField.java のみ，１件ずつ表示に固定されています。
	 * HTML_PAGESIZE = "100" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String HTML_PAGESIZE = "100" ;

	/**
	 * テーブル表示の 各種属性を一括指定します。
	 * 以前は、それぞれの属性を個別に定義するキーワードを持っていましたが、
	 * 現在は、CSS で定義するようになりました。
	 * それでも、場合によっては、table タグに直接属性を設定したいケースがありますので
	 * その場合に、指定してください。
	 * 
	 * HTML_TABLE_PARAM = "frame=\"box\" border=\"1px\" cellpadding=\"0\" cellspacing=\"0\" rules=\"all\" bordercolor=\"#ff0000\"" ;
	 * HTML_TABLE_PARAM = "0" ;
	 *
	 * @og.rev 5.2.3.0 (2010/12/01) テーブル罫線対応
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HTML_TABLE_PARAM = null ;

	/**
	 * テーブル表示の border 属性(外枠の太さ)。
	 * テーブルタグの border 属性に数字で設定します。
	 * USE_HTML_TABLE_CSS="false"の場合のみ使用されます。(通常はCSSで設定します。)
	 * border 外枠の太さ
	 * HTML_BORDER = "0" ;
	 * HTML_BORDER = "2" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_BORDER = "0" ;

	/**
	 * テーブル表示の width 属性(表の幅)。
	 * テーブルタグの width 属性にピクセルまたは,％で設定します。
	 * USE_HTML_TABLE_CSS="false"の場合のみ使用されます。(通常はCSSで設定します。)
	 * 初期値は、指定なしです。
	 * HTML_WIDTH = "" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_WIDTH = "" ;

	/**
	 * テーブル表示の frame 属性(外枠の表示形式)。
	 * テーブルタグの frame 属性に以下の属性のうちどれかを設定します。
	 * void,above,below,lhs,rhs,hsides,vsides,box,border
	 * USE_HTML_TABLE_CSS="false"の場合のみ使用されます。(通常はCSSで設定します。)
	 * HTML_FRAME = "box" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_FRAME = "box" ;

	/**
	 * テーブル表示の rules 属性(セルを区切る線の形式)。
	 * テーブルタグの rules 属性に以下の属性のうちどれかを設定します。
	 * USE_HTML_TABLE_CSS="false"の場合のみ使用されます。(通常はCSSで設定します。)
	 * none,rows,cols,groups,all
	 * HTML_RULES = "all" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_RULES = "all" ;

	/**
	 * テーブル表示の cellspacing 属性(セルの間隔)。
	 * テーブルタグの cellspacing 属性に数字で設定します。
	 * USE_HTML_TABLE_CSS="false"の場合のみ使用されます。(通常はCSSで設定します。)
	 * HTML_CELLSPACING = "3" ;
	 * HTML_CELLSPACING = "0" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_CELLSPACING = "0" ;

	/**
	 * テーブル表示の cellpadding 属性(セル内のマージン)。
	 * テーブルタグの cellpadding 属性に数字で設定します。
	 * USE_HTML_TABLE_CSS="false"の場合のみ使用されます。(通常はCSSで設定します。)
	 * HTML_CELLPADDING = "1" ;
	 * HTML_CELLPADDING = "0" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_CELLPADDING = "0" ;

	/**
	 * テーブル表示の CSSファイル利用の有無。
	 * テーブルタグの HTML_BORDER,HTML_WIDTH,HTML_FRAME,
	 * HTML_RULES,HTML_CELLSPACING,HTML_CELLPADDING パラメータを使用せず、
	 * jsp/common/default.css を使用して、テーブルレイアウトを設定する場合は、
	 * true にセットします。
	 * 過去のイメージとの互換性を重視する場合は、false にしてください。
	 * USE_HTML_TABLE_CSS = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String USE_HTML_TABLE_CSS = "true" ;

	/**
	 * カラム値合成のセパレーター。
	 * 複数のカラム値を合成して表示する場合に使用するセパレーターを設定します。
	 * 初期値は、":" です。
	 * HTML_SEPARATOR = ":" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HTML_SEPARATOR = ":" ;

	/**
	 * カラムタグのテキストフィールドを表示する最大桁数。
	 * カラムタグは、カラム定義の使用桁数(USE_LENGTH)だけのテキストフィールドを
	 * 作成しようとしますが、ここで指定の値を最大値として、表示します。
	 * ただし、表示桁数(VIEW_LENGTH)が設定されている場合は、そちらを優先します。
	 * 入力可能文字数の制限とは異なり、フィールドをどの長さまで表示させるかを
	 * 指定するだけです。
	 * この長さ以下のフィールドについては，元のフィールドの長さで表示します。
	 * HTML_COLUMNS_MAXSIZE = "20" ;
	 * @og.paramLevel 1:設定確認
	 */
	public static final String HTML_COLUMNS_MAXSIZE = "20" ;

	/**
	 * VIEWタグのテキストフィールドを表示する最大桁数。
	 * VIEWタグ で表示する場合の HTML_COLUMNS_MAXSIZE に対する値になります。
	 *
	 * 元々、検索条件を入力する カラムタグ は、多くのフィールド長は不要です。
	 * データを入力する VIEWタグでは、カラムタグで作成するより大きなフィールドを
	 * 必要とします。
	 * 以前は、HTML_COLUMNS_MAXSIZE * 2 がハードコーディングされていました。
	 * この長さ以下のフィールドについては，元のフィールドの長さで表示します。
	 * HTML_VIEW_COLUMNS_MAXSIZE = "40" ;
	 * @og.paramLevel 1:設定確認
	 */
	public static final String HTML_VIEW_COLUMNS_MAXSIZE = "40" ;

	/**
	 * テキストフィールドを並べて表示させる場合の,横方向の表示間隔。
	 * テキストフィールドの自動表示では、縦×横に並べますが、その横の箱同士が
	 * 近すぎると、表示文字とラベルがくっついた状態で見えることがあります。
	 * この箱間の幅をピクセルで指定します。
	 * HTML_COLUMS_SPACE = "10" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HTML_COLUMS_SPACE = "10" ;

	/**
	 * テキストフィールドを並べて表示させる場合の,縦方向の最大表示件数。
	 * 逆にこの件数を超える場合は、横方向の表示を行います。
	 * テキストフィールドの自動表示では、フィールドの数に対して、縦方向に
	 * この数だけ並べます。それ以上の場合は、横のフィールドへ、さらに超える場合は、
	 * その横と、順次並べていきます。
	 * 初期値は、10 個です。
	 * HTML_MAXROW_NUMBER = "10" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HTML_MAXROW_NUMBER = "10" ;

	/**
	 * ラベルセパレーター使用有無。
	 * ラベルとフィールドのセパレーターとして、コロン(：)を使用するかどうか指定します。
	 * 初期値は、使用する(true)です。
	 * HTML_LABEL_SEPARATOR = "true" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String HTML_LABEL_SEPARATOR = "true" ;

	/**
	 * テキストフィールドの大きさ調整を行う倍率。
	 * 従来のShift_JIS画面では、IEでのテキストフィールドが、小さかったので、
	 * 1.3 倍していました。
	 * UTF-8 になると、画面サイズが全角文字分のサイズを取るようで、大きくなって、
	 * しまったので、リソースファイルで指定できるようにします。
	 *
	 * HTML_TEXTFIELD_ADJUSTMENT_MIN_SIZE とともに、設定する必要があります。
	 *
	 * SIZE = "(int)((maxlength-MIN_SIZE)*RATIO+0.5)" ;
	 *                                    ~~~~~
	 *
	 * HTML_TEXTFIELD_ADJUSTMENT_RATIO = "1.0" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_TEXTFIELD_ADJUSTMENT_RATIO = "1.0" ;

	/**
	 * テキストフィールドのCSS調整基準。
	 * テキストフィールドが、この指定値以下の場合は、SIZE="1" で返します。
	 * SIZE="1" の場合は、大きさは,ＣＳＳファイルで指定することになりますので、
	 * S0 + length の class 属性で定義しておく必要があります。
	 *
	 * HTML_TEXTFIELD_ADJUSTMENT_RATIO とともに、設定する必要があります。
	 *
	 * SIZE = "(int)((maxlength-MIN_SIZE)*RATIO+0.5)" ;
	 *                          ~~~~~~~~
	 *
	 * HTML_TEXTFIELD_ADJUSTMENT_MIN_SIZE = "0" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String HTML_TEXTFIELD_ADJUSTMENT_MIN_SIZE = "0" ;

	/**
	 * チェックボックスの全チェック属性の初期値。
	 * viewForm の useCheckControl 属性の初期値を設定します。
	 *
	 * この属性は、selectedTypeがCHECKBOXの時に、チェックボックスの全チェックを選択する
	 * JavaScript の機能を使用するかどうかを指定するのに使います。
	 * useCheckControl=true に設定すると、コントロール用のチェックボックスが現れて、
	 * この機能を使用することができるようになります。
	 * false は、全チェックを使用しません。checked は、初期値を、チェック済みにします。
	 * false は、0でも代用可能です。true は、1でも代用可能です。
	 * checked は、2でも代用可能です。
	 * VIEW_USE_CHECK_CONTROL = "true" ; 全チェックを使用
	 * VIEW_USE_CHECK_CONTROL = "1" ;  true と同じ
	 * VIEW_USE_CHECK_CONTROL = "checked" ;  チェック済みの全チェックを作成
	 *
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_CHECK_CONTROL = "true" ;

	/**
	 * EDITOR の AUTOAREA(自動サイズテキストエリア)の最大列数。
	 *
	 * AUTOAREAは、データ量によって、自動的にサイズを変更するテキストエリアです。
	 * このテキストエリアの最大列数を外部で指定する場合にこのパラメータを使用します。
	 * この大きさ分のテキストエリアの幅で折り返しが発生するとして、
	 * 行数計算も行われます。
	 * 0 が指定された場合は、無制限になります。
	 * AUTOAREA の編集パラメータに、minRow-maxRow,minCol-maxCol として、最小、最大の
	 * それぞれの行列数を指定した場合は、そちらが優先されます。
	 * HTML_AUTOAREA_MAX_COL_SIZE = "80" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HTML_AUTOAREA_MAX_COL_SIZE = "80" ;

	/**
	 * EDITOR の AUTOAREA(自動サイズテキストエリア)の最大行数。
	 *
	 * AUTOAREAは、データ量によって、自動的にサイズを変更するテキストエリアです。
	 * このテキストエリアの最大行数を外部で指定する場合にこのパラメータを使用します。
	 * この大きさ分のテキストエリアの行で打ち切られます。
	 * 0 が指定された場合は、無制限になります。
	 * AUTOAREA の編集パラメータに、minRow-maxRow,minCol-maxCol として、最小、最大の
	 * それぞれの行列数を指定した場合は、そちらが優先されます。
	 * HTML_AUTOAREA_MAX_ROW_SIZE = "30" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String HTML_AUTOAREA_MAX_ROW_SIZE = "30" ;

	/**
	 * マルチキー入力メニューの作成可否。
	 *
	 * 複数文字列のキー入力に反応するプルダウンメニューを追加します。
	 * JavaScriptでの対応ですので、便利機能と位置付けてください。
	 * 使用する場合は、optionAttributes 属性を使用して設定するか、
	 * USE_MULTI_KEY_SELECT=true で設定します。
	 * 初期値は、true (作成する)です。
	 * USE_MULTI_KEY_SELECT = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_MULTI_KEY_SELECT = "true" ;

	/**
	 * 登録時整合性チェックの使用有無。
	 *
	 * Consistency キー による整合性チェックを行うかどうかを指定する
	 * 属性の初期値を設定します。
	 *
	 * 画面よりＤＢ登録時に、ＩＥの戻るボタンで戻り、キャッシュされている画面を
	 * 登録すると、画面と異なるメモリ上のDBTableModel が再び更新されてしまう
	 * というトラブルへ対応する為、画面の hidden パラメータ と DBTableModel が
	 * 一致しないとエラーにするという処理を、行うかどうかを指定できます。
	 * viewFormTag で、整合性パラメータを出力し、entryTag で、チェックします。
	 * 初期値は、true (使用する)です。
	 * USE_CONSISTENCY = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_CONSISTENCY = "true" ;

	/**
	 * 同一セッションでのマルチ起動対策を行うかどうか。
	 *
	 * 通常、同一セッションで登録操作を行うと、『画面とデータの整合性チェックで
	 * エラーが出ました。』エラーが発生するようになっていますが、その前の段階、
	 * つまり、メニューより選択した段階で、マルチセッションになっていない
	 * （つまり、同一セッションで、以前に作成されたメニューからアクセスされた）
	 * 事を判定します。
	 * メニューにMSC=カウンタというパラメータを儲け、session_init.jsp で判定しています。
	 * 初期値は、true (使用する)です。
	 * USE_MULTI_SESSION_CHECK = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_MULTI_SESSION_CHECK = "true" ;

	/*
	 * ======================================================================
	 * org.opengion.hayabusa.taglib.* で使用している変数
	 * ======================================================================
	 */

	/**
	 * メール送信エンコード名。
	 *
	 * E-Mail で日本語を送信する場合のデフォルトエンコード名を指定します。
	 * 現状では、Windows-31J , MS932 , ISO-2022-JP ,それ以外 となっています。
	 * それ以外は、一律に、ISO-2022-JP にて送信されます。
	 * 一般に、E-Mail で日本語を送信する場合、ISO-2022-JP (JISコード)化して、
	 * 7bit でエンコードして送信する必要がありますが、Windows系の特殊文字や、
	 * unicodeと文字のマッピングが異なる文字、半角カタカナなどが、文字化けします。
	 * 社内システムなどの連絡用に使用するのであれば、Windowsシステムに
	 * 特化した、Windows-31J の方が文字化けに関する心配は低減されます。
	 * 初期値は、ISO-2022-JP です。
	 * MAIL_DEFAULT_CHARSET = "ISO-2022-JP" ;
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String MAIL_DEFAULT_CHARSET = "ISO-2022-JP" ;

	/**
	 * 編集データのリクエスト値の転送有無。
	 * DBTableModel からの編集データ(キー__行番号)のリクエストを
	 * frame タグで渡すかどうかを指定します。
	 * 引数が多いと、画面が空白になる現象があるため通常、転送しません。
	 * 初期値は、false (転送しません)。
	 * FRAME_UNDER_BAR_REQUEST_KEY_USED = "false" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String FRAME_UNDER_BAR_REQUEST_KEY_USED = "false" ;

	/**
	 * ページの先頭へのリンクの間隔	。
	 * ViewFormTagで、backLinkCount(ページの先頭へのリンクの間隔)の
	 * 初期値を設定します。
	 * 0 が設定された場合は、ページ先頭へのリンクは作成されません。
	 * (通常、ヘッダー固定を使用する場合は、"0"リンクなしとします。)
	 * 設定がない場合は、NumberFormatException がスローされます。
	 * VIEW_BACK_LINK_COUNT = "0" ;  リンクなし
	 * VIEW_BACK_LINK_COUNT = "10" ; データ１０行毎に先頭へのリンクが作成される。
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String VIEW_BACK_LINK_COUNT = "0" ;

	/**
	 * ヘッダーを出力する間隔	。
	 * ViewFormTagで、headerSkipCount(ヘッダーを出力する間隔)の
	 * 初期値を設定します。
	 * 0 が設定された場合は、ヘッダーは、最初だけで繰り返し出力されません。
	 * (通常、ヘッダー固定を使用する場合は、"0"先頭のみとします。)
	 * 設定がない場合は、NumberFormatException がスローされます。
	 * VIEW_HEADER_SKIP_COUNT = "0" ;  ヘッダーは、先頭のみ作成される。
	 * VIEW_HEADER_SKIP_COUNT = "10" ; ヘッダーは、データ１０行毎に作成される。
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String VIEW_HEADER_SKIP_COUNT = "0" ;

	/**
	 * ヘッダーのソート用リンクの有無	。
	 * ViewFormTagで、useTableSorter(ヘッダーにソート用リンクを作成するかどうか)の
	 * 初期値を設定します。
	 * true が設定された場合は、ヘッダーにソート用リンクを作成します。
	 *
	 * VIEW_USE_TABLE_SORTER = "true" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_TABLE_SORTER = "true" ;

	/**
	 * SVGチャートのテンポラリディレクトリ定義。
	 * SVGチャートを作成する場合、SVGファイルをテンポラリのディレクトリに
	 * 出力して、そこへのアドレスを付けて、HTMLを作成しています。(組み込めないため)
	 * 以前は、ファイルのテンポラリと同じ、FILE_URL で指定したディレクトリの
	 * 下の、ChartTemp フォルダでした。
	 * ここでの定義のフォルダが、そのままテンポラリファイルのフォルダ名になります。
	 * SVGチャートのテンポラリファイルは、Tomcat終了時に自動的に削除されます。
	 * CHART_TEMP_DIR = "filetemp/ChartTemp/" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String CHART_TEMP_DIR = "filetemp/ChartTemp/" ;

	/**
	 * SVGチャートのテンポラリディレクトリ定義。
	 * ChartCreateTag で使用される JFreeChart/画像ファイルのキャッシュ時間(秒)を指定します。
	 * この、キャッシュ時間(秒)を過ぎると、キャッシュを初期化します。
	 * 初期値は、600(秒) = １０分です。
	 * CHART_CACHE_TIME = "600" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String CHART_CACHE_TIME = "600" ;

	/**
	 * 選択行データのフォワード時転送有無。
	 * CommonForwardTag の useTableData 属性の初期値を定義します。
	 * useTableData 属性は、submit タグで、選択行データの値を、URL 引数に
	 * 追加するかどうかを指定します。
	 * 過去との互換性を考慮するなら、false です。
	 * 基本の設定値は、false です。
	 * COMMON_FORWARD_USE_TABLE_DATA = "false" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String COMMON_FORWARD_USE_TABLE_DATA = "false" ;

	/**
	 * ヘッダー固定のスクロールバーの使用有無。
	 * ヘッダー固定のスクロールバーを使用すると、登録ボタンの領域を残したまま、
	 * テーブル部のみスクロールすることが可能になります。
	 * これは、ViewFormTagで、useScrollBar(スクロールバー用のDIV要素を
	 * 出力するかどうか)の初期値を設定しています。
	 * true が設定された場合は、ヘッダー固定のスクロールバーを使用します。
	 * VIEW_USE_SCROLLBAR = "true" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_SCROLLBAR = "true" ;

	/**
	 * 戻るリンク時等の行送り機能の使用有無。
	 * ViewFormTagで、useSelectedRow(指定行の行送りを行うかどうか)の初期値を設定します。
	 * SEL_ROW機能(戻るリンクから戻ったときに、先にクリックされた行番号を返す機能)で、
	 * この機能を有効/無効を指定します。(true:使用する/false:使用しない)
	 * 有効な場合は、選択行まで画面がスクロールします。
	 * VIEW_USE_SELECTED_ROW = "true" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_SELECTED_ROW = "true" ;

	/**
	 * 戻るリンク時等の選択行マーカーの使用有無。
	 * AbstractViewFormで、指定行のマーカーを行う場合の選択行色指定を行うかどうかを設定します。
	 * 選択行の色指定(class="row_sel")を有効/無効を指定します。(true:使用する/false:使用しない)
	 * ここでは、使用するかどうかのみの設定で、実際の色は、CSSファイルで行います。
	 * VIEW_USE_SELROW_COLOR = "true" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_SELROW_COLOR = "true" ;

	/**
	 * 選択行のハイライト表示を行うかどうか。
	 * ViewFormTagで、useHilightRow(選択行のハイライト表示を行うかどうか)の初期値を設定します。
	 * これは、チェックボックスやラジオボタンで行選択した際に、どの行が選択されたか
	 * 判るように行のハイライト表示を行う機能です。
	 * さらに、行の任意の箇所をクリックしても、チェックが入り、ハイライト表示されます。
	 * VIEW_USE_HILIGHT_ROW = "true" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_HILIGHT_ROW = "true" ;

	/**
	 * 選択行のハイライト表示を行った場合に、ON/OFFの制御を行選択のチェックボックスのみで有効にするか。
	 * ViewFormTagで、useHilightCheckOnly(選択行ハイライトの制御をチェックボックスのみで有効にするか)の初期値を設定します。
	 * trueの場合、チェックボックスのみで有効になります。
	 * falseの場合、行全体の各要素で有効になります。
	 * trueにした場合、チェックボックスが存在しない行に対しては、ハイライトすることはできません。
	 * VIEW_USE_HILIGHT_CHECK_ONLY = "false" ;
	 * 
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * 
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String VIEW_USE_HILIGHT_CHECK_ONLY = "false" ;

	/**
	 * QUERY 検索時等の結果を画面に出力するかどうかを指定。
	 * 従来の検索結果は、画面上方に記述していましたが、[X/Y]等も出力される為、
	 * 実質的に不要な項目です。このフラグを true に設定すると画面に表示します。
	 * なお、検索結果が０件の場合は、従来の displayMsg ではなく、notfoundMsg を
	 * 使用するため、こちらは、query タグで明示的に "" をセットしないと表示されます。
	 * VIEW_USE_DISPLAY_MSG = "false" ;
	 * @og.paramLevel 4:個人設定可
	 */
//	public static final String VIEW_USE_DISPLAY_MSG = "false" ;

	/**
	 * QUERY 検索時等の結果を画面に出力する時のメッセージを指定。
	 * 従来の検索結果は、画面上方に記述していましたが、[X/Y]等も出力される為、
	 * 実質的に不要な項目です。
	 * 初期値は、ゼロ文字列とし、何も表示しないようにしておきます。
	 * 従来は、MSG0033 『　件検索しました。』です。
	 * VIEW_DISPLAY_MSG = "" ;
	 * VIEW_DISPLAY_MSG = "MSG0033" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String VIEW_DISPLAY_MSG = "" ;

	/**
	 * 一括入力カラムのアシスト機能の初期値。
	 *
	 * これは、カラムの書き込み時に、一括入力可能なカラムをヘッダーに表示し、
	 * 一括登録できる機能について、初期状態を規定します。
	 * true:に設定すると、何も指定しない限り、一括入力用のカラムが表示されます。
	 * これは、ViewFormTag の clmBulkSet に、"*" を設定したのと同じ状態を作ります。
	 * false:に設定すると、従来どおりです。(null が設定された状態)
	 * どちらが設定されていても、ViewFormTag の clmBulkSet に、カラム名を指定すると
	 * そちらが優先されます。
	 *
	 * VIEW_USE_DEFAULT_BULKSET = "false"
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規追加
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String VIEW_USE_DEFAULT_BULKSET = "false" ;

	/**
	 * 全件無条件でのリセット呼び出しの使用有無。
	 * EntryTagで、command="RESET" 時のアクションとして allReset() を
	 * 呼び出すかどうかを指定します。
	 * true が設定された場合は、allReset() を呼び出します。
	 * 初期値は、false です。
	 * RESET_ACTION_ALL_USE = "false" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String RESET_ACTION_ALL_USE = "false" ;

	/**
	 * オラクル文字化け対応処理の実施有無。
	 * Shift-JISで作成されたオラクルとWindowsとの間の 文字コード変換を行うかどうか
	 * を指定します。
	 * 例えば、"～"は、正常に登録できますが、表示を行う段階で文字化けが発生します。
	 * これは、unicode 0x301c ですが、Windows上では、このコードが認識できず、
	 * 0xff5e を、"～" に認識します。
	 * 登録時は、JDBCが Shift-JIS(0x8160)を、0x301c に変換して登録する為、
	 * 文字化けは発生しません。
	 * 対策として、ＤＢから検索した値 0x301c は、0xff5e に変換して表示させます。
	 * また、UTF-8では、『円マーク』が、『バックスラッシュ』で表示されます。
	 * これは、コードが全く異なる 『円マーク』に置き換えないと、きちんと
	 * 表示されません。この場合は、Set時には、元に戻す必要があります。
	 * 初期値は、false です。
	 * USE_UTF8_CHARACTER_TRANSLATION = "false" ;
	 * @og.paramLevel 7:要再起動
	 */
	//	public static final String USE_UTF8_CHARACTER_TRANSLATION = "false" ;

	/**
	 * シェル実行時のタイムアウト時間(秒)。
	 * 指定時間の単位は、秒です。
	 * 0 秒に設定すると、割り込みが入るまで、待ちつづけます。
	 * 過去の互換性の関係で、大きな時間が入っていますが、基本的には、
	 * １０分、１５分を目安に設定してください。
	 * 初期値は、3600 秒です。
	 * SHELL_TIMEOUT = "3600" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String SHELL_TIMEOUT = "3600" ;

	/**
	 * カラムでリソースファイルの初期値を使うかどうかを指定。
	 * カラムの値や初期値がセットされていないときに、カラムリソースの
	 * 初期値を利用するかどうかを、true/false で指定します。
	 * true の場合は、初期値を使用します。
	 * 過去との互換性の関係で、この属性は、false に設定されています。
	 * USE_DEFAULT_PROPERTIE_VALUE = "false" ;
	 * 
	 * @og.rev 5.2.3.0 (2010/12/01) 新規追加
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_DEFAULT_PROPERTIE_VALUE = "false" ;

	/*
	 * ======================================================================
	 * org.opengion.hayabusa.resource.* で使用している変数
	 *
	 * 基本は、リソースDBを読み取る設定です。ただし、リソースファイルの
	 * 設定があれば、リソースDBは、無視します。
	 * (このファイルの RESOURCE_xx_DB は、コメントにする必要はありません。)
	 *
	 * リソースは、同一キーが存在する場合は、後ろを優先します。
	 * リソースDBでは、KBSAKU(作成区分)でソートしている為、これが大きいほど、
	 * 優先順位が高いことになります。
	 * システム(GE)は、KBSAKU=0 ですので、それ以上の値で各システムプロパティーを
	 * 登録すれば、そちらが優先されます。
	 * ======================================================================
	 */

	/**
	 * カレンダＤＢの使用有無。
	 * CalendarFactoryで、カレンダＤＢを使用するかどうかを指定します。
	 * 使用する場合は、true , 使用しない場合は、false を設定します。
	 * true が設定された場合は、CalendarQuery_XXXX から、事業所と年月より、
	 * カレンダオブジェクトを作成します。
	 * 初期値は、false です。
	 * USE_CALENDAR_DATABASE = "false" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_CALENDAR_DATABASE = "false" ;

	/**
	 * カレンダＤＢの接続先
	 * CalendarFactoryで、カレンダＤＢを使用する場合、どの接続先に接続するか指定します。
	 * ここでは、接続先そのものではなく、接続先のDBID文字列を指定します。
	 * 初期値は、null(標準の接続先) です。
	 * RESOURCE_CALENDAR_DBID = "" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String RESOURCE_CALENDAR_DBID = "" ;

//	/**
//	 * カレンダＤＢの実装クラス定義。
//	 * CalendarFactoryで、カレンダＤＢを使用する場合の実装クラスを指定します。
//	 * クラス名は、CalendarQuery_XXXX をキーとして、指定します。
//	 * 初期設定として、org.opengion.plugin.calendar パッケージの ZY01 が使用可能です。
//	 * CalendarQuery_ZY01 = "org.opengion.plugin.calendar.CalendarQuery_ZY01"  ;
//	 * @og.paramLevel 3:標準推奨
//	 */
//	public static final String CalendarQuery_ZY01 = "org.opengion.plugin.calendar.CalendarQuery_ZY01" ;

	/**
	 * チェックボックスの id を出力可否。
	 * チェックボックスの id を出力するかどうかを指定します。
	 * ガントチャートのスライダー機能で、スライドした行のチェックボックスにチェックを
	 * 付けるために、チェックボックスの id を利用する必要があります。
	 * id は、HybsSystem.ROW_ID_KEY で指定しており、[ROW.ID] で取得できます。
	 * 初期値は、true です。
	 * USE_CHECKBOX_ROW_ID = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_CHECKBOX_ROW_ID = "true" ;

	/**
	 * サーバー名。
	 * このTomcatが実行しているサーバー名で、request.getServerName() で求められる値です。
	 * Webサーバーとしてアクセスされたときのサーバー名になります。
	 * [例： HN50G5]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String SERVER_NAME = "" ;

	/**
	 * ホスト名。
	 * Java VM が実行しているサーバー名で、InetAddress.getLocalHost().getHostName() で
	 * 求められる値です。これは、リクエストとは関係なく、サーバーそのものの名称になります。
	 * [例： HN50G5]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String HOST_NAME = "" ;

	/**
	 * ホストIPアドレス。
	 * Java VM が実行しているホストIPアドレスで、InetAddress.getLocalHost().getHostAddress() で
	 * 求められる値です。
	 * [例： 200.1.50.165]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String HOST_ADRS = "" ;

	/**
	 * サーバー本体の URL
	 * scheme + "://" + serverName + portStr の各情報から作成します。
	 * このシステムの外部へメール＋URLリンク等を作成する場合や、
	 * そのサーバーの別のWebサーバーへのアドレス生成などに使用します。
	 * [例： http://localhost:8824/]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String SERVER_URL = "" ;

	/**
	 * コンテキストの URL
	 * scheme + "://" + serverName + portStr + serverURL + path + "/" の各情報から作成します。
	 * これは、サーバーURL＋コンテキストまで加味したURLになっています。
	 * このシステムの外部へメール＋URLリンク等を作成する場合などに使用します。
	 * [例： http://localhost:8824/gf/]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String CONTEXT_URL = "" ;

	/**
	 * サーバー情報。
	 * ホスト名とホストアドレスをコメント情報として表示する場合に使用します。
	 * [例： HN50G5 ( 200.1.50.165 )]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String SERVER_INFO = "" ;

	/**
	 * サーブレット情報。
	 * [例： Apache Tomcat/5.0.28]
	 * @og.paramLevel 9:自動設定
	 */
	public static final String SERVLET_INFO = "" ;

	/**
	 * ログインユーザーに、ゲストログインを許可するかどうか。
	 * ゲストログインは、リソースファイル、または、データベースにユーザーが存在しない
	 * 場合に、デフォルトロール(*)でログインする機能です。
	 * LDAP等の全社ユーザー使用時に、ローカルにも同数のユーザーをキープするのではなく、
	 * 通常は、ゲストユーザーとして扱い、管理者やその他特定ユーザーだけを、データベース
	 * 等に、登録しておきます。
	 * ver4 では、デフォルトロールという考え方がなくなりましたので、画面のロールに、
	 * (*)を明示的に追加し、RWMODE を指定する必要があります。
	 * 初期値は、false です。
	 * USE_GUEST_USER = "false" ;
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String USE_GUEST_USER = "false" ;

	/**
	 * リソースデータベースのシステムID名。
	 * web.xml で初期値指定する必要があります。
	 * SYSTEM_ID = "GE" ;
	 * @og.paramLevel 9:自動設定
	 */
	public static final String SYSTEM_ID = "" ;

	/**
	 * アプリケーショントップタイトル。
	 * アプリケーションのWindowsに表示するトップタイトル名を指定します。
	 * ブラウザの、上端に表示されます。また、お気に入り等への登録時の名称になります。
	 * GUI_TOP_TITLE = "Web アプリケーションシステム" ;
	 * @og.paramLevel 1:設定確認
	 */
	public static final String GUI_TOP_TITLE = "Web アプリケーションシステム" ;

	/**
	 * 共有リソースデータ DBID。
	 * ユーザーリソース(GEA10)を除く他のリソース(カラム(GEA03),ラベル(GEA08),
	 * コード(GEA04),メッセージ(GEA09),画面(GEA11),システムパラメータ(GE12),
	 * アクセス統計(GE15),ユーザーパラメータ(GE16),URL転送(GE17))
	 * のデータベースを通常の、DEFAULT_DB_URL 設定値以外の場所にアクセスする
	 * 場合に、DBID で指定します。
	 * DBID は、'GE' や 'XX' などの通常は、システムID名を使用しますが、
	 * 何でも構いません。このDBIDは、XX_DB_URL や、XX_DB_USER 、XX_DB_PASSWD として
	 * 登録しておく必要があります。指定がない場合は、DEFAULT_DB_URL を使用します。
	 * RESOURCE_DBID = "" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String RESOURCE_DBID = "" ;

	/**
	 * ユーザーリソースDBID。
	 * リソースDBが、他のデータベースにある場合(統合されているなど)取得データベースを
	 * 通常の、DEFAULT_DB_URL 設定値以外の場所にアクセスする場合に、DBID で指定します。
	 * DBID は、'GE' や 'XX' などの通常は、システムID名を使用しますが、何でも構いません。
	 * このDBIDは、XX_DB_URL や、XX_DB_USER 、XX_DB_PASSWD として登録する必要があります。
	 * 未定義の場合は、RESOURCE_DBID から、それも未定義の場合は デフォルトの接続先を使用します。
	 * RESOURCE_USER_DBID = "" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String RESOURCE_USER_DBID = "" ;

	/**
	 * 共有リソースアクセスアドレス。
	 * 共有リソースを統一した環境でアクセスする場合のサーバアドレスを指定します。
	 * 共有リソース環境を個別に作成する場合、すべtのユーザーで利用可能な転送URLや
	 * 共有メッセージ、アクセス統計管理、ユーザーパラメータなどの画面は、フリーに
	 * しておき、各種リソースは管理者のみアクセスできるようにします。
	 * 未指定の場合は、自分自身のアドレスを使用します。
	 * RESOURCE_ADDRESS = "http://localhost:8823/gf/jsp/index.jsp" ;
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String RESOURCE_ADDRESS = "" ;

	/**
	 * ユーザーリソース検索SQL。
	 * 検索するカラムには、必ず、USERID,LANG,NAME_JA,ROLES,DROLES がこの順番で含まれており、
	 * 絞込み条件(?パラメータ)として、SYSTEM_ID,USERID がこの順番で指定される必要があります。
	 * (カラム名は関係ありません。並び順と意味が重要です。)
	 * また、検索順(ORDER BY)は、優先順位の低い順に検索してください。使用するのは、一番最後に
	 * 検索された行を使用します。
	 * ユーザーリソースは、RESOURCE_USER_DBID で指定のデータベースから取得します。
	 * 未定義の場合は、RESOURCE_DBID から、それも未定義の場合は デフォルトの接続先を使用します。
	 *
	 * @og.rev 4.1.0.0 (2007/01/11) GEA10ではなくGEA10V01を見るように変更
	 * @og.rev 4.4.0.0 (2008/08/02) DROLES(データロールズ)の項目追加
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String RESOURCE_USER_SQL
									= "SELECT USERID,LANG,NAME_JA,ROLES,DROLES"
									+ " FROM GEA10V01"
									+ " WHERE SYSTEM_ID IN ( ?,'**') AND USERID=? AND FGJ='1'"
									+ " ORDER BY SYSTEM_ID, FGDEFAULT DESC" ;

//	public static final String RESOURCE_USER_SQL
//									= "SELECT USERID,LANG,NAME_JA,ROLES"
//									+ " FROM GEA10V01"
//									+ " WHERE SYSTEM_ID IN ( ?,'**') AND USERID=? AND FGJ='1'"
//									+ " ORDER BY SYSTEM_ID, FGDEFAULT DESC" ;

//	public static final String //RESOURCE_USER_SQL = "SELECT USERID,LANG,NAME_JA,ROLES FROM GEA10"
//									RESOURCE_USER_SQL = "SELECT USERID,LANG,NAME_JA,ROLES FROM " // 4.1.0.0(ビュー化のため）
//									+ " GEA10V01 " //4.1.0.0 (2008/01/11) ビューを見るように変更
//					//				+ " (SELECT USERID as userid,LANG as lang,NAME_JA as name,ROLES as roles,SYSTEM_ID,FGDEFAULT,FGJ FROM GEA10" // 4.1.0.0
//					//				+ " UNION ALL select 'RMCN_USR','ja','RemoConUser','manager','**','0','1' FROM DUAL)" // 4.1.0.0
//									+ " WHERE SYSTEM_ID IN ( ?,'**') AND USERID=? AND FGJ='1'"
//							//		+ " AND DYVALID >= TO_CHAR(SYSDATE,'YYYYMMDD')"
//							//		+ " AND DIFF_DATE( DYVALID ) >= 0"
//									+ " ORDER BY SYSTEM_ID, FGDEFAULT DESC" ;

	/**
	 * ユーザーリソース検索SQL(ロール指定)。
	 * 検索するカラムには、必ず、USERID,LANG,NAME_JA,ROLES,DROLES がこの順番で含まれており、
	 * 絞込み条件(?パラメータ)として、SYSTEM_ID,USERID がこの順番で指定される必要があります。
	 * (カラム名は関係ありません。並び順と意味が重要です。)
	 * また、検索順(ORDER BY)は、優先順位の低い順に検索してください。使用するのは、一番最後に
	 * 検索された行を使用します。
	 * ユーザーリソースは、RESOURCE_USER_DBID で指定のデータベースから取得します。
	 * 未定義の場合は、RESOURCE_DBID から、それも未定義の場合は デフォルトの接続先を使用します。
	 * 
	 * @og.rev 4.4.0.0 (2008/08/02) DROLES(データロールズ)の項目追加
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String RESOURCE_USER_ROLE_SQL
									= "SELECT USERID,LANG,NAME_JA,ROLES,DROLES"
									+ " FROM GEA10"
									+ " WHERE SYSTEM_ID IN ( ?,'**') AND USERID=? AND ROLES=? AND FGJ='1'"
									+ " ORDER BY SYSTEM_ID" ;

//	public static final String RESOURCE_USER_ROLE_SQL = "SELECT USERID,LANG,NAME_JA,ROLES FROM GEA10"
//									+ " WHERE SYSTEM_ID IN ( ?,'**') AND USERID=? AND ROLES=? AND FGJ='1'"
//									+ " ORDER BY SYSTEM_ID" ;

	/**
	 * ユーザーリソースの情報を取得するデータソースの種別を指定します。(""(初期値)⇒DB,LDAP)
	 * 何も指定しない場合(初期値)は、DBからユーザーリソースの情報を取得します。
	 * "LDAP"を指定すると、LDAPからユーザーリソースを取得します。
	 *  "LDAP"を使用する場合の設定方法については、RESOURCE_USER_LDAP_CLMの説明を参照して下さい。
	 * 
	 * @og.paramLevel 2:使用時設定
	 */
	// 5.2.0.0 (2010/09/01)
	public static final String RESOURCE_USER_SRC_TYPE = "";

	/**
	 * ユーザーリソース検索LDAP項目。
	 * ユーザーをLDAPから検索する場合のカラムを指定します。
	 * カラムには、必ず、USERID,LANG,NAME_JA,ROLES,DROLES がこの順番で含まれる必要があります。
	 * 検索条件は、RESOURCE_USER_LDAP_FILTER で指定しておいてください。
	 * 対象となる属性名が存在しない場合は、ダミーの属性名を指定することもできます。
	 * 各項目において、値が取得できない場合は、以下のデフォルト値が適用されます。
	 *  lang(言語)             : 'ja'
	 *  cn(名称)               : ログインID
	 *  roles(ロールズ)        : 属性名と同じ文字列
	 *  droles(データロールズ) : null
	 * 
	 * なお、LDAPの基本情報は、LDAP_INITIAL_CONTEXT_FACTORY , LDAP_PROVIDER_URL ,
	 * LDAP_ENTRYDN , LDAP_PASSWORD , LDAP_SEARCH_BASE , LDAP_SEARCH_SCOPE
	 * 定数の値を使用します。
	 * 
	 * ※LDAPからユーザーリソースを取得するには、RESOURCE_USER_SRC_TYPEが"LDAP"に設定されている必要があります。
	 *
	 * @og.paramLevel 2:使用時設定
	 */
	// 5.2.0.0 (2010/09/01)
//	public static final String RESOURCE_USER_LDAP_CLM = "uid,cn,officeName,ou,mail,belongOUID" ;
	public static final String RESOURCE_USER_LDAP_CLM = "uid,lang,cn,roles,droles" ;

	/**
	 * ユーザーリソース検索LDAP条件(ロール指定)。
	 * ユーザーをロール指定でLDAPから検索する場合の検索条件を指定します。
	 * 検索条件で定義される {0}にはシステムIDが、{1}にはログインIDが、{2}にはロールが指定されます。
	 * システムIDの検索条件が不要な場合は、{0}の指定を外すことも可能です。
	 * また、ここで指定された条件は、LDAP検索のフィルターとしてそのまま渡されますので、
	 * 固定条件を追加することもできます。
	 * 
	 * なお、LDAPの基本情報は、LDAP_INITIAL_CONTEXT_FACTORY , LDAP_PROVIDER_URL ,
	 * LDAP_ENTRYDN , LDAP_PASSWORD , LDAP_SEARCH_BASE , LDAP_SEARCH_SCOPE
	 * 定数の値を使用します。
	 * 
	 * ※LDAPからユーザーリソースを取得するには、RESOURCE_USER_SRC_TYPEが"LDAP"に設定されている必要があります。
	 *
	 * @og.paramLevel 2:使用時設定
	 */
	// 5.2.0.0 (2010/09/01)
	public static final String RESOURCE_USER_ROLE_LDAP_FILTER = "(&(sysid={0})(uid={1})(roles={2}))" ;

	/**
	 * ユーザーリソース検索LDAP条件。
	 * ユーザーをLDAPから検索する場合の検索条件を指定します。
	 * 検索条件で定義される {0}にはシステムIDが、{1}にはログインIDが指定されます。
	 * システムIDの検索条件が不要な場合は、{0}の指定を外すことも可能です。
	 * また、ここで指定された条件は、LDAP検索のフィルターとしてそのまま渡されますので、
	 * 固定条件を追加することもできます。
	 * 
	 * なお、LDAPの基本情報は、LDAP_INITIAL_CONTEXT_FACTORY , LDAP_PROVIDER_URL ,
	 * LDAP_ENTRYDN , LDAP_PASSWORD , LDAP_SEARCH_BASE , LDAP_SEARCH_SCOPE
	 * 定数の値を使用します。
	 * 
	 * ※LDAPからユーザーリソースを取得するには、RESOURCE_USER_SRC_TYPEが"LDAP"に設定されている必要があります。
	 *
	 * @og.paramLevel 2:使用時設定
	 */
	// 5.2.0.0 (2010/09/01)
//	public static final String RESOURCE_USER_LDAP_SQL = "(&(objectClass=person)(uid={0})(!(uid=R*))(|(belongOUID=61200)(belongOUID=61100)))" ;
	public static final String RESOURCE_USER_LDAP_FILTER = "(&(sysid={0})(uid={1}))" ;

	/**
	 * ユーザーパラメータ検索SQL。
	 * システムパラメータを個別ユーザー毎にカスタマイズする場合に使用する
	 * ユーザーパラメータを取得するSQLを定義します。
	 * 検索するカラムには、必ず、PARAM_ID,PARAM,FGCRYPT がこの順番で含まれており、
	 * 絞込み条件(?パラメータ)として、SYSTEM_ID,USERID がこの順番で指定される必要があります。
	 * (カラム名は関係ありません。並び順と意味が重要です。)
	 * また、検索順(ORDER BY)は、優先順位の低い順に検索してください。同一PARAM_IDが
	 * 複数存在する場合は、一番最後に検索されたPARAMを使用します。
	 * ユーザーパラメータは、ユーザーリソース と異なり、RESOURCE_DBID から取得します。
	 * それが未定義の場合は デフォルトの接続先を使用します。
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USER_PARAMETER_SQL = "SELECT PARAM_ID,PARAM,FGCRYPT FROM GE16"
									+ " WHERE SYSTEM_ID IN ( ?,'**') AND USERID=? AND FGJ='1'"
									+ " ORDER BY SYSTEM_ID" ;

	/**
	 * アクセス統計テーブル使用設定。
	 * アクセス統計テーブルを使用する場合は、true , 使用しない場合は、false を設定します。
	 * 初期値は、true です。
	 * USE_ACCESS_TOKEI_TABLE = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_ACCESS_TOKEI_TABLE = "true" ;

	/**
	 * アクセス統計テーブルを再編成するまでのデータの生存期間を設定します。
	 * データが作成された日時を基準として、ここで指定された期間よりも古い
	 * データは、物理削除されます。
	 * ACCESS_TOKEI_ALIVE_DAYSが指定されていない(null)場合、データの削除は行われません。
	 * 初期値は、180日です。
	 *
	 * @og.paramLevel 1:設定確認
	 */
	public static final String ACCESS_TOKEI_ALIVE_DAYS = "180" ;

	/**
	 * アクセス統計テーブルを使用したメニュー表示の過去日付の設定日数。
	 * 階層メニューでレベル３の場合、非標準としてデフォルトでは表示されません。
	 * これに、アクセス統計テーブルの個人別過去の訪問履歴より、設定日数以内に
	 * アクセスされていれば、動的にレベル２へ引き上げます。
	 * 前準備として、アクセス統計テーブル使用設定(USE_ACCESS_TOKEI_TABLE)を
	 * true にしておく必要があります。
	 * 初期値は、20日 です。
	 * ACCESS_TOKEI_PAST_DAYS = "20" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String ACCESS_TOKEI_PAST_DAYS = "20" ;

	/**
	 * ＳＱＬインジェクション対策用チェックの実施有無の設定 。
	 * 特定リクエスト情報にクォーティション(')が含まれている場合に、入力エラーに
	 * するかどうかを指定します。
	 * SQLのWHEREパラメータの値に、' or 'A' like 'A のような形式を使用すると、
	 * 意図しない情報を抜き出すことが可能になります。
	 * リクエストが数字タイプの場合は、or 5=5 などのクォーティションを使用しない
	 * コードを埋めても、数字チェックで検出可能です。
	 * エンジンでは、２つのSQLを連続で流す事が出来ないので、上記形式でのチェックで
	 * ある程度のセキュリティは確保できます。
	 * 初期値は、true です。
	 * USE_SQL_INJECTION_CHECK = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_SQL_INJECTION_CHECK = "true" ;

	/**
	 * 個人別お気に入りメニューを登録する場合の画面IDリスト。
	 * このキーには、お気に入りメニューの画面IDをカンマ区切りで記述します。
	 * 個人設定可能ですが、SystemData としては、初期値を持っていません。
	 * GUI_FAVORITE_MENU_LIST = "" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String GUI_FAVORITE_MENU_LIST = "";

	/**
	 * ダイレクトジャンプリンク時のハッシュ化されたURLの有効期間を日数の初期値。
	 * これは、登録された時間から計算して、何日間有効にしておくかを指定します。
	 * ゼロ、またはマイナスが指定された場合は、制限なしに設定されます。
	 * 初期値は、20日 です。
	 * LINK_VALID_DURATION_DAYS = "20" ;
	 * @og.paramLevel 4:個人設定可
	 */
	public static final String LINK_VALID_DURATION_DAYS = "20" ;

	/**
	 * Tomcat起動時初期処理 URL。
	 * これは、Tomcatが起動された初期化処理時に呼び出されるURLを指定します。
	 * エンジンでは、各画面に対してURLを呼び出すことで、処理を実行することが
	 * 可能になります。例えば、帳票デーモンの起動URLを設定しておけば、Tomcat
	 * 起動時に、URLがコールされて、帳票デーモンが自動的に起動します。
	 * このキーで始まるキーで登録します。処理は、キーのソート順です。
	 *
	 * CONTEXT_INITIAL_CALL_URL = "" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String CONTEXT_INITIAL_CALL_URL = "" ;
	
	/**
	 * Tomcat起動時初期処理 URLを起動するユーザー、パス。
	 * これは、Tomcatが起動された初期化処理時に呼び出されるURLを実行する際の
	 * ユーザー及びパスを指定します。
	 * [ユーザー名]:[パスワード]の形式で指定します。
	 * 初期値はadmin:admin(ユーザー、パスワード共にadmin)です。
	 * 
	 * @og.rev 4.2.2.0 (2008/05/22) 新規作成
	 *
	 * CONTEXT_INITIAL_CALL_USERPASS = "admin:admin" ;
	 * @og.paramLevel 7:要再起動
	 */
	public static final String CONTEXT_INITIAL_CALL_USERPASS = "admin:admin" ;

	/**
	 * ファイルダウンロード時の半角カナ-全角カナ変換使用有無。
	 * ファイルダウンロード時のファイル名を半角カナ-全角カナ変換をを行うかどうかを指定します。
	 * FileDownload サーブレットで、漢字(特に半角カタカナ)を含むファイル名でダウンロード
	 * させようとすると、ファイル名が途中で切れてしまう現象が発生します。
	 * 原因は、IEの仕様で、ある程度仕方がないとの事です。
	 * エンジンでは、URLEncodeを行うことで、若干の改善は図られていますが、半角カタカナを
	 * 半角カタカナを全角カタカナに変換することでさらに改善を図ることが可能です。
	 * 初期値は true(半角カナ-全角カナ変換を行う)です。
	 *
	 * USE_FILEDOWNLOAD_HAN_ZEN = "true" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_FILEDOWNLOAD_HAN_ZEN = "true" ;

	/**
	 * PAGE_END_CUT フラグを使用(true)すると、最後の一行が表示されない不具合対応です。
	 * これは、データが存在しなくなった状態で、PAGE_END_CUTという文字列を見つけると
	 * それ以降の、PAGE_END_CUTという文字列が存在している行を削除するという仕様です。
	 * そのときのデータ存在チェックの関係で、最後のデータが存在しているにもかかわらず
	 * その行を削除していました。(判定に rowOver フラグを見ていた)
	 * これを、データがなくなったことを示す、dataOver フラグを見るようにします。
	 * COMPATIBLE_PAGE_END_CUT_RETRIEVAL フラグは、互換性を考慮し、true の場合は、
	 * dataOver フラグを見ます。(つまり正常に動作します)
	 * false(初期値)の場合は、現在の状態のままとして、rowOver フラグを見ます。
	 * 初期値は 互換性を考慮し false(現在の状態のまま)とします。
	 *
	 * @og.rev 3.8.1.2 (2005/12/19) 新規作成
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String COMPATIBLE_PAGE_END_CUT_RETRIEVAL = "false" ;

	/**
	 * ガントヘッダーにおいて、時間ヘッダーの上段に日付を、下段の時間軸の(h)の削除を行います。
	 * 日付は、表示開始時刻の上と、その１２時間後に表示します。
	 * この２段組の対応を使用するかどうかを指定します。
	 * 初期値は、false です。
	 *
	 * @og.rev 3.8.1.4 (2006/03/13) 新規作成
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_GANTTHEADER_MULTIPLE_COLUMN = "false" ;

	/**
	 * ガントヘッダーにおいて、30日、90日、180日のヘッダー表示で、日付まで
	 * 表示するか、月のみ表示するかを指定します。
	 * ガントヘッダーの繰返し表示でバックグラウンドイメージは固定日なので、一月表示を
	 * 30日単位で表示していますが、先頭日付が 1日でないため、判りにくいと苦情がありました。
	 * そこで、30日、90日、180日のヘッダー表示時に、日付を表示しないようにするフラグです。
	 * 月表示は、その月が 20日以降の場合、翌月を表示します。
	 *
	 * 初期値は、互換性の為、false です。
	 *
	 * @og.rev 3.8.5.2 (2006/06/09) 新規作成
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_GANTTHEADER_MONTH_DATE = "false" ;

	/**
	 * ユーザー情報の IDNO で ログインユーザーIDの先頭 C を取り除くかどうか。
	 *
	 * ユーザー情報の IDNO で ログインユーザーIDの先頭 C を取り除くかどうかを
	 * 指定するパラメータ(USE_USER_IDNO_C_SAPLESS)を追加します。
	 * true にすると、現状とおなじ、６桁以上の場合に 先頭 C を取り除きます。
	 * alse にすると、ID と IDNO が同じ値にセットされます。
	 * それ以外の IDNO を使用する場合は、3.8.1.2 (2005/12/19) で対応したように
	 * DNO に直接値を設定する機能を使用して、書き換えてください。
	 * 初期値は、互換性の為、true です。
	 *
	 * @og.rev 3.8.5.3 (2006/06/30) 新規作成
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_USER_IDNO_C_SAPLESS = "true" ;

	/**
	 * カレンダデータの初期設定
	 * カレンダーDBにデータが存在しない場合に、使用されるカレンダークラスを設定します。
	 * 初期設定は、org.opengion.plugin.calendar パッケージの SUNDAY(日曜日のみが休日) です。
	 *
	 * @og.rev 4.0.0.0 (2007/08/28) 新規作成
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String DEFAULT_CALENDAR_CLASS = "org.opengion.hayabusa.resource.CalendarPGData_SUNDAY" ;

	/**
	 * 時間単位のガントヘッダーにおいて、休日を表示するかどうかを判定します。
	 * 初期値は、true です。
	 *
	 * @og.rev 4.0.0.0 (2007/09/05) 新規作成
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_GANTTHEADER_SKIP_HOLIDAY = "true" ;

	/**
	 * 帳票デーモンの検索条件(ORDER BY)を指定します。
	 *
	 * １ページ目と２ページ目で帳票IDを変更し、２ページ目のみを
	 * 大量出力する場合は、検索順（つまり、印字順）を変更することで、対応できます。
	 * ただし、帳票システムは、共通に使用するため、勝手にソート順を変更すると
	 * 大変なことになる為、自システムのみで使用することを確認の上、設定変更を
	 * して下さい。並び替えの項目の前には、必ず"A."(GE50)を付加して下さい。
	 *
	 * @og.rev 4.0.1.0 (2007/12/19) 新規作成 (Ver3より復活)
	 *
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_DAEMON_ORDER_BY = " A.SYSTEM_ID,A.LISTID,A.GROUPID,A.YKNO" ;

	/**
	 * RFID帳票を発行するためのサービスクラスを指定します。
	 * org.opengion.hayabusa.report.RFIDPrintPointService を継承している必要があります。
	 * デフォルト設定は、org.opengion.hayabusa.report.rfid.RFIDPrintPointService_INUYAMAです。
	 *
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String REPORT_RFID_SERVICE_CLASS = "org.opengion.hayabusa.report.rfid.RFIDPrintPointService_INUYAMA" ;

	/**
	 * GEシステムのフォルダ名(URLのコンテキスト名)。
	 * GE_CONTEXT_NAME = "ge4" ;
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String GE_CONTEXT_NAME = "ge4" ;

	/**
	 * 帳票出力に使用するOpenOffice.orgのsoffice.binプロセスの最大数。
	 * 
	 * 初期値は256ですが、デーモンブループの分だけ起動されるため、通常
	 * 意識する必要はありません。
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_MAX_PROCESS_COUNT = "256" ;

	/**
	 * 帳票出力に使用するOpenOffice.orgのsoffice.binプロセスの生存時間(秒)。
	 * 
	 * soffice.binのプロセスは、プールされ各デーモングループで再利用されます。
	 * この時、生成から一定期間が過ぎたプロセスは、一度終了させ、再度起動します。
	 * 初期値は1800秒です。
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_PROCESS_ALIVE = "1800" ;

	/**
	 * 帳票出力時にエラーが発生した場合の再試行回数。
	 * 
	 * 帳票出力処理は、全体で[再試行回数+1回]処理を試みます。
	 * 初期値は0(リトライしない)です。
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_RETRY_COUNT = "0" ;

	/**
	 * 帳票出力に使用するOpenOffice.orgのsoffice.binプロセスに接続する方法を指定します。
	 * 
	 * 特に理由がない限り、初期値のPIPE接続を利用して下さい。
	 * 但し、PIPE接続を利用した場合、64Bit版のJavaから帳票システムを動作させると、
	 * "UnsatisfiedLinkError"が発生することがあります。
	 * このような場合は、この値を"TCP"に変更し、soffice.binのプロセスにTCP接続するようにして下さい。
	 * 初期値は"PIPE"(PIPE接続)です。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_OOO_CONN_TYPE = "PIPE" ;

	/**
	 * 帳票出力で接続方法(REPORT_OOO_CONN_TYPE)を"TCP"(TCP接続)にした場合の初期ポート番号を指定します。
	 * 
	 * soffice.binへの接続方法をTCP接続にした場合の、初期の接続ポート番号を指定します。
	 * このポート番号を基点に、最大で
	 *  [初期ポート番号]+[接続最大プロセス数(REPORT_MAX_PROCESS_COUNT)]-1
	 * のポート番号が使用される可能性がありますので、この間のポート番号は確実に使用されないことを
	 * 確認した上で、この初期ポート番号を指定して下さい。
	 * 初期値は8100です。
	 * 
	 * @og.rev 5.1.7.0 (2010/06/01) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String REPORT_OOO_MIN_PORT = "8100" ;

	/**
	 * 入力値の自動変換を行うかを設定します。
	 * trueが設定された場合、DBTypeに応じて入力値に対する大文字、小文字の変換が
	 * 自動で行われます。自動変換の有無及び仕様については、各DBTypeで定義されて
	 * います。該当のDBTypeが自動変換対象でない場合は、trueに設定しても何も起こ
	 * りません。
	 * 特に理由がない限りは、ユーザービリティ向上のため、trueに設定して下さい。
	 * 初期値はtrue(自動変換する)です。
	 * USE_INPUT_AUTO_CONVERT = "true" ;
	 * 
	 * @og.rev 4.3.0.0 (2008/07/23) 新規作成
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * 
	 * @og.paramLevel 1:設定確認
	 */
//	public static final String USE_INPUT_AUTO_CONVERT = "true" ;
	
	/**
	 * 入力値をリアルタイムでチェックするかを設定します。
	 * trueが設定された場合、リアルタイムでDBTypeに応じたチェックが行われます。
	 * 特に理由がない限りは、ユーザービリティ向上のため、trueに設定して下さい。
	 * チェック用のJSPの初期値は、REAL_TIME_CHECK_JSPで設定します。
	 * 初期値はtrue(リアルタイムチェックを行う)です。
	 * USE_REAL_TIME_CHECK = "true" ;
	 * 
	 * @og.rev 4.3.0.0 (2008/07/23) 新規作成
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * 
	 * @og.paramLevel 1:設定確認
	 */
//	public static final String USE_REAL_TIME_CHECK = "true" ;
	
	/**
	 * 入力値をリアルタイムでチェックする場合の共通JSPを設定します。
	 * この設定は、USE_REAL_TIME_CHECKをtrueに設定した場合のみ有効となります。
	 * 初期値は/jsp/common/realtimecheck.jspです。
	 * REAL_TIME_CHECK_JSP = "/jsp/common/realtimecheck.jsp" ;
	 * 
	 * @og.rev 4.3.0.0 (2008/07/23) 新規作成
	 * @og.rev 4.3.5.0 (2009/02/01) JavaScript整理につき削除
	 * 
	 * @og.paramLevel 1:設定確認
	 */
	// public static final String REAL_TIME_CHECK_JSP = "/jsp/common/realtimecheck.jsp" ;

	/**
	 * 画面遷移なしの登録を行うかどうかを指定します。
	 * trueが指定された場合、画面遷移を行わず選択行のみを動的に切り替えることで、
	 * 編集、登録を行います。サーバーへのデータ送信には、jQueryのFormPluginを
	 * 利用しています。
	 * また、データを送信する対象のフォームには、id="noTransition"を指定して下さい。
	 * (該当のIDのフォームが存在しない場合は、JSPの1番目のフォームが送信されます。)
	 * 
	 * 初期値はfalse(画面遷移なしモードを使用しない)です。
	 * NO_TRANSITION_MODE = "false" ;
	 * 
	 * @og.rev 4.3.1.0 (2008/09/02) 新規作成
	 * 
	 * @og.paramLevel 1:設定確認
	 */
	public static final String NO_TRANSITION_MODE = "false" ;

	/**
	 * 利用するヘッダ固定スクリプト及びスタイルシートを選択します。
	 * 拡張子を抜いた状態でファイルを指定して下さい。
	 * 
	 * 初期値 common/scrollBar
	 * SCROLL_BAR_TYPE = "common/scrollBar"
	 * 
	 * @og.rev 4.3.5.0 (2009/02/01) 新規作成
	 * @og.rev 5.0.0.0 (2009/08/03) scrollBar2を標準とする。
	 * 
	 * @og.paramLevel 1:設定確認
	 */
//	public static final String SCROLL_BAR_TYPE = "common/scrollBar" ;
	public static final String SCROLL_BAR_TYPE = "common/scrollBar2" ;
	
	/**
	 * hideMenuの初期指定の「開く」ボタンの画像を指定します。
	 * 
	 * hideMenuタグで作成されるボタンの画像です。
	 * jsp/imageフォルダ内の画像ファイルを指定して下さい。
	 * 
	 * 初期値 expand_u.png
	 * HIDE_OPEN_BUTTON = "expand_u.png"
	 * 
	 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
	 * 
	 * @og.paramLevel 1:設定確認
	 */
	public static final String HIDE_OPEN_BUTTON = "expand_u.png" ;
	
	/**
	 * hideMenuの初期指定の「閉じる」ボタンの画像を指定します。
	 * 
	 * hideMenuタグで作成されるボタンの画像です。
	 * jsp/imageフォルダ内の画像ファイルを指定して下さい。
	 * 
	 * 初期値 shrink_u.png
	 * HIDE_CLOSE_BUTTON = "shrink_u.png"
	 * 
	 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
	 * 
	 * @og.paramLevel 1:設定確認
	 */
	public static final String HIDE_CLOSE_BUTTON = "shrink_u.png" ;
	
	/**
	 * eventColumn設定時の標準処理JSPファイルの指定します。
	 * 
	 * eventColumnを設定した場合に実際にカラムを作成する標準のJSPを指定します。
	 * ここでの記述はjspフォルダを基底にして下さい
	 * 
	 * 初期値 common/eventColumnMaker.jsp
	 * EVENT_COLUMN_URL = "common/eventColumnMaker.jsp"
	 * 
	 * @og.rev 4.3.6.0 (2009/04/01) 新規作成
	 * 
	 * @og.paramLevel 1:設定確認
	 */
	public static final String EVENT_COLUMN_URL = "common/eventColumnMaker.jsp" ;

	/**
	 * TableUpdateParamで使用される固定カラムを処理するクラスを指定します。
	 * 
	 * TableUpdateParamで通常利用されるクラスは
	 * org.opengion.plugin.query.DBConstValue_"SYSTEM_ID"
	 * ですが、このプラグインが存在しない場合に利用される標準のクラスを設定します。
	 * 
	 * 初期値 null
	 * DEFAULT_CONST_CLASS = null
	 * 
	 * @og.rev 4.3.6.4 (2009/05/01) 新規作成
	 * 
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String DEFAULT_CONST_CLASS = null ;
	
	/**
	 * IE8でIE7エミュレートモードで動作させるタグを吐くかを指定します。
	 * 
	 * trueを指定するとIE8でIE7エミューレートモードで動作するように
	 * og:headタグでmetaタグを出力します。
	 * 通常は初期値のtrue(IE7エミューレートモードで動作させる)を利用します。
	 * 
	 * 初期値 true
	 * USE_IE7_HEADER = "true"
	 * 
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_IE7_HEADER = "true" ;
	
	/**
	 * エラーをポップアップ形式で表示するかを指定します。
	 * 
	 * noTransitionがfalseでも、この値をtrueにする事でajaxを利用して
	 * エラーをポップアップ表示します(command=ENTRY限定)。
	 * (一覧でeventColumnを利用する画面は戻るボタンで戻った場合に
	 * 変更した値が保持されていないため、trueが望ましい）
	 * noTransitionがtrueの場合はこの設定は無視され、全画面で非遷移となります。
	 * 初期値はfalse(利用しない)です。
	 * 
	 * 初期値 false
	 * USE_AJAX_SUBMIT = "false"
	 * 
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * 
	 * @og.paramLevel 2:使用時設定
	 */
	public static final String USE_AJAX_SUBMIT = "false" ;
	
	/**
	 * ビューの件数表示位置をヘッダ部分に変更するかどうかを指定します。
	 * 
	 * ビューの件数表示位置を変更するJavaScriptの読み込み制御を行います。
	 * 特に表示互換性維持のためにシステム全体で変更する場合はこのフラグを
	 * falseにして下さい。
	 * 個別画面で変更する場合はheadタグの属性で制御可能です。
	 * 
	 * 初期値 true
	 * USE_ROWCNT_REPLACE = "true" ;
	 * 
	 * @og.rev 4.3.8.0 (2009/08/01) 新規作成
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
//	public static final String USE_ROWCNT_REPLACE = "true" ;
	
	/**
	 * リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか(true/false)を設定します。
	 *
	 * trueにするとクロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする（true)／かノーチェックか(false)を指定します。
	 * {&#64;xxx}の置換え時にリクエストパラメータのみ適用します。
	 * attributesに対しては動作しません。
	 * 
	 * 初期値 true
	 * USE_XSS_CHECK = "true" ;
	 * 
	 * @og.rev 5.0.0.2 (2009/09/15) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String USE_XSS_CHECK = "true" ;

	/**
	 * 業務ロジックのJavaソースを格納するディレクトリを指定します。
	 *
	 * 業務ロジックのJavaソースを格納するディレクトリを指定します。
	 * ディレクトリは、コンテキストルートを基点とした相対パスで記述します。
	 * 
	 * 初期値 src
	 * BIZLOGIC_SRC_PATH = "src/biz" ;
	 * 
	 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String BIZLOGIC_SRC_PATH = "src" + File.separator + "biz" ;

	/**
	 * 業務ロジックのJavaクラスファイルを格納するディレクトリを指定します。
	 *
	 * 業務ロジックのJavaクラスファイルを格納するディレクトリを指定します。
	 * ディレクトリは、コンテキストルートを基点とした相対パスで記述します。
	 * 
	 * 初期値 classes
	 * BIZLOGIC_CLASS_PATH = "WEB-INF/biz" ;
	 * 
	 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String BIZLOGIC_CLASS_PATH = "WEB-INF" + File.separator + "biz" ;

	/**
	 * 業務ロジックのJavaソースの動的コンパイルを行うかを指定します。
	 *
	 * 業務ロジックのJavaソースの動的コンパイルを行うかを指定します。
	 * trueを指定すると、Javaソースを変更した後、初めてアクセスされた
	 * タイミングで動的にコンパイルが行われます。
	 * 
	 * 初期値 true
	 * BIZLOGIC_AUTO_COMPILE = "true" ;
	 * 
	 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
	 * 
	 * @og.paramLevel 1:設定確認
	 */
	public static final String BIZLOGIC_AUTO_COMPILE ="true";

	/**
	 * 業務ロジックのJavaクラスファイルの動的ロードを行うかを指定します。
	 *
	 * 業務ロジックのJavaクラスファイルの動的ロードを行うかを指定します。
	 * trueを指定すると、Javaクラスファイルが変更された後、初めてアクセスされた
	 * タイミングで動的にクラスロードが行われます。
	 * 
	 * 初期値 true
	 * BIZLOGIC_HOT_DEPLOY = "true" ;
	 * 
	 * @og.rev 5.1.1.0 (2009/12/01) 新規作成
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String BIZLOGIC_HOT_DEPLOY ="true";

	/*******************************************************************************
	 * 
	 * 以下は、Ver4と共通でリソースDB(GExx)を利用する場合にのみ設定します。
	 * 通常、openGionを利用する場合は、確認・設定の必要はありません。
	 * 
	 *******************************************************************************/

	/**
	 * (通常は使いません)Ver4と共通のリソースDB(GExx)を利用する場合に設定します。
	 * 
	 * Ver4と共通のDBを利用する場合は、trueを設定することで、openGion本体(Java)及び
	 * GEシステムの各画面からアクセスする際の項目名が、旧項目名に変換されます。
	 * 
	 * 内部的に項目名が変化される項目一覧は以下の通りです。
	 * 
	 *   テーブルID		(false) ※openGion標準	(true) ※Ver4互換
	 *   ------------	-----------------------	--------------------
	 *   GEA05			ROLES					ROLE
	 *   GEA10			NAME_JA					NAME
	 *   GE50			OUT_FILE				OUTFILE
	 *   GE50			OUT_DIR					OUTDIR
	 *   GE51			TEXT_DATA				TEXT
	 *   GE51			CLM						COLUMN_NAME
	 *   GE53			OUT_FILE				OUTFILE
	 *   GE53			OUT_DIR					OUTDIR
	 *   GE55			HOST_ID					HOST
	 *   GE58			CLM						COLUMN_NAME
	 *   GE30			H_TXT					HEADER
	 *   GE30			F_TXT					FOOTER
	 *   GE31			CONTENTS				CONTENT
	 *   GE32			CONTENTS				CONTENT
	 *   GE33			ADDRESS					MEMBER
	 *   GE33			NAME_JA					NAME
	 *   GE33			KBNAME					NM_KBN
	 * 
	 * @og.paramLevel 3:標準推奨
	 */
	public static final String VER4_COMPATIBLE_MODE = "false";
}
