/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.fukurou.util.StringUtil;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.File;
import java.io.FileFilter;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * ファイルを検索し、DBTableModel にセットするタグです。
 *
 * ファイルの検索結果は、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH"
 * のカラムを持つ DBTableModel にセット されます。このカラムは、固定です。
 * 並び替えについては、このタグで指定しますが、ファイルの選別(where 条件)は、
 * BODY 部に記述する fileWhere タグで指定します。（複数指定可能））
 *
 * [カラム名]      検索するファイルの属性は、以下のカラム名で作成されます。
 *      LEVEL           ディレクトリを展開する場合のレベル。
 *      FILE_TYPE       ファイル(F)かディレクトリ(D)であるか判定。
 *      PARENT          この抽象パス名の親のパス名文字列を返します。
 *      NAME            この抽象パス名が示すファイルまたはディレクトリの名前を返します。
 *      LASTMODIFIED    最後に変更された時刻を返します。
 *      FILE_LENGTH     ファイルの長さを返します。
 *      RWH             読み込み、書き込み、隠し属性をそれぞれ、ｒ,w,h で表します。
 *
 * [from 属性]      検索を開始するファイルまたはディレクトリの名前
 *
 * [multi 属性]     ディレクトリを下位展開するかどうか？
 *      true            下位展開する
 *      false           下位展開しない（初期値）
 *
 * [tableId 属性]       settion に登録する時のID
 * [scope 属性]         settion に登録する時のスコープ "request","page","session","applicaton"
 * [maxRowCount 属性]   検索時の最大検索件数
 * [displayMsg 属性]    検索終了時に表示する メッセージリソースのID
 * [overflowMsg 属性]   オーバーフロー時に表示する メッセージリソースのID
 * [command 属性]       実行を制御するコマンド "NEW" と "RENEW" 時のみ実行する。（初期値 NEW）
 *
 * @og.formSample
 * ●形式：&lt;og:fileQuery from="…" multi="true/false" &gt;
 *             &lt;og:fileWhere … /&gt;
 *              …
 *         &lt;/og:fileQuery&gt;
 * ●body：あり
 *
 * ●使用例
 *    ・一般的な属性でファイルの検索を行います。
 *         &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere endWith=".jsp" /&gt;
 *      &lt;/og:fileQuery&gt;
 *
 *    ・最終変更日で逆順ソートする。対象は、2002/10/01 以降に変更されたファイル。
 *        &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere lastModified="20021001000000" /&gt;
 *        &lt;/og:fileQuery&gt;
 *
 * @og.rev 4.0.0 (2005/01/31) 内部ロジック改定
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileQueryTag extends QueryTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String[] SELECT =
				new String[] { "LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" };

	private static final int LEVEL			= 0;
	private static final int FILE_TYPE		= 1;
	private static final int PARENT			= 2;
	private static final int NAME			= 3;
	private static final int LASTMODIFIED	= 4;
	private static final int FILE_LENGTH	= 5;
	private static final int RWH			= 6;

	private boolean		multi		= false;						// 下位層展開ﾌﾗｸﾞ
	private int			level		= 100;							// 展開ﾚﾍﾞﾙ
	private String      from		= HybsSystem.sys( "FILE_URL" );	// 検索起点ﾌｧｲﾙ
	private transient FileFilter	filter		= null;							// FileWhere で指定したフィルター

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		executeCount = 0;

		table = initDBTable();
		if( maxRowCount < 0 ) {
			maxRowCount	= sysInt( "DB_MAX_ROW_COUNT" ) ;
		}

		execute( new File( from ),0 ) ;

		return(SKIP_BODY);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		multi		= false;
		level		= 100;
		from		= HybsSystem.sys( "FILE_URL" );
		filter		= null;
	}

	/**
	 * FileQuery を実行します。
	 *
	 * @param fin File
	 * @param lvl int
	 */
	protected void execute( final File fin,final int lvl ) {
		if( ( !multi && lvl > 1 ) || lvl > level ) { return; }	// 階層展開する、しない
		if( executeCount > maxRowCount ) { table.setOverflow( true ); return; }

		addFileData( executeCount++,lvl,fin );
		if( fin.isDirectory() ) {
			File[] list = fin.listFiles( filter );
			for( int i = 0; i < list.length; i++ ) {
				execute( list[i],lvl+1 );
			}
		}
	}

	/**
	 * 初期化された DBTableModel を返します。
	 *
	 * @return	 DBTableModel テーブルモデル
	 */
	private DBTableModel initDBTable() {
		DBTableModel tbl = DBTableModelUtil.newDBTable();

		tbl.init( SELECT.length );
		for( int i=0; i<SELECT.length; i++ ) {
			DBColumn dbColumn = getDBColumn( SELECT[i] );
			tbl.setDBColumn( i,dbColumn );
		}

		return tbl ;
	}

	/**
	 * DBTableModel に、ファイル情報をセットします。
	 * ファイル情報は、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" です。
	 *
	 * @param	 rowNo int  セットする行番号
	 * @param	 lvl   int  セットするレベル
	 * @param	 fin   File ファイル情報の元となるファイルオブジェクト
	 */
	private void addFileData( final int rowNo,final int lvl,final File fin ) {
		try {
			File file = fin.getCanonicalFile();

			String rwh = ((file.canRead())?"r":"-" ) +
							((file.canWrite())?"w":"-" ) +
							((file.isHidden())?"h":"-" ) ;

			String lastModified = HybsSystem.getDate( file.lastModified(),"yyyyMMddHHmmss" );

			String[] data = new String[ SELECT.length ];
			data[LEVEL			] = String.valueOf( lvl ) ;
			data[FILE_TYPE		] = (file.isFile())?"F":"D" ;
			data[PARENT			] = file.getParent() ;
			data[NAME			] = file.getName() ;
			data[LASTMODIFIED	] = lastModified ;
			data[FILE_LENGTH	] = String.valueOf( file.length() ) ;
			data[RWH			] = rwh ;

			table.addColumnValues( data );
		}
		catch( IOException ex ) {
			String errMsg = "正式なファイル名の取得に失敗しました。[" + fin + "]"
						+ " ROW=[" + rowNo + "]"
						+ HybsSystem.CR + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
	}

	/**
	 * 【TAG】ファイルの検索元となるディレクトリを指定します。
	 *
	 * @og.tag ファイルの検索元となるディレクトリを指定します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルの検索元となるディレクトリ
	 */
	public void setFrom( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
		}
		furl = StringUtil.urlAppend( from,furl );
		furl = StringUtil.urlAppend( furl,"." );

		from = HybsSystem.url2dir( furl );
	}

	/**
	 * 【TAG】多段階展開するか、１レベル展開するか指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は、false (１レベル) です。
	 *
	 * @param	mlti 多段階展開するか(true:する/false :１レベル)
	 */
	public void setMulti( final String mlti ) {
		multi = nval( getRequestParameter( mlti ),multi );
	}

	/**
	 * 【TAG】多段階展開するレベルを指定します(初期値:100)。
	 *
	 * @og.tag
	 *
	 * @param	lvl 多段階展開するレベル
	 */
	public void setLevel( final String lvl ) {
		level = nval( getRequestParameter( lvl ),level );
	}

	/**
	 * FileFilterオブジェクトをセットします。
	 * これは、BODY 部に登録した、FileWhereタグによって設定された
	 * ファイルフィルターです。
	 *
	 * @param	 filter	FileFilter オブジェクト
	 */
	protected void setFileFilter( final FileFilter filter ) {
		this.filter = filter;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "multi"		,multi	)
				.println( "level"		,level	)
				.println( "from"		,from	)
				.fixForm().toString()
			+ HybsSystem.CR
			+ super.toString() ;
	}
}
