/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval ;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.ChartCreate;
import org.opengion.hayabusa.io.ChartDataset;
import org.opengion.hayabusa.io.HybsURLTagFragmentGenerator;

// import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.util.List;
import java.util.ArrayList;
// import java.util.Iterator ;
import java.util.Map ;
import java.util.HashMap ;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.ChartUtilities;

import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.entity.StandardEntityCollection;
import org.jfree.chart.imagemap.ImageMapUtilities;
import org.jfree.chart.imagemap.StandardToolTipTagFragmentGenerator;
import org.jfree.chart.imagemap.ToolTipTagFragmentGenerator;
import org.jfree.chart.imagemap.URLTagFragmentGenerator;

/**
 * BODY部に指定のSQLの検索結果をチャート(円、棒、線)で表示するタグです。
 *
 * チャート化には、JFreeChart (http://www.jfree.org/jfreechart/) を使用しています。<br />
 * チャートの種類は、chartDataset タグで指定します。これは、複数の異なるチャートを
 * 合成表示する機能が、JFreeChart にある為です。ただし、専門的には、CategoryDataset のみ
 * いまは、合成が可能です。<br />
 * 処理の実行にあたり、キャッシュが有効(useCache="true")であれば、キャッシュデータを
 * 使用します。キャッシュは、masterKey 属性で指定されたキーワード毎に、JFreeChart
 * オブジェクトをキャッシュします。また、seriesPikup 違いの場合は、再検索せずに、
 * キャッシュが使用できます。さらに、seriesPikup が同じ場合(masterKey も同じ)は、
 * すでに画像ファイルが作成済みなので、画像ファイルのみを返します。
 *
 * <p>各属性は、{&#064;XXXX} 変数が使用できます。<br />
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。</p>
 *
 * @og.formSample
 * ●形式：&lt;og:chartCreate title="…" … /&gt;
 * ●body：あり(chartDataset)
 *
 * ●使用例
 *      &lt;og:chartCreate
 *          title       = "JFreeChart Test"             チャートタイトル
 *          domainLabel = "ドメインラベル"              横軸ラベル
 *          width       = "200"                         チャート表示幅
 *          height      = "200"                         チャート表示高さ
 *          showLegend  = "[true/false]"                凡例の表示可否(true/false)
 *          rectangleEdge   = "[TOP|BOTTOM|RIGHT|LEFT]" 凡例の表示箇所
 *          plotOrientation = "[VERTICAL|HORIZONTAL]"   チャートの軸表示方向
 *          chartBackColor  = "WHITE"                   背景色
 *          plotBackColor   = "LIGHT_GRAY"              描画領域色
 *          rotationLabel   = "3"                       横軸ラベルの傾き
 *          domainMarker    = "KING"                    横軸のマーカーライン(縦棒)
 *          useMarkerLabel  = "[true/false]"            マーカーラインの文字有無
 *          useDomainLabel  = "[true/false]"            横軸ラベルの表示有無
 *          categorySkip    = "3"                       横軸ラベルをスキップする間隔
 *          categoryCutNo   = "6"                       横軸ラベルの文字位置指定のキーブレイク
 *          categoryAnchor  = "[START|MIDDLE|END]"      横軸のグリッド(垂直線)の書き出し位置
 *          useDomainLine   = "[false/true]"            横軸のグリッド表示有無(垂直線)
 *          domainLineColor = "LIGHT_GRAY"              横軸のグリッド線の色
 *          useRangeLine    = "[true/false]"            縦軸のグリッド表示有無(水平線)
 *          rangeLineColor  = "LIGHT_GRAY"              縦軸のグリッド線の色
 *          rangeSkip       = "-5"                      縦軸のグリッド(水平線)をスキップする間隔
 *          useCache        = "[false|true]"            キャッシュの有無
 *          masterKey       = "{&#064;XXXX}"            キャッシュ用マスタキー
 *          seriesPikup     = "2"                       ピックアップするシリーズNo
 *          imageMapUrl     = "link.jsp,BLOCK"          クリッカブル・マップ用URL
 *          imageMapTarget  = "CONTENTS"                クリッカブル・マップ用TARGET
 *          categoryMargin  = "[0.0～1.0]"              カテゴリマージン(0.0～1.0)
 *          lowerMargin     = "[0.0～1.0]"              下方マージン(0.0～1.0)
 *          upperMargin     = "[0.0～1.0]"              上方マージン(0.0～1.0)
 *       &gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType}"
 *                  valueLabel  = "{&#064;valueLabel}"
 *                  lowerBound  = "{&#064;lowerBound}"
 *                  upperBound  = "{&#064;upperBound}"
 *                  markValues  = "{&#064;markValues}"
 *                  markColors  = "{&#064;markColors}"
 *                  useGradient   = "{&#064;useGradient}"
 *                  shapesVisible = "{&#064;shapesVisible}"
 *                  useDottedLine = "{&#064;useDottedLine}"
 *                  seriesColors  = "{&#064;seriesColors}"
 *                  valueLabelsVisible = "[true|false]"
 *                  valueMarksVisible  = "[true|false]"
 *              &gt;
 *                         &lt;jsp:text&gt;{&#064;SQL}&lt;/jsp:text&gt;
 *              &lt;/og:chartDataset&gt;
 *      &lt;/og:chartCreate&gt;
 *
 *    複数のグラフを重ね合わせる場合は、chartDataset タグを chartCreate のBODY部に
 *    複数記述します。
 *      &lt;og:chartCreate
 *          title       = "{&#064;title}"
 *          domainLabel = "{&#064;domainLabel}"
 *          width       = "{&#064;width}"
 *          height      = "{&#064;height}" &gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType1}"
 *                  valueLabel  = "{&#064;valueLabel1}"
 *                  lowerBound  = "{&#064;lowerBound1}"
 *                  upperBound  = "{&#064;upperBound1}"
 *                  markValues  = "{&#064;markValues1}"
 *                  markColors  = "{&#064;markColors1}"
 *              &gt;
 *                         &lt;jsp:text&gt;{&#064;SQL1}&lt;/jsp:text&gt;
 *              &lt;/og:chartDataset&gt;
 *              &lt;og:chartDataset
 *                  chartType   = "{&#064;chartType2}"
 *                  valueLabel  = "{&#064;valueLabel2}"
 *                  lowerBound  = "{&#064;lowerBound2}"
 *                  upperBound  = "{&#064;upperBound2}"
 *                  markValues  = "{&#064;markValues2}"
 *                  markColors  = "{&#064;markColors2}"
 *              &gt;
 *                         &lt;jsp:text&gt;{&#064;SQL2}&lt;/jsp:text&gt;
 *              &lt;/og:chartDataset&gt;
 *      &lt;/og:chartCreate&gt;
 *
 *    rectangleEdge属性 は、凡例の表示箇所を示す、RectangleEdge クラスの値を設定します。
 *    [rectangleEdge属性]
 *      TOP      上側
 *      BOTTOM   下側
 *      RIGHT    右側
 *      LEFT     左側
 *
 *    plotOrientation属性 は、チャートの軸表示方向を設定します。
 *    [renderer属性]
 *      VERTICAL    縦方向
 *      HORIZONTAL  横方向
 *
 * @og.group 画面表示
 *
 * @version  0.9.0	2007/06/19
 * @author	 Nakamura
 * @since	 JDK1.4,
 */
public class ChartCreateTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2007/06/29)" ;

	private static final long serialVersionUID = 4000 ;	// 2007/06/29

	private static final String FILE_URL = HybsSystem.sys( "CHART_TEMP_DIR" );

	private		List<ChartDataset>	chartDataset	= null;

	private static final String CHART_CACHE_KEY = HybsSystem.JF_CHART_CACHE_KEY;

	private	String	title			= null;
	private	int		width			= 200;
	private	int		height			= 200;
	private	String	domainLabel		= null;
	private	boolean	showLegend		= true;
	private	String	rectangleEdge	= "BOTTOM";
	private	String	plotOrientation	= "VERTICAL";
	private	String	chartBackColor	= null;			// 背景色の設定
	private	String	plotBackColor	= null;			// 描画領域の設定
	private int		rotationLabel	= 3;			// 横軸ラベルの傾き(1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･ )
	private	String	domainMarker	= null;			// 横軸のマーカーライン
	private	boolean	useDomainLabel	= true;			// 横軸ラベルの表示有無
	private	boolean	useMarkerLabel	= true;			// マーカーライン時の文字有無
	private	boolean	useCache		= false;		// JFreeChart オブジェクトをキャッシュするかどうか
	private int		seriesPikup		= -1;			// ピックアップするシリーズ番号
	private	String	masterKey		= null;			// キャッシュ用マスタキー
	private	transient Map<String,ChartCache>	map	= null;		// キャッシュ用マップ
	private	transient ChartCache	cache		= null;			// キャッシュ
	private	String		imgTag		= null;			// キャッシュ用画像ファイル
	private	String	imageMapUrl		= null;			// クリッカブル・マップ用URL
	private	String	imageMapTarget	= null;			// クリッカブル・マップ用TARGET
	// 3.5.4.7 (2004/02/06) 実行時間測定用のDIV要素を出力します。
	private long dyStart = 0;

	// 4.0.2.0 (2007/12/20)
	private	String		categoryMargin	= null;		// カテゴリマージン(0.0～1.0)
	private	String		lowerMargin		= null;		// 下方マージン(0.0～1.0)
	private	String		upperMargin		= null;		// 上方マージン(0.0～1.0)
//	private String[]	shapeColors		= null;		// 4.0.3.0 (2008/01/07) データ毎にShapeを切り替える時の色指定

	private	boolean	useDomainLine	= false;		// 4.0.3.0 (2008/01/07) 横軸のグリッド表示有無(垂直線)
	private	String	domainLineColor	= null;			// 4.0.3.0 (2008/01/07) 横軸のグリッド線の色
	private	boolean	useRangeLine	= true;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド表示有無(水平線)
	private	String	rangeLineColor	= null;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド線の色
	private int		categorySkip	= 1;			// 4.0.3.0 (2008/01/07) 横軸ラベルをスキップする間隔
	private int		categoryCutNo	= -1;			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のキーブレイク
	private String	categoryAnchor	= null;			// 4.1.1.0 (2008/02/14) 横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)
//	private double	lineOffset		= 0.0;			// 4.1.1.0 (2008/02/04) DomainLineの書き出し位置
	private int		rangeSkip		= 1;			// 4.1.1.0 (2008/02/04) 縦軸のグリッドをスキップする間隔
	private	boolean	useToolTip		= false;		// 4.3.1.0 (2008/08/09) ツールチップスを利用するか
	
	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int 後続処理の指示
	 */
	@SuppressWarnings(value={"unchecked"})
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();

		if( useCache ) {
			// キャッシュから、取り出す。
			map = (Map<String,ChartCache>)getSessionAttribute( CHART_CACHE_KEY );
			if( map != null ) {
				cache = map.get( masterKey );
				if( cache != null && cache.isAvailable() ) {	// キャッシュ有効
					String subKey = String.valueOf( seriesPikup );
					// キャッシュから、イメージを取り出す。存在しなければ、null
					imgTag = cache.getFileAddress( subKey );
					return(SKIP_BODY);		// キャッシュ使用
				}
			}
		}

		return EVAL_BODY_BUFFERED;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) debugPrint() メソッドの処理条件見直し
	 *
	 * @return  int 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		long queryEnd = System.currentTimeMillis();
		jspPrint( "<div id=\"queryTime\" value=\"" + (queryEnd-dyStart) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

//		if( isDebug() ) { debugPrint();	}
		debugPrint();		// 5.2.1.0 (2010/10/01) debugPrint() メソッド自体に、isDebug() が組み込まれている。

		// 画像ファイルのキャッシュがあれば、それを返す。
		if( imgTag != null ) {
			jspPrint( imgTag );
		}
		else {
			JFreeChart chart ;
			if( cache != null ) {
				// キャッシュがあれば、チャートを取り出す。
				chart = cache.getJFreeChart() ;
				// チャートは、変更(modifying)のみ行う。
				chart = ChartCreate.modifying( chart,seriesPikup );
			}
			else {
				// チャートがない、または、無効になっている場合は、新規作成する。
				ChartCreate chartCreate = create();
				chart = chartCreate.getChart();
			}

			File file = getTempFile( FILE_URL );
			try {
				// クリッカブル・マップ
				String fname = file.getName();
				String filename = getContextPath() + "/" + FILE_URL + fname;
				String keyword = fname.substring( 0, fname.length() - 4 ); // ".png" 分を削除
				ToolTipTagFragmentGenerator toolTipGen = null;
				URLTagFragmentGenerator urlGen = null;
				// 4.3.1.0 (2008/08/09) ツールチップスのマップ情報取得
				if( useToolTip ) {
					toolTipGen = new StandardToolTipTagFragmentGenerator();
				}
				if( imageMapUrl != null ) {
					urlGen = new HybsURLTagFragmentGenerator( imageMapTarget );
				}
				if( ( imageMapUrl != null ) || ( useToolTip ) ) {
					ChartRenderingInfo objCri = new ChartRenderingInfo( new StandardEntityCollection() );
					ChartUtilities.saveChartAsPNG( file, chart, width, height, objCri );
					imgTag = ImageMapUtilities.getImageMap( keyword, objCri, toolTipGen, urlGen ) + makeTag2( filename, keyword );
				}
				else {
					ChartUtilities.saveChartAsPNG( file, chart, width, height );
					imgTag = makeTag2( filename, null );
				}
			}
			catch(IOException ex) {
				String errMsg = "ファイル I/O が実行できませんでした。"
					+ HybsSystem.CR + file + HybsSystem.CR
						+ ex.getMessage();
				throw new HybsSystemException( errMsg,ex );
			}

			jspPrint( imgTag );

			// キャッシュ時にセーブする。
			if( useCache ) {
				if( cache == null ) {
					cache = new ChartCache( masterKey );
					cache.setJFreeChart( chart );
				}
				String subKey = String.valueOf( seriesPikup );
				cache.putFileAddress( subKey,imgTag );
				if( map == null ) {
					map = new HashMap<String,ChartCache>();
				}
				map.put( masterKey,cache );
				setSessionAttribute( CHART_CACHE_KEY,map );
			}
			else {
		//		map = (Map<String,ChartCache>)getSessionAttribute( CHART_CACHE_KEY );
		//		if( map != null ) { map.clear() ; }
				removeSessionAttribute( CHART_CACHE_KEY );
			}
		}

		// 3.5.4.7 (2004/02/06)
		long dyEnd = System.currentTimeMillis();
		jspPrint( "<div id=\"viewTime\" value=\"" + (dyEnd-dyStart) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		return EVAL_PAGE;
	}

	/**
	 * タグリブオブジェクトをリリースします。<br />
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。<br />
	 *
	 * @og.rev 4.0.1.0 (2007/12/13) categoryMargin、lowerMargin、upperMargin 属性を追加
	 * @og.rev 4.1.1.0 (2008/02/04) categoryCutNo , categoryAnchor , rangeSkip 属性を追加
	 */
	@Override
	protected void release2() {
		super.release2();
		chartDataset	= null;
		title			= null;
		width			= 200;
		height			= 200;
		domainLabel		= null;
		showLegend		= true;
		rectangleEdge	= "BOTTOM";
		plotOrientation	= "VERTICAL";
		chartBackColor	= null;			// 背景色の設定
		plotBackColor	= null;			// 描画領域の設定
		rotationLabel	= 3;			// 横軸ラベルの傾き(1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･ )
		domainMarker	= null;			// 横軸のマーカーライン
		useDomainLabel	= true;			// 横軸ラベルの表示有無
		useMarkerLabel	= true;			// マーカーライン時の文字有無
		useCache		= false;		// キャッシュするかどうか
		seriesPikup		= -1;			// ピックアップするシリーズ番号
		masterKey		= null;			// キャッシュ用マスタキー
		map				= null;			// キャッシュ用マップ
		cache			= null;			// キャッシュ
		imgTag			= null;			// キャッシュ用画像ファイル
		imageMapUrl		= null;			// クリッカブル・マップ
		imageMapTarget	= null;			// クリッカブル・マップ用TARGET
		categoryMargin	= null;			// 4.0.2.0 (2007/12/20) カテゴリマージン(0.0～1.0)
		lowerMargin		= null;			// 4.0.2.0 (2007/12/20) 下方マージン(0.0～1.0)
		upperMargin		= null;			// 4.0.2.0 (2007/12/20) 上方マージン(0.0～1.0)
//		shapeColors		= null;			// 4.0.3.0 (2008/01/07) データ毎にShapeを切り替える時の色指定
		useDomainLine	= false;		// 4.0.3.0 (2008/01/07) 横軸のグリッド表示有無(垂直線)
		domainLineColor	= null;			// 4.0.3.0 (2008/01/07) 横軸のグリッド線の色
		useRangeLine	= true;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド表示有無(水平線)
		rangeLineColor	= null;			// 4.0.3.0 (2008/01/07) 縦軸のグリッド線の色
		categorySkip	= 1;			// 4.0.3.0 (2008/01/07) 横軸ラベルをスキップする間隔
		categoryCutNo	= -1;			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のキーブレイク
		categoryAnchor	= null;			// 4.1.1.0 (2008/02/14) 横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)
//		lineOffset		= 0.0;			// 4.1.1.0 (2008/02/04) DomainLineの書き出し位置
		rangeSkip		= 1;			// 4.1.1.0 (2008/02/04) 縦軸のグリッドをスキップする間隔
	}

	/**
	 * チャートを表示するためのタグを作成します。
	 *
	 * @param   filename String 画像ファイル
	 * @param   keyword  String クリッカブル・マップの対応づける為のキーワード
	 * @return	タグ文字列
	 */
	private String makeTag2( final String filename,final String keyword ) {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		rtn.append( "<img" );
		if( keyword != null ) {
			rtn.append( " usemap=\"#").append( keyword ).append( "\"");
		}
		rtn.append( " width=\""  ).append( width  ).append( "px\"");
		rtn.append( " height=\"" ).append( height ).append( "px\"");
		rtn.append( " src=\""    ).append( filename ).append( "\" /> ");

		return rtn.toString();
	}

	/**
	 * ChartCreate のオブジェクトを生成します。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) categoryCutNo , categoryAnchor 属性を追加
	 *
	 * @return  ChartCreateオブジェクト
	 */
	private ChartCreate create() {
		ChartCreate tempCreate = new ChartCreate();

		tempCreate.setTitle( title );
		tempCreate.setDomainLabel( domainLabel );
		tempCreate.setShowLegend( showLegend );
		tempCreate.setRectangleEdge( rectangleEdge );
		tempCreate.setPlotOrientation( plotOrientation );
		tempCreate.setChartBackColor( chartBackColor );			// 背景色の設定
		tempCreate.setPlotBackColor( plotBackColor );			// 描画領域の設定
		tempCreate.setRotationLabel( rotationLabel );			// 横軸ラベルの傾き
		tempCreate.setDomainMarker( domainMarker );				// ドメインマーカー
		tempCreate.setUseDomainLabel( useDomainLabel );			// 横軸ラベルの表示有無
		tempCreate.setUseMarkerLabel( useMarkerLabel );			// マーカーの設定値表示
		tempCreate.setSeriesPikup( seriesPikup );				// ピックアップするシリーズ番号
		tempCreate.setImageMapUrl( imageMapUrl );				// クリッカブル・マップ用URL
		tempCreate.setCategoryMargin( categoryMargin );			// 4.0.2.0 (2007/12/20) カテゴリマージン(0.0～1.0)
		tempCreate.setLowerMargin( lowerMargin );				// 4.0.2.0 (2007/12/20) 下方マージン(0.0～1.0)
		tempCreate.setUpperMargin( upperMargin );				// 4.0.2.0 (2007/12/20) 上方マージン(0.0～1.0)
//		tempCreate.setShapeColors( shapeColors );				// 4.0.2.0 (2007/12/20) 上方マージン(0.0～1.0)
		tempCreate.setUseDomainLine( useDomainLine );			// 4.0.3.0 (2008/01/07) 横軸のグリッド表示有無(垂直線)
		tempCreate.setDomainLineColor( domainLineColor );		// 4.0.3.0 (2008/01/07) 横軸のグリッド線の色
		tempCreate.setUseRangeLine( useRangeLine );				// 4.0.3.0 (2008/01/07) 縦軸のグリッド表示有無(水平線)
		tempCreate.setRangeLineColor( rangeLineColor );			// 4.0.3.0 (2008/01/07) 縦軸のグリッド線の色
		tempCreate.setCategorySkip( categorySkip );				// 4.0.3.0 (2008/01/07) 横軸ラベルをスキップする間隔
		tempCreate.setCategoryCutNo( categoryCutNo );			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のキーブレイク
		tempCreate.setCategoryAnchor( categoryAnchor );			// 4.1.1.0 (2008/02/04) 横軸ラベルの文字位置指定のキーブレイク
//		tempCreate.setDomainLineOffset( lineOffset );			// // 4.1.1.0 (2008/02/14) 横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)
		tempCreate.setRangeSkip( rangeSkip );					// 4.1.1.0 (2008/02/04) 縦軸のグリッド線(水平線)をスキップする間隔
		tempCreate.setUseToolTip( useToolTip );                 // 4.3.1.0 (2008/08/09) ツールチップスの利用

		if( isDebug() ) { tempCreate.setDebug( true );	}		// 4.0.2.0 (2007/12/20)

		tempCreate.setDatasetList( chartDataset );

		return tempCreate ;
	}

	/**
	 * テンポラリFile を取得します。
	 *
	 * <p>ここでは、一般的なファイル出力を考慮した テンポラリFile を作成します。</p>
	 *
	 * @param   fileURL String ファイルを作成するディレクトリ
	 * @return  File テンポラリFile
	 */
	private File getTempFile( final String fileURL ) {
		final File file ;

		String directory = HybsSystem.url2dir( fileURL );
		File dir = new File( directory );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
			throw new HybsSystemException( errMsg );
		}

		try {
			file = File.createTempFile( "JFree",".png",dir );
			file.deleteOnExit();
		}
		catch( IOException ex ) {
			String errMsg = "ファイル名がオープン出来ませんでした。"
				+ HybsSystem.CR
					+ "Url:" + fileURL ;
			throw new HybsSystemException( errMsg,ex );
		}

		return file ;
	}

	/**
	 * 【TAG】コマンド(NEW,RENEW)をセットします（初期値:NEW）。
	 *
	 * @og.tag
	 * <p>コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。</p>
	 * command が、NEW または、RENEW の場合のみ、新規にオブジェクトの構築を行います。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.ChartCreateTag.CMD_NEW">コマンド定数</a>
	 */
//	public void setCommand( String cmd ) {
//		String cmd2 = getRequestParameter( cmd );
//		if( cmd2 != null && cmd2.length() != 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
//	}

	/**
	 * 【TAG】チャートのタイトルをセットします。
	 *
	 * @og.tag <p>チャートのタイトルをセットします。</p>
	 *
	 * @param   ttl タイトル
	 */
	public void setTitle( final String ttl ) {
		title = nval( getRequestParameter( ttl ),title );
	}

	/**
	 * 【TAG】チャートの横幅をセットします(初期値:200)。
	 *
	 * @og.tag
	 * <p>タイトルや凡例も含んだ大きさです。データ領域は自動計算されます。</p>
	 *
	 * @og.rev 4.0.1.0 (2007/12/13) 引数に "px" を使用できるようにする。
	 *
	 * @param   wd 横幅
	 */
	public void setWidth( final String wd ) {
		// 引数に、"px" が使用されていた場合は、それを取り除く。
		if( wd != null && wd.length() >= 3 && wd.endsWith( "px" ) ) {
			width = Integer.valueOf( wd.substring( 0,wd.length()-2 ) );
		}
		else {
			width = nval( getRequestParameter( wd ),width );
		}
	}

	/**
	 * 【TAG】チャートの縦幅をセットします(初期値:200)。
	 *
	 * @og.tag
	 * <p>タイトルや凡例も含んだ大きさです。データ領域は自動計算されます。</p>
	 *
	 * @og.rev 4.0.1.0 (2007/12/13) 引数に "px" を使用できるようにする。
	 *
	 * @param   ht 縦幅
	 */
	public void setHeight( final String ht ) {
		// 引数に、"px" が使用されていた場合は、それを取り除く。
		if( ht != null && ht.length() >= 3 && ht.endsWith( "px" ) ) {
			height = Integer.valueOf( ht.substring( 0,ht.length()-2 ) );
		}
		else {
			height = nval( getRequestParameter( ht ),height );
		}
	}

	/**
	 * 【TAG】チャートのドメインラベルを指定します。
	 *
	 * @og.tag
	 * <p>チャートのドメインラベルを指定します。</p>
	 *
	 * @param   dmLbl チャートのドメインラベル
	 */
	public void setDomainLabel( final String dmLbl ) {
		domainLabel = nval( getRequestParameter( dmLbl ),domainLabel );
	}

	/**
	 * 【TAG】チャートの凡例の表示可否(true/false)をセットします(初期値:true[表示する])。
	 *
	 * @og.tag
	 * <p>初期値は、表示する（true) です。</p>
	 *
	 * @param   swLegend 凡例の表示可否 ： 表示する（true) /しない（その他）
	 */
	public void setShowLegend( final String swLegend ) {
		showLegend = nval( getRequestParameter( swLegend ),showLegend );
	}

	/**
	 * 【TAG】チャートの表示箇所を、(TOP、BOTTOM、RIGHT、LEFT)で指定します(初期値:BOTTOM)。
	 *
	 * @og.tag
	 * 表示箇所は、org.jfree.ui.RectangleEdge クラスの設定値を使用します。
	 * 指定できるのは、TOP、BOTTOM、RIGHT、LEFT で、各文字の頭一文字で
	 * 判定してます。つまり、T,B,R,L で、判定を行い、それ以外はエラーになります。
	 * 初期値は、BOTTOM です。
	 *
	 * @param   rectEdge 凡例表示方向 ： (TOP、BOTTOM、RIGHT、LEFT)
	 */
	public void setRectangleEdge( final String rectEdge ) {
		rectangleEdge = nval( getRequestParameter( rectEdge ),rectangleEdge );
	}

	/**
	 * 【TAG】チャートのプロット(Plot)表示方向を、VERTICAL(or V) , HORIZONTAL(or H)で指定します(初期値:VERTICAL)。
	 *
	 * @og.tag
	 * 軸表示方向は、org.jfree.chart.plot.PlotOrientation クラスの設定値を使用します。
	 * 指定できるのは、VERTICAL、HORIZONTALで、各文字の頭一文字で
	 * 判定してます。つまり、V,H で、判定を行い、それ以外はエラーになります。
	 * 初期値は、VERTICAL です。
	 *
	 * @param   orientation プロット表示方向 ： VERTICAL(or V) , HORIZONTAL(or H)
	 */
	public void setPlotOrientation( final String orientation ) {
		plotOrientation = nval( getRequestParameter( orientation ),plotOrientation );
	}

	/**
	 * 【TAG】チャートの背景色を指定します。
	 *
	 * @og.tag
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @param   chBackClr チャートの背景色
	 * @see		java.awt.Color#BLACK
	 */
	public void setChartBackColor( final String chBackClr ) {
		chartBackColor = nval( getRequestParameter( chBackClr ),chartBackColor );
	}

	/**
	 * 【TAG】チャートの描画領域の色を指定します。
	 *
	 * @og.tag
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , (PURPLE) が指定できます。
	 * また、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 *
	 * @param   plBackClr チャートの描画領域色
	 * @see		java.awt.Color#BLACK
	 */
	public void setPlotBackColor( final String plBackClr ) {
		plotBackColor = nval( getRequestParameter( plBackClr ),plotBackColor );
	}

	/**
	 * 【TAG】チャートのカテゴリーラベルの方向を指定します(3:60度)。
	 *
	 * @og.tag
	 * 方向は、上方向に対して、(PI / 指示数) で求まる値に設定します。
	 * この指示数に相当する値を設定します。
	 * 1:180度 、2:90度 、3:60度 、4:45度 、6:30度 ･･･
	 * マイナスは、した方向に回転させます。
	 * 0 を指定した場合は、何も設定しません。
	 * 初期値は、3:60度です。
	 *
	 * @param   rttLabel String カテゴリーラベルの方向
	 */
	public void setRotationLabel( final String rttLabel ) {
		rotationLabel = nval( getRequestParameter( rttLabel ),rotationLabel );
	}

	/**
	 * 【TAG】チャートの横軸の値(ドメイン)に合致する位置にマーカーラインを設定します。
	 *
	 * @og.tag
	 * この属性には、マーカーラインを設定する値を記述します。
	 *
	 * @param   marker ドメインのマーカーライン
	 */
	public void setDomainMarker( final String marker ) {
		domainMarker = nval( getRequestParameter( marker ),domainMarker );
	}

	/**
	 * 【TAG】横軸ラベルのラベルを表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * ドメイン(横軸)が、多数存在する場合、ドメインラベルが見えにくくなります。
	 * そのようなケースで、横軸のラベルそのものを表示しない場合に、false を
	 * 設定します。
	 * 初期値は、表示する(true)です。
	 *
	 * @param   flag 横軸ラベルの表示有無(true:表示する/false:表示しない)
	 */
	public void setUseDomainLabel( final String flag ) {
		useDomainLabel = nval( getRequestParameter( flag ),useDomainLabel );
	}

	/**
	 * 【TAG】マーカーラインに、その設定値を表示するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * ドメイン(横軸)が、多数存在する場合、ドメインラベルが見えない場合があります。
	 * そのようなケースで、見たい値にマーカーラインを設定し、その横に、ドメインラベル
	 * を表示する事で、ピックアップしている軸の値を容易に知ることが可能です。
	 * 初期値は、表示する(true)です。
	 *
	 * @param   flag マーカーラインの設定値表示(true:表示する/false:表示しない)
	 */
	public void setUseMarkerLabel( final String flag ) {
		useMarkerLabel = nval( getRequestParameter( flag ),useMarkerLabel );
	}

	/**
	 * 【TAG】JFreeChart オブジェクトをキャッシュするかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * useCache="true" を設定すると、session に、HybsSystem.JF_CHART_CACHE_KEY
	 * キーで、キャッシュされます。
	 * そのときに、domainMarker 属性と、seriesPikup 属性だけ、再設定可能になっています。
	 *
	 * 初期値は、キャッシュしない(false)です。
	 *
	 * @param   flag JFreeChartをキャッシュするかどうか(true:キャッシュする/false:キャッシュしない)
	 * @see     #setDomainMarker( String )
	 * @see     #setSeriesPikup( String )
	 */
	public void setUseCache( final String flag ) {
		useCache = nval( getRequestParameter( flag ),useCache );
	}

	/**
	 * 【TAG】複数シリーズのピックアップを行う場合のシリーズ番号を指定します。
	 *
	 * @og.tag
	 * 複数シリーズ(検索時に複数項目を同時に検索する場合)では、チャート上に
	 * 複数のグラフが表示されますが、その中の一つをピックアップする場合に、
	 * シリーズ番号を指定します。
	 * シリーズ番号は、0 から始まる数字です。
	 * ここでは、ピックアップされたシリーズは、赤色で表示されます。
	 * それ以外は、グレー色での表示になります。
	 * seriesPikup を使用すると、chartDataset タグの useValueVisible 属性が
	 * 影響を受けます。この属性は、データの値(itemText)を表示しますが、
	 * seriesPikup が指定された場合は、そのシリーズのみにラベル表示します。
	 *
	 * @param   pikup String シリーズ番号
	 */
	public void setSeriesPikup( final String pikup ) {
		seriesPikup = nval( getRequestParameter( pikup ),seriesPikup );
	}

	/**
	 * 【TAG】キャッシュ用マスタキーを指定します。
	 *
	 * @og.tag
	 * useCache="true" に設定した場合、キャッシュを使用できるか確認します。
	 * この場合、seriesPikup 違いの場合は、JFreeChart オブジェクトそのものを
	 * キャッシュしておけば、データベースアクセスなしで、グラフを再描画
	 * させることが可能です。
	 * この、同一 JFreeChart を指定するためのキーを、ここで設定します。
	 * このキーの個数だけ、JFreeChart がキャッシュされます。
	 * なお、キャッシュ保持時間は、５分固定です。
	 *
	 * @param   mkey String キャッシュ用マスタキー
	 */
	public void setMasterKey( final String mkey ) {
		masterKey = nval( getRequestParameter( mkey ),masterKey );
	}

	/**
	 * 【TAG】クリッカブル・マップ用URLを指定します。
	 *
	 * @og.tag
	 * 画像に、クリッカブル・マップを作成する場合の、URL を指定します。
	 * これは、画像上にエリア指定でリンク引数を作成することが可能です。
	 * URL 自身は、? 付きで固定値の引数を連結することが可能です。
	 * クリックしたエリアのカテゴリやインデックスの値(引数)は、自動的に
	 * 設定されます。(指定しない場合はチャートによって異なります)
	 * <pre>
	 * ・Pie      ：category、pieIndex
	 * ・XY       ：series、item
	 * ・Category ：series、category
	 * </pre>
	 * この引数の URL の名称を変更したい場合は、URL に続けて、カンマ(,) で、
	 * 名称を記述してください。
	 * 例：link.jsp,BLOCK
	 *
	 * @param   imap String クリッカブル・マップ用URL
	 * @see #setImageMapTarget( String )
	 */
	public void setImageMapUrl( final String imap ) {
		imageMapUrl = nval( getRequestParameter( imap ),imageMapUrl );
	}

	/**
	 * 【TAG】クリッカブル・マップ用TARGETを指定します。
	 *
	 * @og.tag
	 * 画像に、クリッカブル・マップを作成する場合の、TARGET を指定します。
	 * これは、画像上にエリア指定でリンクを作成する場合のフレーム指定です。
	 *
	 * @param   target String クリッカブル・マップ用TARGET
	 * @see #setImageMapUrl( String )
	 */
	public void setImageMapTarget( final String target ) {
		imageMapTarget = nval( getRequestParameter( target ),imageMapTarget );
	}

	/**
	 * 【TAG】カテゴリマージン(0.0～1.0)を指定します。
	 *
	 * @og.tag
	 * カテゴリ(グラフの横軸に相当)の表示間隔(マージン)の比率を指定します。
	 * この比率は、％ ではなく、数字(double)での設定になります。
	 * 何も指定しない場合は、デフォルトで自動調整されます。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param   margin String カテゴリマージン(0.0～1.0)
	 */
	public void setCategoryMargin( final String margin ) {
		categoryMargin = nval( getRequestParameter( margin ),categoryMargin );
	}

	/**
	 * 【TAG】下方マージン(0.0～1.0)を指定します。
	 *
	 * @og.tag
	 * カテゴリ(グラフの横軸に相当)の下方側(左側)のマージンの比率を指定します。
	 * この比率は、％ ではなく、数字(double)での設定になります。
	 * 何も指定しない場合は、デフォルトで自動調整されます。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param   margin String 下方マージン(0.0～1.0)
	 */
	public void setLowerMargin( final String margin ) {
		lowerMargin = nval( getRequestParameter( margin ),lowerMargin );
	}

	/**
	 * 【TAG】上方マージン(0.0～1.0)を指定します。
	 *
	 * @og.tag
	 * カテゴリ(グラフの横軸に相当)の上方側(右側)のマージンの比率を指定します。
	 * この比率は、％ ではなく、数字(double)での設定になります。
	 * 何も指定しない場合は、デフォルトで自動調整されます。
	 *
	 * @og.rev 4.0.2.0 (2007/12/20) 新規追加
	 *
	 * @param   margin String 上方マージン(0.0～1.0)
	 */
	public void setUpperMargin( final String margin ) {
		upperMargin = nval( getRequestParameter( margin ),upperMargin );
	}

	/**
	 * 【TAG】横軸のグリッド表示有無(垂直線)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ドメイン(横軸)に対する、グリッドラインを表示するかどうか指定します。
	 *
	 * 何も指定しない場合は、表示しません。(false)
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param   useLine String 横軸のグリッド表示有無(垂直線)
	 */
	public void setUseDomainLine( final String useLine ) {
		useDomainLine = nval( getRequestParameter( useLine ),useDomainLine );
	}

	/**
	 * 【TAG】横軸のグリッド線の色を指定します。
	 *
	 * @og.tag
	 * ドメイン(横軸)に対する、グリッドラインの表示色を指定します。
	 * 何も指定しない場合は、デフォルトで自動設定されます。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param   color String 横軸のグリッド線の色
	 */
	public void setDomainLineColor( final String color ) {
		domainLineColor = nval( getRequestParameter( color ),domainLineColor );
	}

	/**
	 * 【TAG】横軸ラベルをスキップする間隔を指定します。
	 *
	 * @og.tag
	 * 横軸ラベル(カテゴリラベル表示)する際に、スキップする間隔を指定します。
	 * "1" (初期値)では、１つづつ表示（つまり、すべて表示する）します。
	 * "2" とすると、１つおきに、"3" とすると、２つおきに表示します。
	 * 初期値は、"1" (すべて表示)です。
	 * なお、先頭から表示を開始します。
	 *
	 * 注意：これとは別に、ラベル先頭に "_" を付けた場合は、ラベルを表示しません。
	 * また、categoryCutNo が指定された場合は、categorySkip は使用されません。
	 *
	 * @param   step String スキップする間隔
	 * @see		#setCategoryCutNo( String )
	 */
	public void setCategorySkip( final String step ) {
		categorySkip = nval( getRequestParameter( step ),categorySkip );
	}

	/**
	 * 【TAG】横軸ラベルの文字位置指定のキーブレイクを指定します。
	 *
	 * @og.tag
	 * 横軸ラベル(カテゴリラベル表示)する際に、ラベルの先頭から、この指定文字数だけ
	 * カットして、表示します。
	 * その際、前回作成したカットラベルと、同一ラベルの場合は、表示しません。
	 * 例えば、データは、年月日で、年と月のみ(先頭６文字)を指定すると、
	 * 日のデータは、ラベルが表示されません。
	 * 指定される数字は、１以上の整数としてください。
	 * 初期値は、すべて表示です。
	 *
	 * 注意：これとは別に、ラベル先頭に "_" を付けた場合は、ラベルを表示しません。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param   cutNo String キーブレイク位置
	 * @see		#setCategorySkip( String )
	 */
	public void setCategoryCutNo( final String cutNo ) {
		categoryCutNo = nval( getRequestParameter( cutNo ),categoryCutNo );
	}

	/**
	 * 【TAG】横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)を指定します。
	 *
	 * @og.tag
	 * 横軸のグリッド(垂直線)を、グラフのどの位置に記述するかを指定します。
	 * 具体的な値は、CategoryAnchor オブジェクトの値になります。
	 * ここでは、文字列で(START,MIDDLE,END)を指定します。(先頭一文字で判定)
	 * 何も指定しない場合は、デフォルト(MIDDLE)です。
	 *
	 * @og.rev 4.1.1.0 (2008/02/14) 新規追加
	 *
	 * @param   anchor String 横軸のグリッド(垂直線)の書き出し位置(START,MIDDLE,END)
	 */
	public void setCategoryAnchor( final String anchor ) {
		categoryAnchor = nval( getRequestParameter( anchor ),categoryAnchor );

		if( categoryAnchor != null && categoryAnchor.length() > 0 ) {
			if( ! "START".equals( categoryAnchor )  &&
				! "MIDDLE".equals( categoryAnchor ) &&
				! "END".equals( categoryAnchor ) ) {
					String errMsg = "指定のAnchorは適用できません。[" + anchor + "]"
							+ HybsSystem.CR
							+ "START,MIDDLE,END の中から、指定してください。" ;
					throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】DomainLineの書き出し位置Offsetを指定します。
	 *
	 * @og.tag
	 * 横軸ライン(カテゴリライン表示)する際に、ラインの描画位置を、
	 * 先頭から、この指定数だけずらします。
	 * プラスの場合は、右へ、マイナスの場合は、左へずらします。
	 * 
	 * これにより、ライン(縦線)の位置を、グラフの中心から、ずらす事が
	 * 可能になります。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param   domainLineOffset String DomainLineの書き出し位置Offset
	 */
//	public void setDomainLineOffset( final String domainLineOffset ) {
//		String offset = StringUtil.nval( getRequestParameter( domainLineOffset ),null );
//		if( offset != null ) {
//			lineOffset = Double.parseDouble( offset );
//		}
//	}

	/**
	 * 【TAG】縦軸のグリッド表示有無(水平線)を指定します(初期値:true)。
	 *
	 * @og.tag
	 * レンジ(縦軸)に対する、グリッドラインを表示するかどうか指定します。
	 *
	 * 何も指定しない場合は、表示しません。(false)
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param   useLine String 横軸のグリッド表示有無(垂直線)
	 */
	public void setUseRangeLine( final String useLine ) {
		useRangeLine = nval( getRequestParameter( useLine ),useRangeLine );
	}

	/**
	 * 【TAG】縦軸のグリッド線の色を指定します。
	 *
	 * @og.tag
	 * レンジ(縦軸)に対する、グリッドラインの表示色を指定します。
	 * 何も指定しない場合は、デフォルトで自動設定されます。
	 *
	 * @og.rev 4.0.3.0 (2008/01/07) 新規追加
	 *
	 * @param   color String 縦軸のグリッド線の色
	 */
	public void setRangeLineColor( final String color ) {
		rangeLineColor = nval( getRequestParameter( color ),rangeLineColor );
	}

	/**
	 * 【TAG】縦軸のグリッド線(水平線)をスキップする間隔を指定します。
	 *
	 * @og.tag
	 * 縦軸のグリッド線(水平線)を表示する際に、スキップする間隔を指定します。
	 * 通常は、ラベルと同じだけのグリッド線が掛かれますが、ラベルよりも
	 * 少ない数のグリッド線（例えば、２つおき）を出す場合に、値を設定します。
	 * "1" (初期値)では、１つづつ表示（つまり、すべて表示する）します。
	 * "2" とすると、１つおきに、"3" とすると、２つおきに表示します。
	 * 初期値は、"1" (すべて表示)です。
	 * なお、先頭から表示を開始します。
	 *
	 * @og.rev 4.1.1.0 (2008/02/04) 新規追加
	 *
	 * @param   rngSkip String 縦軸のグリッド線(水平線)をスキップする間隔
	 */
	public void setRangeSkip( final String rngSkip ) {
		rangeSkip = nval( getRequestParameter( rngSkip ),rangeSkip );
	}

	/**
	 * 【TAG】ツールチップスを利用（true）/利用しない(false)を指定します。
	 *
	 * @og.tag
	 * ラベルを利用する際に、ラベルと図面、隣のラベル同士が重なることがあります。
	 * この場合、ツールチップスの利用をお勧めします。
	 * 初期値は'false'です。
	 * 
	 * @og.rev 4.3.1.0 (2008/08/09) 新規追加
	 *
	 * @param   toolTip String ツールチップスを利用（true）/利用しない(false)
	 */
	public void setUseToolTip( final String toolTip ) {
		useToolTip = nval( getRequestParameter( toolTip ), useToolTip );
	}
	
	/**
	 * BODY要素に記述した、chartDataset タグの属性設定クラスをListに追加します。
	 *
	 * BODY部に記述された、chartDataset タグの属性は、チャートのデータ取得Queryや、
	 * チャートの種類などの情報を管理しています。
	 * チャートの種類(Category系)によっては、チャートを合成して表示させることが可能です。
	 * それらの複数のチャートの指定が、chartDataset タグ で行われます。
	 *
	 * @param   chDataset ChartDataset chartDataset タグの属性設定クラス
	 */
	protected void addChartDataset( final ChartDataset chDataset ) {
		if( chartDataset == null ) { chartDataset = new ArrayList<ChartDataset>(); }
		chartDataset.add( chDataset );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2007/11/29) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2007/11/29) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		rtn.append( "[" ).append( this.getClass().getName() ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "title           [" ).append( title           ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "width           [" ).append( width           ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "height          [" ).append( height          ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "domainLabel     [" ).append( domainLabel     ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "showLegend      [" ).append( showLegend      ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "rectangleEdge   [" ).append( rectangleEdge   ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "plotOrientation [" ).append( plotOrientation ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "chartBackColor  [" ).append( chartBackColor  ).append( "]" ).append( HybsSystem.CR );
		rtn.append( "plotBackColor   [" ).append( plotBackColor   ).append( "]" ).append( HybsSystem.CR );

		return rtn.toString();
	}

	/**
	 * JFreeChart や 画像ファイルのアドレスをキャッシュするクラスです。
	 * このキャッシュは、JFreeChart 単位に保持されます。
	 * JFreeChart と、画像ファイルのキャッシュを行います。
	 * JFreeChart オブジェクトをキャッシュしなおしたタイミングで、
	 * 画像ファイルのキャッシュは初期化(クリア)されます。
	 * キャッシュの有効期間は、システムリソースの CHART_CACHE_TIME で定義します。
	 */
	private static final class ChartCache {
		private final static int CACHE_TIME = HybsSystem.sysInt( "CHART_CACHE_TIME" );
		private final static long	MAX_CACHE_TIME = CACHE_TIME * 1000L;

		private final Map<String,String> fileAdrs = new HashMap<String,String>();
		private final String		mkey ;
		private 	  long			createLimitTime = 0L;
		private		  JFreeChart	chart			= null ;

		/**
		 * コンストラクター
		 * JFreeChart オブジェクトをキャッシュするキーを指定します。
		 * 内部的には、このキー(mkey)で、比較や hashCode が計算されています。
		 *
		 * @param   mkey String キャッシュするキー
		 */
		public ChartCache( final String mkey ) {
			this.mkey = mkey;
		}

		/**
		 * JFreeChart オブジェクトをキャッシュします。
		 * 画像ファイルのキャッシュは初期化(クリア)されます。
		 * このキャッシュのタイミングで、キャッシュ時間を初期化します。
		 *
		 * @param   chart JFreeChart JFreeChartオブジェクト
		 */
		public void setJFreeChart( final JFreeChart chart ) {
			this.chart = chart;
			fileAdrs.clear();
			createLimitTime = System.currentTimeMillis() + MAX_CACHE_TIME;
		}

		/**
		 * JFreeChart オブジェクトを返します。
		 * キャッシュされていれば、そのキャッシュを、そうでなければ、null を
		 * 返します。
		 * キャッシュの有効期間は、システムリソースの CHART_CACHE_TIME で定義します。
		 * オブジェクトの作成時間が、それを超える場合は、キャッシュを初期化します。
		 *
		 * @return   JFreeChart JFreeChartオブジェクト
		 */
		public JFreeChart getJFreeChart() {
			return chart;
		}

		/**
		 * 画像ファイルをキャッシュします。
		 * 画像ファイルのキャッシュはサブキー単位に行われます。
		 * つまり、このオブジェクトの中に、複数の画像ファイルのアドレスが
		 * キャッシュされています。
		 *
		 * @param   chart JFreeChart JFreeChartオブジェクト
		 */
		public void putFileAddress( final String skey,final String adrs ) {
			fileAdrs.put( skey,adrs );
		}

		/**
		 * 画像ファイルのアドレスを返します。
		 * キャッシュされていれば、そのキャッシュを、そうでなければ、null を
		 * 返します。
		 * キャッシュの有効期間は、システムリソースの CHART_CACHE_TIME で定義します。
		 * オブジェクトの作成時間が、それを超える場合は、キャッシュを初期化します。
		 *
		 * @return   JFreeChart JFreeChartオブジェクト
		 */
		public String getFileAddress( final String skey ) {
			return fileAdrs.get( skey ) ;
		}

		/**
		 * キャッシュが、有効かどうかを返します。(有効:true)
		 * キャッシュは、システムリソースの CHART_CACHE_TIME (秒)パラメータ
		 * 指定された時間が経過すると、無効と判断されます。
		 * 有効であれば、true を返します。
		 *
		 * @return   boolean 有効かどうか(有効:true/無効:false)
		 */
		public boolean isAvailable() {
			return ( System.currentTimeMillis() < createLimitTime ) ;
		}

		/**
		 * このオブジェクトと他のオブジェクトが等しいかどうかを示します。
		 * キャッシュのキーが等しいかどうかで判断します。
		 *
		 * @param   object 比較対象の参照オブジェクト
		 * @return  obj 引数に指定されたオブジェクトとこのオブジェクトが等しい場合は true、そうでない場合は false
		 */
		public boolean equals( final Object object ) {
			if( object instanceof ChartCache ) {
				return this.mkey.equals( ((ChartCache)object).mkey );
			}
			return false ;
		}

		/**
		 * オブジェクトのハッシュコード値を返します。
		 * このメソッドは、java.util.Hashtable によって提供されるような
		 * ハッシュテーブルで使用するために用意されています。
		 * equals( Object ) メソッドをオーバーライトした場合は、hashCode() メソッドも
		 * 必ず 記述する必要があります。
		 * この実装では、mkey.hashCode() と同値を返します。
		 *
		 * @return  このオブジェクトのハッシュコード値
		 */
		public int hashCode() {
			return mkey.hashCode() ;
		}
	}
}
