/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.LogWriter;

import java.util.Set ;
import java.util.Map ;
import java.util.LinkedHashMap ;
import java.net.InetAddress;
import java.net.UnknownHostException;

import java.sql.Connection;

/**
 * Process_DBParam は、他のプロセスへ共通のデータベース接続を割り当てる為の、
 * ParamProcess インターフェースの実装クラスです。
 *
 * DB接続 が必要な Process (DBCountFilter、DBMerge、DBReader、DBWriterなど)を
 * 使用して処理する場合に、接続を指定することができます。
 * DBID は、Process_DBParam の -configFile で指定する DBConfig.xml ファイルを使用します。
 *
 * @og.formSample
 *  Process_DBParam -infoUSER=C00000 -infoPGID=GE1234 -configFile=DBConfig.xml
 *
 *     -infoUSER=実行ユーザー         ： DB接続履歴取得用の実行ユーザー(例:C00000)
 *     -infoPGID=実行プログラムID     ： DB接続履歴取得用の実行プログラムID(例:GE1234)
 *     -configFile=実行プログラムID   ： DB接続情報設定 XMLファイル(例:DBConfig.xml)
 *
 * @og.rev 4.0.0.0 (2007/11/22) DBConfig.xml による DBID(接続先)指定に変更。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBParam extends AbstractProcess implements ParamProcess {
	/** 実行しているサーバーの名称 */
	private static final String HOST_NAME ;
	/** 実行しているサーバーのIPアドレス */
	private static final String HOST_ADRS ;

	static {
		String dmnHost ;
		String dnmAdrs ;
		try {
			InetAddress address = InetAddress.getLocalHost();
			dmnHost = address.getHostName() ;
			dnmAdrs = address.getHostAddress() ;
		}
		catch( UnknownHostException ex ) {
			dmnHost = "Unknown";
			dnmAdrs = "Unknown";
		}
		HOST_NAME = dmnHost;
		HOST_ADRS = dnmAdrs;
	}

	private ApplicationInfo appInfo = null;

	// 5.3.4.0 (2011/04/01) bulkData 関係のメソッドを追加
	private Set<String> bulkData ;

	private final static Map<String,String> mustProparty   ;	// ［プロパティ］必須チェック用 Map
	private final static Map<String,String> usableProparty ;	// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "infoUSER",	"DB接続履歴取得用の実行ユーザー" );
		mustProparty.put( "infoPGID",	"DB接続履歴取得用の実行プログラムID" );
		mustProparty.put( "configFile",	"DB接続情報設定 XMLファイル" );

		usableProparty = new LinkedHashMap<String,String>();
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBParam() {
		super( "org.opengion.fukurou.process.Process_DBParam",mustProparty,usableProparty );
	}

	/**
	 * ApplicationInfo オブジェクトを登録します。
	 * これは、通常の初期処理ではなく、タグリブから起動される場合のみ
	 * 呼ばれるメソッドです。
	 * 初期処理メソッド（init）では、appInfo がセット済みの場合は、
	 * ConnectionFactoryの初期化を行いません。
	 *
	 * @og.rev 4.3.1.1 (2008/09/04) 新規追加（taglib呼出専用）
	 *
	 * @param   appInfo ApplicationInfo
	 */
	public void setAppInfo( final ApplicationInfo appInfo ) {
		this.appInfo = appInfo;
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理（ファイルオープン、ＤＢオープン等）に使用します。
	 *
	 * @og.rev 4.3.1.1 (2008/09/04) taglib呼出時は、ConnectionFactoryの初期化を行わない
	 *
	 * @param   paramProcess ParamProcess
	 */
	public void init( final ParamProcess paramProcess ) {
		// 4.3.1.1 (2008/09/04) taglib呼出時は、ConnectionFactoryの初期化を行わない
		if( appInfo == null ) {
			Argument arg = getArgument();

			String infoUSER = arg.getProparty( "infoUSER" );	// DB接続履歴取得用の実行ユーザー
			String infoPGID = arg.getProparty( "infoPGID" );	// DB接続履歴取得用の実行プログラムID
			String configFile = arg.getProparty( "configFile" );	// DB接続情報設定 XMLファイル

			appInfo = new ApplicationInfo();
			// JavaVM 起動時のユーザーID,IPアドレス,ホスト名をセットします。
			appInfo.setClientInfo( infoUSER,HOST_ADRS,HOST_NAME );

			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( infoPGID,null,"fukurou" );

			// DBID接続情報の取得先の設定
			ConnectionFactory.init( null,configFile );
		}
	}

	/**
	 * 指定の DBID に対する コネクションを返します。
	 *
	 * @param	key	String DBID
	 * @return	Connection コネクション
	 * @throws	RuntimeException DB接続先が未設定の場合
	 */
	public Connection getConnection( final String key ) {
		return ConnectionFactory.connection( key,appInfo );
	}

	/**
	 * 検索した結果が設定された Set オブジェクトを設定します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	bulkData	Set<String> 検索した結果が設定された Set オブジェクト
	 */
	public void setBulkData( final Set<String> bulkData ) {
		this.bulkData = bulkData;
	}

	/**
	 * 検索した結果が設定された Set オブジェクトを返します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @return	Set<String> 検索した結果が設定された Set オブジェクト
	 */
	public Set<String> getBulkData() {
		return bulkData ;
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理（ファイルクローズ、ＤＢクローズ等）に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 *
	 * @param   isOK トータルで、OKだったかどうか(true:成功/false:失敗）
	 */
	public void end( final boolean isOK ) {
		// 何もありません。(PMD エラー回避)
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ ConnectionFactory.information();

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	String このクラスの使用方法
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_DBParam は、他のプロセスへ共通のデータベース接続を割り当てる為の、"	).append( CR );
		buf.append( "ParamProcess インターフェースの実装クラスです。"								).append( CR );
		buf.append( CR );
		buf.append( "DB接続 が必要な Process (DBCountFilter、DBMerge、DBReader、DBWriterなど)を"	).append( CR );
		buf.append( "使用して処理する場合に、接続を指定することができます。"						).append( CR );
		buf.append( "DBID は、-configFile で指定する DBConfig.xml ファイルを使用します。"			).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR ).append( CR );

		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args String[]
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBParam().usage() );
	}
}
