/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.transfer;

import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.Query;
import org.opengion.hayabusa.db.QueryFactory;

/**
 * 伝送要求に対してのPL/SQLを実行します。
 *
 * 実行するPL/SQLのPGIDは、実行対象で指定します。
 * 指定形式は、PG名( ?, ?, ... )  例) GEP00002.GEP00002( ?,?,0 ) です。
 * 引数は、従来のPL/SQLの実行が可能なように、第一引数はエラーコード、第二引数は、
 * エラーメッセージを返してきます。第三引数以降は、自由に指定できます。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferExec_PLSQL implements TransferExec {

	/**
	 * PL/SQLを実行します。
	 * 
	 * @param vals 伝送データ(配列)
	 * @param config 伝送設定オブジェクト
	 * @param tran トランザクションオブジェクト
	 */
	@Override
	public void execute( final String[] vals, final TransferConfig config, final Transaction tran ) {
		Query query = QueryFactory.newInstance( "JDBCCallable" );
		try {
			query.setTransaction( config.getExecDbid(), tran );
			// { call GEP00002.GEP00002( ?,?,&#39;{&#064;GUI.KEY}&#39;,&#39;{&#064;USER.ID}&#39; )
			String stmt = "{ call " + config.getExecObj() + "}";
			query.setStatement( stmt );
			query.execute();

			int errCode = query.getErrorCode();
			ErrorMessage errMessage = query.getErrorMessage();
			String errMsg = null;
			if( errMessage != null ) {
				errMsg = errMessage.toString();
			}

			if( query.isUpdate() ) {
				if( errCode < ErrorMessage.WARNING ) { // 警告以下の場合
					query.commit();
				}
				else {
					query.rollback();
					errMsg = "PL/SQL実行時に例外が発生しました。" + errMsg;
					throw new HybsSystemException( errMsg );
				}
			}
		}
		catch( Throwable ex ) {
			query.rollback();
			String errMsg = "PL/SQL呼出時に例外が発生しました。";
			throw new HybsSystemException( errMsg, ex );
		}
		finally {
			QueryFactory.close( query );
		}
	}
}
