/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.daemon;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.mail.MailTX;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.HybsTimerTask;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.transfer.TransferConfig;
import org.opengion.hayabusa.transfer.TransferExec;

/**
 * 【伝送システム】旧伝送DB(CB01)を監視して、実行方法に応じた処理プログラムを呼び出します。
 * 
 * このデーモンは、伝送定義マスタの読取方法が、旧伝送DB読取(CB01)の定義を対象として実行されます。
 * 読取対象は、旧伝送DB(CB01)で、データコード、送り先、テキスト種別、状況='1'を条件に読み込まれます。
 * 伝送定義マスタ上では、読取対象にて、以下の形式で定義する必要があります。
 *   (データコード) (送り先) (テキスト種別)   例):"3 D9 B119"
 * 処理実行後は、読み取ったヘッダーデータの状況を'2'に更新します。
 * 但し、読取パラメーターに"NOUPDATE"を指定した場合、処理後の更新は行われません。
 * 
 * トランザクションは、読取対象の単位になります。
 * 同じ読取対象で、異なる実行方法、実行対象を定義した場合、同じデータに対して複数回処理が行われます。
 * しかし、この場合においても、トランザクションは読取対象の単位で生成されるため、複数回の処理の内、
 * 1回でもエラーが発生した場合は、同じ読取対象でそれまでに処理した分についてもrollbackされます。
 * 
 * このクラスは、HybsTimerTask を継承した タイマータスククラスです。
 * startDaemon() がタイマータスクによって、呼び出されます。
 *
 * @og.group デーモン
 *
 * @version  5.0
 * @author   Hiroki Nakamura
 * @since    JDK6.0,
 */
public class Daemon_Transfer_CB01 extends HybsTimerTask {
	
	private final static String CLS_BASE = "org.opengion.hayabusa.transfer.TransferExec_" ;

	private static final String CB_SELECT =
		"SELECT B.READOBJ,B.READPRM,B.KBEXEC,B.EXECDBID,B.EXECOBJ,B.EXECPRM,A.HTCNO" +
		" FROM CB01 A,GE62 B" +
		" WHERE A.HCDD = SUBSTR(B.READOBJ,1,INSTR(B.READOBJ,' ',1,1)-1)" + // "3 D9 B119"の"3"
		" AND A.HTO    = RPAD(SUBSTR(B.READOBJ,INSTR(B.READOBJ,' ',1,1)+1,INSTR(B.READOBJ,' ',1,2)-INSTR(B.READOBJ,' ',1,1)-1),8)" + // "3 D9 B119"の"D9"
		" AND A.HSYU   = RPAD(SUBSTR(B.READOBJ,INSTR(B.READOBJ,' ',1,2)+1),4)" +  // "3 D9 B119"の"B119"
		" AND A.HCDJ = '1'" +
		" AND B.FGJ = '1'";

	// コネクションにアプリケーション情報を追記するかどうか指定
	private static final boolean USE_DB_APPLICATION_INFO  = HybsSystem.sysBool( "USE_DB_APPLICATION_INFO" ) ;

	private static final int LOOP_COUNTER = 24;		// カウンタを24回に設定

	private boolean running = true;
	private int loopCnt		= 0;

	private String cbSelect = null;
	private String dmnName = null;
	
	private ApplicationInfo appInfo = null;
	private boolean debug = false;

	/**
	 * このタイマータスクによって初期化されるアクションです。
	 * パラメータを使用した初期化を行います。
	 *
	 */
	public void initDaemon() {
		debug = StringUtil.nval( getValue( "DEBUG" ),debug );

		dmnName = getName();

		StringBuilder buf = new StringBuilder();
		buf.append( CB_SELECT );

		// SYSTEM_ID は、指定がなければ、全件検索対象になります。
		String systemId = getValue( "SYSTEM_ID" );
		if( StringUtil.isNull( systemId ) ) {
			String errMsg = "システムID方法は必須指定です。" ;
			throw new HybsSystemException( errMsg );
		}
		else {
			buf.append( " AND B.SYSTEM_ID='" ).append( systemId ).append( "'" );
		}

		// 読取方法は必須指定
		String kbRead = getValue( "KBREAD" );
		if( StringUtil.isNull( kbRead ) ) {
			String errMsg = "読取方法は必須指定です。" ;
			throw new HybsSystemException( errMsg );
		}
		else {
			buf.append( " AND B.KBREAD='" ).append( kbRead ).append( "'" );
		}

		// デーモングループは必須指定
		String dmnGroup = getValue( "DMN_GRP" );
		if( StringUtil.isNull( dmnGroup ) ) {
			String errMsg = "デーモングループは必須指定です。" ;
			throw new HybsSystemException( errMsg );
		}
		else {
			buf.append( " AND B.DMN_GRP='" ).append( dmnGroup ).append( "'" );
		}

		buf.append( " ORDER BY A.HTC" );

		cbSelect = buf.toString() ;

		if( debug ) {
			System.out.println( "DMN_NAME=[" + dmnName + "]" );
			System.out.println( "QUERY=[" + cbSelect + "]" );
		}

		if( USE_DB_APPLICATION_INFO ) {
			appInfo = new ApplicationInfo();
			// ユーザーID,IPアドレス,ホスト名
			appInfo.setClientInfo( systemId,HybsSystem.HOST_ADRS,HybsSystem.HOST_NAME );
			// 画面ID,操作,プログラムID
			appInfo.setModuleInfo( "TransferDaemon",dmnName,dmnName );
		}
		else {
			appInfo = null;
		}
	}

	/**
	 * タイマータスクのデーモン処理の開始ポイントです。
	 *
	 */
	protected void startDaemon() {
		if( loopCnt % LOOP_COUNTER == 0 ) {
			loopCnt = 1;
			System.out.println();
			System.out.print( toString() + " " + new Date()  + " " );
		}
		else {
			System.out.print( "." );
			loopCnt++ ;
		}

		// 伝送DB読取
		String[][] vals  = null;
		CB01Data cb01Data = new CB01Data();
		try {
			vals = DBUtil.dbExecute( cbSelect,null,appInfo );
			if( vals != null && vals.length > 0 ) {
				for( int row=0; running && row<vals.length; row++ ) {
					cb01Data.addData( vals[row] );
				}
			}
		}
		catch( Throwable ex ) {
			String header = "伝送読取エラー：DMN_NAME=[" + dmnName + "] , DMN_HOST=[" + HybsSystem.HOST_NAME + "] , QUERY=[" + cbSelect + "]";
			String errMsg = header + HybsSystem.CR + StringUtil.stringStackTrace( ex ) ;
			System.out.println( errMsg );
			LogWriter.log( errMsg );
			sendMail( header, errMsg );
		}

		// 処理実行
		// 読取対象の単位でトランザクションを生成します。
		for( String tranKey : cb01Data.getTranSet() ) {
			Transaction tran = null;
			TransferConfig conf = null;
			String[] htcnoArr = null;
			boolean isUpdate = true;
			try {
				tran = new TransactionReal( appInfo );

				// 読取対象+実行方法+実行対象の単位で処理を行います。
				for( String confKey : cb01Data.getExecKeySet( tranKey ) ) {
					conf = cb01Data.getConfig( confKey );
					htcnoArr = cb01Data.getHtcno( confKey );

					// デバッグ情報を出力します。
					if( debug ) {
						System.out.println();
						System.out.print( " START = " + new Date() );
						System.out.print( "[" + dmnName + "]:[" + StringUtil.array2csv( htcnoArr ) + "]:[" + conf.toString() + "]" );
					}

					// 伝送データを読み出します。
					String[] val = read( htcnoArr, tran );
					// 実行方法のオブジェクトを生成します。
					TransferExec exec = (TransferExec)StringUtil.newInstance( CLS_BASE + conf.getKbExec() );
					// 処理を実行します。
					exec.execute( val, conf, tran );

					// デバッグ情報を出力します。
					if( debug ) {
						System.out.println();
						System.out.print( " END = " + new Date() );
						System.out.print( "[" + dmnName + "]:[" + StringUtil.array2csv( htcnoArr ) + "]:[" + conf.toString() + "]" );
					}

					// 対象となるマスタの内、読取パラメーターに１つでも"NOUPDATE"が指定されている場合は、CB01の状況を更新しない
					if( "NOUPDATE".equalsIgnoreCase( conf.getReadPrm() ) ) {
						isUpdate = false;
					}
				}

				// 対象となるマスタの内、読取パラメーターに１つでも"NOUPDATE"が指定されている場合は、CB01の状況を更新しない
				if( isUpdate ) {
					complete( htcnoArr, tran );
				}
			}
			catch( Throwable ex ) {
				if( tran != null ) {
					tran.rollback();
					tran.close();
					tran = null; // エラー発生時は、接続を終了します。(次の状況更新でデッドロックになるため)
				}

				if( htcnoArr != null && htcnoArr.length > 0 ) {
					error( htcnoArr ); // エラー発生時はCB01>状態を9:エラーに更新
				}

				String header = "伝送エラー：DMN_NAME=[" + dmnName + "] , DMN_HOST=[" + HybsSystem.HOST_NAME + "]";
				if( htcnoArr != null && htcnoArr.length > 0 ) {
					header += " , HTCNO=[" + StringUtil.array2csv( htcnoArr ) + "]";
				}
				if( conf != null ) {
					header += " , CONFIG=[" + conf.toString() + "]";
				}

				String errMsg = header + HybsSystem.CR + StringUtil.stringStackTrace( ex ) ;
				System.out.println( errMsg );
				LogWriter.log( errMsg );
				sendMail( header, errMsg );
			}
			finally {
				if( tran != null ) { tran.close(); }
			}
		}
	}

	/**
	 * このタイマータスクのcancel() メソッドをオーバーライドします。<br />
	 * HybsTimerTaskManager#cancelTask( int ) を実行します。
	 *
	 * @see java.util.TimerTask#cancel()
	 */
	public boolean cancel() {
		running = false;
		return super.cancel();
	}


	/**
	 * CB01を読み込みデータを配列で返します。
	 * 
	 * @param htcnoArr 読取対象の通番NO(配列)
	 * @param tran トランザクション
	 * @return データ(配列)
	 */
	private String[] read( final String[] htcnoArr, final Transaction tran ) {
		if( htcnoArr == null || htcnoArr.length == 0 ) { return new String[0]; }

		String htcnos = StringUtil.array2csv( htcnoArr );
		StringBuilder buf = new StringBuilder();
		buf.append( "SELECT A.HTEXT" );
		buf.append( " FROM CB01 A" );
		buf.append( " WHERE A.HCDJ = '5'" );
		buf.append( " AND A.HTCNO IN (" );
		buf.append( htcnos );
		buf.append( ") ORDER BY A.HTC, A.HTCNO" );

		String[][] vals = DBUtil.dbExecute( buf.toString(),null,tran );
		String[] rtn = new String[vals.length];
		for( int i=0; i<vals.length; i++ ) {
			rtn[i] = vals[i][0];
		}
		return rtn;
	}

	/**
	 * CB01のヘッダーデータの状況を2:抜出済みに更新します。
	 * 
	 * @param htcnoArr 更新対象の通番NO(配列)
	 * @param tran トランザクション
	 */
	private void complete( final String[] htcnoArr, final Transaction tran ) {
		if( htcnoArr == null || htcnoArr.length == 0 ) { return; }

		String htcnos = StringUtil.array2csv( htcnoArr );
		StringBuilder buf = new StringBuilder();
		buf.append( "UPDATE CB01 A" );
		buf.append( " SET A.HCDJ = '2'" );
		buf.append( " WHERE A.HCDJ = '1'" );
		buf.append( " AND A.HTCNO IN (" );
		buf.append( htcnos );
		buf.append( ")" );

		DBUtil.dbExecute( buf.toString(),null,tran );
	}

	/**
	 * CB01のヘッダーデータの状況を9:エラーに更新します。
	 * 
	 * @param htcnoArr 更新対象の通番NO(配列)
	 */
	private void error( final String[] htcnoArr ) {
		if( htcnoArr == null || htcnoArr.length == 0 ) { return; }

		String htcnos = StringUtil.array2csv( htcnoArr );
		StringBuilder buf = new StringBuilder();
		buf.append( "UPDATE CB01 A" );
		buf.append( " SET A.HCDJ = '9'" );
		buf.append( " WHERE A.HCDJ in ('1','2')" ); // 既に実行PGで抜出済みに更新されている可能性がある
		buf.append( " AND A.HTCNO IN (" );
		buf.append( htcnos );
		buf.append( ")" );

		DBUtil.dbExecute( buf.toString(),null,appInfo ); // エラー更新はトランザクションを分けて処理する
	}

	/**
	 * エラー情報のメール送信を行います。
	 * エラーメールは、システムパラメータ の COMMON_MAIL_SERVER(メールサーバー)と
	 * ERROR_MAIL_FROM_USER(エラーメール発信元)と、ERROR_MAIL_TO_USERS(エラーメール受信者)
	 * がすべて設定されている場合に、送信されます。
	 *
	 * @param inHeader String ヘッダーメッセージ
	 * @param inErrMsg String エラーメッセージ
	 */
	private void sendMail( final String inHeader, final String inErrMsg ) {

		String   host = HybsSystem.sys( "COMMON_MAIL_SERVER" );
		String   from = HybsSystem.sys( "ERROR_MAIL_FROM_USER" );
		String[] to = StringUtil.csv2Array( HybsSystem.sys( "ERROR_MAIL_TO_USERS" ) );

		if( host != null && from != null && to.length > 0 ) {
			try {
				MailTX tx = new MailTX( host );
				tx.setFrom( from );
				tx.setTo( to );
				tx.setSubject( inHeader );
				tx.setMessage( inErrMsg );
				tx.sendmail();
			}
			catch( Throwable ex ) {
				String errMsg = "エラー時メール送信に失敗しました。" + HybsSystem.CR
							+ " SUBJECT:" + inHeader				+ HybsSystem.CR
							+ " HOST:" + host						+ HybsSystem.CR
							+ " FROM:" + from						+ HybsSystem.CR
							+ " TO:"   + Arrays.toString( to )		+ HybsSystem.CR
							+ ex.getMessage();
				LogWriter.log( errMsg );
				LogWriter.log( ex );
			}
		}
	}

	/**
	 * 旧伝送DBから読み出したデータを管理します。
	 */
	private static class CB01Data {

		// トランザクションを生成するキーのセット(読取対象単位)
		private final Set<String> tranSet = new LinkedHashSet<String>();
		// トランザクションキー(読取対象)に対する、処理キー(読取対象+実行方法+実行対象)のセット
		private final Map<String,Set<String>> tran2ExecKeySet = new LinkedHashMap<String,Set<String>>();
		// 処理キー(読取対象+実行方法+実行対象)に対する設定オブジェクトのマッピング
		private final Map<String,TransferConfig> execKey2Conf = new HashMap<String,TransferConfig>();
		// 処理キー(読取対象+実行方法+実行対象)に対する通番NO(配列)のマッピング
		private final Map<String,List<String>> execKey2HtcnoArr = new LinkedHashMap<String,List<String>>();

		/**
		 * CB01読取データを追加します。
		 * 
		 * @param data CB01読取データ
		 */
		private void addData( final String[] data ) {
			String readObj	= data[0];
			String readPrm	= data[1];
			String kbExec	= data[2];
			String execDbid	= data[3];
			String execObj	= data[4];
			String execPrm	= data[5];
			String htcno	= data[6];

			String tranKey = readObj;
			tranSet.add( tranKey );

			// 読取対象 + 実行方法 + 実行対象 単位に処理対象通番NOを集約する
			String execKey = readObj + kbExec + execObj;
			Set<String> execKeySet = tran2ExecKeySet.get( tranKey );
			if( execKeySet == null ) {
				execKeySet = new LinkedHashSet<String>();
			}
			execKeySet.add( execKey );
			tran2ExecKeySet.put( tranKey, execKeySet );

			// 伝送設定オブジェクトのマップ
			TransferConfig conf = execKey2Conf.get( execKey );
			if( conf == null ) {
				conf = new TransferConfig( readObj, readPrm, kbExec, execDbid, execObj, execPrm );
				execKey2Conf.put( execKey, conf );
			}

			// 通番NOのマップ
			List<String> htcnoArr = execKey2HtcnoArr.get( execKey );
			if( htcnoArr == null ) {
				htcnoArr = new ArrayList<String>();
			}
			htcnoArr.add( htcno );
			execKey2HtcnoArr.put( execKey, htcnoArr );
		}

		/**
		 * トランザクション生成キー(読取対象)のセットを返します。
		 * 
		 * @return トランザクション生成キー(読取対象)のセット
		 */
		private Set<String> getTranSet() {
			return tranSet;
		}

		/**
		 * トランザクション生成キー(読取対象)に対する処理キー(読取対象+実行方法+実行対象)のセットを返します。
		 * 
		 * @param tranKey トランザクション生成キー(読取対象)
		 * @return トランザクション生成キー(読取対象)に対する処理キー(読取対象+実行方法+実行対象)のセット
		 */
		private Set<String> getExecKeySet( final String tranKey ) {
			return tran2ExecKeySet.get( tranKey );
		}

		/**
		 * 処理キー(読取対象+実行方法+実行対象)に対する設定オブジェクトを返します。
		 * 
		 * @param execKey 処理キー(読取対象+実行方法+実行対象)
		 * @return 設定オブジェクト
		 */
		private TransferConfig getConfig( final String execKey ) {
			return execKey2Conf.get( execKey );
		}

		/**
		 * 処理キー(読取対象+実行方法+実行対象)に対する通番NO(配列)を返します。
		 * 
		 * @param execKey 処理キー(読取対象+実行方法+実行対象)
		 * @return 通番NO(配列)
		 */
		private String[] getHtcno( final String execKey ) {
			List<String> l = execKey2HtcnoArr.get( execKey );
			if( l == null ) { return new String[0]; }
			else { return l.toArray( new String[0] ); }
		}
	}
}
