/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * ProcessTag に対して、引数(パラメータ)を渡す、入れ子の子クラスです。
 *
 * @og.formSample
 * ●形式：&lt;og:param key="ZZZ" value="AAA,BBB" /&gt;
 * ●body：あり
 *
 * ●Tag定義：
 *   &lt;og:param
 *       key              ○【TAG】リクエスト情報 に登録するキーをセットします。(必須)
 *       value              【TAG】リクエスト情報 に登録する値をセットします
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:param&gt;
 *
 * ●使用例
 *     一般的な変数の渡し方
 *     &lt;og:process ･････ &gt;
 *        &lt;og:param key="AAA" value="111" /&gt;
 *        &lt;og:param key="BBB" value="222" /&gt;
 *        &lt;og:param key="CCC" value="333" /&gt;
 *        &lt;og:param key="DDD" value="444" /&gt;
 *     &lt;/og:process &gt;
 *
 *     BODY 部に記述した変数の渡し方
 *     &lt;og:process ･････ &gt;
 *        &lt;og:param key="SQL" &gt;
 *          SELECT * FROM DUAL
 *        &lt;/og:param&gt;
 *     &lt;/og:process &gt;
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ParamTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private String	key		= null;
	private String	value	= null;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		return ( EVAL_BODY_BUFFERED );		// Body を評価する
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		if( value == null || value.length() <= 0 ) {
			value = getBodyString();
		}

		return ( SKIP_BODY );
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		ProcessTag process = (ProcessTag)findAncestorWithClass( this,ProcessTag.class );
		if( process == null ) {
			String errMsg = "<b>このタグは、ProcessTag の内側(要素)に記述してください。</b>";
			throw new HybsSystemException( errMsg );
		}

		process.addParam( key,value );

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		key		= null;
		value	= null;
	}

	/**
	 * 【TAG】リクエスト情報 に登録するキーをセットします。
	 *
	 * @og.tag リクエスト情報 に登録するキーをセットします。
	 *
	 * @param	key1 リクエスト情報に登録するキー
	 */
	public void setKey( final String key1 ) {
		key = nval( getRequestParameter( key1 ),key ) ;
	}

	/**
	 * 【TAG】リクエスト情報 に登録する値をセットします。
	 *
	 * @og.tag リクエスト情報 に登録する値をセットします。
	 *
	 * @param	val	リクエスト情報に登録する値
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"				,VERSION			)
				.println( "key"					,key				)
				.println( "value"				,value				)
				.fixForm().toString() ;
	}
}
