/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.LabelInterface;
import org.opengion.fukurou.util.XHTMLTag;

import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Locale ;

/**
 * サブミットボタンを表示するHTML拡張タグです(forward.jsp の commonForward タグと同時に使用します)。
 *
 * このタグは、value に指定された値＋".jsp" の画面へサブミットします。
 * その際、command に設定された値を 送信します。
 * 従来は、value に、copy/modify/delete など、処理するJSPを個別に設定していましたので、
 * command と lbl が未設定時には、value の値を使用していました。(下位互換性のため残しています)
 * Ver5 になって、update.jsp ですべての処理を行う方向で開発するに当たり、
 * command と lbl を記述する必要がでてきました。
 * そこで、現在最新版では、action 属性を用意し、command を記述するだけで良くなりました。
 *
 *    [action属性]  [初期設定されるパラメータ郡]
 *      INSERT       value="update" command="INSERT" lbl="INSERT"  accesskey="I"
 *      COPY         value="update" command="COPY"   lbl="COPY"    accesskey="C"
 *      MODIFY       value="update" command="MODIFY" lbl="MODIFY"  accesskey="M"
 *      DELETE       value="update" command="DELETE" lbl="DELETE"  accesskey="Z"
 *      ENTRY        value="entry"  command="ENTRY"  lbl="ENTRY"   accesskey="E"
 *      RESET        value="reset"  command="RESET"  lbl="ENTRY"   accesskey="R"
 *
 * columnWritable , noWritable の属性追加(5.2.2.0 (2010/11/01))
 * これは、各コマンドごとに、次ページの view のカラム属性の設定に利用する属性です。
 * Ver4 では、keys,vals の一般引数として設定していましたが、ここでは、専用属性として
 * 用意しています。
 *
 * @og.formSample
 * ●形式：&lt;og:submit value="…" lbl="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:submit
 *       action             【TAG】アクション(INSERT,COPY,MODIFY,DELETE,ENTRY,RESET)を指定します
 *       value              【TAG】forward したいＪＳＰファイル名を記述します(例:insert,copy,modify,delete など)
 *       command            【TAG】処理コマンドを登録します(初期値:大文字の value 属性値[INSERT,COPY,MODIFY,DELETE など])
 *       gamenId            【TAG】gamenId 属性を登録します
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       target             【TAG】サブミット先の文書を表示させるフレーム、またはウィンドウの名前を指定します
 *       keys               【TAG】ボタン専用のリクエストキーをCSV形式で複数指定します
 *       vals               【TAG】ボタン専用のリクエスト値をCSV形式で複数指定します
 *       roles              【TAG】ロールをセットします
 *       dbkeys             【TAG】commonForward の dbkeys にカラム指定を行います
 *       optionAttributes   【TAG】JavaScript などの HTML基本タグ以外の属性を、そのままタグとして使用します
 *       columnWritable     【TAG】書き込み可能カラム名を、カンマ区切りで与えます
 *       noWritable         【TAG】書き込み不可カラム名を、カンマ区切りで与えます
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       readonly           【TAG】その部品に対して変更が出来ないように(readonly)指定します(サーバーに送信される)
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】アクセスキー(alt+キーで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       img                【TAG】画像ボタンを作る場合の、画像ファイルを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     Ver5 推奨ケース：指定の action 属性に設定された値に応じた、value , command , lbl , accesskey を初期化します。
 *                      (5.2.2.0 (2010/11/01)以降)
 *     &lt;og:writeCheck&gt;
 *         &lt;og:submit  action="COPY"    noWritable="FGJ,UNIQ" /&gt;
 *         &lt;og:submit  action="MODIFY"  noWritable="CLM,FGJ,UNIQ" /&gt;
 *         &lt;og:submit  action="DELETE"  columnWritable="null" /&gt;
 *         &lt;og:submit  gamenId="GF9110" value="index" dbkeys="SYSTEM_ID,LANG,CLM" command="NEW" lbl="GF9100" target="CONTENTS" /&gt;
 *         &lt;br /&gt;
 *     &lt;/og:writeCheck&gt;
 *
 *     Ver4 一般的なケース：value は共通になったため、command と lbl の設定が必要(下位互換性のための設定)
 *     &lt;og:writeCheck&gt;
 *         &lt;og:submit  value="update"   command="COPY"   lbl="COPY"    accesskey="C" /&gt;
 *         &lt;og:submit  value="update"   command="MODIFY" lbl="MODIFY"  accesskey="M" /&gt;
 *         &lt;og:submit  value="update"   command="DELETE" lbl="DELETE"  accesskey="Z" /&gt;
 *         &lt;og:submit  gamenId="GF9110" value="index" dbkeys="SYSTEM_ID,LANG,CLM" command="NEW" lbl="GF9100" target="CONTENTS" /&gt;
 *         &lt;br /&gt;
 *     &lt;/og:writeCheck&gt;
 *
 *     従来のケース：value に設定された値JSPに対してサブミットされます。(下位互換性のための設定)
 *     &lt;og:writeCheck&gt;
 *         &lt;og:submit value="copy"     lbl="MSG0035" accesskey="C"  /&gt;
 *         &lt;og:submit value="modify"   lbl="MSG0036" accesskey="M"  /&gt;
 *         &lt;og:submit value="delete"   lbl="MSG0037" accesskey="Z"  /&gt;
 *         &lt;og:submit  gamenId="GF9110" value="index" dbkeys="SYSTEM_ID,LANG,CLM" command="NEW" lbl="GF9100" target="CONTENTS" /&gt;
 *         &lt;br /&gt;
 *     &lt;/og:writeCheck&gt;
 *
 * @og.rev 3.1.1.0 (2003/03/28) 新規作成
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SubmitTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	//  5.2.2.0 (2010/11/01) 新規追加
	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String ACT_INSERT  = "INSERT" ;
	/** command 引数に渡す事の出来る コマンド  複写 {@value} */
	public static final String ACT_COPY    = "COPY" ;
	/** command 引数に渡す事の出来る コマンド  変更 {@value} */
	public static final String ACT_MODIFY  = "MODIFY" ;
	/** command 引数に渡す事の出来る コマンド  削除 {@value} */
	public static final String ACT_DELETE  = "DELETE" ;
	/** command 引数に渡す事の出来る コマンド  エントリー {@value} */
	public static final String ACT_ENTRY   = "ENTRY" ;
	/** command 引数に渡す事の出来る コマンド  リセット {@value} */
	public static final String ACT_RESET   = "RESET" ;

	//  5.2.2.0 (2010/11/01) 新規追加
	private static final String[] ACTION_LIST = new String[] {
		ACT_INSERT , ACT_COPY , ACT_MODIFY , ACT_DELETE , ACT_ENTRY , ACT_RESET };

	//  5.2.2.0 (2010/11/01) 新規追加
	private enum ENUM_ACTION {
		// action  value      command    lbl       accesskey
		   INSERT( "update" , "INSERT" , "INSERT" , "I" ),
		   COPY  ( "update" , "COPY"   , "COPY"   , "C" ),
		   MODIFY( "update" , "MODIFY" , "MODIFY" , "M" ),
		   DELETE( "update" , "DELETE" , "DELETE" , "Z" ),
		   ENTRY ( "entry"  , "ENTRY"  , "ENTRY"  , "E" ),
		   RESET ( "reset"  , "RESET"  , "RESET"  , "R" ) ;

		private final String defVal ;
		private final String defCmd ;
		private final String defLbl ;
		private final String defKey ;

		ENUM_ACTION( final String val , final String cmd , final String lbl , final String key ) {
			defVal = val;
			defCmd = cmd;
			defLbl = lbl;
			defKey = key;
		}

		public String getDefVal() { return defVal; }
		public String getDefCmd() { return defCmd; }
		public String getDefLbl() { return defLbl; }
		public String getDefKey() { return defKey; }
	}

	private static final String TYPE = "submit" ;
	private static final String NAME = "command" ;
	private static final String MARGIN = "<span style=\"width: 3px;\" ></span>" ;

	private static final String IMG_PRE = "background: url(";			// 5.5.0.0 (2012/03/01)
	private static final String IMG_SUF = ") left top no-repeat;";		// 5.5.0.0 (2012/03/01)

	private String  action		= null;		// 5.2.2.0 (2010/11/01)
	private String  command		= null;
	private String  value		= null;
	private String  gamenId		= null;
	private String  target		= null;		// 3.5.5.2 (2004/04/02)
	private String  dbkeys		= null;		// 4.0.0 (2007/05/23)

	private String	columnWritable	= null;	// 5.2.2.0 (2010/11/01)
	private String	noWritable		= null;	// 5.2.2.0 (2010/11/01)

	private String bgImg		= null;		// 5.5.0.0 (2012/03/01)

	private String[] keys		= null;		// 3.5.5.5 (2004/04/23)
	private String[] vals		= null;		// 3.5.5.5 (2004/04/23)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			// 5.1.8.0 (2010/07/01) 同時設定不可(columnWritable,noWritable) チェック
			if( columnWritable != null && noWritable != null ) {
				String errMsg = "columnWritable と noWritable は同時に指定できません。"
								+ "columnWritable = [" + columnWritable
								+ "] , noWritable = [" + noWritable
								+ "]";
				throw new HybsSystemException( errMsg );
			}

			// 5.2.2.0 (2010/11/01) action 属性による初期値の設定
			if( action != null ) {
				ENUM_ACTION eact = ENUM_ACTION.valueOf( action );
				if( value		== null ) { value   = eact.getDefVal(); }
				if( command		== null ) { command = eact.getDefCmd(); }
				if( getMsglbl() == null )        { setLbl( eact.getDefLbl() ); }
				if( get( "accesskey" ) == null ) { set( "accesskey",eact.getDefKey() ); }
			}
		}
		return( SKIP_BODY );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.5.2 (2004/04/02) target 属性の追加
	 * @og.rev 3.5.5.5 (2004/04/23) keys,vals 属性の追加
	 * @og.rev 5.2.2.0 (2010/11/01) action,columnWritable,noWritable 属性の追加
	 * @og.rev 5.5.0.0 (2012/03/01) bgImg追加
	 */
	@Override
	protected void release2() {
		super.release2();
		action		= null;		// 5.2.2.0 (2010/11/01)
		command		= null;
		value		= null;
		gamenId		= null;
		target		= null;		// 3.5.5.2 (2004/04/02)
		keys		= null;		// 3.5.5.5 (2004/04/23)
		vals		= null;		// 3.5.5.5 (2004/04/23)
		dbkeys		= null;		// 4.0.0 (2007/05/23)
		columnWritable	= null;	// 5.2.2.0 (2010/11/01)
		noWritable		= null;	// 5.2.2.0 (2010/11/01)
		bgImg		= null;		// 5.5.0.0 (2012/03/01)
	}

	/**
	 * サブミットボタンを作成します。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) ForwardManager クラスの廃止。飛び先のキャッシュを廃止します。
	 * @og.rev 3.5.5.2 (2004/04/02) target 属性の追加
	 * @og.rev 3.5.5.9 (2004/06/07) target 属性を、set ではなく add で追加。
	 * @og.rev 3.5.5.9 (2004/06/07) target 属性を、set ではなく add で追加。
	 * @og.rev 4.0.0 (2005/11/30) title 属性が未設定時の処理追加
	 * @og.rev 5.5.0.0 (2012/03/01) bgImg対応
	 *
	 * @return  サブミットボタンタグ
	 */
	@Override
	protected String makeTag() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		if( value == null ) {
			String errMsg = "value に null がセットされています。";
			throw new HybsSystemException( errMsg );
		}

		String lbl = nval( getMsglbl(), value.toUpperCase( Locale.JAPAN ) );
		String accesskey = get( "accesskey" );
		if( accesskey != null ) {
			lbl = lbl + "(" + accesskey + ")" ;
		}

		// キャッシュエントリ
		command = nval( command,value.toUpperCase(Locale.JAPAN) );
		String valueLink = getValueLink( gamenId , value );
		if( valueLink == null ) { return ""; }	// 	アクセス不可時は null

		// 3.3.1.1 (2003/07/03) ForwardManager クラスの廃止。飛び先のキャッシュを廃止します。
		rtn.append( getHiddenTag( command, lbl, valueLink ) );
		rtn.append( HybsSystem.CR );

		set( "type",TYPE );
		set( "name",NAME );
		set( "value",lbl );

		// 3.5.5.2 (2004/04/02) target 属性の追加
		if( target != null ) {
			add( "onClick","this.form.target='" + target + "'",";" );	// 3.5.5.9 (2004/06/07)
		}

		// 4.0.0 (2005/11/30) title 属性が未設定時の処理追加
		if( get( "title" ) == null ) {
			LabelInterface msglbl = getLabelInterface();
			if( msglbl != null ) {
				String desc = msglbl.getDescription();
				if( desc != null && desc.length() > 0 ) {
					set( "title",desc );
				}
			}
		}

		// 5.5.0.0 Img対応 スタイル属性に追加する。
		if( bgImg != null && bgImg.length() > 0 ){
			add("style",IMG_PRE+bgImg+IMG_SUF);
		}

		rtn.append( XHTMLTag.input( getAttributes() ) );
		rtn.append( MARGIN );
		rtn.append( HybsSystem.CR );

		return rtn.toString() ;
	}

	/**
	 * 画面ＩＤとvalue から、指定のＵＲＬを作成します。
	 * 画面へのアクセス許可が与えられていない場合は、null を返します。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) URLを求めるのに、GUIInfo#getRealAddress() を使用する。
	 * @og.rev 4.0.0 (2005/01/31) GUIInfoの実アドレスのパラメータを考慮する。
	 *
	 * @param   gamenId	画面ＩＤ
	 * @param   value	飛ばし先(XXXX.jspのXXXX部分)
	 *
	 * @return	ＵＲＬ文字列(アクセス不可時は null)
	 */
	private String getValueLink( final String gamenId,final String value ) {
		String link = value + ".jsp";

		if( gamenId != null && gamenId.length() > 0 ) {
			GUIInfo guiInfo = getGUIInfo( gamenId );		// 4.0.0 (2005/01/31)
			if( guiInfo == null ) { return null; }	// 見つからない場合は、アクセス不可

			String address = guiInfo.getRealAddress( link );
			link = getRequestParameter( address );
		}

		return link ;
	}

	/**
	 * 【TAG】アクション(INSERT,COPY,MODIFY,DELETE,ENTRY,RESET)を指定します。
	 *
	 * @og.tag
	 * Ver5 になって、update.jsp ですべての処理を行う方向で開発するに当たり、
	 * command と lbl を記述する必要がでてきました。
	 * そこで、現在最新版では、action 属性を用意し、command を記述するだけで良くなりました。
	 *
	 *    [action属性]  [初期設定されるパラメータ郡]
	 *      INSERT       value="update" command="INSERT" lbl="INSERT"  accesskey="I"
	 *      COPY         value="update" command="COPY"   lbl="COPY"    accesskey="C"
	 *      MODIFY       value="update" command="MODIFY" lbl="MODIFY"  accesskey="M"
	 *      DELETE       value="update" command="DELETE" lbl="DELETE"  accesskey="Z"
	 *      ENTRY        value="entry"  command="ENTRY"  lbl="ENTRY"   accesskey="E"
	 *      RESET        value="reset"  command="RESET"  lbl="ENTRY"   accesskey="R"
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @param	act アクション文字列
	 * @see		<a href="{&#064;docRoot}/constant-values.html#org.opengion.hayabusa.taglib.SubmitTag.ACT_COPY">アクション定数</a>
	 */
	public void setAction( final String act ) {
		action = getRequestParameter( act );

		if( ! check( action, ACTION_LIST ) ) {

			StringBuilder errMsg = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
			errMsg.append( "指定のアクションは実行できません。アクションエラー" );
			errMsg.append( HybsSystem.CR );
			errMsg.append( "action=[" ).append( action ).append( "] " );
			errMsg.append( HybsSystem.CR );

			for( int i=0; i<ACTION_LIST.length; i++ ) {
				errMsg.append( " | " );
				errMsg.append( ACTION_LIST[i] );
			}
			errMsg.append( " | " );
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 【TAG】forward したいＪＳＰファイル名を記述します(例:insert,copy,modify,delete など)。
	 *
	 * @og.tag
	 * ＪＳＰファイル名は、標準で、insert,copy,modify,delete などと指定します。
	 * 実際には、各JSP画面(insert.jsp,copy.jsp,modify.jsp,delete.jsp )に
	 * リクエストが転送されます。
	 * このＪＳＰファイル名は、同一画面ＩＤ内のフォルダに属している必要があります。
	 * ここのＩＤは、ＪＳＰ画面そのものですので、大文字小文字は区別されます。
	 *
	 * @param   val ＪＳＰファイル名(insert,copy,modify,delete など)
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
		if( value == null ) {
			String errMsg = "value に null がセットされています。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】gamenId 属性を登録します。
	 *
	 * @og.tag
	 * gamenId 属性は、別の画面にforward する場合に使用します。
	 * 実際は、forward ではなく、sendRedirect されます。
	 *
	 * @param   id gamenId 属性
	 */
	public void setGamenId( final String id ) {
		gamenId = nval( getRequestParameter( id ),gamenId );
	}

	/**
	 * 隠し属性タグを取得します。
	 * 各ボタンに設定された値を、隠しフィールドに設定して受け渡しします。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) ForwardManager クラスの廃止。飛び先のキャッシュを廃止します。
	 * @og.rev 3.5.5.5 (2004/04/23) 余計なボタン関連情報を転送しない為に、キーを変更します。
	 * @og.rev 3.5.5.5 (2004/04/23) keys,vals 属性で指定した値を、出力します。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.8.0.8 (2005/10/03) gamenId が指定されている場合は、BACK_GAMENID を出力する。
	 * @og.rev 5.2.2.0 (2010/11/01) columnWritable,noWritable 属性の追加
	 *
	 * @param	command		コマンド
	 * @param	lbl			ラベルID
	 * @param	valueLink	飛び先URL
	 *
	 * @return	隠し属性タグ
	 */
	private String getHiddenTag( final String command, final String lbl, final String valueLink ) {

		StringBuilder strRet = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		String prefix = HybsSystem.NO_XFER_KEY + lbl ;
		strRet.append( XHTMLTag.hidden( prefix        ,valueLink ) );	// 3.5.5.5 (2004/04/23)
		strRet.append( XHTMLTag.hidden( prefix + "CMD",command   ) );	// 3.5.5.5 (2004/04/23)

		// 3.5.5.5 (2004/04/23) keys,vals 属性で指定した値を、出力します。
		if( keys != null ) {
			if( keys.length != vals.length ) {
				String errMsg = "SubmitTag の keys と vals の引数の個数が異なります。" + HybsSystem.CR
							+ "keys=[" + StringUtil.array2csv( keys ) + "], vals=["
							+ StringUtil.array2csv( vals ) + "]" ;
				throw new HybsSystemException( errMsg );
			}
			for( int i=0; i<keys.length; i++ ) {
				strRet.append( XHTMLTag.hidden( prefix + "KEY_" + keys[i],vals[i] ) );	// 3.5.5.5 (2004/04/23)
			}
		}

		// 4.0.0 (2007/05/23) dbkeys が指定されている場合
		if( dbkeys != null && dbkeys.length() > 0 ) {
			strRet.append( XHTMLTag.hidden( prefix + "KEY_dbkeys",dbkeys ) );	// 4.0.0 (2007/05/23)
		}

		// 3.8.0.8 (2005/10/03) gamenId が指定されている場合は、BACK_GAMENID を出力する。
		if( gamenId != null && gamenId.length() > 0 ) {
			String backGamenId = getGUIInfoAttri( "KEY" );
			strRet.append( XHTMLTag.hidden( prefix + "KEY_BACK_GAMENID",backGamenId ) );
		}

		// 5.2.2.0 (2010/11/01) columnWritable,noWritable 属性の追加
		if( columnWritable != null ) {
			strRet.append( XHTMLTag.hidden( prefix + "KEY_columnWritable",columnWritable ) );
		}
		if( noWritable != null ) {
			strRet.append( XHTMLTag.hidden( prefix + "KEY_noWritable",noWritable ) );
		}

		return strRet.toString();
	}

	/**
	 * 【TAG】処理コマンドを登録します(初期値:大文字の value 属性値[INSERT,COPY,MODIFY,DELETE など])。
	 *
	 * @og.tag
	 * command 属性を指定しない場合は、このvalue 属性値が、コマンドになります。
	 * value 属性に、insert,copy,modify,delete などと指定されていた場合は、
	 * それぞれ、INSERT,COPY,MODIFY,DELETE というコマンドになります。
	 * コマンドは、大文字です。
	 *
	 * @param	cmd コマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
		if( command != null ) { command = command.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】サブミット先の文書を表示させるフレーム、またはウィンドウの名前を指定します。
	 *
	 * @og.tag サブミット先のフレーム名(ターゲット属性)を設定します。
	 *
	 * @og.rev 3.5.5.2 (2004/04/02) 新規追加
	 *
	 * @param	flag サブミット先の文書のフレーム名(ターゲット属性)
	 */
	public void setTarget( final String flag ) {
		target = nval( getRequestParameter( flag ),target );
	}

	/**
	 * 【TAG】ボタン専用のリクエストキーをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * このサブミットボタンが押された場合のみ、転送されるリクエスト情報の
	 * キーを設定できます。カンマ区切りで複数指定できます。
	 * vals 属性には、キーに対応する値を、設定してください。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	key ボタンが押された時に転送するキー
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】ボタン専用のリクエスト値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * キーに対応した値を、カンマ区切り文字で複数指定出来ます。
	 * 指定順序は、キーと同じにしておいて下さい。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規追加
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	val keys属性に対応する値
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【TAG】commonForward の dbkeys にカラム指定を行います。
	 *
	 * @og.tag
	 * カラム指定は、CSV形式(カンマ区切り)で指定してください。
	 * なお、引数は、自動的に受けるのではなく、commonForward タグに、
	 * dbkeys="{&#064;dbkeys}" の記述は必要です。
	 *
	 * @og.rev 4.0.0 (2007/05/23) 新規追加
	 *
	 * @param	keys	dbkeys のにカラム指定
	 */
	public void setDbkeys( final String keys ) {
		dbkeys = nval( getRequestParameter( keys ),dbkeys );
	}

	/**
	 * 【TAG】書き込み可能カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、書き込み不可カラム名の指定(noWritable)と同時にセットする
	 * ことは出来ません。
	 * なお、カラム名の代わりに、"null" を指定すると、なにも指定しないこと
	 * になります。つまり、noWritable にすべてのカラムを指定することと
	 * 同じになります。(デフォルトなので、あまり意味はありません。)
	 * "*" を指定すると、すべてのカラムを(columnWritable)指定したことになります。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @param	columnName	例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * @see		#setNoWritable( String )
	 */
	public void setColumnWritable( final String columnName ) {
		columnWritable = nval( getRequestParameter(columnName),null );
	}

	/**
	 * 【TAG】書き込み不可カラム名を、カンマ区切りで与えます。
	 *
	 * @og.tag
	 * これは、書き込み可能カラム名の指定(columnWritable)と同時にセットする
	 * ことは出来ません。
	 * なお、カラム名の代わりに、"null" を指定すると、なにも指定しないこと
	 * になります。つまり、columnWritable にすべてのカラムを指定することと
	 * 同じになります。
	 * "*" を指定すると、すべてのカラムを(noWritable)指定したことになります。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @param	columnName	例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * @see		#setColumnWritable( String )
	 */
	public void setNoWritable( final String columnName ) {
		noWritable = nval( getRequestParameter(columnName),null );
	}

	/**
	 * 【TAG】画像ボタンを作る場合の、画像ファイルを指定します。
	 *
	 * @og.tag
	 * 画像ボタン作成支援の属性です。
	 * inputタグでtype=imageにした場合、IEではname,valueのセットが
	 * 次の画面に渡されない仕様になっているためエンジンのsubmitでは
	 * 利用できません。(どのボタンが押されたか分からない)
	 * そこで、typeはsubmitのままcssの背景画像としてここで指定した
	 * 画像を配置します。
	 * 内部的にはbackground: url(imgFile) left top no-repeat;
	 * をstyleタグに書く事と同じです。
	 * 高さ、幅は把握できないため、別途style属性でhight,widthを指定して下さい。
	 *
	 * @og.rev 5.5.0.0 (2012/03/01) 新規追加
	 *
	 * @param	image	ボタンの背景画像
	 */
	public void setImg( final String image ) {
		bgImg = nval( getRequestParameter(image),null );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "command"		,command	)
				.println( "value"		,value		)
				.println( "gamenId"		,gamenId	)
				.println( "target"		,target		)
				.println( "keys"		,keys		)
				.println( "vals"		,vals		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
