/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.io.IOException;
import java.io.Reader;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Locale;

import java.text.DateFormat;
import java.text.SimpleDateFormat;

import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.StringUtil;

/**
 * データベース関連の便利なメソッドを集めた簡易ユーティリティークラスです。
 * 全てのメソッドは、static メソッドになっています。
 *
 * @og.rev 2.1.1.1 (2002/11/15) Serializable インターフェースを削除する。
 * @og.rev 4.0.0.0 (2007/10/16) DBアクセス関係のメソッドのみをパッケージ移動(hayabusa/db > fukurou/db)
 * @og.group ＤＢ/Shell制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBUtil {

	/** システム依存の改行記号をセットします。4.0.0.0(2007/10/17) */
	private static final String CR = System.getProperty( "line.separator" );

	/**
	 * インスタンスを作らないので、コンストラクタは、private に設定します。
	 */
	private DBUtil() {}

	/**
	 * 初期データベースに接続して、Queryを実行します(互換性確保のため残しています)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * <del>検索するデータベースは、DEFAULT です。</del>
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args ,final ApplicationInfo appInfo ) {
//		return dbExecute( stmt ,args,appInfo,"DEFAULT" );
//		return dbExecute( stmt, args, appInfo, null );

//		Transaction tran = new TransactionReal( null,appInfo );
		Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//		return dbExecute( stmt, args, tran, null, false );

		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
		try {
			return dbExecute( stmt, args, tran, null, false );
		}
		finally {
			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
			tran.close();
		}
	}

	/**
	 * 初期データベースに接続して、Queryを実行します(Transaction 対応)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * ここでは、Transactionオブジェクトから、Connection を取り出して使用します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args ,final Transaction tran ) {
		return dbExecute( stmt, args, tran, null, false );
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(互換性確保のため残しています)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) 検索のみのクエリーから、何でもありのクエリーに変更
	 * @og.rev 2.3.1.3 (2003/01/28) Open Cursor が、大量に残る件の対応。ResultSet を close()
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo, final String dbid ) {
//		return dbExecute( stmt, args, appInfo, dbid, false );

//		Transaction tran = new TransactionReal( dbid,appInfo );
		Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//		return dbExecute( stmt, args, tran, dbid, false  );

		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
		try {
			return dbExecute( stmt, args, tran, dbid, false  );
		}
		finally {
			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
			tran.close();
		}
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(Transaction 対応)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 * ここでは、Transactionオブジェクトから、Connection を取り出して使用します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final Transaction tran , final String dbid ) {
		return dbExecute( stmt, args, tran, dbid, false );
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(互換性確保のため残しています)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 * @param   useHeader 1行目にヘッダーを含めるか
	 *
	 * @return  検索結果の配列
	 */
	public static String[][] dbExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo, final String dbid, final boolean useHeader ) {
//		Transaction tran = new TransactionReal( dbid,appInfo );
		Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//		return dbExecute( stmt, args, tran, dbid, useHeader );

		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
		try {
			return dbExecute( stmt, args, tran, dbid, useHeader );
		}
		finally {
			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
			tran.close();
		}
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します(Transaction 対応)。
	 *
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 追加：検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 * @param   useHeader 1行目にヘッダーを含めるか
	 *
	 * @return  検索結果の配列
	 */
//	public static String[][] dbExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo, final String dbid, final boolean useHeader ) {
	public static String[][] dbExecute( final String stmt ,final String[] args, final Transaction tran, final String dbid, final boolean useHeader ) {
//		Connection conn = null;				// 5.1.9.0 (2010/08/01) Transaction 対応
		PreparedStatement pstmt = null;
		ResultSet resultSet = null;
		String[][] rtn = null;
//		boolean errFlag = true;
		try {
//			conn = ConnectionFactory.connection( dbid,appInfo );
			Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			pstmt = conn.prepareStatement( stmt );
			if( args != null ) {
				// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
//				boolean useParamMetaData = ApplicationInfo.useParameterMetaData( conn );
				boolean useParamMetaData = ConnectionFactory.useParameterMetaData( dbid );	// 5.3.8.0 (2011/08/01)
				if( useParamMetaData ) {
					ParameterMetaData pMeta = pstmt.getParameterMetaData();
					for( int i=0; i<args.length; i++ ) {
						int type = pMeta.getParameterType( i+1 );
						// 5.3.8.0 (2011/08/01) setNull 対応
//						pstmt.setObject( i+1,args[i],type );
						String val = args[i];
						if( val == null || val.isEmpty() ) {
							pstmt.setNull( i+1, type );
						}
						else {
							pstmt.setObject( i+1, val, type );
						}
					}
				}
				else {
					for( int i=0; i<args.length; i++ ) {
						pstmt.setObject( i+1,args[i] );
					}
				}
			}
			boolean status = pstmt.execute();
			if( status ) {
				resultSet = pstmt.getResultSet();
//				rtn = DBUtil.resultToArray( resultSet,false );
				rtn = DBUtil.resultToArray( resultSet,useHeader ); // 4.3.7.0 (2009/06/01)
			}
			else {
//				conn.commit();
				tran.commit();			// 5.1.9.0 (2010/08/01) Transaction 対応
			}
//			errFlag = false;	// エラーでない
		}
		catch ( SQLException ex ) {
//			Closer.rollback( conn );
			tran.rollback();			// 5.1.9.0 (2010/08/01) Transaction 対応
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ "SQL=[" + stmt + "]" + CR
						+ "ARG=[" + StringUtil.array2csv( args ) + "]" + CR
						+ "DBID=[" + dbid + "]" + CR;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( pstmt );

//			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }
//			else {			ConnectionFactory.close( conn,dbid );  }
			// 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
//			tran.close( errFlag );		// 5.1.9.0 (2010/08/01) Transaction 対応
		}
		return rtn;
	}

	/**
	 * 初期データベースに接続して、CallableStatement(PL/SQL)を実行します(互換性確保のため残しています)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 * <del>検索するデータベースは、DEFAULT です。</del>
	 *
	 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo ) {
//		return dbCallExecute( stmt ,args,appInfo,"DEFAULT" );
//		return dbCallExecute( stmt ,args, appInfo, null );

//		Transaction tran = new TransactionReal( null,appInfo );
		Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//		return dbCallExecute( stmt ,args, tran, null );

		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
		try {
			return dbCallExecute( stmt ,args, tran, null );
		}
		finally {
			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
			tran.close();
		}
	}

	/**
	 * 初期データベースに接続して、CallableStatement(PL/SQL)を実行します(Transaction 対応)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 * <del>検索するデータベースは、DEFAULT です。</del>
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final Transaction tran ) {
		return dbCallExecute( stmt ,args, tran, null );
	}

	/**
	 * 検索するデータベースを指定して、CallableStatement(PL/SQL)を実行します(互換性確保のため残しています)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 * 検索するデータベースは、DEFAULT です。
	 *
	 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final ApplicationInfo appInfo ,final String dbid ) {
//		Transaction tran = new TransactionReal( dbid,appInfo );
		Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//		return dbCallExecute( stmt ,args, tran, dbid );

		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
		try {
			return dbCallExecute( stmt ,args, tran, dbid );
		}
		finally {
			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
			tran.close();
		}
	}

	/**
	 * 検索するデータベースを指定して、CallableStatement(PL/SQL)を実行します(Transaction 対応)。
	 * ステートメントと引数により、CallableStatement クエリーを実行します。
	 * 結果は,ステータスとエラーメッセージを返します。便宜的に、String配列に
	 * 設定して返します。
	 * ステートメント文字列には、 { call PLSQL( ?,?,?･･･ ) } となります。
	 * 第一引数、第二引数は、OUT属性で、結果(STATUS)とエラー時の内容(ERR_CODE)を返します。
	 * 第三引数以降の ? には、オブジェクトの引数配列 が順に割り当てられます。
	 * 検索するデータベースは、DEFAULT です。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  実行結果([0]=ステータス、[1]=エラーメッセージ
	 */
	public static String[] dbCallExecute( final String stmt ,final String[] args, final Transaction tran ,final String dbid ) {
//		Connection conn = null ;				// 5.1.9.0 (2010/08/01) Transaction 対応
		CallableStatement callStmt = null ;

		String[] rtn = new String[2] ;

//		boolean errFlag = true;
		try {
//			conn = ConnectionFactory.connection( dbid,appInfo );
			Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			callStmt = conn.prepareCall( stmt );

			callStmt.registerOutParameter( 1, Types.INTEGER );
			callStmt.registerOutParameter( 2, Types.VARCHAR );
			if( args != null ) {
				for( int i=0; i<args.length; i++ ) {
					callStmt.setObject( i+3,args[i] );
				}
			}
			callStmt.execute();

			rtn[0] = String.valueOf( callStmt.getInt(1) );	// 結果ステータス
			rtn[1] = callStmt.getString(2);					// 内容(エラーメッセージ)

//			conn.commit();
			tran.commit();					// 5.1.9.0 (2010/08/01) Transaction 対応
//			errFlag = false;	// エラーでない
		}
		catch ( SQLException ex ) {
//			Closer.rollback( conn );
			tran.rollback();				// 5.1.9.0 (2010/08/01) Transaction 対応
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ "SQL=[" + stmt + "]" + CR
						+ "ARG=[" + StringUtil.array2csv( args ) + "]" + CR
						+ "DBID=[" + dbid + "]" + CR;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.stmtClose( callStmt );
//			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }
//			else {			ConnectionFactory.close( conn,dbid );  }
			// 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
//			tran.close( errFlag );		// 5.1.9.0 (2010/08/01) Transaction 対応
		}
		return rtn;
	}

	/**
	 * SQL文の実行結果において、データの件数を取得します(互換性確保のため残しています)。
	 * ステートメントと引数により、Prepared クエリーの検索を実行します。
	 * 結果は、件数を数値で返します。
	 * あくまで、存在チェックに必要な処理のみ行っているため、通常の検索より高速です。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) 新規作成
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.1 (2007/12/03) try ～ catch ～ finally をきちんと行う。
	 * @og.rev 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応します。
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果(データの件数)
	 */
	public static int dbExist( final String stmt ,final String[] args, final ApplicationInfo appInfo , final String dbid ) {
//		Transaction tran = new TransactionReal( dbid,appInfo );
		Transaction tran = new TransactionReal( appInfo );			// 5.3.7.0 (2011/07/01) 引数変更
//		return dbExist( stmt ,args, tran , dbid );

		// 5.3.8.0 (2011/08/01) TransactionReal と close() 処理をセットで実行する。
		try {
			return dbExist( stmt ,args, tran , dbid );
		}
		finally {
			// エラー発生時は、tran.rollback() が呼ばれているため、close()時にエラーフラグをセットする必要はない。
			tran.close();
		}
	}

	/**
	 * SQL文の実行結果において、データの件数を取得します(Transaction 対応)。
	 * ステートメントと引数により、Prepared クエリーの検索を実行します。
	 * 結果は、件数を数値で返します。
	 * あくまで、存在チェックに必要な処理のみ行っているため、通常の検索より高速です。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規作成 Transaction 対応
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
	 * @og.rev 5.3.8.0 (2011/08/01) useParamMetaData を ConnectionFactory経由で取得。(PostgreSQL対応)、setNull 対応
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   tran Transactionオブジェクト
	 * @param   dbid 接続先ID
	 *
	 * @return  検索結果(データの件数)
	 */
	public static int dbExist( final String stmt ,final String[] args, final Transaction tran , final String dbid ) {
//		Connection conn = null;				// 5.1.9.0 (2010/08/01) Transaction 対応
		PreparedStatement pstmt = null;
		ResultSet resultSet = null;
		int rtnCnt = -1;

//		boolean errFlag = true;
		try {
//			conn = ConnectionFactory.connection( dbid,appInfo );
			Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			pstmt = conn.prepareStatement( stmt );
			if( args != null ) {
				// 5.1.2.0 (2010/01/01) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
//				boolean useParamMetaData = ApplicationInfo.useParameterMetaData( conn );
				boolean useParamMetaData = ConnectionFactory.useParameterMetaData( dbid );	// 5.3.8.0 (2011/08/01)
				if( useParamMetaData ) {
					ParameterMetaData pMeta = pstmt.getParameterMetaData();
					for( int i=0; i<args.length; i++ ) {
						int type = pMeta.getParameterType( i+1 );
						// 5.3.8.0 (2011/08/01) setNull 対応
//						pstmt.setObject( i+1,args[i],type );
						String val = args[i];
						if( val == null || val.isEmpty() ) {
							pstmt.setNull( i+1, type );
						}
						else {
							pstmt.setObject( i+1, val, type );
						}
					}
				}
				else {
					for( int i=0; i<args.length; i++ ) {
						pstmt.setObject( i+1,args[i] );
					}
				}
			}

			resultSet = pstmt.executeQuery();
			if( resultSet.next() ) {
				rtnCnt = resultSet.getInt(1);
			}
//			errFlag = false;	// エラーでない
		}
		catch ( SQLException ex ) {
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + CR
						+ "SQL=[" + stmt + "]" + CR
						+ "ARG=[" + StringUtil.array2csv( args ) + "]" + CR
						+ "DBID=[" + dbid + "]" + CR;
			throw new RuntimeException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( pstmt );

//			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }
//			else {			ConnectionFactory.close( conn,dbid );  }
			// 5.3.8.0 (2011/08/01) Transaction を引数で受け取った場合は、close() しない。
//			tran.close( errFlag );		// 5.1.9.0 (2010/08/01) Transaction 対応
		}
		return rtnCnt;
	}

	/**
	 * ResultSet より、結果の文字列配列を作成します。
	 *
	 * 結果は,すべて文字列に変換されて格納されます。
	 * 移動したメソッドで使われているのでこれも移動
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) Vector を使用している箇所で、非同期でも構わない箇所を、ArrayList に置換え。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 4.0.0 (2005/01/31) private ⇒ public , ヘッダー情報の取得有無フラグの追加
	 *
	 * @param   resultSet ResultSetオブジェクト
	 * @param   useHeader true:ヘッダーを第一行に含める/false:含めない
	 *
	 * @return  ResultSetの検索結果配列
	 */
	public static String[][] resultToArray( final ResultSet resultSet,final boolean useHeader ) {
		ArrayList<String[]> data = new ArrayList<String[]>();
		try {
			ResultSetMetaData metaData  = resultSet.getMetaData();
			int numberOfColumns =  metaData.getColumnCount();

			if( useHeader ) {
				String[] columnNames = new String[numberOfColumns];
				for( int column = 0; column < numberOfColumns; column++ ) {
					columnNames[column]	 = ( metaData.getColumnLabel(column+1) ).toUpperCase(Locale.JAPAN) ;
				}
				data.add( columnNames );
			}

			while( resultSet.next() ) {
				String[] columnValues = new String[numberOfColumns];
				for( int i = 1; i <= numberOfColumns; i++ ) {
					Object obj = resultSet.getObject(i);
					if( obj == null ) {
						columnValues[i-1] = "";
					}
					else {
						columnValues[i-1] = String.valueOf( obj );
					}
				}
				data.add( columnValues );
			}
		}
		catch ( SQLException ex ) {
			String errMsg = "処理結果を実行できませんでした。"
						+ CR + ex.getMessage() ;
			throw new RuntimeException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		int size = data.size();
		String[][] rtn = new String[size][];
		for( int i=0; i<size; i++ ) {
			rtn[i] = data.get(i);
		}

		return rtn;
	}

	/**
	 * 検索結果オブジェクトから値を取り出します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 集計機能対応によりメソッド化
	 * @og.rev 5.5.5.4 (2012/08/18) if文をcase文に置き換え。
	 * @og.rev 5.5.5.4 (2012/08/18) TIMESTAMP の処理を追加。
	 *
	 * @param res 検索結果オブジェクト
	 * @param col カラム(0から始まる値。このメソッドの内部で、＋１しています)
	 * @param type データタイプ(java.sql.Types.XXXX)
	 *
	 * @return 値
	 * @throws SQLException
	 */
	public static String getValue( final ResultSet res, final int col, final int type ) throws SQLException {
		String val = null;

		Object obj = res.getObject(col+1);
		if( obj == null ) {
			val = "";
		}
		else {
			switch( type ) {
				case Types.CLOB : 		val = getClobData( (Clob)obj ) ;	break;
				case Types.ROWID: 		val = res.getString(col+1);			break;
	//			case Types.TIMESTAMP : 	val = DATE_FMT.format( (java.sql.Timestamp)obj );	break;
				default : 				val = String.valueOf( obj );
			}
		}

		return val;

//		5.5.5.4 (2012/08/18) if文をcase文に置き換え
//		if( type == Types.CLOB ) {
//			Object obj = res.getObject(col+1);
//			val = getClobData( (Clob)obj ) ;
//		}
//		else if( type == Types.ROWID ) {
//			String obj = res.getString(col+1);
//			if( obj == null ) {
//				val = "";
//			}
//			else {
//				val = obj;
//			}
//		}
//		else {
//			Object obj = res.getObject(col+1);
//			if( obj == null ) {
//				val = "";
//			}
//			else {
//				val = String.valueOf( obj );
//			}
//		}
//
//		return val;
	}

	// 5.5.5.4 (2012/08/18) DATE,TIMESTAMP の処理を追加
	//	private static final DateFormat DATE_FMT = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );

	/**
	 * Clob オブジェクトから文字列を取り出します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
	 *
	 * @param	clobData Clobオブジェクト
	 *
	 * @return	Clobオブジェクトから取り出した文字列
	 * @throws	SQLException
	 */
	private static String getClobData( final Clob clobData ) throws SQLException {
		if( clobData == null ) { return ""; }

		Reader reader = null;
		StringBuilder buf = new StringBuilder( 10000 );

		try {
			reader = clobData.getCharacterStream();
			char[] ch = new char[10000];
			int  len ;
			while( (len = reader.read( ch )) >= 0 ) {
				buf.append( ch,0,len );
			}
		}
		catch( IOException ex ) {
			String errMsg = "CLOBデータの読み込みに失敗しました。";
//			throw new HybsSystemException( errMsg,ex );
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
		}
		return buf.toString();
	}
}
