/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * 各フィールド情報から、動的にカラムを作成する動的カラム一覧表示クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLDynamic extends ViewForm_HTMLTable  {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/** カラムの値を返す場合の、カラムキー名称 {@value} */
	public static final String COLUMN_RETURN_KEY = "COLUMN_RETURN";
	private int rtnColumnNo = -1;	// column_return カラムの番号

	// 4.3.4.4 (2009/01/01)
//	/**
//	 * デフォルトコンストラクター
//	 *
//	 */
//	public ViewForm_HTMLDynamic() {
//		super();
//	}

	/**
	 * 初期化します。
	 * ここでは、内部で使用されているキャッシュをクリアし、
	 * 新しいモデル(DBTableModel)と言語(lang) を元に内部データを再構築します。
	 * ただし、設定情報は、以前の状態がそのままキープされています。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 *
	 * @param	table	DBTableModelオブジェクト
	 */
	@Override
	public void init( final DBTableModel table ) {
		super.init( table );
		int clmCnt = getColumnCount();
		for( int i=0; i<clmCnt; i++ ) {
			if( COLUMN_RETURN_KEY.equalsIgnoreCase( getColumnName(i) )) {
				rtnColumnNo = i;
				break;
			}
		}
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) getBgColorCycleClass の返す文字列を変更する。
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 *
	 * @param  startNo    表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		int lastNo = getLastNo( startNo, pageSize );

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		out.append( getCountForm( startNo,pageSize ) );
		out.append( getHeader() );

		int rowIndex = 0;
		out.append("<tbody>").append( HybsSystem.CR );
		out.append("<tr").append( getBgColorCycleClass( rowIndex++ ) ).append(">");
		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for( int row=startNo; row<lastNo; row++ ) {
			for(int column = 0; column < clmCnt; column++) {
				if( isColumnReturn( row,column ) ) {
					out.append("</tr>");
					out.append("<tr").append( getBgColorCycleClass( rowIndex++ ) ).append(">");
				}
				else if( isColumnDisplay( column ) ) {
					out.append("  <td>");
					out.append( getValueLabel(row,column) );
					out.append("</td>").append( HybsSystem.CR );
				}
			}
		}
		out.append("</tr>").append( HybsSystem.CR );
		out.append("</tbody>").append( HybsSystem.CR );
		out.append("</table>").append( HybsSystem.CR );

		out.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)
		return out.toString();
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @return  テーブルのタグ文字列
	 */
	@Override
	protected String getTableHead() {
		// ヘッダーは，不要です。
		return "";
	}

	/**
	 * カラムが表示可能かどうかを返します。
	 * もし，表示不可の場合は,このカラムの全データが，表示対象から外されます。
	 *
	 * @param   row		行番号
	 * @param   column	カラム番号
	 *
	 * @return  表示可能(true)／不可能(false)
	 */
	private boolean isColumnReturn( final int row,final int column ) {
		boolean rtnFlag = false;

		if( rtnColumnNo == column &&
			"1".equals( getValue( row,column ) ) ) {
				rtnFlag = true;
		}

		return rtnFlag;
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか(false:不可能)
	 */
	@Override
	public boolean isEditable() {
		return false;
	}
}
